﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;

using MedRed.Services;
using Shared.Model.Config.MDWS;
using Shared.Model;
using BT.Health.SchedulingWeb.UISession;
using MedRed.Services.Interfaces;

namespace BT.Health.SchedulingWeb.Patient
{
    public partial class PatientAppointmentAdd : System.Web.UI.Page
    {
        //----------------------------------------------------

        #region Page and control events

        //----------------------------------------------------
        // On a new search, an Appointment searcher does not exist
        // We create one only when we first go and get the first resources
        //----------------------------------------------------
        protected void Page_Load(object sender, EventArgs e)
        {
            try
            {
                lblMessage.Text = string.Empty;

                if (!IsPostBack)
                {
                    // must have a patient to proceed
                    UserSession userSession = Helper.GetUserSession(Page);

                    if (userSession.PatientList.GetCurrentPatient() != null)
                    {
                        // check if we are fulfillling an appointment request
                        VerifyAppointmentRequest(userSession.PatientList.GetCurrentPatient());

                        LoadFacilityDropDownListForVistaInstance();
                        LoadAppointmentRelationTypeDDL();

                        // when not set by an appointment request - assume UTC now
                        if (!WucDateTimePicker1.SelectedDateTimeUTC.HasValue)
                            WucDateTimePicker1.SelectedDateTimeUTC = DateTime.UtcNow;

                        // assume no telehealth
                        tbodyTeleHealth.Visible = false;

                        // verify if the appointment is a walk-in type
                        if (Request.QueryString["Walk-in"] != null)
                        {
                            if (Request.QueryString["Walk-in"].ToLower() == "true")
                            {
                                // index 1 displays Walk-in
                                mvAppointmentType.ActiveViewIndex = 1;
                                tbDaysToSearch.Text = "0";
                            }
                        }
                        else
                        {
                            if (Request.QueryString["telehealth"] != null)
                            {
                                if (Request.QueryString["telehealth"].ToLower() == "true")
                                {
                                    // index 1 displays Walk-in
                                    mvAppointmentType.ActiveViewIndex = 2;

                                    tbodyTeleHealth.Visible = true;
                                }
                            }
                        }
                    }
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void Page_PreRender(object sender, EventArgs e)
        {
            try
            {
                // perform actions based on wizard view displayed
                switch (mvWizard.ActiveViewIndex)
                {
                    case 0:
                        break;

                    case 1:
                        FillStep2Table(tblStep2);
                        ShowExistingAppointments();
                        break;

                    case 2:
                        //ddlIsRescheduleHighPriority.SelectedIndex = 0; // assume NO
                        FillStep3Table(tblStep3);
                        break;

                    case 3:
                        break;
                }

                //-----------------------------------------------------
                // assume cannot proceed
                btnGotoStep3.Enabled = false;

                // we can only move to next step when a time is selected
                var selectedSlotTime = this.SelectedSlotTimeLocalTimeZone;

                // assume no over-book message required
                trOverbooking.Visible = false;

                if (selectedSlotTime.HasValue)
                {
                    if (SelectedSlotRemaining <= 0)
                    {
                        trOverbooking.Visible = true;

                        if (Helper.GetUserSession(Page).UserActions.HasAction("action|CanOverbookAppointment"))
                        {
                            btnGotoStep3.Enabled = true;
                        }
                        else
                        {
                            lblMessage.Text = "You are not authorized to overbook this appointment";
                        }
                    }
                    else
                    {
                        btnGotoStep3.Enabled = true;
                    }
                }

                //-----------------------------------------------------
                if (!IsPostBack)
                {
                    // assume no additional add allowed
                    hAddAnotherAppointment.Visible = false;

                    int appointmentRequestId = 0;
                    int.TryParse(hfAppointmentRequestId.Value, out appointmentRequestId);

                    if (appointmentRequestId < 1)
                    {
                        // only makes sense to add another appointment when not a walk in
                        // OK when "Appointment" or "Telehealth"
                        if (mvAppointmentType.ActiveViewIndex == 0 || mvAppointmentType.ActiveViewIndex == 2)
                            hAddAnotherAppointment.Visible = true;
                    }

                    //if (pnlAppointmentRequest.Visible)
                    //{
                    // when the page first loads and it's to make an appointment for an appointment request ...
                    // set drop down choices so they match the request
                    //    ShowInitialAppointmentRequirementsFromRequest();
                    // }
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void ShowInitialAppointmentRequirementsFromRequest()
        {
            int requestId = 0;
            int.TryParse(hfAppointmentRequestId.Value, out requestId);

            if (requestId > 0)
            {
                var patient = Helper.GetUserSession(Page).PatientList.GetCurrentPatient();

                Factory factory = Helper.GetFactory(Page);
                var service = factory.GetAppointmentService();
                var appointmentRequest = (from r in service.GetAppointmentRequestsForPatient(patient)
                                          where r.Id == requestId
                                          select r).FirstOrDefault();

                if (appointmentRequest != null)
                {
                    if (appointmentRequest.RequestedSection != null)
                    {
                        // we have a section - get the facility
                        int facilityId = appointmentRequest.RequestedSection.Facility.Id;

                        ListItem liFacility = ddlFacility.Items.FindByValue(facilityId.ToString());

                        if (liFacility != null)
                        {
                            // select facility if found
                            ddlFacility.SelectedIndex = -1;
                            liFacility.Selected = true;
                            LoadClinicDropDownListForFaclility(facilityId);

                            // select clinic
                            ListItem liClinic = ddlClinic.Items.FindByValue(appointmentRequest.RequestedSection.Id.ToString());

                            if (liClinic != null)
                            {
                                ddlClinic.SelectedIndex = -1;
                                liClinic.Selected = true;

                                // select provider (if there is one)
                                if (appointmentRequest.RequestedResource != null)
                                {
                                    ListItem liProvider = ddlProviders.Items.FindByValue(appointmentRequest.RequestedResource.Id.ToString());

                                    if (liProvider != null)
                                    {

                                        ddlProviders.SelectedIndex = -1;
                                        liProvider.Selected = true;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        protected void ddlFacility_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                LoadClinicDropDownListForFaclility(ddlFacility.SelectedValueAsInt());
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void ddlClinic_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                Factory factory = Helper.GetFactory(Page);
                LoadResourcesDropDownLists(factory, ddlClinic.SelectedValueAsInt());
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        #endregion

        //----------------------------------------------------

        /// <summary>
        /// Check if we are fulfilling an appointment request
        /// If so, display info for it and add appointment to it when we eventually save
        /// </summary>
        void VerifyAppointmentRequest(Shared.Model.Patient patient)
        {
            // check if we are adding an appointment to an existing request
            if (Request.QueryString["AppointmentRequestId"] != null)
            {
                int requestId = 0;

                // querystring has a value so process it
                if (int.TryParse(Request.QueryString["AppointmentRequestId"], out requestId))
                {
                    if (requestId > 0)
                    {
                        Factory factory = Helper.GetFactory(Page);
                        var service = factory.GetAppointmentService();

                        var appointmentRequest = (from r in service.GetAppointmentRequestsForPatient(patient)
                                                  where r.Id == requestId
                                                  select r).FirstOrDefault();

                        if (appointmentRequest != null)
                        {
                            if (appointmentRequest.ResultingAppointment == null)
                            {
                                TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();

                                // save for later
                                hfAppointmentRequestId.Value = requestId.ToString();
                                // show Request panel
                                pnlAppointmentRequest.Visible = true;

                                // fill panel with reuqest data
                                if (appointmentRequest.RequestedSection != null)
                                {
                                    tdArRequestedSection.InnerHtml = appointmentRequest.RequestedSection.Facility.Site.Name + "<br/>" + appointmentRequest.RequestedSection.Facility.Name + "<br />" + appointmentRequest.RequestedSection.Name;

                                    // if the current VistA site is not the one required - change it
                                    var userSession = Helper.GetUserSession(Page);

                                    if (userSession.CurrentSite != null &&
                                        userSession.CurrentSite.Id != appointmentRequest.RequestedSection.Facility.Site.Id)
                                    {
                                        // set as current site
                                        userSession.CurrentSite = appointmentRequest.RequestedSection.Facility.Site;
                                        ExtensionMethods.ShowSiteChangedBanner(this);
                                    }
                                }

                                if (appointmentRequest.DesiredDate.HasValue)
                                {
                                    WucDateTimePicker1.SelectedDateTimeUTC = appointmentRequest.DesiredDate.Value;

                                    if (WucDateTimePicker1.SelectedDateTimeUTC.HasValue)
                                        tdArDesiredDate.InnerText = WucDateTimePicker1.SelectedDateTimeUTC.Value.ToMediumFormatFromUtc(tz);

                                }

                                string strPriority = "Non Urgent";
                                if (appointmentRequest.Priority.ToString() == "1")
                                {
                                    strPriority = "Urgent";
                                }

                                tdArPriority.InnerText = strPriority;
                                tdArStatus.InnerText = appointmentRequest.Status.ToString();
                                tdArReason.InnerText = appointmentRequest.Reason.ToString();
                                tdArNotes.InnerHtml = appointmentRequest.Notes;
                                //tdRequestedResource.InnerText = appointmentRequest.RequestedResource.Name;
                                if (appointmentRequest.RequestedResource != null)
                                    tdArRequestedResource.InnerText = appointmentRequest.RequestedResource.Name;
                            }
                            else
                            {
                                lblBadAppointmentRequestReason.Text = "Appointment Request already has an appointment";
                                mvWizard.ActiveViewIndex = 4;
                            }
                        }
                        else
                        {
                            lblBadAppointmentRequestReason.Text = "Appointment Request not found for this Id";
                            mvWizard.ActiveViewIndex = 4;
                        }
                    }
                    else
                    {
                        lblBadAppointmentRequestReason.Text = "Appointment Request Id must be greater than zero";
                        mvWizard.ActiveViewIndex = 4;
                    }
                }
                else
                {
                    lblBadAppointmentRequestReason.Text = "Appointment Request Id is invalid";
                    mvWizard.ActiveViewIndex = 4;
                }
            }
        }

        void ShowExistingAppointments()
        {
            int patientId = Helper.GetCurrentPatient(Page).Id;

            Factory factory = Helper.GetFactory(Page);
            var service = factory.GetAppointmentService();


            TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();

            // existing appointments that are not closed or checked-in - from date time of site "now"
            var appointmentTimeFromUtcs = (from a in service.GetForPatient(patientId, null, null)
                                           where a.Time > DateTime.UtcNow.Date &&
                                           a.Status == AppointmentStatus.Scheduled || a.Status == AppointmentStatus.Unknown
                                           orderby a.Time
                                           select new
                                           {
                                               Id = a.Id,
                                               Time = a.Time.ToMediumFormatFromUtc(tz)
                                           }).ToList();

            if (appointmentTimeFromUtcs.Count > 0)
            {
                trExistingAppointments.Visible = true;

                // show times of appointment with appointment ID as the key
                blExistingAppointments.DataValueField = "Id";
                blExistingAppointments.DataTextField = "Time";
                blExistingAppointments.DataSource = appointmentTimeFromUtcs;
                blExistingAppointments.DataBind();

                //----------------------------------------------
                // Linked appointments
                // show times of appointment
                ddlLinkedAppointment.DataTextField = "Time";
                ddlLinkedAppointment.DataValueField = "Id";
                ddlLinkedAppointment.DataSource = appointmentTimeFromUtcs;
                ddlLinkedAppointment.DataBind();
            }
            else
            {
                blExistingAppointments.Items.Clear();
                trExistingAppointments.Visible = false;

                ddlLinkedAppointment.Items.Clear();
            }

            // add an entry in the linked appointment drop down for NO appointment to link to
            ddlLinkedAppointment.Items.Insert(0, new ListItem("Not linked", "0"));
        }

        /// <summary>
        /// Gets the appointment times displayed in the existing appointments list
        /// </summary>
        /// <returns></returns>
        List<DateTime> GetExistingAppointmentTimes()
        {
            List<DateTime> list = new List<DateTime>();

            foreach (ListItem li in blExistingAppointments.Items)
            {
                list.Add(Convert.ToDateTime(li.Text));
            }

            return list;
        }

        void LoadAppointmentRelationTypeDDL()
        {
            //            ExtensionMethods.FillDropDownListWithEnumValues(typeof(AppointmentRelationType), ddlAppointmentRelationship);
            ExtensionMethods.FillWithRecheduleOptions(ddlAppointmentRelationship);
        }


        /// <summary>
        /// Show all facilities for the current VistA site
        /// </summary>
        void LoadFacilityDropDownListForVistaInstance()
        {
            // Data Hierarchy :- VistA Instance -> Facility -> Clinic
            // we have the Vista Site Id, now show all facilities within it
            // We'll sort out which clinic to show later in the page process

            int siteId = UISession.Helper.GetUserSession(Page).CurrentSite.Id;
            Factory factory = Helper.GetFactory(Page);

            var facilityService = factory.GetFacilityService();


            ddlFacility.DataValueField = "Id";
            ddlFacility.DataTextField = "Name";
            ddlFacility.DataSource = (from f in facilityService.GetAll(siteId)
                                      orderby f.Name
                                      select f).ToList();

            ddlFacility.SelectedIndex = -1;
            ddlFacility.DataBind();

            if (ddlFacility.Items.Count > 0)
            {
                ddlFacility.SelectedIndex = 0;
                LoadClinicDropDownListForFaclility(ddlFacility.SelectedValueAsInt());
            }
            else
            {
                LoadClinicDropDownListForFaclility(0);
            }
        }

        /// <summary>
        /// Load all of the clinics for a facility
        /// </summary>
        /// <param name="facilityId"></param>
        void LoadClinicDropDownListForFaclility(int facilityId)
        {
            Factory factory = Helper.GetFactory(Page);
            var service = factory.GetSectionService();

            // Show ONLY active clicnics
            var clinics = (from f in service.GetAll(facilityId)
                           where f.Active==true
                           orderby f.Name
                           select f).ToList();

            if (clinics.Count > 0)
            {
                // One or more clinics to show
                ddlClinic.DataTextField = "Name";
                ddlClinic.DataValueField = "Id";
                ddlClinic.DataSource = clinics;
                ddlClinic.SelectedIndex = -1;
                ddlClinic.DataBind();

                if (ddlClinic.Items.Count > 0)
                {
                    ddlClinic.SelectedIndex = 0;
                    LoadResourcesDropDownLists(factory, ddlClinic.SelectedValueAsInt());
                    return;
                }
            }

            // display "No clinics" to user
            ddlClinic.Items.Clear();
            // ID of zero implies no clinic
            ddlClinic.Items.Add(new ListItem("No Clinics", "0"));

            LoadResourcesDropDownLists(factory, 0);
        }

        void LoadResourcesDropDownLists(Factory factory, int clinicId)
        {
            ddlProviders.Items.Clear();
            ddlRooms.Items.Clear();
            ddlEquipment.Items.Clear();
            ddlAppointmentType.Items.Clear();

            if (clinicId > 0)
            {
                UserSession userSession = Helper.GetUserSession(Page);
                int siteId = userSession.CurrentSite.Id;

                var sectionService = factory.GetSectionService();
                var clinic = sectionService.Get(clinicId);

                // get available resources for the clinic
                var resourceService = factory.GetResourceService();
                var resources = resourceService.GetAll(clinicId);

                //-----------------------------------------------
                var providerService = factory.GetProviderService();
                var providers = providerService.GetAllForSite(siteId).ToList();

                // get providers who are resources at the clinic
                var resourceProviders = (from r in resources
                                         join p in providers
                                         on r.FullfillingResourceId equals p.Id
                                         where r.Type == ResourceType.Provider
                                         orderby p.Person.LastName
                                         select new BasicResourceData
                                         {
                                             ResourceId = r.Id,
                                             ResourceType = Shared.Model.ResourceType.Provider,
                                             ResourceName = r.Name + " (" + p.Person.GetFullName() + ")"
                                         }).ToList();

                ddlProviders.DataValueField = "ResourceId";
                ddlProviders.DataTextField = "ResourceName";
                ddlProviders.DataSource = resourceProviders;
                ddlProviders.DataBind();

                //-----------------------------------------------
                var roomService = factory.GetRoomService();
                var allRooms = roomService.GetAllForSite(siteId).ToList();
                var allRoomResources = resourceService.GetAllOfType(clinicId, ResourceType.Room);

                var rooms = (from r in allRoomResources
                             join a in allRooms
                             on r.FullfillingResourceId equals a.Id
                             orderby r.Name
                             select new BasicResourceData
                             {
                                 ResourceId = r.Id,
                                 ResourceName = r.Name + " (" + a.Name + " - floor + " + a.Floor + ")"
                             }).ToList();

                ddlRooms.DataValueField = "ResourceId";
                ddlRooms.DataTextField = "ResourceName";
                ddlRooms.DataSource = rooms;
                ddlRooms.DataBind();

                // rooms are optional - prepend an empty selection
                ddlRooms.Items.Insert(0, new ListItem("No room required", "0"));
                ddlRooms.SelectedIndex = 0;

                //-----------------------------------------------
                var equipmentService = factory.GetEquipmentService();
                var allEquipment = equipmentService.GetAllForSite(siteId);
                var allEquipmentResources = resourceService.GetAllOfType(clinicId, ResourceType.Equipment);

                var equipment = (from r in allEquipmentResources
                                 join e in allEquipment
                                 on r.FullfillingResourceId equals e.Id
                                 orderby r.Name
                                 select new BasicResourceData
                                 {
                                     ResourceId = r.Id,
                                     ResourceName = r.Name + " (" + e.Name + " - type " + e.Type + ")"
                                 }).ToList();

                ddlEquipment.DataValueField = "ResourceId";
                ddlEquipment.DataTextField = "ResourceName";
                ddlEquipment.DataSource = equipment;
                ddlEquipment.DataBind();

                // equipment optional
                ddlEquipment.Items.Insert(0, new ListItem("No equipment required", "0"));
                ddlEquipment.SelectedIndex = 0;

                //-----------------------------------------------
                var appointmentTypes = (from t in clinic.AppointmentTypes
                                        orderby t.Name
                                        select new BasicResourceData
                                        {
                                            ResourceId = t.Id,
                                            ResourceName = t.Name
                                        }).ToList();

                ddlAppointmentType.DataValueField = "ResourceId";
                ddlAppointmentType.DataTextField = "ResourceName";
                ddlAppointmentType.DataSource = appointmentTypes;
                ddlAppointmentType.DataBind();
            }

            if (ddlProviders.Items.Count == 0)
                ddlProviders.Items.Add(new ListItem("No Providers", "0"));

            if (ddlRooms.Items.Count == 0)
                ddlRooms.Items.Add(new ListItem("No Rooms", "0"));

            if (ddlEquipment.Items.Count == 0)
                ddlEquipment.Items.Add(new ListItem("No Equipment", "0"));

            if (ddlAppointmentType.Items.Count == 0)
                ddlAppointmentType.Items.Add(new ListItem("No Appointment Types", "0"));
        }

        protected void ddlSlotDays_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                SelectSlotsForDay(Convert.ToDateTime(ddlSlotDays.SelectedValue));
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void SelectSlotsForDay(DateTime dt)
        {
            lbSlots.Items.Clear();

            if (ddlSlotDays.SelectedIndex >= 0)
            {
                if (Session["BT.Health.SchedulingWeb.Patient.PatientAppointmentSearch"] != null &&
                    Session["BT.Health.SchedulingWeb.Patient.PatientAppointmentSearch"] is Dictionary<DateTime, List<SlotInstance>>)
                {
                    var slots = Session["BT.Health.SchedulingWeb.Patient.PatientAppointmentSearch"] as Dictionary<DateTime, List<SlotInstance>>;

                    List<string> slotTimesFromUtc = new List<string>();

                    TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();

                    foreach (var slotTime in slots[dt])
                    {
                        DateTime dtTimeZoneTime = TimeZoneInfo.ConvertTimeFromUtc(slotTime.Time, tz);

                        ListItem li = new ListItem();

                        // set value to local timezone date/time, a | separator and the remaining slots
                        li.Value = dtTimeZoneTime.ToMediumFormat() + "|" + slotTime.Remaining.ToString();

                        string slotText = dtTimeZoneTime.ToString("HH:mm");

                        if (slotTime.Remaining <= 0)
                        {
                            // when we get a slot but the capcity is less than one, it's overbookable
                            if (slotTime.Remaining == 0)
                                li.Text = slotText + " [Booked to capacity]";
                            else
                                li.Text = slotText + " [Over-booked by " + (Math.Abs(slotTime.Remaining)).ToString() + "]";

                            // simple class to set red text
                            li.Attributes.Add("class", "slotOverBooked");
                        }
                        else
                        {
                            li.Text = slotText;
                        }

                        lbSlots.Items.Add(li);
                    }
                }
            }

            if (lbSlots.Items.Count == 0)
            {
                lbSlots.Items.Add(new ListItem("No slots found", string.Empty));
            }
        }

        /// <summary>
        /// gets the selected slot time in the local timezone (as displayed) (or null if no slot selected)
        /// </summary>
        DateTime? SelectedSlotTimeLocalTimeZone
        {
            get
            {
                // need selected item
                if (lbSlots.SelectedIndex >= 0)
                {
                    // expecting a date, separator and remaining slots
                    string[] parts = lbSlots.SelectedItem.Value.Split(new char[] { '|' });

                    // must have two parts
                    if (parts.GetLength(0) == 2)
                    {
                        // see if valid date
                        DateTime dt = DateTime.Now;
                        if (DateTime.TryParse(parts[0], out dt))
                            return dt;
                    }
                }

                return null;
            }
        }

        /// <summary>
        /// gets the selected slot time remaining capacity, or null if not found or selected
        /// </summary>
        int? SelectedSlotRemaining
        {
            get
            {
                // need selected item
                if (lbSlots.SelectedIndex >= 0)
                {
                    // expecting a date, separator and remaining slots
                    string[] parts = lbSlots.SelectedItem.Value.Split(new char[] { '|' });

                    // must have two parts
                    if (parts.GetLength(0) == 2)
                    {
                        // see if valid date
                        int r = 0;
                        if (int.TryParse(parts[1], out r))
                            return r;
                    }
                }

                return null;
            }
        }

        void ShowSlots(Dictionary<DateTime, List<SlotInstance>> slots)
        {
            if (slots.Keys.Count > 0)
            {
                ddlSlotDays.DataTextFormatString = "{0:dd-MMM-yyyy}";
                ddlSlotDays.DataSource = slots.Keys;
                ddlSlotDays.DataBind();
                ddlSlotDays.SelectedIndex = 0;
                SelectSlotsForDay(Convert.ToDateTime(ddlSlotDays.SelectedItem.Text));
            }
            else
            {
                ddlSlotDays.Items.Clear();
                ddlSlotDays.Items.Add(new ListItem("No slots found", "-1"));

                lbSlots.Items.Clear();
            }
        }

        /// <summary>
        /// Search for free slots
        /// Get the slots for all selected resources then check each has the same slot time free
        /// </summary>
        /// <param name="dtSearchStartDate"></param>
        /// <param name="dtSearchEndDate"></param>
        Dictionary<DateTime, List<SlotInstance>> SearchForSlots(DateTime dtSearchStartDate, DateTime dtSearchEndDate)
        {
            // to search for free slots we must get the slots for all selected resources then check each has the same slots free
            // get the resource Id's selected - we must have at leat a provider
            int providerResourceId = ddlProviders.SelectedValueAsInt();
            int roomResourceId = ddlRooms.SelectedValueAsInt();
            int equipmentResourceId = ddlEquipment.SelectedValueAsInt();

            int siteId = UISession.Helper.GetUserSession(Page).CurrentSite.Id;
            Factory factory = Helper.GetFactory(Page);
            var resourceService = factory.GetResourceService();

            System.Collections.ObjectModel.Collection<int> resourceIds = new System.Collections.ObjectModel.Collection<int>();

            // will have a provider
            resourceIds.Add(providerResourceId);

            // may have a room
            if (roomResourceId > 0)
                resourceIds.Add(roomResourceId);

            // may have equipment
            if (equipmentResourceId > 0)
                resourceIds.Add(equipmentResourceId);

            var clinic = factory.GetSectionService().Get(int.Parse(ddlClinic.SelectedValue));
            var apptType = (from t in clinic.AppointmentTypes
                            where t.Id == ddlAppointmentType.SelectedValueAsInt()
                            select t).FirstOrDefault();

            var slotDays = resourceService.GetAvailability(resourceIds, dtSearchStartDate, dtSearchEndDate, apptType);

            var slotDaysWithFreeSlots = new Dictionary<DateTime, List<SlotInstance>>();

            foreach (var slotDay in slotDays)
            {
                if (slotDay.Value.Count > 0)
                    slotDaysWithFreeSlots.Add(slotDay.Key, slotDay.Value);
            }

            return slotDaysWithFreeSlots;
        }

        void Stage1_SearchForSlots()
        {
            if (WucDateTimePicker1.SelectedDateTimeUTC.HasValue)
            {
                int daysToSearch = 0;

                if (int.TryParse(tbDaysToSearch.Text, out daysToSearch))
                {
                    // check dates make sense - we work in UTC
                    DateTime dtSearchStartDate = WucDateTimePicker1.SelectedDateTimeUTC.Value.Date;
                    DateTime utcNow = DateTime.UtcNow.Date;

                    if (dtSearchStartDate >= utcNow)
                    {
                        if (daysToSearch >= 0)
                        {
                            DateTime dtSearchEndDate = dtSearchStartDate.Date.AddDays(daysToSearch).Date;

                            // get the free slots (UTC dates)
                            var slots = SearchForSlots(dtSearchStartDate, dtSearchEndDate);

                            if (slots != null)
                            {
                                // show slots
                                mvWizard.ActiveViewIndex = 1;
                                Session.Add("BT.Health.SchedulingWeb.Patient.PatientAppointmentSearch", slots);
                                ShowSlots(slots);
                            }
                            else
                            {
                                lblMessage.Text = "No slots found for the requested period and resources";
                            }
                        }
                        else
                        {
                            lblMessage.Text = "Days to search must be zero or greater";
                        }
                    }
                    else
                    {
                        lblMessage.Text = "Your search cannot start prior to today";
                    }
                }
                else
                {
                    lblMessage.Text = "Days to search is invalid";
                }
            }
            else
            {
                lblMessage.Text = "Preferred date is invalid";
            }
        }

        protected void btnSearchForSlots_Click(object sender, EventArgs e)
        {
            if (ddlFacility.SelectedValueAsInt() > 0)
            {
                if (ddlClinic.SelectedValueAsInt() > 0)
                {
                    // local provider or telehealth to go on
                    if (ddlProviders.SelectedValueAsInt() > 0)
                    {
                        if (ddlAppointmentType.SelectedValueAsInt() > 0)
                        {
                            if (mvAppointmentType.ActiveViewIndex == 2)
                            {
                                if (tbTelehealthPatientLocation.Text.Trim().Length == 0)
                                {
                                    lblMessage.Text = "For Tele-Health, you must select a Patient Location";
                                    return;
                                }
                            }

                            Stage1_SearchForSlots();
                        }
                        else
                        {
                            lblMessage.Text = "Please select an appointment type";
                        }
                    }
                    else
                    {
                        lblMessage.Text = "Please select a provider";
                    }
                }
                else
                {
                    lblMessage.Text = "Please select a clinic";
                }
            }
            else
            {
                lblMessage.Text = "Please select a facility";
            }
        }

        protected void btnGotoStep1_Click(object sender, EventArgs e)
        {
            mvWizard.ActiveViewIndex = 0;
        }

        protected void btnGotoStep3_Click(object sender, EventArgs e)
        {
            try
            {
                if (SelectedSlotTimeLocalTimeZone.HasValue)
                {
                    // check patient doesn't already have an appointment at the time of the slot selected
                    // we are comparing date/times already on the client (timezone) and not converting to UTC to do this
                    List<DateTime> existingAppointments = GetExistingAppointmentTimes();

                    if (existingAppointments.Contains(SelectedSlotTimeLocalTimeZone.Value))
                    {
                        // check for telehealth
                        if (mvAppointmentType.ActiveViewIndex == 2)
                        {
                            // get appointment at the same time - if it's in a different Vista Site, that's OK
                            string dateTextToCheck = SelectedSlotTimeLocalTimeZone.Value.ToMediumFormat();

                            List<int> appointmentIdList = new List<int>();
                            foreach (ListItem li in blExistingAppointments.Items)
                            {
                                if (li.Text == dateTextToCheck)
                                {
                                    appointmentIdList.Add(int.Parse(li.Value));
                                }
                            }

                            bool appointmentThisSite = false;

                            if (appointmentIdList.Count > 0)
                            {
                                foreach (int apptId in appointmentIdList)
                                {
                                    if (IsAppointmentAtCurrentVistaSite(apptId))
                                    {
                                        appointmentThisSite = true;
                                        break;
                                    }
                                }
                            }

                            if (!appointmentThisSite)
                            {
                                // We can proceed - trying to book same time appointment as a telehealth
                                // with appointment for same time in another Vista Site
                                mvWizard.ActiveViewIndex = 2;
                                return;
                            }
                        }

                        // not telehealth, so can't have two appointments at the same time
                        lblMessage.Text = "The patient already has an appointment at this date and time";
                    }
                    else
                    {
                        mvWizard.ActiveViewIndex = 2;
                    }
                }
                else
                {
                    lblMessage.Text = "Select a slot for the appointment";
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        bool IsAppointmentAtCurrentVistaSite(int appointmentId)
        {
            // assumed true - we have to prove it is not at current site
            bool result = true;

            try
            {
                Factory factory = Helper.GetFactory(Page);
                var service = factory.GetAppointmentService();
                var appointment = service.Get(appointmentId);

                if (appointment != null)
                {
                    if (appointment.Section != null && appointment.Section.Facility != null && appointment.Section.Facility.Site != null)
                    {
                        if (appointment.Section.Facility.Site.Id != Helper.GetUserSession(Page).CurrentSite.Id)
                            result = false;
                    }
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }

            return result;
        }

        protected void btnGotoStep2_Click(object sender, EventArgs e)
        {
            mvWizard.ActiveViewIndex = 1;
        }

        protected void btnGotoStep4_Click(object sender, EventArgs e)
        {
            try
            {
                var a = CreateAppointment();

                if (a != null)
                {
                    mvWizard.ActiveViewIndex = 3;
                    TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();
                    hProviderRoster.NavigateUrl = "~/Patient/ProviderRoster.aspx?appointmentId=" + a.Id.ToString();

                    // show appointment data
                    DateTime? desirerdDateTimeUtc = WucDateTimePicker1.SelectedDateTimeUTC;

                    // there should be a desired data displayed, so show the displayed date (as it's always in the local time zone)
                    if (WucDateTimePicker1.DisplayedDateTime.HasValue)
                        AddTableRow(tblStep4, "Requested Time", WucDateTimePicker1.DisplayedDateTime.Value.ToMediumFormat());

                    AddTableRow(tblStep4, "Appointment Time", a.Time.ToMediumFormatFromUtc(tz));
                    AddTableRow(tblStep4, "Length", a.Length.ToString() + " minutes");

                    if (a.AppointmentType != null)
                        AddTableRow(tblStep4, "Appointment Type", a.AppointmentType.Name);

                    AddTableRow(tblStep4, "Status", a.Status.ToString());

                    foreach (var r in a.Resources)
                        AddTableRow(tblStep4, r.Type.ToString(), r.Name);

                    //--------------------------------------------------
                    // check if its a child appointment and show parent
                    int patientId = Helper.GetCurrentPatient(Page).Id;

                    Factory factory = Helper.GetFactory(Page);
                    var service = factory.GetAppointmentService();

                    // possible parent appointments...
                    var parentAppointments = (from p in service.GetForPatient(patientId, null, null)
                                              orderby p.Time
                                              select p).ToList();

                    foreach (var parentAppointment in parentAppointments)
                    {
                        var childAppointmentIds = (from c in parentAppointment.ChildAppointments
                                                   where c.ChildAppointment.Id == a.Id
                                                   select c).ToList();

                        if (childAppointmentIds.Count > 0)
                        {
                            // our new on is a child
                            AddTableRow(tblStep4, "Linked to appointment on", parentAppointment.Time.ToMediumFormat());
                        }
                    }
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void FillStep2Table(Table tbl)
        {
            tbl.Rows.Clear();
            AddTableRow(tbl, "Facility", ddlFacility.SelectedItem.Text);
            AddTableRow(tbl, "Clinic", ddlClinic.SelectedItem.Text);
            AddTableRow(tbl, "Provider", ddlProviders.SelectedItem.Text);

            if (ddlRooms.SelectedValueAsInt() > 0)
                AddTableRow(tbl, "Rooms", ddlRooms.SelectedItem.Text);

            if (ddlEquipment.SelectedValueAsInt() > 0)
                AddTableRow(tbl, "Equipment", ddlEquipment.SelectedItem.Text);
        }

        void FillStep3Table(Table tbl)
        {
            // show all data in step 2
            FillStep2Table(tbl);

            AddTableRow(tbl, "Slot", SelectedSlotTimeLocalTimeZone.Value.ToMediumFormat());
        }

        void AddTableRow(Table tbl, string itemName, string itemValue)
        {
            TableRow tr = new TableRow();
            tbl.Rows.Add(tr);

            TableCell tdName = new TableCell();
            tr.Cells.Add(tdName);
            tdName.Text = itemName;

            TableCell tdValue = new TableCell();
            tr.Cells.Add(tdValue);
            tdValue.Text = itemValue;
        }

        Shared.Model.Appointment CreateAppointment()
        {
            // are we fulfilling an appointment request?
            int appointmentRequestId = 0;
            int.TryParse(hfAppointmentRequestId.Value, out appointmentRequestId);

            // save
            string vistaSiteId = UISession.Helper.GetUserSession(Page).CurrentSite.VistaSiteId;

            Shared.Model.User user = UISession.Helper.GetCurrentUserAccount(this);
            var patient = UISession.Helper.GetCurrentPatient(this);

            //-----------------------------------------
            Factory factory = Helper.GetFactory(Page);
            //-----------------------------------------
            var siteService = factory.GetSectionService();
            var clinic = siteService.Get(int.Parse(ddlClinic.SelectedValue));

            //-----------------------------------------
            var resourceService = factory.GetResourceService();
            var providerResource = resourceService.Get(ddlProviders.SelectedValueAsInt());

            //-----------------------------------------
            var service = factory.GetAppointmentService();

            var nationalSystemService = factory.GetNationalSystemService();
            //-----------------------------------------

            Shared.Model.AppointmentRequest r = null;
            Shared.Model.Appointment a = null;

            DateTime? desirerdDateTimeUtc = WucDateTimePicker1.SelectedDateTimeUTC;

            if (desirerdDateTimeUtc == null)
                throw new ApplicationException("Desired date is not valid");

            TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();

            //UTC time
            DateTime appointmentDateTimeUtc = TimeZoneInfo.ConvertTimeToUtc(SelectedSlotTimeLocalTimeZone.Value, tz);

            //Get Appointment Type so we can get length in minutes
            int appointmentLength = GetAppointmentLengthMinutes(appointmentDateTimeUtc, SelectedSlotTimeLocalTimeZone.Value);

            if (appointmentRequestId > 0)
            {
                var request = (from ar in service.GetAppointmentRequestsForPatient(patient)
                               where ar.Id == appointmentRequestId
                               select ar).FirstOrDefault();

                if (request != null)
                {
                    r = request;
                }
                else
                {
                    lblMessage.Text = "Unable to find appointment request to add this appointment";
                    return null;
                }
            }

            if (r == null)
            {
                // create new request to pair with appointment
                r = new Shared.Model.AppointmentRequest();
                r.CreatingUser = user;
                r.Patient = patient;
                r.RequestMethod = Shared.Model.RequestMethod.Other;
                r.RequestedSection = clinic;
                r.DesiredDate = desirerdDateTimeUtc;
                r.Time = DateTime.UtcNow;
                r.RequestedResource = providerResource;
                r.RequestedSection = clinic;

                service.AddAppointmentRequest(r);
            }

            //--------------------------------------ExtensionMethods.AddACommunicationTemplate(
            //AddCommunicationTemplate(r, clinic);
            //--------------------------------------

            a = new Shared.Model.Appointment();
            a.Patient = patient;
            a.VistaId = vistaSiteId;
            a.Section = clinic;

            r.Status = Shared.Model.RequestStatus.Complete;
            a.Status = Shared.Model.AppointmentStatus.Scheduled;
            r.Time = appointmentDateTimeUtc;

            a.Time = appointmentDateTimeUtc;
            a.Length = appointmentLength;

            // get priority from calculated values
            r.Priority = patient.Priority();

            string trimmedNotes = tbNotes.Text.Trim();

            //----------------------------------------
            // unless specifically NOT selected - assume high priority
            //if (ddlIsRescheduleHighPriority.SelectedIndex == 0)
            //    a.HighPriorityReschedule = false;
            //else
            //    a.HighPriorityReschedule = true;
            //----------------------------------------

            var apptType = (from t in clinic.AppointmentTypes
                            where t.Id == ddlAppointmentType.SelectedValueAsInt()
                            select t).FirstOrDefault();

            a.AppointmentType = apptType;

            //AppointmentTypeCategory etc
            if (trimmedNotes != string.Empty)
            {
                PatientInstructions i = new PatientInstructions();
                i.Instructions = trimmedNotes;
                i.Type = InstructionType.PreAppointment;
                apptType.AddPatientInstructions(i);
            }

            //----------------------------------------
            // resources
            a.Resources.Add(providerResource);

            // add Room resource (if there is one)
            if (ddlRooms.SelectedValueAsInt() > 0)
                a.Resources.Add(resourceService.Get(ddlRooms.SelectedValueAsInt()));

            // add Equipment resource (if there is one)
            if (ddlEquipment.SelectedValueAsInt() > 0)
                a.Resources.Add(resourceService.Get(ddlEquipment.SelectedValueAsInt()));

            //----------------------------------------

            // Telehealth
            if (mvAppointmentType.ActiveViewIndex == 2)
            {
                string telehealthPatientLocation = tbTelehealthPatientLocation.Text.Trim();
                if (telehealthPatientLocation.Length > 0)
                {
                    a.TelehealthLocation = telehealthPatientLocation;
                }
                else
                {
                    // this will ahve been validated previously - belt and braces
                    throw new ApplicationException("You must set a patient location for a Tele-Health appointment");
                }
            }

            if (mvAppointmentType.ActiveViewIndex == 0 || mvAppointmentType.ActiveViewIndex == 2)
            {
                if (!chkRecuring.Checked)
                {
                    // standard appointment
                    service.CreateAppointment(a);
                }
                else
                {
                    var recur = new Recur()
                    {
                        RecurringFrequency = ddlFrequency.SelectedValue,
                        RecurringInterval = txtInterval.Text,
                        RecurringStart = a.Time,
                        RecurringEnd = WucRecEnd.SelectedDateTimeUTC.Value
                    };
                    service.CreateRecurringAppointments(a, recur);
                }
            }
            else
            {
                // walk-in appointment
                service.CreateWalkInAppointment(a);
            }

            ExtensionMethods.AddACommunicationTemplate(a, CommunicationType.PreAppointment, this.Page);

            // add
            r.ResultingAppointment = a;

            // update the request
            service.UpdateAppointmentRequest(r);

            //----------------------------------------
            // Linked appointments
            if (ddlLinkedAppointment.SelectedIndex > 0)
            {
                // 1..n implies a real appointment selected to link to
                var parentAppointment = service.Get(int.Parse(ddlLinkedAppointment.SelectedValue));

                if (parentAppointment != null)
                {
                    AppointmentRelationType art = (AppointmentRelationType)ddlAppointmentRelationship.SelectedValueAsInt();
                    parentAppointment.AddChildAppointment(a, art);
                    service.Update(parentAppointment);
                }
                else
                {
                    throw new ApplicationException("Unable to find selected appointment linked to");
                }
            }

            return a;
        }

        //private void AddCommunicationTemplate(Appointment r, Section clinic)
        //{

        //    //UserSession userSession = Helper.GetUserSession(Page);
        //    Factory factory = Helper.GetFactory(Page);
        //    MedRed.Services.Interfaces.IFacilityService facilityService = factory.GetFacilityService();
        //    MedRed.Services.Interfaces.IAppointmentService appointmentService = factory.GetAppointmentService();

        //    MedRed.Services.Interfaces.ISectionService sectionService = factory.GetSectionService();
        //    IList<CommunicationTemplate> templateList = null;
        //    Section sec = sectionService.Get(clinic.Id);
        //    templateList = sec.CommunicationTemplates;

        //    CommunicationTemplate t = null;
        //    foreach (CommunicationTemplate template in templateList)
        //    {
        //        if (template.Type == CommunicationType.PreAppointment)
        //        {
        //            t = template;
        //        }
        //    }

        //    if (t != null)
        //    {    //A matching communication template exists so create a queue item
        //        CommunicationQueueItem item = new CommunicationQueueItem();
        //        appointmentService.AddToCommunicationQueue(t, r);
        //        lblQueueMessage.Text = "Pre Appointment Queue item added";
        //    }

        //}

        //private void AddCommunicationTemplate(AppointmentRequest r, Section clinic)
        //{

        //    //UserSession userSession = Helper.GetUserSession(Page);
        //    Factory factory = Helper.GetFactory(Page);
        //    MedRed.Services.Interfaces.IFacilityService facilityService = factory.GetFacilityService();
        //    MedRed.Services.Interfaces.IAppointmentService appointmentService = factory.GetAppointmentService();

        //    MedRed.Services.Interfaces.ISectionService sectionService = factory.GetSectionService();
        //    IList<CommunicationTemplate> templateList = null;
        //    Section sec = sectionService.Get(clinic.Id);
        //    templateList = sec.CommunicationTemplates;

        //    CommunicationTemplate t = null;
        //    foreach (CommunicationTemplate template in templateList)
        //    {
        //        if (template.Type == CommunicationType.PreAppointment)
        //        {
        //            t = template;
        //        }
        //    }

        //    if (t != null)
        //    {    //A matching communication template exists so create a queue item
        //        CommunicationQueueItem item = new CommunicationQueueItem();
        //        appointmentService.AddToCommunicationQueue(t, r);
        //        lblQueueMessage.Text = "Pre Appointment Queue item added";
        //    }

        //}

        protected void ddlLinkedAppointment_SelectedIndexChanged(object sender, EventArgs e)
        {
            // show appointment relationship when required
            int parentApointmentId = ddlLinkedAppointment.SelectedValueAsInt();

            trAppointmentRelationship.Visible = (parentApointmentId > 0);
        }

        protected void lbSlots_SelectedIndexChanged(object sender, EventArgs e)
        {
            // we use the info in pre-render - just make sure we post back
        }

        protected int GetAppointmentLengthMinutes(DateTime slotTimeUtc, DateTime localTime)
        {
            var slots = Session["BT.Health.SchedulingWeb.Patient.PatientAppointmentSearch"] as Dictionary<DateTime, List<SlotInstance>>;

            DateTime slotDay = localTime.Date;

            var slot = (from s in slots[slotDay]
                        where s.Time == slotTimeUtc
                        select s).FirstOrDefault();

            if (slot != null)
                return slot.Length;
            else
                throw new ApplicationException("Unable to find slot length");
        }

        protected void chkRecuring_CheckedChanged(object sender, EventArgs e)
        {
            if (chkRecuring.Checked)
            {
                tblRecur.Visible = true;
                RequiredFieldValidator2.Enabled = true;
                RegularExpressionValidator1.Enabled = true;
            }
            else
            {
                tblRecur.Visible = false;
                RequiredFieldValidator2.Enabled = false;
                RegularExpressionValidator1.Enabled = false;
            }
        }

    }
}
