﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.HtmlControls;

using MedRed.Services;
using Shared.Model.Config.MDWS;
using Shared.Model;
using BT.Health.SchedulingWeb.UISession;
using MedRed.Services.Interfaces;


namespace BT.Health.SchedulingWeb.Patient
{
    public partial class GroupAppointmentsList : System.Web.UI.Page
    {
        //--------------------------------------------------
        /// <summary>
        /// True to refresh group appointment list in view "0"
        /// </summary>
        bool _refreshCalendarData = false;

        /// <summary>
        /// True to refresh the patient appointment list in view "1"
        /// </summary>
        bool _refreshGroupAppointmentData = false;

        /// <summary>
        /// The date and time of the appointment to view - in the time zone of the site
        /// </summary>
        DateTime _appointmentTimeSelectedSiteTimeZone = DateTime.MinValue;

        /// <summary>
        /// The length of the selected appointment
        /// </summary>
        int _appointmentTimeSelectedLength = 0;

        /// <summary>
        /// during a databind, whether there are appointments to cancel (status=scheduled)
        /// </summary>
        bool _appointmentsToCancel = false;

        /// <summary>
        /// The capcity of the group appointment
        /// </summary>
        int _remainingGroupCapacity = 0;

        /// <summary>
        /// Whether the lost can be over-booked
        /// </summary>
        bool _isOverBookable = false;
        //--------------------------------------------------
        protected override void OnInit(EventArgs e)
        {
            Page.RegisterRequiresControlState(this);
            base.OnInit(e);

            WucPatientAppointmentViewCalendar1.CalendarEventClick += WucPatientAppointmentViewCalendar1_CalendarEventClick;
            WucPatientAppointmentViewCalendar1.DateSelected += WucPatientAppointmentViewCalendar1_DateSelected;

            //--------------------
            wucChangeProvider1.ProviderChanged += wucChangeProvider1_ProviderChanged;
        }

        void wucChangeProvider1_ProviderChanged(object sender, User_Controls.ProviderChangedArgs e)
        {
            try
            {
                ListItem li = ddlProvider.Items.FindByValue(e.ProviderId.ToString());

                if (li != null)
                {
                    ddlProvider.SelectedIndex = -1;
                    li.Selected = true;
                }

                _refreshCalendarData = true;
                _refreshGroupAppointmentData = true;
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void WucPatientAppointmentViewCalendar1_DateSelected(object sender, User_Controls.DateSelected e)
        {
            _refreshCalendarData = true;
        }

        void WucPatientAppointmentViewCalendar1_CalendarEventClick(object sender, DayPilot.Web.Ui.Events.EventClickEventArgs e)
        {
            try
            {
                // An event in the calendar was clicked which is a group appointment
                // the time in the displayed appointment is in the timezone of the site
                // show a view that allows the user to view the appointment status
                var parts = e.Value.Split(new char[] { '|' }).ToList();

                if (parts.Count == 2)
                {
                    // we use the date for conversion so any will do
                    DateTime dtAppointmentStart = DateTime.UtcNow;

                    if (DateTime.TryParse(parts[0], out dtAppointmentStart))
                    {
                        int appointmentLengthMinutes = 0;

                        if (int.TryParse(parts[1], out appointmentLengthMinutes))
                        {
                            _appointmentTimeSelectedSiteTimeZone = dtAppointmentStart;
                            _appointmentTimeSelectedLength = appointmentLengthMinutes;

                            // show appointments for slot
                            mv.ActiveViewIndex = 1;

                            // get remaining capcity
                            CalculateRemainingSlotCapacity();

                            // refresh the data viewed
                            _refreshGroupAppointmentData = true;
                        }
                        else
                        {
                            lblMessage.Text = "Appointment minutes value is invalid";
                        }
                    }
                    else
                    {
                        lblMessage.Text = "Appointment start time is invalid";
                    }
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected override object SaveControlState()
        {
            Dictionary<string, object> dic = new Dictionary<string, object>();

            dic.Add("_appointmentTimeSelected", _appointmentTimeSelectedSiteTimeZone);
            dic.Add("_appointmentTimeSelectedLength", _appointmentTimeSelectedLength);

            dic.Add("_remainingGroupCapacity", _remainingGroupCapacity);
            dic.Add("_isOverBookable", _isOverBookable);

            dic.Add("base", base.SaveControlState());

            return dic;
        }

        protected override void LoadControlState(object savedState)
        {
            Dictionary<string, object> dic = savedState as Dictionary<string, object>;

            if (dic != null)
            {
                if (dic["_appointmentTimeSelected"] != null && dic["_appointmentTimeSelected"] is DateTime)
                    _appointmentTimeSelectedSiteTimeZone = (DateTime)dic["_appointmentTimeSelected"];

                if (dic["_appointmentTimeSelectedLength"] != null && dic["_appointmentTimeSelectedLength"] is int)
                    _appointmentTimeSelectedLength = (int)dic["_appointmentTimeSelectedLength"];

                if (dic["_remainingGroupCapacity"] != null && dic["_remainingGroupCapacity"] is int)
                    _remainingGroupCapacity = (int)dic["_remainingGroupCapacity"];

                if (dic["_isOverBookable"] != null && dic["_isOverBookable"] is bool)
                    _isOverBookable = (bool)dic["_isOverBookable"];

                base.LoadControlState(dic["base"]);
            }
            else
            {
                base.LoadControlState(savedState);
            }
        }

        protected void Page_Load(object sender, EventArgs e)
        {
            lblMessage.Text = string.Empty;

            if (!IsPostBack)
            {
                LoadFacilitiesDdl();
            }
        }

        protected void Page_PreRender(object sender, EventArgs e)
        {
            if (_refreshCalendarData)
            {
                ShowSlotsForProvider();
            }

            if (_refreshGroupAppointmentData)
            {
                ShowGroupAppointmentData();
            }

            ShowAvailability(lblRemainingCapacity);

            // we see the appointment info in all pages except the calendar
            divAppointmentInfo.Visible = (mv.ActiveViewIndex > 0);
        }

        void LoadFacilitiesDdl()
        {
            try
            {
                Factory factory = Helper.GetFactory(Page);
                var facilityService = factory.GetFacilityService();

                int siteId = Helper.GetUserSession(Page).CurrentSite.Id;

                ddlFacility.DataValueField = "Id";
                ddlFacility.DataTextField = "Name";
                ddlFacility.DataSource = (from f in facilityService.GetAll(siteId)
                                          orderby f.Name
                                          select new
                                          {
                                              Id = f.Id,
                                              Name = f.Name
                                          }).ToList();
                ddlFacility.DataBind();

                if (ddlFacility.Items.Count > 0)
                    ddlFacility.SelectedIndex = 0;

                ShowClinicsForFacility();
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void ShowClinicsForFacility()
        {
            ddlClinic.Items.Clear();

            if (ddlFacility.SelectedValueAsInt() > 0)
            {
                Factory factory = Helper.GetFactory(Page);
                var sectionService = factory.GetSectionService();

                ddlClinic.DataValueField = "Id";
                ddlClinic.DataTextField = "Name";
                // Show ONLY active clicnics
                ddlClinic.DataSource = (from s in sectionService.GetAll(ddlFacility.SelectedValueAsInt())
                                        where s.Active==true
                                        orderby s.Name
                                        select new
                                        {
                                            Id = s.Id,
                                            Name = s.Name
                                        }).ToList();
                ddlClinic.DataBind();
            }

            if (ddlClinic.Items.Count == 0)
            {
                ddlClinic.Items.Clear();
                ddlClinic.Items.Add(new ListItem("No clinics", "-1"));
            }

            ShowProvidersForClicnic();

            _refreshCalendarData = true;
        }

        void ShowProvidersForClicnic()
        {
            try
            {
                Factory factory = Helper.GetFactory(Page);

                // get all providers for a clinic
                var resourceService = factory.GetResourceService();
                var providerResources = resourceService.GetAllOfType(ddlClinic.SelectedValueAsInt(), ResourceType.Provider);

                List<Resource> providerResourceWithGroupSlots = new List<Resource>();

                foreach (var providerResource in providerResources)
                {
                    // we need to break out of inner-ineer loop which is not clean to do -
                    // keep track when we find at least one resource with slot capcity more than 1
                    bool foundForResource = false;

                    foreach (var schedulingPolicy in providerResource.SchedulingPolicies)
                    {
                        foreach (var slot in schedulingPolicy.Slots)
                        {
                            if (slot.GroupAppointment)
                            {
                                providerResourceWithGroupSlots.Add(providerResource);
                                foundForResource = true;
                                break;
                            }
                        }

                        if (foundForResource)
                            break;
                    }
                }

                ddlProvider.DataValueField = "Id";
                ddlProvider.DataTextField = "Name";
                ddlProvider.DataSource = (from p in providerResourceWithGroupSlots
                                          orderby p.Name
                                          select new
                                          {
                                              Id = p.Id,
                                              Name = p.Name
                                          }).ToList();
                ddlProvider.DataBind();

                if (ddlProvider.Items.Count == 0)
                {
                    ddlProvider.Items.Clear();
                    ddlProvider.Items.Add(new ListItem("No providers", "-1"));
                }

                _refreshCalendarData = true;
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void ddlFacility_SelectedIndexChanged(object sender, EventArgs e)
        {
            ShowClinicsForFacility();
        }

        protected void ddlClinic_SelectedIndexChanged(object sender, EventArgs e)
        {
            ShowProvidersForClicnic();
        }

        protected void ddlProvider_SelectedIndexChanged(object sender, EventArgs e)
        {
            _refreshCalendarData = true;

        }
        void ShowSlotsForProvider()
        {
            try
            {
                List<CalendarData> list = new List<CalendarData>();

                if (ddlProvider.SelectedValueAsInt() > 0)
                {
                    Factory factory = Helper.GetFactory(Page);

                    // get all providers for a clinic
                    var resourceService = factory.GetResourceService();

                    // need to get slots for all resources involved and whether free slots
                    var slotDays = resourceService.GetGroupAppointmentSlots(ddlProvider.SelectedValueAsInt(), WucPatientAppointmentViewCalendar1.WeekStartDate, WucPatientAppointmentViewCalendar1.WeekEndingDate);

                    TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();

                    foreach (var day in slotDays.Keys)
                    {
                        // For VALUE - we'll send a composite string of the start time of the appointment and the length
                        list.AddRange((from s in slotDays[day]
                                       orderby s.Time
                                       select new CalendarData
                                       {
                                           DataStart = TimeZoneInfo.ConvertTimeFromUtc(s.Time, tz),
                                           DataEnd = TimeZoneInfo.ConvertTimeFromUtc(s.Time, tz).AddMinutes(s.Length),
                                           Text = s.Slot.AppointmentType.Name,
                                           Value = s.Time.ToMediumFormatFromUtc(tz) + "|" + s.Length.ToString()
                                       }).ToList());
                    }
                }

                WucPatientAppointmentViewCalendar1.BindCalendar(list);
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }


        //---------------------------------------------------------
        // Group Appointment list

        void ShowGroupAppointmentData()
        {
            try
            {
                // show local time of appointmemnt when cancelling
                lblAppointmentTimeBeingCancelledForPatient.Text = _appointmentTimeSelectedSiteTimeZone.ToMediumFormat();
                lblAppointmentTimeBeingCancelledForAllPatient.Text = lblAppointmentTimeBeingCancelledForPatient.Text;
                //-----------------------------------

                Factory factory = Helper.GetFactory(Page);

                // get all providers for a clinic
                var resourceService = factory.GetResourceService();
                var providerResources = resourceService.GetAllOfType(ddlClinic.SelectedValueAsInt(), ResourceType.Provider);

                var service = factory.GetAppointmentService();
                List<Appointment> appts = new List<Appointment>();

                int providerResourceId = ddlProvider.SelectedValueAsInt();

                if (providerResourceId > 0)
                {
                    // need to add a call to service layer to get all regardless of type

                    // get all appointments - need to convert displayed date to UTC
                    TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();
                    DateTime appointmentTimeSelectedUtc = TimeZoneInfo.ConvertTimeToUtc(_appointmentTimeSelectedSiteTimeZone, tz);
                    // get in UTC
                    appts.AddRange(service.GetForResource(providerResourceId, appointmentTimeSelectedUtc, appointmentTimeSelectedUtc, AppointmentStatus.Unknown));
                }

                // refresh display of remaining capcity
                CalculateRemainingSlotCapacity();

                // order by checked-in then, last name
                lv.DataSource = (from a in appts
                                 orderby a.Status, a.Patient.Person.LastName
                                 select a).ToList();

                //assume no appointments to cancel
                _appointmentsToCancel = false;

                lv.DataBind();

                // check for group cancel allowed - and change provider
                lbCancelAllAppointments.Visible = _appointmentsToCancel;
                lbChangeProvider.Visible = _appointmentsToCancel;


                // made unavailable for now
                //lbChangeProvider.Visible = false;
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void lv_ItemDataBound(object sender, ListViewItemEventArgs e)
        {
            if (e.Item.ItemType == ListViewItemType.DataItem)
            {
                Appointment a = e.Item.DataItem as Appointment;
                Label lblPatient = e.Item.RecursiveFindControl("lblPatient") as Label;
                HtmlTableCell tdAppointmentStatus = e.Item.RecursiveFindControl("tdAppointmentStatus") as HtmlTableCell;
                HiddenField hfAppointmentId = e.Item.RecursiveFindControl("hfAppointmentId") as HiddenField;
                LinkButton lbAppointmentActions = e.Item.RecursiveFindControl("lbAppointmentActions") as LinkButton;

                if (a != null && lblPatient != null && tdAppointmentStatus != null &&
                    hfAppointmentId != null && lbAppointmentActions != null)
                {
                    // if amy appointment id schedules, a GROUP cancel can be performed
                    if (a.Status == AppointmentStatus.Scheduled)
                        _appointmentsToCancel = true;

                    hfAppointmentId.Value = a.Id.ToString();
                    lblPatient.Text = a.Patient.Person.GetFullName();

                    tdAppointmentStatus.InnerText = a.Status.ToString();

                    // actions only make sens for scheduled or checked in
                    lbAppointmentActions.Visible = (a.Status == AppointmentStatus.Scheduled || a.Status == AppointmentStatus.CheckedIn);
                }
            }
        }

        protected void lv_ItemCommand(object sender, ListViewCommandEventArgs e)
        {
            try
            {
                if (e.CommandName == "AppointmentActions")
                {
                    ShowPatientAppointmentActions(int.Parse(e.CommandArgument.ToString()));
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void CheckIn(int appointmentId)
        {
            try
            {
                Factory factory = Helper.GetFactory(Page);
                var appointmentService = factory.GetAppointmentService();

                var appointment = appointmentService.Get(appointmentId);

                if (appointment != null)
                {
                    if (appointment.Status == AppointmentStatus.Scheduled)
                    {
                        appointmentService.CheckInAppointment(appointment);
                        lblMessage.Text = "Appointment Checked-In";
                        _refreshGroupAppointmentData = true;
                    }
                    else
                    {
                        lblMessage.Text = "Appointment must be in the [scheduled] state before it can be checked in";
                    }
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void CheckOut(int appointmentId, string disposition)
        {
            try
            {
                Factory factory = Helper.GetFactory(Page);
                var appointmentService = factory.GetAppointmentService();

                var appointment = appointmentService.Get(appointmentId);

                if (appointment != null)
                {
                    if (appointment.Status == AppointmentStatus.CheckedIn)
                    {
                        appointmentService.CheckOutAppointment(appointment, disposition);
                        lblMessage.Text = "Appointment Checked-Out";
                        _refreshGroupAppointmentData = true;
                    }
                    else
                    {
                        lblMessage.Text = "Appointment must be in the [Checked-In] state before it can be checked out";
                    }
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void NoShow(int appointmentId)
        {
            try
            {
                Factory factory = Helper.GetFactory(Page);
                var appointmentService = factory.GetAppointmentService();

                var appointment = appointmentService.Get(appointmentId);

                if (appointment != null)
                {
                    if (appointment.Status == AppointmentStatus.Scheduled)
                    {
                        appointmentService.NoShowAppointment(appointment);
                        lblMessage.Text = "Appointment Checked-Out as No-Show";
                        _refreshGroupAppointmentData = true;
                    }
                    else
                    {
                        lblMessage.Text = "Appointment must be in the [Checked-In] state before it can be checked out (no-show)";
                    }
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void lbViewCalendar_Click(object sender, EventArgs e)
        {
            // update calcendar with any changes before showing it (whether from this page or another user at another browser)
            _refreshCalendarData = true;

            mv.ActiveViewIndex = 0;
        }

        protected void lbBackToAppointmentList_Click(object sender, EventArgs e)
        {
            mv.ActiveViewIndex = 1;
        }

        int CalculateRemainingSlotCapacity()
        {
            try
            {
                // assume no capacity
                _remainingGroupCapacity = 0;
                // assume not over-bookable
                _isOverBookable = false;

                // to search for free slots we must get the slots for all selected resources then check each has the same slots free
                // get the resource Id's selected - we must have at leat a provider
                int providerResourceId = ddlProvider.SelectedValueAsInt();
                int roomResourceId = ddlRooms.SelectedValueAsInt();
                int equipmentResourceId = ddlEquipment.SelectedValueAsInt();

                int siteId = UISession.Helper.GetUserSession(Page).CurrentSite.Id;
                Factory factory = Helper.GetFactory(Page);
                var resourceService = factory.GetResourceService();

                System.Collections.ObjectModel.Collection<int> resourceIds = new System.Collections.ObjectModel.Collection<int>();

                // will have a provider
                resourceIds.Add(providerResourceId);

                // may have a room
                if (roomResourceId > 0)
                    resourceIds.Add(roomResourceId);

                // may have equipment
                if (equipmentResourceId > 0)
                    resourceIds.Add(equipmentResourceId);

                TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();

                DateTime appointmentTimeUtc = TimeZoneInfo.ConvertTimeToUtc(_appointmentTimeSelectedSiteTimeZone, tz);

                var slotDays = resourceService.GetAvailability(resourceIds, appointmentTimeUtc, appointmentTimeUtc);

                if (slotDays.Count == 1)
                {
                    // get selected slot date-time from the first (only) slot day

                    var slot = (from s in slotDays[slotDays.Keys.First()]
                                where s.Time == appointmentTimeUtc
                                select s).FirstOrDefault();

                    if (slot != null)
                    {
                        // found capacity
                        _remainingGroupCapacity = slot.Remaining;

                        // get provider over-bookability
                        var provider = resourceService.Get(providerResourceId);

                        if (provider != null)
                        {
                            lblAppointmentInfo.Text = provider.Name + " : " +
                                slot.Time.ToMediumFormatFromUtc(tz) +
                                " [" + slot.Slot.AppointmentType.Name + "]";
                            _isOverBookable = provider.AllowOverbooking;
                        }
                        else
                        {
                            lblAppointmentInfo.Text = "No Provider " + " : " +
                                slot.Time.ToMediumFormatFromUtc(tz) +
                                " [" + slot.Slot.AppointmentType.Name + "]";
                        }
                    }
                }
                else
                {
                    // can't find free slot so must have reached capacity
                    _remainingGroupCapacity = 0;

                    // can't find free slot so not overbookable
                    _isOverBookable = false;
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }

            return 0;
        }

        protected void btnAddPatientsToGroup_Click(object sender, EventArgs e)
        {
            try
            {
                List<int> patientsSelected = new List<int>();

                foreach (ListItem li in cblPatients.Items)
                {
                    if (li.Selected)
                    {
                        patientsSelected.Add(int.Parse(li.Value));
                    }
                }

                if (patientsSelected.Count == 0)
                {
                    lblMessage.Text = "You have not selected any patients";
                    return;
                }

                if (!_isOverBookable)
                {
                    // when not over-bookable, _remainingGroupCapacity is greater than zero for availability, or zero when none
                    if (patientsSelected.Count > _remainingGroupCapacity)
                    {
                        lblMessage.Text = "Cannot add " + patientsSelected.Count.ToString() +
                            " patient(s) to " + _remainingGroupCapacity.ToString() + " available place(s)";
                        return;
                    }
                }

                // save
                string vistaSiteId = UISession.Helper.GetUserSession(Page).CurrentSite.VistaSiteId;

                Shared.Model.User user = UISession.Helper.GetCurrentUserAccount(this);

                //-----------------------------------------
                Factory factory = Helper.GetFactory(Page);
                //-----------------------------------------
                var siteService = factory.GetSectionService();
                var clinic = siteService.Get(int.Parse(ddlClinic.SelectedValue));

                //-----------------------------------------
                var resourceService = factory.GetResourceService();
                var providerResource = resourceService.Get(ddlProvider.SelectedValueAsInt());

                if (providerResource == null)
                {
                    lblMessage.Text = "There is no provider selected";
                    return;
                }

                //-----------------------------------------
                var service = factory.GetAppointmentService();
                var patientService = factory.GetPatientService();
                var nationalSystemService = factory.GetNationalSystemService();
                //-----------------------------------------
                int patientsAdded = 0;

                var tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();
                var appointmentTimeSelectedUtc = TimeZoneInfo.ConvertTimeToUtc(_appointmentTimeSelectedSiteTimeZone, tz);

                foreach (int patientId in patientsSelected)
                {
                    var patient = patientService.Get(patientId);

                    Shared.Model.AppointmentRequest r = null;
                    Shared.Model.Appointment a = null;

                    r = new Shared.Model.AppointmentRequest();
                    a = new Shared.Model.Appointment();

                    r.CreatingUser = user;

                    // aded patients
                    r.Patient = patient;
                    a.Patient = patient;
                    a.VistaId = vistaSiteId;

                    r.RequestMethod = Shared.Model.RequestMethod.Other;

                    r.RequestedSection = clinic;
                    a.Section = clinic;

                    r.Status = Shared.Model.RequestStatus.Complete;
                    a.Status = Shared.Model.AppointmentStatus.Scheduled;

                    r.DesiredDate = appointmentTimeSelectedUtc;
                    r.Time = appointmentTimeSelectedUtc;

                    a.Time = appointmentTimeSelectedUtc;
                    a.Length = _appointmentTimeSelectedLength;

                   // if (ddlUrgency.SelectedValue != null && ddlUrgency.SelectedValue == "1")
                       // r.Priority = 1;
                    //else
                        r.Priority = 0;

                    //----------------------------------------
                    // unless specifically NOT selected - assume high priority
                    //if (ddlIsRescheduleHighPriority.SelectedIndex == 0)
                        a.HighPriorityReschedule = false;
                   // else
                       // a.HighPriorityReschedule = true;

                    //----------------------------------------
                    var apptType = (from t in clinic.AppointmentTypes
                                    where t.Id == ddlAppointmentType.SelectedValueAsInt()
                                    select t).FirstOrDefault();

                    a.AppointmentType = apptType;

                    //----------------------------------------
                    // resources
                    r.RequestedResource = providerResource;
                    r.RequestedSection = clinic;

                    a.Resources.Add(providerResource);

                    // add Room resource (if there is one)
                    if (ddlRooms.SelectedValueAsInt() > 0)
                        a.Resources.Add(resourceService.Get(ddlRooms.SelectedValueAsInt()));

                    // add Equipment resource (if there is one)
                    if (ddlEquipment.SelectedValueAsInt() > 0)
                        a.Resources.Add(resourceService.Get(ddlEquipment.SelectedValueAsInt()));

                    //----------------------------------------

                    service.CreateAppointment(a);
                    service.AddAppointmentRequest(r);

                    // add
                    r.ResultingAppointment = a;

                    // save
                    service.UpdateAppointmentRequest(r);

                    patientsAdded += 1;
                }

                lblMessage.Text = patientsAdded.ToString() + " patient(s) added";

                // refresh the group list
                _refreshGroupAppointmentData = true;

                // show the group list
                mv.ActiveViewIndex = 1;
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void lbShowPatientAddView_Click(object sender, EventArgs e)
        {
            try
            {
                // show where we are adding the patients to
                tdFacilityName.InnerText = ddlFacility.SelectedItem.Text;
                tdClinicName.InnerText = ddlClinic.SelectedItem.Text;
                tdProviderName.InnerText = ddlProvider.SelectedItem.Text;

                // drop down lists
                LoadResourcesDropDownLists();

                // Show patients not already booked into the slot
                ShowPatientsNotInGroup();

                ShowAvailability(lblAvailability);

                mv.ActiveViewIndex = 2;
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        /// <summary>
        /// Show availability in a label
        /// </summary>
        /// <param name="lbl"></param>
        void ShowAvailability(Label lbl)
        {
            if (_isOverBookable)
            {
                lbShowPatientAddView.Visible = true;

                if (_remainingGroupCapacity > 0)
                {
                    lbl.Text = "Slots remaining : " + _remainingGroupCapacity.ToString();
                }
                else
                {
                    if (Helper.GetUserSession(Page).UserActions.HasAction("action|CanOverbookAppointment"))
                    {
                        if (_remainingGroupCapacity == 0)
                        {
                            lbl.Text = "Booked to capacity : can be over-booked";
                        }
                        else
                        {
                            lbl.Text = "Over-booked by : " +
                                (Math.Abs(_remainingGroupCapacity).ToString()) +
                                " : can be over-booked";
                        }
                    }
                    else
                    {
                        lbl.Text = "You are not authorized to overbook this appointment";
                        lbShowPatientAddView.Visible = false;
                    }
                }
            }
            else
            {
                lbl.Text = "Slots remaining : " + _remainingGroupCapacity.ToString();
                lbShowPatientAddView.Visible = (_remainingGroupCapacity > 0);
            }
        }

        void LoadResourcesDropDownLists()
        {
            int clinicId = ddlClinic.SelectedValueAsInt();

            ddlRooms.Items.Clear();
            ddlEquipment.Items.Clear();
            ddlAppointmentType.Items.Clear();

            Factory factory = Helper.GetFactory(Page);

            UserSession userSession = Helper.GetUserSession(Page);
            int siteId = userSession.CurrentSite.Id;

            var sectionService = factory.GetSectionService();
            var clinic = sectionService.Get(clinicId);

            // get available resources for the clinic
            var resourceService = factory.GetResourceService();
            var resources = resourceService.GetAll(clinicId);

            //-----------------------------------------------
            var roomService = factory.GetRoomService();
            var allRooms = roomService.GetAllForSite(siteId).ToList();
            var allRoomResources = resourceService.GetAllOfType(clinicId, ResourceType.Room);

            var rooms = (from r in allRoomResources
                         join a in allRooms
                         on r.FullfillingResourceId equals a.Id
                         orderby r.Name
                         select new BasicResourceData
                         {
                             ResourceId = r.Id,
                             ResourceName = r.Name + " (" + a.Name + " - floor + " + a.Floor + ")"
                         }).ToList();

            ddlRooms.DataValueField = "ResourceId";
            ddlRooms.DataTextField = "ResourceName";
            ddlRooms.DataSource = rooms;
            ddlRooms.DataBind();

            // rooms are optional - prepend an empty selection
            ddlRooms.Items.Insert(0, new ListItem("No room required", "0"));
            ddlRooms.SelectedIndex = 0;

            //-----------------------------------------------
            var equipmentService = factory.GetEquipmentService();
            var allEquipment = equipmentService.GetAllForSite(siteId);
            var allEquipmentResources = resourceService.GetAllOfType(clinicId, ResourceType.Equipment);

            var equipment = (from r in allEquipmentResources
                             join e in allEquipment
                             on r.FullfillingResourceId equals e.Id
                             orderby r.Name
                             select new BasicResourceData
                             {
                                 ResourceId = r.Id,
                                 ResourceName = r.Name + " (" + e.Name + " - type " + e.Type + ")"
                             }).ToList();

            ddlEquipment.DataValueField = "ResourceId";
            ddlEquipment.DataTextField = "ResourceName";
            ddlEquipment.DataSource = equipment;
            ddlEquipment.DataBind();

            // equipment optional
            ddlEquipment.Items.Insert(0, new ListItem("No equipment required", "0"));
            ddlEquipment.SelectedIndex = 0;

            //-----------------------------------------------
            var appointmentTypes = (from t in clinic.AppointmentTypes
                                    orderby t.Name
                                    select new BasicResourceData
                                    {
                                        ResourceId = t.Id,
                                        ResourceName = t.Name
                                    }).ToList();

            ddlAppointmentType.DataValueField = "ResourceId";
            ddlAppointmentType.DataTextField = "ResourceName";
            ddlAppointmentType.DataSource = appointmentTypes;
            ddlAppointmentType.DataBind();

            if (ddlRooms.Items.Count == 0)
                ddlRooms.Items.Add(new ListItem("No Rooms", "0"));

            if (ddlEquipment.Items.Count == 0)
                ddlEquipment.Items.Add(new ListItem("No Equipment", "0"));

            if (ddlAppointmentType.Items.Count == 0)
                ddlAppointmentType.Items.Add(new ListItem("No Appointment Types", "0"));

            SetResourcesToExistingAppointment(factory);
        }

        /// <summary>
        /// Sets the DDL's from the first appointment in the group - if there is one
        /// </summary>
        /// <param name="f"></param>
        void SetResourcesToExistingAppointment(Factory f)
        {
            if (lv.Items.Count > 0)
            {
                // get first item and use this to set resources
                var appointmentService = f.GetAppointmentService();
                HiddenField hfAppointmentId = lv.Items[0].RecursiveFindControl("hfAppointmentId") as HiddenField;

                if (hfAppointmentId != null)
                {
                    var appt = appointmentService.Get(int.Parse(hfAppointmentId.Value));

                    if (appt != null)
                    {
                        bool roomResourceSelected = false;
                        bool equipmentResourceSelected = false;

                        foreach (var resource in appt.Resources)
                        {
                            switch (resource.Type)
                            {
                                // provider already selected

                                case ResourceType.Room:
                                    if (!roomResourceSelected)
                                    {
                                        if (SetResourceDdlItem(ddlRooms, resource.Id))
                                            roomResourceSelected = true;
                                    }
                                    break;

                                case ResourceType.Equipment:
                                    if (!equipmentResourceSelected)
                                    {
                                        if (SetResourceDdlItem(ddlEquipment, resource.Id))
                                            equipmentResourceSelected = true;
                                    }
                                    break;
                            }
                        }
                    }
                }
            }
        }

        bool SetResourceDdlItem(DropDownList ddl, int val)
        {
            ListItem li = ddl.Items.FindByValue(val.ToString());
            if (li != null)
            {
                ddl.SelectedIndex = -1;
                li.Selected = true;
                return true;
            }

            return false;
        }

        void ShowPatientsNotInGroup()
        {
            Factory factory = Helper.GetFactory(Page);
            var appointmentService = factory.GetAppointmentService();

            List<int> patientsInGroup = new List<int>();

            foreach (var item in lv.Items)
            {
                // get stored appointment ID
                HiddenField hfAppointmentId = item.RecursiveFindControl("hfAppointmentId") as HiddenField;

                // get appointment status
                HtmlTableCell tdAppointmentStatus = item.RecursiveFindControl("tdAppointmentStatus") as HtmlTableCell;
                AppointmentStatus currentStatus = (AppointmentStatus)Enum.Parse(typeof(AppointmentStatus), tdAppointmentStatus.InnerText);

                // if a current appointment then cannot add again to the appoinment
                // if not a current appointment (e.g. cancelled - we can add them again)
                if (currentStatus == AppointmentStatus.CheckedIn || currentStatus == AppointmentStatus.Scheduled)
                {
                    var appt = appointmentService.Get(int.Parse(hfAppointmentId.Value));
                    patientsInGroup.Add(appt.Patient.Id);
                }
            }

            // fill patient list
            string vistaSiteId = Helper.GetUserSession(Page).CurrentSite.VistaSiteId;

            var patientService = factory.GetPatientService();

            cblPatients.DataTextField = "Name";
            cblPatients.DataValueField = "Id";

            // get patients not in the group
            cblPatients.DataSource = (from p in patientService.Get()
                                      where p.PatientSites.Keys.Contains(vistaSiteId) &&
                                      !patientsInGroup.Contains(p.Id)
                                      orderby p.Person.LastName
                                      select new
                                      {
                                          Id = p.Id,
                                          Name = p.Person.GetFullName()
                                      }).Distinct().ToList();

            cblPatients.DataBind();
        }

        void ShowPatientAppointmentActions(int appointmentId)
        {
            // get appointment and patient details
            Factory factory = Helper.GetFactory(Page);
            var appointmentService = factory.GetAppointmentService();
            var appointment = appointmentService.Get(appointmentId);

            // show patient
            ShowPatientDetails(appointment.Patient);

            // show actions
            lbCheckIn.Visible = false;
            lbAppointmentKept.Visible = false;
            lpAppointmentNoShow.Visible = false;
            lbAppointmentLeftWithoutBeingSeen.Visible = false;
            lbCancelAppointment.Visible = false;

            switch (appointment.Status)
            {
                case AppointmentStatus.Scheduled:
                    // only these states can be set when scheduled
                    lbCheckIn.Visible = true;
                    lbCheckIn.CommandArgument = appointment.Id.ToString();

                    lpAppointmentNoShow.Visible = true;
                    lpAppointmentNoShow.CommandArgument = appointment.Id.ToString();

                    lbCancelAppointment.Visible = true;
                    lbCancelAppointment.CommandArgument = appointment.Id.ToString();
                    break;

                case AppointmentStatus.CheckedIn:
                    // Once check in, these states can be set
                    lbAppointmentKept.Visible = true;
                    lbAppointmentKept.CommandArgument = appointment.Id.ToString();

                    lbAppointmentLeftWithoutBeingSeen.Visible = true;
                    lbAppointmentLeftWithoutBeingSeen.CommandArgument = appointment.Id.ToString();
                    break;
            }

            mv.ActiveViewIndex = 3;
        }

        void PerformPatientAction(string commandName, string commandArgument)
        {
            try
            {
                switch (commandName)
                {
                    case "CheckIn":
                        CheckIn(int.Parse(commandArgument));
                        break;

                    case "CheckOutAsAppointmentKept":
                        CheckOut(int.Parse(commandArgument), "Appointment Kept");
                        break;

                    case "CheckOutAsLeftWithoutBeingSeen":
                        CheckOut(int.Parse(commandArgument), "Left without being seen");
                        break;

                    case "CheckOutAsNoShow":
                        NoShow(int.Parse(commandArgument));
                        break;
                }

                // refresh data
                _refreshGroupAppointmentData = true;

                // appointment list
                mv.ActiveViewIndex = 1;
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void ShowPatientDetails(Shared.Model.Patient selectedPatient)
        {
            try
            {
                if (selectedPatient != null)
                {
                    var p = selectedPatient.Person;
                    // show details
                    tdName.InnerText = p.GetFullName();

                    tdAddress.InnerHtml = p.Address.Street1 + "<br />" +
                                          p.Address.Street2 + "<br />" +
                                          p.Address.Street3;

                    if (p.DateOfBirth.HasValue)
                        tdDateOfBirth.InnerText = p.DateOfBirth.Value.ToString("dd-MMM-yyyy");
                    else
                        tdDateOfBirth.InnerText = string.Empty;

                    tdEMail.InnerText = p.Email;
                    tdEthnicity.InnerText = p.Ethnicity;
                    tdMaritalStatus.InnerText = p.MaritalStatus;
                    tdPhone.InnerText = p.Phone;
                    tdSex.InnerText = p.Sex;
                    tdSSN.InnerText = Helper.GetSSN(p.SSN);

                    return;
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }

            lblMessage.Text = "Patient not found";
        }

        protected void btnCancelAppointment_Click(object sender, EventArgs e)
        {
            try
            {
                string reason = tbReason.Text.Trim();

                if (reason.Length > 0)
                {
                    int appointmentId = int.Parse(lbCancelAppointment.CommandArgument);

                    Factory factory = Helper.GetFactory(Page);
                    var appointmentService = factory.GetAppointmentService();
                    var appointment = appointmentService.Get(appointmentId);

                    if (appointment != null)
                    {
                        if (appointment.Status == AppointmentStatus.Scheduled)
                        {
                            TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();

                            string appointmentDetail = null;

                            if (appointment.Section != null)
                                appointmentDetail = "Appointment at " + appointment.Section.Name + " on " + appointment.Time.ToMediumFormatFromUtc(tz) + " is cancelled";
                            else
                                appointmentDetail = "Appointment on " + appointment.Time.ToMediumFormatFromUtc(tz) + " is cancelled";

                            appointmentService.CancelAppointment(appointment, ddlCancelledBy.SelectedItem.Value, reason, string.Empty);

                            if (ddlReschedule.SelectedValue == "1")
                                AddAppointmentRequest(appointment, DateTime.UtcNow);

                            lblMessage.Text = appointmentDetail;

                            ExtensionMethods.AddACommunicationTemplate(appointment, CommunicationType.Cancellation, this.Page);
                            //AddCancellationCommunicationTemplate(appointment, CommunicationType.Cancellation);

                            // back to list
                            mv.ActiveViewIndex = 1;

                            _refreshGroupAppointmentData = true;
                        }
                        else
                        {
                            lblMessage.Text = "Can only cancel an appointment that is scheduled";
                        }
                    }
                    else
                    {
                        lblMessage.Text = "Cannot find the appointment to cancel";
                    }
                }
                else
                {
                    lblMessage.Text = "Please provide a reason for the cancellation";
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }


        //void AddCancellationCommunicationTemplate(Appointment appointment, CommunicationType commsType)
        //{
        //    // Does section have a comms template?
        //    CommunicationTemplate ct = appointment.Section.CommunicationTemplates.Where(a => a.Type == commsType).FirstOrDefault();

        //    // Try facility level if not
        //    if (ct == null)
        //        ct = appointment.Section.Facility.CommunicationTemplates.Where(a => a.Type == commsType).FirstOrDefault();

        //    if (ct != null)
        //    {
        //        Factory factory = Helper.GetFactory(Page);
        //        var appointmentService = factory.GetAppointmentService();

        //        appointmentService.AddToCommunicationQueue(ct, appointment);
        //    }
        //}

        protected void lbCheckIn_Click(object sender, EventArgs e)
        {
            PerformPatientAction(lbCheckIn.CommandName, lbCheckIn.CommandArgument);
        }

        protected void lbAppointmentKept_Click(object sender, EventArgs e)
        {
            PerformPatientAction(lbAppointmentKept.CommandName, lbAppointmentKept.CommandArgument);
        }

        protected void lbAppointmentLeftWithoutBeingSeen_Click(object sender, EventArgs e)
        {
            PerformPatientAction(lbAppointmentLeftWithoutBeingSeen.CommandName, lbAppointmentLeftWithoutBeingSeen.CommandArgument);
        }

        protected void lpAppointmentNoShow_Click(object sender, EventArgs e)
        {
            PerformPatientAction(lpAppointmentNoShow.CommandName, lpAppointmentNoShow.CommandArgument);
        }

        protected void lbBackToAPpointmentsFromPatientView_Click(object sender, EventArgs e)
        {
            mv.ActiveViewIndex = 1;
        }

        protected void lbCancelAppointment_Click(object sender, EventArgs e)
        {
            // cancel
            mv.ActiveViewIndex = 4;
            tbReason.Text = string.Empty; // clear any previous reason

            //lbCancelAppointment.CommandArgument = lbCancelAppointment.CommandArgument;
        }

        protected void lbReturnToListFromCancelAppointment_Click(object sender, EventArgs e)
        {
            mv.ActiveViewIndex = 1;
        }

        protected void lbReturnToListFromCancelAllAppointments_Click(object sender, EventArgs e)
        {
            mv.ActiveViewIndex = 1;
        }

        protected void btnCancelGroup_Click(object sender, EventArgs e)
        {
            try
            {
                string reason = tbGroupCancelReason.Text.Trim();

                if (reason.Length > 0)
                {
                    Factory factory = Helper.GetFactory(Page);
                    var appointmentService = factory.GetAppointmentService();

                    int cancelCount = 0;

                    DateTime utcNow = DateTime.UtcNow;

                    foreach (var item in lv.Items)
                    {
                        HiddenField hfAppointmentId = item.RecursiveFindControl("hfAppointmentId") as HiddenField;

                        if (hfAppointmentId != null)
                        {
                            var appointment = appointmentService.Get(int.Parse(hfAppointmentId.Value));

                            if (appointment != null)
                            {
                                if (appointment.Status == AppointmentStatus.Scheduled)
                                {
                                    TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();
                                    string appointmentDetail = null;

                                    if (appointment.Section != null)
                                        appointmentDetail = "Appointment at " + appointment.Section.Name + " on " + appointment.Time.ToMediumFormatFromUtc(tz) + " is cancelled";
                                    else
                                        appointmentDetail = "Appointment on " + appointment.Time.ToMediumFormatFromUtc(tz) + " is cancelled";

                                    // cancelled by clicnic is only reasonable cancel option (patients don't do a group cancel)
                                    appointmentService.CancelAppointment(appointment, "C", reason, string.Empty);

                                    if (ddlRescheuleGroup.SelectedValue == "1")
                                        AddAppointmentRequest(appointment, utcNow);
                                    ExtensionMethods.AddACommunicationTemplate(appointment, CommunicationType.Cancellation, this.Page);
                                    //AddCancellationCommunicationTemplate(appointment, CommunicationType.Cancellation);

                                    cancelCount += 1;
                                }
                            }
                        }
                    }

                    lblMessage.Text = "Scheduled appointments cancelled: " + lv.Items.Count;

                    // back to list
                    mv.ActiveViewIndex = 1;

                    _refreshGroupAppointmentData = true;
                }
                else
                {
                    lblMessage.Text = "Please provide a reason for the group cancellation";
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void lbCancelAllAppointments_Click1(object sender, EventArgs e)
        {
            mv.ActiveViewIndex = 5;
        }

        protected void lbChangeProvider_Click(object sender, EventArgs e)
        {
            try
            {
                Factory factory = Helper.GetFactory(Page);
                var appointmentService = factory.GetAppointmentService();

                List<Appointment> appointments = new List<Appointment>();

                foreach (var item in lv.Items)
                {
                    HiddenField hfAppointmentId = item.RecursiveFindControl("hfAppointmentId") as HiddenField;

                    if (hfAppointmentId != null)
                    {
                        var appointment = appointmentService.Get(int.Parse(hfAppointmentId.Value));

                        if (appointment != null)
                        {
                            if (appointment.Status == AppointmentStatus.Scheduled ||
                                appointment.Status == AppointmentStatus.CheckedIn)
                                appointments.Add(appointment);
                        }
                        else
                        {
                            lblMessage.Text = "Unable to find appointment with Id=" + hfAppointmentId.Value;
                            return;
                        }
                    }
                    else
                    {
                        lblMessage.Text = "Unable to find appointment";
                    }
                }

                wucChangeProvider1.InitialiseForAppointment(appointments);
                mv.ActiveViewIndex = 6;
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void lbReturnToAppointmentListFromChangeProvider_Click(object sender, EventArgs e)
        {
            mv.ActiveViewIndex = 1;
            _refreshGroupAppointmentData = true;
        }

        void AddAppointmentRequest(Appointment appointment, DateTime rescheduleDate)
        {
            try
            {
                string vistaSiteId = UISession.Helper.GetUserSession(Page).CurrentSite.VistaSiteId;

                Shared.Model.User user = UISession.Helper.GetCurrentUserAccount(this);
                // fix for SC-622 - patient to come from appointment passed and NOT current patient
                var patient = appointment.Patient;

                //-----------------------------------------
                Factory factory = Helper.GetFactory(Page);
                var service = factory.GetAppointmentService();
                //-----------------------------------------

                Shared.Model.AppointmentRequest r = null;

                // create new request
                r = new Shared.Model.AppointmentRequest();
                r.CreatingUser = user;
                r.Patient = patient;
                r.RequestMethod = Shared.Model.RequestMethod.Automatic;
                r.RequestedSection = appointment.Section;
                r.DesiredDate = rescheduleDate;

                // need to know what TIME versus desired time is??????
                r.Time = rescheduleDate;

                foreach (var resource in appointment.Resources)
                {
                    if (resource.Type == ResourceType.Provider)
                    {
                        // pick the first provider resource
                        r.RequestedResource = resource;
                        break;
                    }
                }

                r.Notes = "Auto-Reschedule after cancellation";
                r.Reason = RequestReason.Cancellation;
                r.Status = Shared.Model.RequestStatus.Pending;
                r.Priority = 1; // assumed to be higher priority if rescheduled

                service.AddAppointmentRequest(r);
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }
    }
}