﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.HtmlControls;

using MedRed.Services;
using Shared.Model.Config.MDWS;
using Shared.Model;
using BT.Health.SchedulingWeb.UISession;
using MedRed.Services.Interfaces;

namespace BT.Health.SchedulingWeb.Patient
{
    public partial class ClinicSchedule : System.Web.UI.Page
    {
        /// <summary>
        /// The list of appointments to show
        /// </summary>
        List<Appointment> _appointments = null;

        /// <summary>
        /// The selected resources for a clinic
        /// </summary>
        List<Resource> _clinicResourcesWithSlots = null;

        /// <summary>
        /// Local time zone
        /// </summary>
        TimeZoneInfo _tz = null;

        int clinicStartHour = 6;

        int clinicEndHour = 20;

        /// <summary>
        /// Keep a track of a resources table cell by resouceId
        /// </summary>
        Dictionary<int, TableCell> _resourceCells = new Dictionary<int, TableCell>();

        protected void Page_Load(object sender, EventArgs e)
        {
            lblMessage.Text = string.Empty;
            _tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();

            if (!IsPostBack)
            {
                tbSelectDate.Text = TimeZoneInfo.ConvertTimeFromUtc(DateTime.UtcNow, _tz).Date.ToString("dd-MMM-yyyy");

                //--------
                LoadResourceTypesDdl();
                LoadFacilitiesDdl();

                int lastFacility = 0;
                int lastClinic = 0;

                if (Session["ClinicSchedule_LastFacility"] != null && Session["ClinicSchedule_LastFacility"] is int)
                {
                    lastFacility = (int)Session["ClinicSchedule_LastFacility"];

                    ListItem liFacility = ddlFacility.Items.FindByValue(lastFacility.ToString());

                    if (liFacility != null)
                    {
                        ddlFacility.SelectedIndex = -1;
                        liFacility.Selected = true;

                        ShowClinicsForFacility();
                    }

                }

                if (Session["ClinicSchedule_LastClinic"] != null && Session["ClinicSchedule_LastClinic"] is int)
                {
                    lastClinic = (int)Session["ClinicSchedule_LastClinic"];

                    ListItem liClinic = ddlClinic.Items.FindByValue(lastClinic.ToString());

                    if (liClinic != null)
                    {
                        ddlClinic.SelectedIndex = -1;
                        liClinic.Selected = true;
                    }
                }

                if (Session["ClinicSchedule_LastDate"] != null && Session["ClinicSchedule_LastDate"] is string)
                {
                    tbSelectDate.Text = (string)Session["ClinicSchedule_LastDate"];
                }

                if (Session["ClinicSchedule_LastResourceType"] != null && Session["ClinicSchedule_LastResourceType"] is int)
                {
                    int lastResourceType = (int)Session["ClinicSchedule_LastResourceType"];

                    ListItem liLastResourceType = ddlResourceType.Items.FindByValue(lastResourceType.ToString());

                    if (liLastResourceType != null)
                    {
                        ddlResourceType.SelectedIndex = -1;
                        liLastResourceType.Selected = true;
                    }
                }
            }
        }

        protected void Page_PreRender(object sender, EventArgs e)
        {
            // refreshed every get/post as it's dynamic
            ShowResourcesSchedule();

            Session.Add("ClinicSchedule_LastFacility", ddlFacility.SelectedValueAsInt());
            Session.Add("ClinicSchedule_LastClinic", ddlClinic.SelectedValueAsInt());
            Session.Add("ClinicSchedule_LastDate", tbSelectDate.Text);
            Session.Add("ClinicSchedule_LastResourceType", ddlResourceType.SelectedValueAsInt());
        }

        void ShowResourcesSchedule()
        {
            try
            {
                // clear the list
                _appointments = new List<Appointment>();

                DateTime dtLocal = DateTime.Parse(tbSelectDate.Text).Date;
                DateTime dtUtc = TimeZoneInfo.ConvertTimeToUtc(dtLocal, _tz);
                DateTime dtUtcNextDay = dtUtc.AddDays(1);

                var factory = Helper.GetFactory(Page);

                var sectionService = factory.GetSectionService();
                var clinic = sectionService.Get(ddlClinic.SelectedValueAsInt());

                ResourceType t = (ResourceType)Enum.Parse(typeof(ResourceType), ddlResourceType.SelectedItem.Text);

                // get all resources for a clinic of the type specified
                var resourceService = factory.GetResourceService();
                var allResources = (from r in resourceService.GetAll(ddlClinic.SelectedValueAsInt())
                                    where r.Type == t
                                    orderby r.Name
                                    select r).ToList();

                // get resources that have slots
                _clinicResourcesWithSlots = new List<Resource>();
                GetResourcesWithSLots(allResources, dtUtc);

                DateTime? earliestLocalSlot = null;
                DateTime? latestLocalSlot = null;

                // draw table based on resources with slots
                DrawTable(earliestLocalSlot, latestLocalSlot, dtUtc);

                // iterate each resource with slots
                foreach (var resource in _clinicResourcesWithSlots)
                {
                    // draw any slots and return eearliest and latest times of slot and splan of lasty slot
                    ClinicSlots cs = ShowResourceSlots(resource, dtUtc);

                    // set earlitest time found from resource
                    if (cs != null)
                    {
                        if (cs.EarliestTime.HasValue)
                        {
                            if (earliestLocalSlot.HasValue)
                            {
                                if (cs.EarliestTime.Value < earliestLocalSlot.Value)
                                {
                                    earliestLocalSlot = cs.EarliestTime;
                                }
                            }
                            else
                            {
                                earliestLocalSlot = cs.EarliestTime;
                            }
                        }

                        // set latest time found from resource
                        if (cs.LatestTime.HasValue)
                        {
                            if (latestLocalSlot.HasValue)
                            {
                                if (cs.LatestTime.Value > latestLocalSlot.Value)
                                {
                                    latestLocalSlot = cs.LatestTime.Value;
                                }
                            }
                            else
                            {
                                latestLocalSlot = cs.LatestTime.Value;
                            }
                        }
                    }
                }

                //----------------------------------------------
                // show clinic hours
                lblClinicHours.Text = "Clinic Hours : ";

                if (clinic != null && clinic.Hours != null)
                {
                    lblClinicHours.Text += clinic.Hours;
                }

                //----------------------------------------------
                var appointmentService = factory.GetAppointmentService();

                // iterate the resources
                foreach (Resource r in _clinicResourcesWithSlots)
                {
                    // get appointments of all types
                    var appts = appointmentService.GetForResource(r.Id, null, null, AppointmentStatus.Unknown).ToList();

                    foreach (var appt in appts)
                    {
                        // get appointments that are not cancelled and on the required day
                        if (appt.Time >= dtUtc && appt.Time < dtUtcNextDay && appt.Status != AppointmentStatus.Cancelled)
                        {
                            // only add appointment once - it has a full resource list anyway, so we don't need multiple copies
                            if ((from a in _appointments
                                 where a.Id == appt.Id
                                 select a).FirstOrDefault()
                                == null)
                            {
                                _appointments.Add(appt);

                                // show in display
                                ShowAppointment(r, appt);
                            }
                        }
                    }
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void DrawTable(DateTime? earliest, DateTime? latest, DateTime dtUtc)
        {
            tbl.Rows.Clear();
            DrawTableRows(earliest, latest, dtUtc);
        }

        void DrawTableRows(DateTime? earliest, DateTime? latest, DateTime dtUtc)
        {
            // Header
            TableHeaderRow thr = new TableHeaderRow();
            tbl.Rows.Add(thr);

            // time under this TH
            TableHeaderCell thTime = new TableHeaderCell();
            thr.Cells.Add(thTime);

            // resources header
            foreach (var resource in _clinicResourcesWithSlots)
            {
                TableHeaderCell thResource = new TableHeaderCell();
                thr.Cells.Add(thResource);
                thResource.Text = resource.Name;

            }

            //----------------------------------------------------------
            // Resource slots

            TableRow tr = new TableRow();
            tbl.Rows.Add(tr);

            // Time slot
            TableCell tdTime = new TableCell();
            tr.Cells.Add(tdTime);
            tdTime.CssClass = "tdClinicScheduleTimeBar";
            AddTimeBar(earliest, latest, tdTime);

            foreach (var resource in _clinicResourcesWithSlots)
            {
                TableCell tdResource = new TableCell();
                tr.Cells.Add(tdResource);
                _resourceCells.Add(resource.Id, tdResource);
            }
        }

        void AddTimeBar(DateTime? earliest, DateTime? latest, TableCell tdTime)
        {
            if (earliest.HasValue)
            {
                this.clinicStartHour = earliest.Value.Hour;
            }
            if (latest.HasValue)
            {
                this.clinicEndHour = latest.Value.Hour;
            }
            for (int hour = clinicStartHour; hour <= clinicEndHour; hour += 1)
            {
                Panel p = new Panel();

                p.CssClass = "divClinicSlot";
                p.Style.Add(HtmlTextWriterStyle.Top, ((hour - clinicStartHour) * 60).ToString() + "px");

                Label lbl = new Label();
                lbl.Text = hour.ToString("00") + ":00";
                p.Controls.Add(lbl);
                tdTime.Controls.Add(p);
            }
        }

        void GetResourcesWithSLots(List<Resource> resources, DateTime dtUtc)
        {
            short dayOfWeek = (short)dtUtc.DayOfWeek;

            foreach (var r in resources)
            {
                // get highest priority scheduling policy for the day of the week that covers the date passed
                // order by descending priority and the first one will be the highest
                var policy = (from p in r.SchedulingPolicies
                              where p.DayOfWeek == dayOfWeek &&
                        dtUtc >= p.StartEffectDate &&
                        dtUtc <= p.EndEffectDate
                              orderby p.Priority descending
                              select p).FirstOrDefault();

                if (policy != null)
                {
                    if (policy.Slots.Count > 0)
                    {
                        ClinicSlots cs = new ClinicSlots();

                        // date passed to us falls within the effective period and is on the day in question
                        foreach (var slot in (from s in policy.Slots orderby s.Time select s).ToList())
                        {
                            // add slot to display
                            //--------------------
                            // start and end times
                            DateTime startLocal = TimeZoneInfo.ConvertTimeFromUtc(dtUtc.Date.Add(slot.Time), _tz);
                            DateTime endLocal = startLocal.AddMinutes(slot.Length);

                            // add in the first time as it is the earliest, since we sorted the slots by time ASC
                            if (!cs.EarliestTime.HasValue)
                            {
                                cs.EarliestTime = startLocal;
                            }

                            // set when the last slot finishes
                            if (cs.LatestTime.HasValue)
                            {
                                // set if later
                                if (endLocal > cs.LatestTime)
                                {
                                    cs.LatestTime = endLocal;
                                }
                            }
                            else
                            {
                                // set forst time
                                cs.LatestTime = endLocal;
                            }

                            //--------------------------------------------------
                            // for now - don't show slots outside of 6am to 8pm
                            if (startLocal.Hour > 6 && endLocal.Hour <= 20 && endLocal.Hour != 0)
                            {
                                _clinicResourcesWithSlots.Add(r);

                                // we found a slot that fits so, onto next resource
                                break;
                            }
                        }
                    }
                }
            }
        }

        ClinicSlots ShowResourceSlots(Resource r, DateTime dtUtc)
        {
            ClinicSlots cs = null;
            short dayOfWeek = (short)dtUtc.DayOfWeek;

            // get highest priority scheduling policy for the day of the week that covers the date passed
            // order by descending priority and the first one will be the highest
            var policy = (from p in r.SchedulingPolicies
                          where p.DayOfWeek == dayOfWeek &&
                    dtUtc >= p.StartEffectDate &&
                    dtUtc <= p.EndEffectDate
                          orderby p.Priority descending
                          select p).FirstOrDefault();

            if (policy != null)
            {
                if (policy.Slots.Count > 0)
                {
                    cs = new ClinicSlots();

                    // date passed to us falls within the effective period and is on the day in question
                    foreach (var slot in (from s in policy.Slots orderby s.Time select s).ToList())
                    {
                        // add slot to display
                        //--------------------
                        // start and end times
                        DateTime startLocal = TimeZoneInfo.ConvertTimeFromUtc(dtUtc.Date.Add(slot.Time), _tz);
                        DateTime endLocal = startLocal.AddMinutes(slot.Length);

                        // add in the first time as it is the earliest, since we sorted the slots by time ASC
                        if (!cs.EarliestTime.HasValue)
                        {
                            cs.EarliestTime = startLocal;
                        }

                        // set when the last slot finishes
                        if (cs.LatestTime.HasValue)
                        {
                            // set if later
                            if (endLocal > cs.LatestTime)
                            {
                                cs.LatestTime = endLocal;
                            }
                        }
                        else
                        {
                            // set forst time
                            cs.LatestTime = endLocal;
                        }


                        //--------------------------------------------------
                        // for now - don't show slots outside of 6am to 8pm
                        // need to check the end hour isn't zero - midnight
                        if (startLocal.Hour > 6 && endLocal.Hour <= 20 && endLocal.Hour != 0)
                        {
                            Panel p = new Panel();
                            // set the ID to the provider plus the start time - unique and can be used to identify later
                            p.ID = r.Id.ToString() + "-" + startLocal.TimeOfDay.Minutes.ToString();

                            p.CssClass = "divClinicSlot";
                            p.Style.Add(HtmlTextWriterStyle.Top, (startLocal.TimeOfDay.TotalMinutes - (clinicStartHour * 60)).ToString() + "px");
                            p.Style.Add(HtmlTextWriterStyle.Height, slot.Length.ToString() + "px");
                            p.Style.Add("line-height", slot.Length.ToString() + "px");

                            Label lbl = new Label();
                            string apptTypeName = string.Empty;

                            if (slot.AppointmentType != null)
                            {
                                apptTypeName = slot.AppointmentType.Name;
                            }
                            else
                            {
                                apptTypeName = "Open appointment type";
                            }

                            lbl.Text = "Open";
                            lbl.ToolTip = startLocal.ToString("HH:mm") + " to " + endLocal.ToString("HH:mm") +
                                " " + apptTypeName + " - Capacity:" + slot.Capacity.ToString();
                            p.Controls.Add(lbl);

                            TableCell td = _resourceCells[r.Id];
                            td.Controls.Add(p);
                        }
                    }
                }
            }

            return cs;
        }

        void ShowAppointment(Resource r, Appointment a)
        {
            DateTime startLocal = TimeZoneInfo.ConvertTimeFromUtc(a.Time, _tz);
            DateTime endLocal = startLocal.AddMinutes(a.Length);
            //--------------------------------------------------
            // for now - don't show appts outside of 6am to 8pm - and not ending at midnight
            if (startLocal.Hour > 6 && endLocal.Hour <= 20 && endLocal.Hour != 0)
            {
                Panel p = new Panel();
                p.CssClass = "divClinicAppointment";

                p.Style.Add(HtmlTextWriterStyle.Top, (startLocal.TimeOfDay.TotalMinutes - (clinicStartHour * 60)).ToString() + "px");
                p.Style.Add(HtmlTextWriterStyle.Height, a.Length.ToString() + "px");
                p.Style.Add("line-height", a.Length.ToString() + "px");

                Label lbl = new Label();
                lbl.Text = "Booked";
                lbl.ToolTip = startLocal.ToString("HH:mm") + " to " + endLocal.ToString("HH:mm") + " " + a.Patient.Person.GetFullName();
                p.Controls.Add(lbl);

                //HyperLink h = new HyperLink();
                //p.Controls.Add(h);
                //h.NavigateUrl = "~/Patient/PatientAppointmentList.aspx?appointmentId=" + a.Id.ToString();

                //HtmlGenericControl div = new HtmlGenericControl("div");
                //h.Controls.Add(div);
                //div.InnerHtml = a.Patient.Person.GetFullName();

                //h.ToolTip = startLocal.ToString("HH:mm") + " to " + endLocal.ToString("HH:mm") + " " + a.Patient.Person.GetFullName();

                //if (a.AppointmentType != null)
                //{
                //    div.InnerHtml += "<br />" + a.AppointmentType.Name;
                //    h.ToolTip += " (" + a.AppointmentType.Name + ")";
                //}

                //h.ToolTip += " - Click to view in Patient's appointment list";

                //            p.Controls.Add(h);

                TableCell td = _resourceCells[r.Id];
                td.Controls.Add(p);
            }
        }

        /// <summary>
        /// Show a list of the types of resource (which there will always be)
        /// </summary>
        void LoadResourceTypesDdl()
        {
            // get the resource types
            ExtensionMethods.FillDropDownListWithEnumValues(typeof(ResourceType), ddlResourceType);

            // make first item the selected one
            if (ddlResourceType.Items.Count > 0)
                ddlResourceType.SelectedIndex = 0;
        }

        void LoadFacilitiesDdl()
        {
            try
            {
                Factory factory = Helper.GetFactory(Page);
                var facilityService = factory.GetFacilityService();

                int siteId = Helper.GetUserSession(Page).CurrentSite.Id;

                ddlFacility.DataValueField = "Id";
                ddlFacility.DataTextField = "Name";
                ddlFacility.DataSource = (from f in facilityService.GetAll(siteId)
                                          orderby f.Name
                                          select new
                                          {
                                              Id = f.Id,
                                              Name = f.Name
                                          }).ToList();
                ddlFacility.DataBind();

                if (ddlFacility.Items.Count > 0)
                    ddlFacility.SelectedIndex = 0;

                ShowClinicsForFacility();
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void ShowClinicsForFacility()
        {
            ddlClinic.Items.Clear();

            if (ddlFacility.SelectedValueAsInt() > 0)
            {
                Factory factory = Helper.GetFactory(Page);
                var sectionService = factory.GetSectionService();

                ddlClinic.DataValueField = "Id";
                ddlClinic.DataTextField = "Name";

                // Show ONLY active clicnics
                ddlClinic.DataSource = (from s in sectionService.GetAll(ddlFacility.SelectedValueAsInt())
                                        where s.Active==true
                                        orderby s.Name
                                        select new
                                        {
                                            Id = s.Id,
                                            Name = s.Name
                                        }).ToList();
                ddlClinic.DataBind();
            }

            if (ddlClinic.Items.Count == 0)
            {
                ddlClinic.Items.Clear();
                ddlClinic.Items.Add(new ListItem("No clinics", "-1"));
            }
        }

        protected void ddlFacility_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                ShowClinicsForFacility();
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void ddlClinic_SelectedIndexChanged(object sender, EventArgs e)
        {
        }

        protected void btnPrevious_Click(object sender, EventArgs e)
        {
            TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();
            tbSelectDate.Text = (DateTime.Parse(tbSelectDate.Text).AddDays(-1)).ToString("dd-MMM-yyyy");
        }

        protected void btnThis_Click(object sender, EventArgs e)
        {
            TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();
            tbSelectDate.Text = TimeZoneInfo.ConvertTimeFromUtc(DateTime.UtcNow, tz).Date.ToString("dd-MMM-yyyy");
        }

        protected void tbSelectDate_TextChanged(object sender, EventArgs e)
        {
        }

        protected void btnNext_Click(object sender, EventArgs e)
        {
            TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();
            tbSelectDate.Text = (DateTime.Parse(tbSelectDate.Text).AddDays(1)).ToString("dd-MMM-yyyy");
        }

        protected void ddlResourceType_SelectedIndexChanged(object sender, EventArgs e)
        {
        }
    }

    public class ResourceScheduleItem
    {
        public int ResourceId { get; set; }
        public string ResourceName { get; set; }
        public int AppointmentId { get; set; }
        public DateTime TimeLocalTimeZone { get; set; }
        public int LengthMinutes { get; set; }
    }

    public class ClinicSlots
    {
        public DateTime? EarliestTime { get; set; }
        public DateTime? LatestTime { get; set; }
    }
}