﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Data;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.HtmlControls;
using Shared.Model;
using BT.Health.SchedulingWeb.UISession;
using MedRed.Services;

namespace BT.Health.SchedulingWeb
{
    static public class ExtensionMethods
    {

        static public void ShowSiteChangedBanner(System.Web.UI.Page page)
        {
            string script = "var _vistaSiteChanged = true;";
            page.ClientScript.RegisterClientScriptBlock(typeof(Page), "ShowSiteChangedBanner", script, true);
        }

        /// <summary>
        /// Gets the priority of a patient based on Service Connectivity and Percentage disability
        /// ONLY availble in service layer calls...
        /// Patient Get(string vistaSiteId, string dfn);
        /// Patient GetFullDetails(string vistaSiteId, int id);
        /// </summary>
        /// <param name="patient"></param>
        /// <returns></returns>
        public static int Priority(this Shared.Model.Patient patient)
        {
            if (patient.ServiceConnected)
            {
                return 1;
            }
            else
            {
                if (patient.SCPercent > 50)
                {
                    return 1;
                }
            }

            return 0;
        }

        /// <summary>
        /// Log the exception and any  inner exception to a repository (file, database, etc.)
        /// </summary>
        /// <param name="ex"></param>
        public static void Log(this Exception ex)
        {
            try
            {
                if (HttpContext.Current != null && HttpContext.Current.Server != null)
                {
                    string userName = "User Unknown";

                    if (HttpContext.Current.User != null && HttpContext.Current.User.Identity != null)
                        userName = HttpContext.Current.User.Identity.Name;

                    string filename = DateTime.UtcNow.ToString("yyyy-MM-dd HH-mm-ss") +
                                      " User[" + userName +
                                      "] - Exception " +
                                      " " +
                                      Guid.NewGuid().ToString("D") +
                                      ".txt";

                    using (var sw = System.IO.File.CreateText(HttpContext.Current.Server.MapPath("~/Exceptions/" + filename)))
                    {
                        sw.WriteLine("VA Exception Log Entry");
                        sw.WriteLine("----------------------");
                        sw.WriteLine();
                        sw.WriteLine();
                        sw.WriteLine("Exception");
                        sw.WriteLine("---------");

                        LogException(ex, sw);

                        if (ex.InnerException != null)
                        {
                            sw.WriteLine();
                            sw.WriteLine();
                            sw.WriteLine("Inner Exception");
                            sw.WriteLine("---------------");
                            LogException(ex.InnerException, sw);
                        }

                        sw.Flush();
                        sw.Close();
                    }
                }
            }
            catch
            {
            }
        }

        public static void LogException(Exception ex, System.IO.StreamWriter sw)
        {
            sw.WriteLine();
            sw.WriteLine();
            sw.WriteLine("Source");
            sw.WriteLine("------");
            sw.WriteLine(ex.Source);

            sw.WriteLine();
            sw.WriteLine();
            sw.WriteLine("Target Site");
            sw.WriteLine("-----------");
            sw.WriteLine(ex.TargetSite);

            sw.WriteLine();
            sw.WriteLine();
            sw.WriteLine("Message");
            sw.WriteLine("-------");
            sw.WriteLine(ex.Message);

            sw.WriteLine();
            sw.WriteLine();
            sw.WriteLine("Stack Trace");
            sw.WriteLine("-----------");
            sw.WriteLine(ex.StackTrace);
        }

        public static Control RecursiveFindControl(this Control c, string id)
        {
            if (c.ID == id)
                return c;

            foreach (Control cChild in c.Controls)
            {
                if (cChild.ID == id)
                    return cChild;

                Control childControl = RecursiveFindControl(cChild, id);

                if (childControl != null)
                    return childControl;

            }

            return null;
        }

        //---------------------------------------

        /// <summary>
        /// Gets the date as a medium format string dd-MMM-yyyy HH:mm
        /// </summary>
        /// <param name="dt"></param>
        /// <returns></returns>
        public static string ToMediumFormat(this DateTime dt)
        {
            return dt.ToString("dd-MMM-yyyy HH:mm");
        }



        /// <summary>
        /// Gets the date supplied as UTC in medium format string dd-MMM-yyyy HH:mm of the Time Zone Info
        /// </summary>
        /// <param name="dt"></param>
        /// <param name="tz"></param>
        /// <returns></returns>
        public static string ToMediumFormatFromUtc(this DateTime dt, TimeZoneInfo tz)
        {
            return TimeZoneInfo.ConvertTimeFromUtc(dt, tz).ToMediumFormat();
        }

        /// <summary>
        /// Gets the date supplied as UTC in short format string dd-MMM-yyyy of the Time Zone Info
        /// </summary>
        /// <param name="dt"></param>
        /// <param name="tz"></param>
        /// <returns></returns>
        public static string ToShortFormatFromUtc(this DateTime dt, TimeZoneInfo tz)
        {
            return TimeZoneInfo.ConvertTimeFromUtc(dt, tz).ToShortDateString();
        }


        public static DateTime GetNowForTimeZone(this TimeZoneInfo tz)
        {
            return TimeZoneInfo.ConvertTimeFromUtc(DateTime.UtcNow, tz);
        }

        /// <summary>
        /// Gets the date supplied as UTC in medium format string dd-MMM-yyyy HH:mm of the Time Zone Info
        /// </summary>
        /// <param name="dt"></param>
        /// <param name="tz"></param>
        /// <returns></returns>
        public static string ToMediumFormatFromUtcFromNullableDateTime(this DateTime? dt, TimeZoneInfo tz)
        {
            if (dt != null)
            {
                DateTime d = (DateTime)dt;
                return TimeZoneInfo.ConvertTimeFromUtc(d, tz).ToMediumFormat();
            }
            else
            {
                return "";
            }
        }

        /// <summary>
        /// Get a timespan in the local time zone from one in UTC
        /// </summary>
        /// <param name="timeInUtcOffset"></param>
        /// <param name="tz"></param>
        /// <returns></returns>
        public static TimeSpan GetLocalFtomUtc(this TimeSpan timeInUtcOffset, TimeZoneInfo tz)
        {
            // We work in comparison to midnight (any day will do)
            // Get the UTC timespan offset for the zone passed to us
            TimeSpan timezoneOffset = tz.GetUtcOffset(DateTime.Today);

            // Add the offset to move the time from a time in UTC to a time in the timezone
            return timeInUtcOffset + timezoneOffset;
        }

        /// <summary>
        /// Get a timespan in UTC from one in the local timezone
        /// </summary>
        /// <param name="timeLocal"></param>
        /// <param name="tz"></param>
        /// <returns></returns>
        public static TimeSpan GetUtcFromLocal(this TimeSpan timeLocal, TimeZoneInfo tz)
        {
            // We work in comparison to midnight (any day will do)
            // Get the UTC timespan offset for the zone passed to us
            TimeSpan timezoneOffset = tz.GetUtcOffset(DateTime.Today);

            // Subtract the offset to move the time from a time in the timezone to UTC
            return timeLocal - timezoneOffset;
        }

        /// <summary>
        /// Gets midnight for the first day of the week
        /// </summary>
        /// <param name="dt"></param>
        /// <returns></returns>
        public static DateTime ToStartOfWeekDate(this DateTime dt)
        {
            return (dt.AddDays(DayOfWeek.Monday - dt.DayOfWeek)).Date;
        }

        /// <summary>
        /// Gets the text as a date (time is 00:00:00) - null if not a valid date
        /// </summary>
        /// <param name="tb"></param>
        /// <returns></returns>
        static public DateTime? ToDate(this TextBox tb)
        {
            DateTime dt = DateTime.MinValue;

            if (DateTime.TryParse(tb.Text, out dt))
                return dt.Date;
            else
                return null;
        }

        /// <summary>
        /// Gets the text as a date-time - null if not a valid date-time
        /// </summary>
        /// <param name="tb"></param>
        /// <returns></returns>
        static public DateTime? ToDateTime(this TextBox tb)
        {
            DateTime dt = DateTime.MinValue;

            if (DateTime.TryParse(tb.Text, out dt))
                return dt;
            else
                return null;
        }

        /// <summary>
        /// Gets the text as an int - null if not a valid int
        /// </summary>
        /// <param name="tb"></param>
        /// <returns></returns>
        static public int? ToInt(this TextBox tb)
        {
            int val = 0;

            if (int.TryParse(tb.Text, out val))
                return val;
            else
                return null;
        }

        /// <summary>
        /// Gets the TimeZoneInfo Object for a site
        /// </summary>
        /// <param name="site"></param>
        /// <returns></returns>
        public static TimeZoneInfo GetTimeZoneInfo(this Shared.Model.Site site)
        {
            if (site != null)
                return TimeZoneInfo.FindSystemTimeZoneById(site.TimeZoneId);
            else
                return null;
        }

        //---------------------------------------
        /// <summary>
        /// Gets a full name for a person
        /// </summary>
        /// <param name="p"></param>
        /// <returns></returns>
        public static string GetFullName(this Shared.Model.Person p)
        {
            if (p.MiddleName != string.Empty)
                return p.FirstName + " " + p.MiddleName + " " + p.LastName;
            else
                return p.FirstName + " " + p.LastName;
        }

        /// <summary>
        /// Gets a full name for a patient search result
        /// </summary>
        /// <param name="p"></param>
        /// <returns></returns>
        public static string GetFullName(this Shared.Model.PatientSearchResult p)
        {
            if (p.MiddleName != string.Empty)
                return p.FirstName + " " + p.MiddleName + " " + p.LastName;
            else
                return p.FirstName + " " + p.LastName;
        }

        /// <summary>
        /// Gets the selected value as an integer
        /// Assumes the value is an int, otherwise returns zero
        /// </summary>
        static public int SelectedValueAsInt(this DropDownList ddl)
        {
            if (ddl.SelectedIndex >= 0)
            {
                int val = 0;
                int.TryParse(ddl.SelectedValue, out val);
                return val;
            }

            return 0;
        }

        /// <summary>
        /// Sets a DDL value from INT passed - returns false if not set
        /// </summary>
        /// <param name="ddl"></param>
        /// <param name="val"></param>
        /// <returns></returns>
        static public bool SetSelectedValueFromInt(this DropDownList ddl, int val)
        {
            ListItem li = ddl.Items.FindByValue(val.ToString());

            if (li != null)
            {
                ddl.SelectedIndex = -1;
                li.Selected = true;
                return true;
            }

            return false;
        }


        /// <summary>
        /// Add a communication template queue item
        /// </summary>
        /// <param name="appointment">The appointment</param>
        /// <param name="page">The current Page</param>
        static public void AddACommunicationTemplate(Appointment appointment, CommunicationType commsType, Page page)
        {
            CommunicationTemplate ct = appointment.Section.CommunicationTemplates.Where(a => a.Type == commsType).FirstOrDefault();
            if (ct == null)
                ct = appointment.Section.Facility.CommunicationTemplates.Where(a => a.Type == commsType).FirstOrDefault();

            if (ct != null)
            {
                Factory factory = Helper.GetFactory(page);
                var appointmentService = factory.GetAppointmentService();
                appointmentService.AddToCommunicationQueue(ct, appointment);
            }
        }

        /// <summary>
        /// Add a communication template queue item
        /// </summary>
        /// <param name="appointment">The appointment</param>
        /// <param name="page">The current Page</param>
        static public void AddACommunicationTemplate(AppointmentRequest appointment, CommunicationType commsType, Page page, Section clinic)
        {

            Factory factory = Helper.GetFactory(page);
            MedRed.Services.Interfaces.IFacilityService facilityService = factory.GetFacilityService();

            MedRed.Services.Interfaces.ISectionService sectionService = factory.GetSectionService();
            Section sec = sectionService.Get(clinic.Id);

            CommunicationTemplate ct = sec.CommunicationTemplates.Where(a => a.Type == commsType).FirstOrDefault();
            if (ct == null)
                ct = sec.Facility.CommunicationTemplates.Where(a => a.Type == commsType).FirstOrDefault();

            if (ct != null)
            {
                var appointmentService = factory.GetAppointmentService();
                appointmentService.AddToCommunicationQueue(ct, appointment);
            }
        }


        /// <summary>
        /// Gets a list of ENUM name/value pairs for the requested enym type
        /// </summary>
        /// <param name="enumType"></param>
        /// <returns></returns>
        static public List<EnumValue> GetEnumValues(Type enumType)
        {
            // Not strictly an extension class, but an ENUM helper
            var list = new List<EnumValue>();

            var names = Enum.GetNames(enumType);


            foreach (string n in names)
                list.Add(new EnumValue { Value = (int)Enum.Parse(enumType, n), Name = n });

            return list;
        }

        /// <summary>
        /// Fills a dro down list with ENUM name/value pairs for the requested enym type
        /// </summary>
        /// <param name="enumType"></param>
        /// <param name="ddl"></param>
        static public void FillDropDownListWithEnumValues(Type enumType, DropDownList ddl)
        {
            ddl.DataTextField = "Name";
            ddl.DataValueField = "Value";
            ddl.DataSource = GetEnumValues(enumType);
            ddl.DataBind();
        }

        /// <summary>
        /// Fills a dro down list with ENUM name/value pairs for the requested enym type
        /// Selects the Name requried (the string part of the name/value pair)
        /// </summary>
        /// <param name="enumType"></param>
        /// <param name="ddl"></param>
        static public void FillDropDownListWithEnumValues(Type enumType, DropDownList ddl, string selectedName)
        {
            ddl.DataTextField = "Name";
            ddl.DataValueField = "Value";
            ddl.DataSource = GetEnumValues(enumType);
            ddl.DataBind();

            if (selectedName != null)
            {
                ddl.SelectedIndex = -1;

                foreach (ListItem li in ddl.Items)
                {
                    if (li.Text == selectedName)
                    {
                        li.Selected = true;
                        return;
                    }
                }
            }

            if (ddl.Items.Count > 0)
                ddl.SelectedIndex = 0;
        }

        /// <summary>
        /// Fills a dro down list with ENUM name/value pairs for the requested enym type
        /// Selects the Value requried (the integer part of the name/value pair)
        /// </summary>
        /// <param name="enumType"></param>
        /// <param name="ddl"></param>
        static public void FillDropDownListWithEnumValues(Type enumType, DropDownList ddl, int selectedValue)
        {
            ddl.DataTextField = "Name";
            ddl.DataValueField = "Value";
            ddl.DataSource = GetEnumValues(enumType);
            ddl.DataBind();

            ddl.SelectedIndex = -1;
            string valueSstring = selectedValue.ToString();

            foreach (ListItem li in ddl.Items)
            {
                if (li.Value == valueSstring)
                {
                    li.Selected = true;
                    return;
                }
            }

            if (ddl.Items.Count > 0)
                ddl.SelectedIndex = 0;
        }

        static public void FillWithRecheduleOptions(DropDownList ddl)
        {
            var list = new List<EnumValue>();

            list.Add(new EnumValue { Value = (int)AppointmentRelationType.RequiredAfter, Name = AppointmentRelation.DisplayText(AppointmentRelationType.RequiredAfter) });
            list.Add(new EnumValue { Value = (int)AppointmentRelationType.RequiredBefore, Name = AppointmentRelation.DisplayText(AppointmentRelationType.RequiredBefore) });
            list.Add(new EnumValue { Value = (int)AppointmentRelationType.TravelConvenience, Name = AppointmentRelation.DisplayText(AppointmentRelationType.TravelConvenience) });

            ddl.DataTextField = "Name";
            ddl.DataValueField = "Value";
            ddl.DataSource = list;
            ddl.DataBind();
        }

    }

    /// <summary>
    /// Class containing ENUM value/text pairs
    /// </summary>
    public class EnumValue
    {
        public EnumValue()
        {
        }

        public int Value { get; set; }
        public string Name { get; set; }
    }
}