﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;

using MedRed.Services;
using MedRed.Services.Utils;
using Shared.Model.Config.MDWS;
using Shared.Model;

using BT.Health.SchedulingWeb.UISession;
namespace BT.Health.SchedulingWeb.Administration.Users
{
    public partial class Users : System.Web.UI.Page
    {
        /// <summary>
        /// true to refresh the user list
        /// </summary>
        bool _refreshData = false;

        protected void Page_Load(object sender, EventArgs e)
        {
            lblMessage.Text = string.Empty;

            if (!IsPostBack)
            {
                _refreshData = true;
            }
        }

        protected void Page_PreRender(object sender, EventArgs e)
        {
            if (_refreshData)
            {
                LoadUsers();
            }
        }

        void LoadUsers()
        {
            try
            {
                Factory factory = Factory.GetFactory(MDWSConfiguration_ConfigHelper.GetDefaultConfig());
                var service = factory.GetAccountService();
                lvUsers.DataSource = (from u in service.GetUsers()
                                      orderby u.Username
                                      select u).ToList();
                lvUsers.DataBind();
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }


        protected void lvUsers_ItemDataBound(object sender, ListViewItemEventArgs e)
        {
            if (e.Item.ItemType == ListViewItemType.DataItem)
            {
                Shared.Model.User u = e.Item.DataItem as Shared.Model.User;
                Image imgAdmin = e.Item.RecursiveFindControl("imgAdmin") as Image;
                LinkButton lbUser = e.Item.RecursiveFindControl("lbUser") as LinkButton;
                Label lblAdminUser = e.Item.RecursiveFindControl("lblAdminUser") as Label;

                LinkButton lbChangePassword = e.Item.RecursiveFindControl("lbChangePassword") as LinkButton;
                LinkButton lbViewRoles = e.Item.RecursiveFindControl("lbViewRoles") as LinkButton;
                LinkButton lbDeleteUser = e.Item.RecursiveFindControl("lbDeleteUser") as LinkButton;

                if (u != null && imgAdmin != null &&
                    lbUser != null && lblAdminUser != null && lbChangePassword != null &&
                    lbViewRoles != null && lbDeleteUser != null)
                {
                    string commandArgument = u.Id.ToString();

                    lbViewRoles.CommandName = "ViewUserRoles";
                    lbViewRoles.CommandArgument = commandArgument;

                    var adminRole = (from a in u.Roles
                                     where a.Role.Name == "Administrator"
                                     select a).FirstOrDefault();

                    if (adminRole != null)
                    {
                        // user is admin - this is fixed in start-up scripts so cannot edit
                        lblAdminUser.Visible = true;
                        lbUser.Visible = false;

                        lblAdminUser.Text = u.Username;

                        imgAdmin.ImageUrl = "~/graphics/user.png";
                        imgAdmin.ToolTip = "User is an administrator and cannot be edited";
                        imgAdmin.AlternateText = "User is an administrator";

                        lbChangePassword.Visible = false;
                        lbDeleteUser.Visible = false;
                        lbViewRoles.Visible = false;
                    }
                    else
                    {
                        // user is not an admin user and can be edited
                        lblAdminUser.Visible = false;
                        lbUser.Visible = true;

                        lbUser.Text = u.Username;
                        lbUser.ToolTip = "Edit " + u.Username;
                        lbUser.CommandName = "EditUser";
                        lbUser.CommandArgument = commandArgument;

                        imgAdmin.ImageUrl = "~/graphics/empty.png";
                        imgAdmin.AlternateText = "Standard user";

                        lbDeleteUser.CommandName = "DeleteUser";
                        lbDeleteUser.CommandArgument = commandArgument;

                        lbChangePassword.CommandName = "ChangeUserPassword";
                        lbChangePassword.CommandArgument = commandArgument;
                    }
                }
            }
        }

        protected void lvUsers_ItemCommand(object sender, ListViewCommandEventArgs e)
        {
            try
            {
                int id = 0;
                int.TryParse(e.CommandArgument.ToString(), out id);

                switch (e.CommandName)
                {
                    case "EditUser":
                        EditUser(id);
                        break;

                    case "ChangeUserPassword":
                        ChangeUserPassword(id);
                        break;

                    case "ViewUserRoles":
                        Response.Redirect("~/Administration/Users/UsersInRoles.aspx?userid=" + id.ToString(), false);
                        break;

                    case "DeleteUser":
                        DeleteUser(id);
                        break;

                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void DeleteUser(int id)
        {
            try
            {
                ClearEditControls();

                Factory factory = Factory.GetFactory(MDWSConfiguration_ConfigHelper.GetDefaultConfig());
                var service = factory.GetAccountService();

                var user = (from u in service.GetUsers()
                            where u.Id == id
                            select u).FirstOrDefault();

                if (user != null)
                {
                    string userName = user.Username;

                    service.DeleteUser(id);

                    lblMessage.Text = userName + " deleted";
                    _refreshData = true;
                }
                else
                {
                    lblMessage.Text = "Unable to find user";
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void EditUser(int id)
        {
            try
            {
                ClearEditControls();

                Factory factory = Factory.GetFactory(MDWSConfiguration_ConfigHelper.GetDefaultConfig());
                var service = factory.GetAccountService();

                var user = (from u in service.GetUsers()
                            where u.Id == id
                            select u).FirstOrDefault();

                if (user != null)
                {
                    tbFirstName.Text = user.Person.FirstName;
                    tbLastName.Text = user.Person.LastName;
                    tbUserName.Text = user.Username;

                    // there is a separate view to edit the password
                    trPassword.Visible = false;

                    EditUserId = id;
                    lblAddEdit.Text = "Edit User";
                    mv.ActiveViewIndex = 1;
                    tbUserName.Focus();
                }
                else
                {
                    lblMessage.Text = "Unable to find user";
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void ChangeUserPassword(int id)
        {
            try
            {
                ClearEditControls();

                Factory factory = Factory.GetFactory(MDWSConfiguration_ConfigHelper.GetDefaultConfig());
                var service = factory.GetAccountService();

                var user = (from u in service.GetUsers()
                            where u.Id == id
                            select u).FirstOrDefault();

                if (user != null)
                {
                    EditUserId = id;
                    mv.ActiveViewIndex = 2;
                }
                else
                {
                    lblMessage.Text = "Unable to find user";
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void SaveNewUser()
        {
            try
            {
                string newUserNameTrimmed = tbUserName.Text.Trim();

                if (newUserNameTrimmed == string.Empty)
                {
                    lblMessage.Text = "New User name cannot be empty";
                    return;
                }

                if (tbPassword.Text == string.Empty)
                {
                    lblMessage.Text = "Password cannot be empty";
                    return;
                }

                string newFirstNameTrimmed = tbFirstName.Text.Trim();

                if (newFirstNameTrimmed == string.Empty)
                {
                    lblMessage.Text = "First name cannot be empty";
                    return;
                }

                string newLastNameTrimmed = tbLastName.Text.Trim();

                if (newLastNameTrimmed == string.Empty)
                {
                    lblMessage.Text = "Last name cannot be empty";
                    return;
                }

                Factory factory = Factory.GetFactory(MDWSConfiguration_ConfigHelper.GetDefaultConfig());
                var accountService = factory.GetAccountService();

                // get all users to check name unique
                List<User> allUsers = accountService.GetUsers().ToList();

                if ((from u in allUsers
                     where u.Username.ToLower() == newUserNameTrimmed.ToLower()
                     select u).FirstOrDefault() == null)
                {
                    // user does not exist, so we can add
                    User newUser = new User()
                    {
                        Username = newUserNameTrimmed,
                        Person = new Person()
                        {
                            FirstName = newFirstNameTrimmed,
                            LastName = newLastNameTrimmed
                        }
                    };

                    PasswordHelper.SetUserPassword(ref newUser, tbPassword.Text);

                    accountService.AddUser(newUser);

                    ClearEditControls();

                    lblMessage.Text = newUserNameTrimmed + " added";
                    mv.ActiveViewIndex = 0;
                    _refreshData = true;
                }
                else
                {
                    lblMessage.Text = "A user with this name already exists";
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void SaveExistingUser(int userId)
        {
            try
            {
                Factory factory = Factory.GetFactory(MDWSConfiguration_ConfigHelper.GetDefaultConfig());
                var service = factory.GetAccountService();

                var user = (from u in service.GetUsers()
                            where u.Id == userId
                            select u).FirstOrDefault();

                if (user != null)
                {
                    string userNameTrimmed = tbUserName.Text.Trim();

                    if (userNameTrimmed == string.Empty)
                    {
                        lblMessage.Text = "New User name cannot be empty";
                        return;
                    }

                    string firstNameTrimmed = tbFirstName.Text.Trim();

                    if (firstNameTrimmed == string.Empty)
                    {
                        lblMessage.Text = "First name cannot be empty";
                        return;
                    }

                    string lastNameTrimmed = tbLastName.Text.Trim();

                    if (lastNameTrimmed == string.Empty)
                    {
                        lblMessage.Text = "Last name cannot be empty";
                        return;
                    }

                    // get all users (except the current user being edited) to check name unique
                    var accountService = factory.GetAccountService();
                    var allUsersExceptEditingUser = (from a in accountService.GetUsers()
                                                     where a.Id != userId
                                                     select a).ToList();

                    if ((from u in allUsersExceptEditingUser
                         where u.Username.ToLower() == userNameTrimmed.ToLower()
                         select u).FirstOrDefault() ==
                         null)
                    {
                        // user name has not ben changed to some other user's name
                        user.Username = userNameTrimmed;
                        user.Person.FirstName = firstNameTrimmed;
                        user.Person.LastName = lastNameTrimmed;
                        service.UpdateUser(user);

                        ClearEditControls();

                        lblMessage.Text = userNameTrimmed + " updated";
                        mv.ActiveViewIndex = 0;
                        _refreshData = true;
                    }
                    else
                    {
                        lblMessage.Text = "A user with this name already exists";
                    }
                }
                else
                {
                    lblMessage.Text = "User not found";
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void ClearEditControls()
        {
            tbFirstName.Text = string.Empty;
            tbLastName.Text = string.Empty;
            tbPassword.Text = string.Empty;
            tbUserName.Text = string.Empty;
            tbChangedPassword.Text = string.Empty;
        }

        protected void btnShowAddUser_Click(object sender, EventArgs e)
        {
            EditUserId = 0;
            trPassword.Visible = true;
            ClearEditControls();
            lblAddEdit.Text = "Add User";
            mv.ActiveViewIndex = 1;
            tbUserName.Focus();
        }

        protected void btnCancel_Click(object sender, EventArgs e)
        {
            mv.ActiveViewIndex = 0;
        }

        protected void btnSave_Click(object sender, EventArgs e)
        {
            int userId = this.EditUserId;

            if (userId > 0)
                SaveExistingUser(userId);
            else
                SaveNewUser();
        }

        /// <summary>
        /// Gets or sets the edit user Id
        /// Zero implies a new user
        /// </summary>
        int EditUserId
        {
            get
            {
                int id = 0;
                int.TryParse(hfEditUserId.Value, out id);
                return id;
            }

            set
            {
                hfEditUserId.Value = value.ToString();
            }
        }

        protected void btnSaveNewPassword_Click(object sender, EventArgs e)
        {
            try
            {
                int userId = EditUserId;

                if (userId > 0)
                {
                    Factory factory = Helper.GetFactory(Page);
                    var service = factory.GetAccountService();

                    var user = (from u in service.GetUsers()
                                where u.Id == userId
                                select u).FirstOrDefault();

                    if (user != null)
                    {
                        if (tbChangedPassword.Text == string.Empty)
                        {
                            lblMessage.Text = "Password cannot be empty";
                            return;
                        }

                        PasswordHelper.SetUserPassword(ref user, tbChangedPassword.Text);
                        lblMessage.Text = user.Username + " password changed";
                        mv.ActiveViewIndex = 0;
                        _refreshData = true;
                    }
                    else
                    {
                        lblMessage.Text = "User not found";
                    }
                }
                else
                {
                    lblMessage.Text = "No user selected to change password";
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

    }
}