﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using Shared.Model;
using BT.Health.SchedulingWeb.UISession;
using MedRed.Services;

namespace BT.Health.SchedulingWeb.Administration.Schedules
{
    public partial class SlotEdit : System.Web.UI.Page
    {
        /// <summary>
        /// When greater than zero - The schedule id of the schedule being edited.
        /// When zero, we are adding a new policy
        /// </summary>
        public int _policyId = 0;

        /// <summary>
        /// The resoure within the schedule being edited
        /// </summary>
        public int _resourceId = 0;

        /// <summary>
        /// The section (aka clinic) for which the schdeulig policy applies
        /// </summary>
        public int _sectionId = 0;

        /// <summary>
        /// ID of the slot being edited
        /// When zero, we are adding a new slot
        /// </summary>
        public int _slotId = 0;

        protected override void OnInit(EventArgs e)
        {
            base.OnInit(e);
            Page.RegisterRequiresControlState(this);
        }

        protected override object SaveControlState()
        {
            Dictionary<string, object> dic = new Dictionary<string, object>();
            dic.Add("_sectionId", _sectionId);
            dic.Add("_resourceId", _resourceId);
            dic.Add("_policyId", _policyId);
            dic.Add("_slotId", _slotId);
            dic.Add("base", base.SaveControlState());

            return dic;
        }

        protected override void LoadControlState(object savedState)
        {
            Dictionary<string, object> dic = savedState as Dictionary<string, object>;

            if (dic != null)
            {
                if (dic["_sectionId"] != null && dic["_sectionId"] is int)
                    _sectionId = (int)dic["_sectionId"];

                if (dic["_resourceId"] != null && dic["_resourceId"] is int)
                    _resourceId = (int)dic["_resourceId"];

                if (dic["_policyId"] != null && dic["_policyId"] is int)
                    _policyId = (int)dic["_policyId"];

                if (dic["_slotId"] != null && dic["_slotId"] is int)
                    _slotId = (int)dic["_slotId"];

                base.LoadControlState(dic["base"]);
            }
            else
            {
                base.LoadControlState(savedState);
            }
        }

        protected void Page_Load(object sender, EventArgs e)
        {
            lblMessage.Text = string.Empty;

            if (!IsPostBack)
            {
                try
                {
                    // load data from quesrystrings on first GET
                    LoadQuerystringData();

                    // load appointment types on first GET
                    LoadAppointmentTypesDdl();

                    // show edit mode
                    ShowEditUpdateMode();
                }
                catch (System.Exception ex)
                {
                    lblMessage.Text = ex.Message; ex.Log();
                }
            }
        }

        protected void Page_PreRender(object sender, EventArgs e)
        {
            // if adding new slots, show how many the user will add using the current on screen info
            if (_slotId == 0)
            {
                int slotCount = GetSlotsFromStartAndEndTimes().Count;

                if (slotCount > 1)
                    btnAddUpdate.Text = "Add " + slotCount.ToString() + " slots";
                else
                    btnAddUpdate.Text = "Add 1 slot";
            }
        }

        /// <summary>
        /// Load local values from querystring values
        /// </summary>
        void LoadQuerystringData()
        {
            // get editing mode
            if (Request.QueryString["mode"] != null && Request.QueryString["mode"].ToLower() == "edit")
            {
                // assumes edit - if zero, we add new by default
                _slotId = Helper.GetQueryStringInt(Request.QueryString["slotid"]);
            }
            else
            {
                // assumes add new
                _slotId = 0;
            }

            _resourceId = Helper.GetQueryStringInt(Request.QueryString["resourceid"]);
            _sectionId = Helper.GetQueryStringInt(Request.QueryString["sectionid"]);
            _policyId = Helper.GetQueryStringInt(Request.QueryString["policyid"]);
        }

        void LoadAppointmentTypesDdl()
        {
            Factory factory = Helper.GetFactory(Page);
            var sectionService = factory.GetSectionService();
            Section clinic = sectionService.Get(_sectionId);

            ddlAppointmentType.DataSource = (from t in clinic.AppointmentTypes
                                             orderby t.Name
                                             select t).ToList();

            ddlAppointmentType.DataValueField = "Id";
            ddlAppointmentType.DataTextField = "Name";
            ddlAppointmentType.DataBind();

            // add empty item at top of list
            ddlAppointmentType.Items.Insert(0, new ListItem("", ""));
        }

        void ShowEditUpdateMode()
        {
            // when a schedule, we can edit
            if (_slotId > 0)
            {
                ShowSlotEditMode();
            }
            else
            {
                ShowAddNewSlotMode();
            }
        }

        void ShowSlotEditMode()
        {
            // when editing, adding slots is irrelevant
            bodyAddMultiple.Visible = false;

            // initially hide the update button
            btnAddUpdate.Visible = false;

            // dont want auto time shifting of drop downs
            ddlStartTime.AutoPostBack = false;
            ddlEndTime.AutoPostBack = false;
            ddlLengthOfSlot.AutoPostBack = false;

            Factory factory = Helper.GetFactory(Page);
            var resourceService = factory.GetResourceService();
            var resource = resourceService.Get(_resourceId);
            var policy = resource.SchedulingPolicies.Single(p => p.Id == _policyId);

            Slot s = policy.Slots.Single(p => p.Id == _slotId);

            if (s != null)
            {
                txtCapacity.Text = s.Capacity.ToString();

                TimeZoneInfo tz = TimeZoneInfo.FindSystemTimeZoneById(Helper.GetUserSession(Page).CurrentSite.TimeZoneId);
                var selectedStartTimeInLocalTimezone = s.Time.GetLocalFtomUtc(tz);

                ShowSlotTimes(new TimeSpan(6, 0, 0), new TimeSpan(20, 0, 0), 15, ddlStartTime, selectedStartTimeInLocalTimezone);

                if (s.AppointmentType != null)
                {
                    ddlAppointmentType.SelectedIndex = ddlAppointmentType.Items.IndexOf(ddlAppointmentType.Items.FindByValue(s.AppointmentType.Id.ToString()));
                }

                ddlLengthOfSlot.SelectedIndex = ddlLengthOfSlot.Items.IndexOf(ddlLengthOfSlot.Items.FindByValue(s.Length.ToString()));

                btnAddUpdate.Text = "Update";
                btnAddUpdate.Visible = true;
            }
            else
            {
                lblMessage.Text = "Slot not found";
            }
        }

        void ShowAddNewSlotMode()
        {
            // this will get changed in the PreRender when we know how many slots to add
            btnAddUpdate.Text = "Add";

            ShowSlotTimes(new TimeSpan(6, 0, 0), new TimeSpan(20, 0, 0), 15, ddlStartTime, null);
            ShowSlotTimes(new TimeSpan(6, 0, 0), new TimeSpan(20, 0, 0), 15, ddlEndTime, null);
            btnAddUpdate.Visible = true;

            // when editing, adding slots is irrelevant
            bodyAddMultiple.Visible = true;

            // allow auto time shifting of drop downs
            ddlStartTime.AutoPostBack = true;
            ddlEndTime.AutoPostBack = true;
            ddlLengthOfSlot.AutoPostBack = true;
        }

        void ShowSlotTimes(TimeSpan timeStart, TimeSpan timeEnd, int minutesInterval, DropDownList ddl, TimeSpan? selectedTime)
        {
            ddl.Items.Clear();

            TimeSpan t = timeStart;

            while (t < timeEnd)
            {
                // get ticks as a time of day
                var dt = new DateTime(t.Ticks);

                // new list item - show as hours and minutes
                ListItem li = new ListItem(dt.ToString("HH:mm"));

                if (selectedTime.HasValue && t == selectedTime.Value)
                    li.Selected = true;

                ddl.Items.Add(li);

                t += new TimeSpan(0, minutesInterval, 0);
            }
        }

        protected void btnAddUpdate_Click(object sender, EventArgs e)
        {
            try
            {
                int capcity = 0;
                int.TryParse(txtCapacity.Text, out capcity);

                if (capcity < 1)
                {
                    lblMessage.Text = "Capacity must be greater than zero";
                    return;
                }

                if (capcity > 1000)
                {
                    lblMessage.Text = "Capacity must be less than 1000";
                    return;
                }

                Factory factory = Helper.GetFactory(Page);
                var resourceService = factory.GetResourceService();
                var resource = resourceService.Get(_resourceId);
                var policy = resource.SchedulingPolicies.Single(p => p.Id == _policyId);

                var sectionService = factory.GetSectionService();
                var section = sectionService.Get(_sectionId);

                int appointmentTypeId = ddlAppointmentType.SelectedValueAsInt();
                var appointmenttype = section.AppointmentTypes.SingleOrDefault(p => p.Id == appointmentTypeId);

                short length = Convert.ToInt16(ddlLengthOfSlot.SelectedValue);

                if (_slotId > 0)
                {
                    // update existing slot
                    Slot slot = policy.Slots.Single(p => p.Id == _slotId);

                    DateTime sTime = Convert.ToDateTime(ddlStartTime.SelectedItem.ToString());
                    TimeZoneInfo tz = TimeZoneInfo.FindSystemTimeZoneById(Helper.GetUserSession(Page).CurrentSite.TimeZoneId);
                    slot.Time = sTime.TimeOfDay.GetUtcFromLocal(tz);

                    slot.Capacity = capcity;
                    slot.Length = length;

                    if (appointmenttype != null)
                    {
                        slot.AppointmentType = appointmenttype;
                    }

                    // verify no overlap within this schedule
                    string result = VerifyNoOverlap(slot);

                    if (result == string.Empty)
                    {
                        resourceService.Update(resource);
                        lblMessage.Text = "Slot updated";
                    }
                    else
                    {
                        lblMessage.Text = result;
                    }
                }
                else
                {
                    string result = AddSlots(policy, appointmenttype, capcity, length);

                    if (result == string.Empty)
                    {
                        resourceService.Update(resource);

                        // redirect to the slots list page
                        Response.Redirect("~/Administration/Schedules/SlotList.aspx?policyid=" + _policyId.ToString() + "&resourceid=" + _resourceId.ToString() + "&sectionid=" + _sectionId.ToString(),
                            false);
                    }
                    else
                    {
                        lblMessage.Text = result;
                    }
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        string AddSlots(SchedulingPolicy policy, AppointmentType appointmenttype, int capacity, short length)
        {
            TimeZoneInfo tz = TimeZoneInfo.FindSystemTimeZoneById(Helper.GetUserSession(Page).CurrentSite.TimeZoneId);
            var times = GetSlotsFromStartAndEndTimes();

            if (times.Count > 0)
            {
                foreach (var t in times)
                {
                    Slot slot = new Slot();

                    slot.Time = t.GetUtcFromLocal(tz);
                    slot.Capacity = capacity;
                    slot.Length = length;


                    if (appointmenttype != null)
                    {
                        slot.AppointmentType = appointmenttype;
                    }

                    string result = VerifyNoOverlap(slot);

                    if (result == string.Empty)
                    {
                        policy.Slots.Add(slot);
                    }
                    else
                    {
                        return result;
                    }
                }
            }
            else
            {
                return "No slots found";
            }

            return string.Empty;
        }

        protected void ddlStartTime_SelectedIndexChanged(object sender, EventArgs e)
        {
            // only makes sens when adding new slots
            if (_slotId == 0)
            {
                DateTime tStart = Convert.ToDateTime(ddlStartTime.SelectedItem.ToString());
                DateTime tEnd = Convert.ToDateTime(ddlEndTime.SelectedItem.ToString());

                if (tEnd < tStart)
                {
                    // see if we can get the next slot
                    short lengthMinutes = Convert.ToInt16(ddlLengthOfSlot.SelectedValue);
                    DateTime nextSlot = tStart.AddMinutes(lengthMinutes);

                    // find next start
                    ListItem li = ddlEndTime.Items.FindByValue(nextSlot.ToString("HH:mm"));

                    // if not found, find start time
                    if (li == null)
                    {
                        li = ddlEndTime.Items.FindByValue(tStart.ToString("HH:mm"));
                    }

                    // hopefully we found something
                    if (li != null)
                    {
                        ddlEndTime.SelectedIndex = -1;
                        li.Selected = true;
                    }
                }
            }
        }

        protected void ddlEndTime_SelectedIndexChanged(object sender, EventArgs e)
        {
            // only makes sens when adding new slots
            if (_slotId == 0)
            {
                DateTime tStart = Convert.ToDateTime(ddlStartTime.SelectedItem.ToString());
                DateTime tEnd = Convert.ToDateTime(ddlEndTime.SelectedItem.ToString());

                if (tEnd < tStart)
                {
                    // see if we can get the previous slot
                    short lengthMinutes = Convert.ToInt16(ddlLengthOfSlot.SelectedValue);
                    DateTime previousSlot = tEnd.AddMinutes(-lengthMinutes);

                    // find previous start
                    ListItem li = ddlStartTime.Items.FindByValue(previousSlot.ToString("HH:mm"));

                    // if not found, find end time
                    if (li == null)
                    {
                        li = ddlStartTime.Items.FindByValue(tStart.ToString("HH:mm"));
                    }

                    // hopefully we found something
                    if (li != null)
                    {
                        ddlStartTime.SelectedIndex = -1;
                        li.Selected = true;
                    }
                }
            }
        }

        protected void ddlLengthOfSlot_SelectedIndexChanged(object sender, EventArgs e)
        {
            // only makes sense when adding new slots
            if (_slotId == 0)
            {
                DateTime tStart = Convert.ToDateTime(ddlStartTime.SelectedItem.ToString());
                DateTime tEnd = Convert.ToDateTime(ddlEndTime.SelectedItem.ToString());

                // get the next slot
                short lengthMinutes = Convert.ToInt16(ddlLengthOfSlot.SelectedValue);
                DateTime nextSlot = tStart.AddMinutes(lengthMinutes);

                // if the next slot passes the selected end time, see if we can extend the end time
                if (nextSlot > tEnd)
                {
                    // find next end
                    ListItem li = ddlEndTime.Items.FindByValue(nextSlot.ToString("HH:mm"));

                    // hopefully we found something
                    if (li != null)
                    {
                        ddlEndTime.SelectedIndex = -1;
                        li.Selected = true;
                    }
                }
            }
        }

        List<TimeSpan> GetSlotsFromStartAndEndTimes()
        {
            List<TimeSpan> list = new List<TimeSpan>();

            DateTime tStart = Convert.ToDateTime(ddlStartTime.SelectedItem.ToString());
            DateTime tEnd = Convert.ToDateTime(ddlEndTime.SelectedItem.ToString());

            // get the next slot
            short lengthMinutes = Convert.ToInt16(ddlLengthOfSlot.SelectedValue);

            DateTime t = tStart;
            while (t < tEnd)
            {
                // add a time of day to the list
                list.Add(new TimeSpan(t.TimeOfDay.Ticks));

                // next time to check
                t = t.AddMinutes(lengthMinutes);
            }

            return list;
        }

        /// <summary>
        /// Verify that the slot does not overlap with other slots.
        /// A slot ID of less than one assumes a new slot
        /// </summary>
        /// <param name="slotId"></param>
        /// <returns></returns>
        string VerifyNoOverlap(Shared.Model.Slot slotToCheck)
        {
            Factory factory = Helper.GetFactory(Page);
            var resourceService = factory.GetResourceService();
            var resource = resourceService.Get(_resourceId);

            // get policy
            var policy = resource.SchedulingPolicies.SingleOrDefault(p => p.Id == _policyId);

            var startTimeToCheck = slotToCheck.Time;
            var endTimeToCheck = slotToCheck.Time.Add(new TimeSpan(0, slotToCheck.Length, 0));

            // will be ither 0,1, or 2)
            foreach (var slot in policy.Slots)
            {
                // will always work = either the same slot with ID>0 is checked or new slot to check has Id - which no slot in DB has
                if (slot.Id != slotToCheck.Id)
                {
                    var startTime = slot.Time;
                    var endTime = slot.Time.Add(new TimeSpan(0, slot.Length, 0));

                    // starts at anothers time
                    if (startTimeToCheck == startTime)
                    {
                        return GetOverlapErrorForSLot(slot);
                    }

                    // starts between a another slot period
                    if (startTimeToCheck > startTime && startTimeToCheck < endTime)
                    {
                        return GetOverlapErrorForSLot(slot);
                    }

                    // ends between a another slot period
                    if (endTimeToCheck > startTime && endTimeToCheck < endTime)
                    {
                        return GetOverlapErrorForSLot(slot);
                    }

                    // an existing slot starts during the slot to check
                    if (startTime > startTimeToCheck && startTime < endTimeToCheck)
                    {
                        return GetOverlapErrorForSLot(slot);
                    }

                    // an existing slot ends during the slot to check
                    if (endTime > startTimeToCheck && endTime < endTimeToCheck)
                    {
                        return GetOverlapErrorForSLot(slot);
                    }
                }
            }

            // no error
            return string.Empty;
        }

        string GetOverlapErrorForSLot(Slot slot)
        {
            TimeZoneInfo tz = TimeZoneInfo.FindSystemTimeZoneById(Helper.GetUserSession(Page).CurrentSite.TimeZoneId);
            DateTime dtLocalTime = new DateTime(slot.Time.GetLocalFtomUtc(tz).Ticks);

            return "Overlaps with slot starting at " + dtLocalTime.ToString("HH:mm");
        }
    }
}
