﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using Shared.Model;
using BT.Health.SchedulingWeb.UISession;
using MedRed.Services;

namespace BT.Health.SchedulingWeb.Administration.Schedules
{
    public partial class ScheduleEdit : System.Web.UI.Page
    {
        /// <summary>
        /// When greater than zero - The schedule id of the schedule being edited.
        /// WHen zero, we are adding a new policy
        /// </summary>
        public int _policyId = 0;

        /// <summary>
        /// The resoure within the schedule being edited
        /// </summary>
        public int _resourceId = 0;

        /// <summary>
        /// The section (aka clinic) for which the schdeulig policy applies
        /// </summary>
        public int _sectionId = 0;

        protected override void OnInit(EventArgs e)
        {
            base.OnInit(e);
            Page.RegisterRequiresControlState(this);
        }

        protected override object SaveControlState()
        {
            Dictionary<string, object> dic = new Dictionary<string, object>();
            dic.Add("_sectionId", _sectionId);
            dic.Add("_resourceId", _resourceId);
            dic.Add("_policyId", _policyId);
            dic.Add("base", base.SaveControlState());

            return dic;
        }

        protected override void LoadControlState(object savedState)
        {
            Dictionary<string, object> dic = savedState as Dictionary<string, object>;

            if (dic != null)
            {
                if (dic["_sectionId"] != null && dic["_sectionId"] is int)
                    _sectionId = (int)dic["_sectionId"];

                if (dic["_resourceId"] != null && dic["_resourceId"] is int)
                    _resourceId = (int)dic["_resourceId"];

                if (dic["_policyId"] != null && dic["_policyId"] is int)
                    _policyId = (int)dic["_policyId"];

                base.LoadControlState(dic["base"]);
            }
            else
            {
                base.LoadControlState(savedState);
            }
        }

        protected void Page_Load(object sender, EventArgs e)
        {
            try
            {
                lblMessage.Text = string.Empty;

                if (!IsPostBack)
                {
                    // MUST DO FIRST -  get the querystring data on FIRST page load only - we manage therafter
                    // later calls rely on having this data
                    LoadQuerystringData();

                    ShowEditUpdateMode();
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void ShowEditUpdateMode()
        {
            // when a schedule, we can edit
            if (_policyId > 0)
            {
                ShowPolicyEditMode();
            }
            else
            {
                ShowAddNewPolicyMode();
            }

        }

        /// <summary>
        /// Load local values from querystring values
        /// </summary>
        void LoadQuerystringData()
        {
            // get editing mode
            if (Request.QueryString["mode"] != null && Request.QueryString["mode"].ToLower() == "edit")
            {
                // assumes edit - if zero, we add new by default
                _policyId = Helper.GetQueryStringInt(Request.QueryString["policyid"]);
            }
            else
            {
                // assumes add new
                _policyId = 0;
            }

            _resourceId = Helper.GetQueryStringInt(Request.QueryString["resourceid"]);
            _sectionId = Helper.GetQueryStringInt(Request.QueryString["sectionid"]);
        }

        void ShowPolicyEditMode()
        {
            // there are slots to view
            lnkSlots.Visible = true;

            // add navigation to the slots created for the schedule
            lnkSlots.NavigateUrl = "SlotList.aspx?policyid=" + _policyId.ToString() + "&resourceid=" + _resourceId.ToString() + "&sectionid=" + _sectionId.ToString();

            // load the existing policy details
            Factory factory = Helper.GetFactory(Page);
            var resourceService = factory.GetResourceService();
            Resource resource = resourceService.Get(_resourceId);

            // Expecting only one policy with the specified ID
            SchedulingPolicy pol = resource.SchedulingPolicies.Single(p => p.Id == _policyId);

            if (pol.EndEffectDate == null)
            {
                tbDateTo.Visible = false;
            }
            else
            {
                tbDateFrom.Visible = true;
                tbDateFrom.Text = pol.StartEffectDate.ToString("dd-MMM-yyyy");

                tbDateTo.Visible = true;
                tbDateTo.Text = pol.EndEffectDate.Value.ToString("dd-MMM-yyyy");
            }

            ddlDay.SelectedValue = pol.DayOfWeek.ToString();

            ddlPriority.SelectedIndex = ddlPriority.Items.IndexOf(ddlPriority.Items.FindByText(pol.Priority.ToString()));

            // updating an existing policy, so show UPDATE as the action
            btnAddUpdate.Text = "Update";

            // we have a policy so can manage slots
            trViewSlots.Visible = true;
        }

        void ShowAddNewPolicyMode()
        {
            // no slots to view
            lnkSlots.Visible = false;

            // set NOW date of the browser - no point in setting against any timezone
            // we don't need the time part,  just the date
            DateTime now = DateTime.Now.Date;

            // show dates for today and a year ahead 
            tbDateFrom.Text = now.ToString("dd-MMM-yyyy");
            tbDateTo.Text = now.AddYears(1).ToString("dd-MMM-yyyy");

            // a new policy, so show ADD as the action
            btnAddUpdate.Text = "Add";

            // no policy yet so cannot edit slots
            trViewSlots.Visible = false;

            ddlDay.SelectedValue = "1";
        }

        protected void btnAddUpdate_Click(object sender, EventArgs e)
        {
            try
            {
                var dateFrom = DateTime.Parse(tbDateFrom.Text).Date;
                var dateTo = DateTime.Parse(tbDateTo.Text).Date;

                if (dateFrom > dateTo)
                {
                    lblMessage.Text = "The policy end date must be on or after the start date";
                    return;
                }

                // get the priority for the policy - we need this do do date checks later
                int priority = int.Parse(ddlPriority.SelectedItem.Text);
                short dayOfWeek = Convert.ToInt16(ddlDay.SelectedValue);

                Factory factory = Helper.GetFactory(Page);
                var resourceService = factory.GetResourceService();
                Resource resource = resourceService.Get(_resourceId);

                // policies to check against for date clashes - will have same day of week and priority
                List<SchedulingPolicy> policiesToCheck = null;

                // the new or updating policy
                SchedulingPolicy policy = null;

                if (_policyId > 0)
                {
                    // the policy to edit
                    policy = resource.SchedulingPolicies.Single(p => p.Id == _policyId);

                    // the other resource policies except the editing one
                    policiesToCheck = (from p in resource.SchedulingPolicies
                                       where p.Id != policy.Id &&
                                       p.Priority == priority &&
                                       p.DayOfWeek == dayOfWeek
                                       select p).ToList();
                }
                else
                {
                    // a new policy
                    policy = new SchedulingPolicy();

                    // check against all policies
                    policiesToCheck = (from p in resource.SchedulingPolicies
                                       where p.Priority == priority &&
                                       p.DayOfWeek == dayOfWeek
                                       select p).ToList();
                }

                //-----------------------------------------------------------------------
                // Check date of new or updated policy does not clash with existing ones
                // Check against policies of the same priority. Check no date overlap
                //-----------------------------------------------------------------------
                foreach (var policyToCheck in policiesToCheck)
                {
                    if (policyToCheck.EndEffectDate.HasValue)
                    {
                        // policy must start and end before the next one
                        if (dateFrom < policyToCheck.StartEffectDate && dateTo < policyToCheck.StartEffectDate)
                        {
                            // we are OK - our policy starts and ends before the start of the one tested against
                            // we don't need to chech anything else
                        }
                        else
                        {
                            // one or other of the start or end dates is after the start of the policy - check we start and end after it
                            if (dateFrom > policyToCheck.EndEffectDate.Value && dateTo > policyToCheck.EndEffectDate.Value)
                            {
                                // we are OK, bith start and end date are after the policy checked against
                            }
                            else
                            {
                                lblMessage.Text = "Cannot save this policy - it overlaps with one from [" + policyToCheck.StartEffectDate.ToString("dd-MMM-yyyy") + " to " + policyToCheck.EndEffectDate.Value.ToString("dd-MMM-yyyy") + "]";
                                return;
                            }
                        }
                    }
                    else
                    {
                        if (dateTo > policyToCheck.StartEffectDate)
                        {
                            lblMessage.Text = "Cannot save this policy - it overlaps with one from " + policyToCheck.StartEffectDate.ToString("dd-MMM-yyyy") + " with no end date";
                            return;
                        }
                    }
                }

                //-----------------------------------------------------------------------
                // if here - OK to save
                policy.StartEffectDate = dateFrom;
                policy.EndEffectDate = dateTo;
                policy.Priority = priority;
                policy.DayOfWeek = dayOfWeek;

                if (_policyId == 0)
                {
                    // new policy - add it
                    resource.AddShedulingPolicy(policy);
                }

                // all update to save changes to existing, or get new ID
                resourceService.Update(resource);

                // should have this after saving
                _policyId = policy.Id;

                lblMessage.Text = "Scheduling policy saved";

                // show database data by reloading
                ShowEditUpdateMode();
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }
    }
}