package gov.va.ccd.consol.component.impl;

import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import gov.va.ccd.consol.AbstractConsolComponent;
import gov.va.ccd.consol.component.transform.TransformerConsolCore;
import gov.va.ccd.mapper.TemplateMapper;
import gov.va.ccd.service.value.object.SectionValueObject;

import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.emf.common.util.EList;
import org.openhealthtools.mdht.uml.cda.Act;
import org.openhealthtools.mdht.uml.cda.Observation;
import org.openhealthtools.mdht.uml.cda.StrucDocText;
import org.openhealthtools.mdht.uml.cda.consol.AllergiesSection;
import org.openhealthtools.mdht.uml.cda.consol.ContinuityOfCareDocument;
import org.openhealthtools.mdht.uml.hl7.datatypes.ANY;
import org.openhealthtools.mdht.uml.hl7.datatypes.CD;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Component will parse through all of the allergies in the document. The maps
 * created should then be passed to the approprite freemarker template for
 * creating the Allergy narrative.
 * 
 * @author ChmaraC
 * 
 */
public class AllergyConsolComponent extends AbstractConsolComponent<AllergiesSection>
{
	private static final Logger logger = LoggerFactory.getLogger(AllergyConsolComponent.class);

	public AllergyConsolComponent(ContinuityOfCareDocument ccd)
	{
		super(ccd.getAllergiesSection(), ccd);
	}

	@Deprecated
	public List<SectionValueObject> getSectionData()
	{
		throw new UnsupportedOperationException("Get Section data is for legacy code.");
	}

	@Override
	protected String generateXml()
	{
		if(values.isEmpty())
			return StringUtils.EMPTY;

		try
		{
			Configuration cfg = TemplateMapper.getInstance().getCfg();
			Template temp = cfg.getTemplate("ccda/Allergy.ftl");
			/* Merge data-model with template */

			StringWriter out = new StringWriter();
			temp.process(values, out);
			return out.toString();
		}
		catch(IOException e)
		{
			logger.error("Error occurred", e);
		}
		catch(TemplateException e)
		{
			logger.error("Error occurred", e);
		}
		return StringUtils.EMPTY;
	}

	@Override
	protected void createObjects()
	{
		List<Map<String, Object>> allergies = new ArrayList<Map<String, Object>>();
		for(Act act : section.getActs())
		{
			Map<String, Object> map = new HashMap<String, Object>();

			mapAllergyItem(map, act, narrativeText);
			if(!map.isEmpty())
			{
				allergies.add(map);
			}
		}

		if(CollectionUtils.isNotEmpty(allergies))
		{
			values.put("allergies", allergies);
		}
	}

	/**
	 * Allergies processing.
	 * 
	 * @param results
	 * @param act
	 * @param narrative
	 */
	public static void mapAllergyItem(final Map<String, Object> results, final Act act, final StrucDocText narrative)
	{
		// Default Items
		// id
		TransformerConsolCore.mapIds(results, act.getIds());
		// Event type
		// TODO: Needs to be updated to proper path. Not sure if there is an
		// equivelant.
		TransformerConsolCore.mapCode(results, "", act.getCode(), narrative);
		// time
		TransformerConsolCore.mapEffectiveTime(results, act.getEffectiveTime());
		// statusCode
		TransformerConsolCore.mapWrappedCode(results, "status", act.getStatusCode(), narrative);
		// author
		TransformerConsolCore.mapAuthors(results, "", act.getAuthors());

		mapAllergyObservations(results, act.getObservations(), narrative);

		// Find other sources
		TransformerConsolCore.mapSources(results, act.getAuthors(), act.getPerformers(), act.getParticipants(), act.getInformants());
	}

	public static void mapAllergyObservations(final Map<String, Object> results, final EList<Observation> observations, final StrucDocText narrative)
	{

		if(CollectionUtils.isNotEmpty(observations))
		{
			for(Observation obs : observations)
			{
				// Get Event Type
				for(ANY any : obs.getValues())
				{
					if(any instanceof CD)
					{
						TransformerConsolCore.mapWrappedCode(results, "eventType", (CD) any, narrative);
					}
				}
				// Get Allergen Names.
				if(CollectionUtils.isNotEmpty(obs.getParticipants()))
				{
					TransformerConsolCore.mapParticipants(results, "", obs.getParticipants(), narrative);
				}

				TransformerConsolCore.mapReactions(results, obs, narrative);
			}
		}
	}

}
