package gov.va.ccd.components.transform;

import gov.va.ccd.service.util.Utils;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.emf.common.util.EList;
import org.openhealthtools.mdht.uml.cda.Act;
import org.openhealthtools.mdht.uml.cda.Author;
import org.openhealthtools.mdht.uml.cda.EntryRelationship;
import org.openhealthtools.mdht.uml.cda.Observation;
import org.openhealthtools.mdht.uml.cda.StrucDocText;
import org.openhealthtools.mdht.uml.cda.SubstanceAdministration;
import org.openhealthtools.mdht.uml.cda.Supply;
import org.openhealthtools.mdht.uml.hl7.datatypes.CD;
import org.openhealthtools.mdht.uml.hl7.datatypes.EIVL_TS;
import org.openhealthtools.mdht.uml.hl7.datatypes.IVL_TS;
import org.openhealthtools.mdht.uml.hl7.datatypes.PIVL_TS;
import org.openhealthtools.mdht.uml.hl7.datatypes.PQ;
import org.openhealthtools.mdht.uml.hl7.datatypes.RTO_PQ_PQ;
import org.openhealthtools.mdht.uml.hl7.datatypes.SXCM_TS;
import org.openhealthtools.mdht.uml.hl7.vocab.SetOperator;
import org.openhealthtools.mdht.uml.hl7.vocab.x_ActRelationshipEntryRelationship;
import org.openhealthtools.mdht.uml.hl7.vocab.x_DocumentSubstanceMood;

public class SubstanceAdminTransformer
{
	/**
	 * Loops through all of the entry relationships to find specific items.
	 * 
	 * @param results
	 * @param subAdm
	 * @param narrative
	 * @see #processFulfillment(Map, EntryRelationship, StrucDocText)
	 * @see #processMedIndication(Map, EntryRelationship, StrucDocText)
	 * @see #processMedStatus(Map, EntryRelationship, StrucDocText)
	 * @see #processMedType(Map, EntryRelationship, StrucDocText)
	 * @see #processOrder(Map, EntryRelationship, StrucDocText)
	 * @see #processPatInstructions(Map, EntryRelationship, StrucDocText)
	 * @see #processReaction(Map, EntryRelationship, StrucDocText)
	 */
	public static void mapEntryRels(final Map<String, Object> results, final SubstanceAdministration subAdm, final StrucDocText narrative)
	{
		if(subAdm != null)
		{
			EList<EntryRelationship> rels = subAdm.getEntryRelationships();
			for(EntryRelationship rel : rels)
			{
				processFulfillment(results, rel, narrative);
				processOrder(results, rel, narrative);
				processPatInstructions(results, rel, narrative);
				processReaction(results, rel, narrative);
				processMedType(results, rel, narrative);
				processMedIndication(results, rel, narrative);
				processMedStatus(results, rel, narrative);
			}
		}
	}

	/**
	 * Maps the original text from the entryRel/Obs for instructions into its
	 * own map and then places it in results key <b>instructions</b>
	 * {@link Transformer#mapOriginalText(Map, String, org.openhealthtools.mdht.uml.hl7.datatypes.ED, StrucDocText)}
	 * 
	 * @param results
	 * @param rel
	 * @param narrative
	 * @see Transformer#mapOriginalText(Map, String,
	 *      org.openhealthtools.mdht.uml.hl7.datatypes.ED, StrucDocText)
	 */
	public static void processPatInstructions(final Map<String, Object> results, final EntryRelationship rel, StrucDocText narrative)
	{
		if(rel != null)
		{
			Act act = rel.getAct();
			if(act != null && Utils.containsTemplateId(act.getTemplateIds(), "2.16.840.1.113883.10.20.1.49"))
			{
				Map<String, Object> instructions = new HashMap<String, Object>();
				Transformer.mapOriginalText(instructions, "", act.getText(), narrative);
				if(!instructions.isEmpty())
				{
					results.put("instructions", instructions);
				}
			}

		}
	}

	/**
	 * Will add a map a value like a code for <b>medType</b> using
	 * {@link Transformer#mapWrappedCode(Map, String, CD, StrucDocText)}
	 * 
	 * @param results
	 * @param rel
	 * @param narrative
	 */
	public static void processMedType(final Map<String, Object> results, final EntryRelationship rel, StrucDocText narrative)
	{
		if(rel != null && x_ActRelationshipEntryRelationship.SUBJ.equals(rel.getTypeCode()))
		{
			Observation obs = rel.getObservation();

			if(obs != null && Utils.containsTemplateId(obs.getTemplateIds(), "2.16.840.1.113883.3.88.11.83.8.1"))
			{
				Transformer.mapObsValueOrCode(results, "medType", obs, narrative);
			}
		}
	}

	/**
	 * Will add a map a value like a code for <b>medStatus</b> using
	 * {@link Transformer#mapWrappedCode(Map, String, CD, StrucDocText)}
	 * 
	 * @param results
	 * @param rel
	 * @param narrative
	 */
	public static void processMedStatus(final Map<String, Object> results, final EntryRelationship rel, StrucDocText narrative)
	{
		if(rel != null && x_ActRelationshipEntryRelationship.REFR.equals(rel.getTypeCode()))
		{
			Observation obs = rel.getObservation();

			if(obs != null && Utils.containsTemplateId(obs.getTemplateIds(), "2.16.840.1.113883.10.20.1.47"))
			{
				Transformer.mapObsValueOrCode(results, "medStatus", obs, narrative);
			}
		}
	}

	/**
	 * * Will add a map a value like a code for <b>medIndication</b> using
	 * {@link Transformer#mapWrappedCode(Map, String, CD, StrucDocText)}
	 * 
	 * @param results
	 * @param rel
	 * @param narrative
	 */
	public static void processMedIndication(final Map<String, Object> results, final EntryRelationship rel, StrucDocText narrative)
	{
		if(rel != null && x_ActRelationshipEntryRelationship.RSON.equals(rel.getTypeCode()))
		{
			Observation obs = rel.getObservation();

			if(obs != null && Utils.containsTemplateId(obs.getTemplateIds(), "2.16.840.1.113883.10.20.1.28"))
			{
				Transformer.mapObsValueOrCode(results, "medIndication", obs, narrative);
			}
		}
	}

	/**
	 * Maps the original text from the entryRel/Obs for reaction into results
	 * key <b>reaction</b>
	 * {@link Transformer#mapOriginalText(Map, String, org.openhealthtools.mdht.uml.hl7.datatypes.ED, StrucDocText)}
	 * 
	 * @param results
	 * @param rel
	 * @param narrative
	 */
	public static void processReaction(final Map<String, Object> results, final EntryRelationship rel, StrucDocText narrative)
	{
		if(rel != null)
		{
			Observation obs = rel.getObservation();

			if(obs != null && Utils.containsTemplateId(obs.getTemplateIds(), "2.16.840.1.113883.10.20.1.54"))
			{
				Map<String, Object> reaction = new HashMap<String, Object>();
				Transformer.mapOriginalText(reaction, "", obs.getText(), narrative);
				if(!reaction.isEmpty())
				{
					results.put("reaction", reaction);
				}
			}

		}
	}

	/**
	 * <pre>
	 * Creates a map <b>fulfillment</b> in results from EntryRel/act.  With the following objects in the map:
	 * <ul>
	 * <li>{@link Transformer#mapOriginalText(Map, String, org.openhealthtools.mdht.uml.hl7.datatypes.ED, StrucDocText)} of Text block.</li>
	 * <li>{@link Transformer#mapIds(Map, List)} of all the ids.</li>
	 * <li>{@link Transformer#mapPQValue(Map, String, org.openhealthtools.mdht.uml.hl7.datatypes.PQ, StrucDocText)} for <b>quantity</b></li>
	 * <li>{@link Transformer#mapEffectiveTime(Map, org.openhealthtools.mdht.uml.hl7.datatypes.SXCM_TS)} for the first available effectiveTime</li>
	 * <li>{@link Transformer#mapPerformers(Map, String, EList)} for Provider/Pharmacy Information</li>
	 * <li>{@link Transformer#mapWrappedCode(Map, String, CD, StrucDocText)} for <b>fillStatus</b></li>
	 * <li>{@link Transformer#mapString(Map, String, String)} for <b>fillNumber</b></li>
	 * 
	 * </ul>
	 * </pre>
	 * 
	 * @param results
	 * @param rel
	 * @param narrative
	 */
	public static void processFulfillment(final Map<String, Object> results, final EntryRelationship rel, StrucDocText narrative)
	{
		if(rel != null)
		{
			Act act = rel.getAct();
			if(act != null && Utils.containsTemplateId(act.getTemplateIds(), "2.16.840.1.113883.10.20.1.43"))
			{
				if(rel.getSupply() != null)
				{
					Supply supply = rel.getSupply();
					if(x_DocumentSubstanceMood.EVN.equals(supply.getMoodCode()))
					{
						Map<String, Object> fulfill = new HashMap<String, Object>();
						// Fulfillment instructions
						Transformer.mapOriginalText(fulfill, "", act.getText(), narrative);
						// This is the item we are looking for for Order
						// information.
						Transformer.mapIds(fulfill, supply.getIds());
						// Quantity
						Transformer.mapPQValue(fulfill, "quantity", supply.getQuantity(), narrative);
						// Get the first effective time if available.
						if(CollectionUtils.isNotEmpty(supply.getEffectiveTimes()))
						{
							Transformer.mapEffectiveTime(fulfill, supply.getEffectiveTimes().get(0));
						}
						// Get Provider/Pharmacy information
						Transformer.mapPerformers(fulfill, "", supply.getPerformers());
						Transformer.mapWrappedCode(fulfill, "fillStatus", supply.getStatusCode(), narrative);

						if(CollectionUtils.isNotEmpty(supply.getEntryRelationships()))
						{
							for(EntryRelationship entryRel : supply.getEntryRelationships())
							{
								if(x_ActRelationshipEntryRelationship.COMP.equals(entryRel.getTypeCode()))
								{
									// fillNumber
									Transformer.mapString(fulfill, "fillNumber", entryRel.getSequenceNumber().getValue().toString());
								}
							}
						}
						if(!fulfill.isEmpty())
						{
							results.put("fulfillment", fulfill);
						}
					}
				}
			}
		}
	}

	/**
	 * <pre>
	 * Creates an <b>order</b> in results from EntryRel/supply.  With the following objects in each map:
	 * <ul>
	 * <li>{@link Transformer#mapIds(Map, List)} of all the ids.</li>
	 * <li>{@link Transformer#mapString(Map, String, String)} for <b>fill</b></li>
	 * <li>{@link Transformer#mapPQValue(Map, String, org.openhealthtools.mdht.uml.hl7.datatypes.PQ, StrucDocText)} for <b>quantity</b></li>
	 * <li>{@link Transformer#mapEffectiveTime(Map, org.openhealthtools.mdht.uml.hl7.datatypes.SXCM_TS)} for the first available effectiveTime</li>
	 * <li>{@link Transformer#mapEffectiveTime(Map, org.openhealthtools.mdht.uml.hl7.datatypes.TS)} into a map then into results as <b>orderTime</b></li>
	 * <li>{@link Transformer#mapString(Map, String, String)} to add <b>provider</b> names</li>
	 * 
	 * </ul>
	 * </pre>
	 * 
	 * @param results
	 * @param rel
	 * @param narrative
	 */
	public static void processOrder(final Map<String, Object> results, final EntryRelationship rel, StrucDocText narrative)
	{
		if(rel != null && rel.getSupply() != null)
		{
			Supply supply = rel.getSupply();
			if(x_ActRelationshipEntryRelationship.REFR.equals(rel.getTypeCode()) && x_DocumentSubstanceMood.INT.equals(supply.getMoodCode()))
			{
				Map<String, Object> order = new HashMap<String, Object>();
				// This is the item we are looking for for Order information.
				Transformer.mapIds(order, supply.getIds());
				// Fills
				if(supply.getRepeatNumber() != null && supply.getRepeatNumber().getValue() != null)
				{
					Transformer.mapString(order, "fill", supply.getRepeatNumber().getValue().toString());
				}
				// Quantity
				Transformer.mapPQValue(order, "quantity", supply.getQuantity(), narrative);
				// Get the first effective time if available.
				if(CollectionUtils.isNotEmpty(supply.getEffectiveTimes()))
				{
					Transformer.mapEffectiveTime(order, supply.getEffectiveTimes().get(0));
				}
				if(CollectionUtils.isNotEmpty(supply.getAuthors()))
				{
					Author author = supply.getAuthors().get(0);
					Map<String, Object> orderTime = new HashMap<String, Object>();
					Transformer.mapEffectiveTime(orderTime, author.getTime());
					if(!orderTime.isEmpty())
					{
						order.put("orderTime", orderTime);
					}
					if(author.getAssignedAuthor() != null && author.getAssignedAuthor().getAssignedPerson() != null)
					{
						String name = Transformer.createNames(author.getAssignedAuthor().getAssignedPerson().getNames());
						if(StringUtils.isNotBlank(name))
						{
							Transformer.mapString(order, "provider", name);
						}
					}
				}

				if(!order.isEmpty())
				{
					results.put("order", order);
				}
			}
		}
	}

	public static void mapSubstanceTimes(final Map<String, Object> results, final SubstanceAdministration subAdm, StrucDocText narrative)
	{
		if(subAdm != null && CollectionUtils.isNotEmpty(subAdm.getEffectiveTimes()))
		{

		}
	}

	public static void mapSubstanceEffectiveTimes(final SubstanceAdministration subAdmin, final Map<String, Object> results)
	{
		StringBuilder ret = new StringBuilder();

		for(SXCM_TS time : subAdmin.getEffectiveTimes())
		{
			if(time instanceof PIVL_TS)
			{
				ret.append(processPIVL((PIVL_TS) time)).append(" ");
			}
			else if(time instanceof EIVL_TS)
			{
				ret.append(processEIVL((EIVL_TS) time)).append(" ");
			}
			else if(time instanceof IVL_TS)
			{
				ret.append(processIVL((IVL_TS) time, results)).append(" ");
			}
			else if(StringUtils.isNotBlank(time.getValue()))
			{
				// Fall through test
				ret.append("Once on " + time.getValue());
			}
		}

		if(ret.length() > 0)
		{
			results.put("adminTiming", ret.toString());
		}
	}

	/**
	 * Will set Indicate med stopped only if time.getHigh != null and
	 * time.getLow == null
	 * 
	 * Otherwise this will return the time string for indicating timing.
	 * 
	 * @param time
	 * @param vo
	 * @return
	 */
	private static String processIVL(final IVL_TS time, final Map<String, Object> results)
	{
		if(time == null)
		{
			return StringUtils.EMPTY;
		}
		String timeLow;
		String timeHigh;

		if(time.getLow() != null)
		{
			timeLow = time.getLow().getValue();
		}
		else
		{
			timeLow = StringUtils.EMPTY;
		}

		if(time.getHigh() != null)
		{
			timeHigh = time.getHigh().getValue();
		}
		else
		{
			timeHigh = StringUtils.EMPTY;
		}

		if(StringUtils.isNotBlank(timeHigh) && StringUtils.isBlank(timeLow))
		{
			// Indicate MedStopped
			results.put("medStopped", timeHigh);
			return StringUtils.EMPTY;
		}

		if(StringUtils.isNotBlank(timeLow) && StringUtils.isNotBlank(timeHigh))
		{
			return "From: " + timeLow + " To: " + timeHigh;
		}

		if(StringUtils.isNotBlank(time.getValue()))
		{
			return "Once on " + time.getValue();
		}

		if(StringUtils.isNotBlank(timeLow))
		{
			return "From: " + timeLow;
		}

		return StringUtils.EMPTY;
	}

	private static String processPIVL(final PIVL_TS time)
	{
		if(time == null)
		{
			return null;
		}

		StringBuilder effectiveTime = new StringBuilder();

		if(time.getOperator() != null && time.getOperator().equals(SetOperator.A) && time.getPeriod() != null && time.getPeriod().getValue() != null)
		{
			if(time.getInstitutionSpecified() && "h".equalsIgnoreCase(time.getPeriod().getUnit()))
			{
				Double value = time.getPeriod().getValue().doubleValue();
				Double period = 0.0;
				if(value != null)
				{
					if(value < 1.0)
					{
						period = 24.0 * value;
					}
					else
					{
						period = 24.0 / value;
					}
				}

				effectiveTime.append(period);
				effectiveTime.append(" ");
				effectiveTime.append("time(s) a day");
			}
			else
			{
				effectiveTime.append("every ");
				effectiveTime.append(time.getPeriod().getValue());
				effectiveTime.append(" ");
				effectiveTime.append(time.getPeriod().getUnit()).append(" ");
				String phase = processPhase(time.getPhase());
				if(StringUtils.isNotBlank(phase))
				{
					effectiveTime.append("for ").append(phase).append(" ");
				}

			}
		}
		else if(time.getPhase() != null)
		{
			effectiveTime.append(" ");
			effectiveTime.append("for ");
			effectiveTime.append(processPhase(time.getPhase()));
		}
		return effectiveTime.toString().trim();
	}

	private static String processPhase(final IVL_TS phase)
	{
		StringBuilder sb = new StringBuilder();

		if(phase != null)
		{
			sb.append(phase.getWidth().getValue());
			sb.append(" ");
			sb.append(phase.getWidth().getUnit());

			if(phase.getLow() != null)
			{
				String time = phase.getLow().getValue();
				if(StringUtils.isNotBlank(time))
				{
					sb.append(" at ");
					sb.append(time.substring(time.length() - 4));
				}
			}
		}
		return sb.toString().trim();
	}

	private static String processEIVL(final EIVL_TS time)
	{
		if(time == null)
		{
			return null;
		}

		StringBuilder ret = new StringBuilder();

		if(SetOperator.A.equals(time.getOperator()))
		{
			if(time.getEvent() != null && StringUtils.isNotBlank(time.getEvent().getCode()))
			{
				// Cycle through all of the possible code values to get the
				// proper return.
				if("AC".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("before meal");
				}
				else if("ACD".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("before lunch");
				}
				else if("ACM".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("before breakfast");
				}
				else if("ACV".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("before dinner");
				}
				else if("HS".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("hour of sleep");
				}
				else if("IC".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("between meals");
				}
				else if("ICD".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("between lunch and dinner");
				}
				else if("ICM".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("between breakfast and lunch");
				}
				else if("ICV".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("between dinner and hour of sleep");
				}
				else if("PC".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("after meal");
				}
				else if("PCD".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("after lunch");
				}
				else if("PCM".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("after breakfast");
				}
				else if("PCV".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("after dinner");
				}
			}
			if(ret.length() > 1)
			{
				return ret.toString();
			}
		}
		return null;
	}

	public static void mapDoseRestriction(final SubstanceAdministration subAdmin, final Map<String, Object> results)
	{
		final RTO_PQ_PQ maxDoseQnty = subAdmin.getMaxDoseQuantity();

		final StringBuilder maxDoseString = new StringBuilder();

		if(maxDoseQnty != null)
		{

			if(maxDoseQnty.getNumerator() != null)
			{
				PQ num = maxDoseQnty.getNumerator();
				if(num.getValue() != null)
				{
					maxDoseString.append(num.getValue().toString());
				}
				if(StringUtils.isNotBlank(num.getUnit()))
				{
					maxDoseString.append(num.getUnit());
				}
			}

			if(maxDoseQnty.getDenominator() != null)
			{
				maxDoseString.append(" / ");
				if(maxDoseQnty.getDenominator().getValue() != null)
				{
					maxDoseString.append(maxDoseQnty.getDenominator().getValue().toString());
				}
				if(StringUtils.isNotBlank(maxDoseQnty.getDenominator().getUnit()))
				{
					maxDoseString.append(maxDoseQnty.getDenominator().getUnit());
				}
			}
		}

		if(maxDoseString.length() > 0)
		{
			results.put("doseRestriction", maxDoseString.toString());
		}
	}

}
