package gov.va.soapui;

import java.util.Collections;
import java.util.List;

import org.apache.ws.security.components.crypto.Crypto;
import org.apache.ws.security.saml.ext.SAMLCallback;
import org.apache.ws.security.saml.ext.bean.ActionBean;
import org.apache.ws.security.saml.ext.bean.AttributeBean;
import org.apache.ws.security.saml.ext.bean.AttributeStatementBean;
import org.apache.ws.security.saml.ext.bean.AuthDecisionStatementBean;
import org.apache.ws.security.saml.ext.bean.AuthenticationStatementBean;
import org.apache.ws.security.saml.ext.bean.SubjectBean;
import org.apache.ws.security.saml.ext.bean.SubjectLocalityBean;

import com.eviware.soapui.impl.wsdl.support.wss.entries.AutomaticSAMLEntry;
import com.eviware.soapui.impl.wsdl.support.wss.saml.callback.AbstractSAMLCallbackHandler;
import com.eviware.soapui.support.UISupport;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.apache.ws.security.WSConstants;
import org.apache.ws.security.message.WSSecEncryptedKey;
import org.apache.ws.security.saml.ext.bean.KeyInfoBean;
import org.opensaml.Configuration;
import org.opensaml.saml2.core.AttributeValue;
import org.opensaml.saml2.core.AuthnContext;
import org.opensaml.xml.XMLObjectBuilderFactory;
import org.opensaml.xml.schema.XSAny;
import org.opensaml.xml.schema.impl.XSAnyBuilder;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/*
 * @author Erik R. Yverling
 * 
 * A base implementation of a Callback Handler for a SAML assertion. By
 * default it creates an authentication assertion.
 * 
 */
public abstract class AbstractCONNECTSAMLCallbackHandler extends AbstractSAMLCallbackHandler implements CONNECTSAMLCallbackHandler {
    String issuerFormat;
    String subjectId;
    String organization;
    String organizationId;
    String homeCommunityId;
    String resourceId;
    String roleName;
    String roleCode;
    String roleCodeSystem;
    String roleCodeSystemName;
    String roleDisplayName;
    String purposeCode = "TREATMENT";
    String purposeSystem = "2.16.840.1.113883.3.18.7.1";
    String purposeSystemName = "nhin-purpose";
    String purposeDisplay;
    
    private String customAttributeName;
    protected AbstractCONNECTSAMLCallbackHandler.Statement statement;
    private final XSAnyBuilder xsAnyBuilder;
    private static XMLObjectBuilderFactory builderFactory = Configuration.getBuilderFactory();
    
    public enum Statement {
        AUTHN, ATTR, AUTHZ, CONNECT
    }

    ;

    /**
     * Use this is for signed assertions
     */
    public AbstractCONNECTSAMLCallbackHandler(Crypto crypto, String alias, String assertionTypeFriendlyName, String confirmationMethodFriendlyName) {
        super(crypto, alias, assertionTypeFriendlyName, confirmationMethodFriendlyName);
        xsAnyBuilder = (XSAnyBuilder) builderFactory.getBuilder(XSAny.TYPE_NAME);
        setStatement("CONNECT");
    }
    
    /**
     * Use this is for unsigned assertions
     */
    public AbstractCONNECTSAMLCallbackHandler(String assertionTypeFriendlyName, String confirmationMethodFriendlyName) {
        super(assertionTypeFriendlyName, confirmationMethodFriendlyName);
        xsAnyBuilder = (XSAnyBuilder) builderFactory.getBuilder(XSAny.TYPE_NAME);
        setStatement("CONNECT");
    }
    
    public void setIssuerFormat(String issuerFormat) {
        this.issuerFormat = issuerFormat;
    }
    
    @Override
    public void setStatement(String statement) {
        if (statement.equals(AutomaticSAMLEntry.AUTHENTICATION_ASSERTION_TYPE)) {
            this.statement = AbstractCONNECTSAMLCallbackHandler.Statement.AUTHN;
        } else if (statement.equals(AutomaticSAMLEntry.ATTRIBUTE_ASSERTION_TYPE)) {
            this.statement = AbstractCONNECTSAMLCallbackHandler.Statement.ATTR;
        } else if (statement.equals(AutomaticSAMLEntry.AUTHORIZATION_ASSERTION_TYPE)) {
            this.statement = AbstractCONNECTSAMLCallbackHandler.Statement.AUTHZ;
        } else if (statement.equals(CONNECTSAMLEntry.CONNECT_ASSERTION_TYPE)) {
            this.statement = AbstractCONNECTSAMLCallbackHandler.Statement.CONNECT;
        }
    }
 
    /**
     * Note that the SubjectBean parameter should be null for SAML2.0
     */
    @Override
    protected void createAndSetStatement(SubjectBean subjectBean, SAMLCallback callback) {
        if (statement == Statement.AUTHN) {
            AuthenticationStatementBean authBean = new AuthenticationStatementBean();
            if (subjectBean != null) {
                authBean.setSubject(subjectBean);
            }
            if (subjectLocalityIpAddress != null || subjectLocalityDnsAddress != null) {
                SubjectLocalityBean subjectLocality = new SubjectLocalityBean();
                subjectLocality.setIpAddress(subjectLocalityIpAddress);
                subjectLocality.setDnsAddress(subjectLocalityDnsAddress);
                authBean.setSubjectLocality(subjectLocality);
            }
            authBean.setAuthenticationMethod("Password");
            callback.setAuthenticationStatementData(Collections.singletonList(authBean));
        } else if (statement == Statement.ATTR) {
            AttributeStatementBean attrBean = new AttributeStatementBean();
            if (subjectBean != null) {
                attrBean.setSubject(subjectBean);
            }
            AttributeBean attributeBean = new AttributeBean();
            attributeBean.setSimpleName(customAttributeName);
            if (customAttributeValues != null) {
                attributeBean.setCustomAttributeValues(customAttributeValues);
            }

            // TODO This should be removed
            else {
                attributeBean.setAttributeValues(Collections.singletonList("user"));
            }

            attrBean.setSamlAttributes(Collections.singletonList(attributeBean));
            callback.setAttributeStatementData(Collections.singletonList(attrBean));
        } else if (statement == Statement.CONNECT) {
            //set Authn
            AuthenticationStatementBean authBean = new AuthenticationStatementBean();
            if (subjectBean != null) {
                authBean.setSubject(subjectBean);
            }
            if (subjectLocalityIpAddress != null || subjectLocalityDnsAddress != null) {
                SubjectLocalityBean subjectLocality = new SubjectLocalityBean();
                subjectLocality.setIpAddress(subjectLocalityIpAddress);
                subjectLocality.setDnsAddress(subjectLocalityDnsAddress);
                authBean.setSubjectLocality(subjectLocality);
            }
            authBean.setAuthenticationMethod(AuthnContext.X509_AUTHN_CTX);
            callback.setAuthenticationStatementData(Collections.singletonList(authBean));
            
            //set Attributes
            List<AttributeStatementBean> attributeStatementBeans = new ArrayList<AttributeStatementBean>();
            
             //subject name
            AttributeStatementBean subjectAttrBean = new AttributeStatementBean();
            if (subjectBean != null) {
                subjectAttrBean.setSubject(subjectBean);
            }
            AttributeBean subjectIDAttributeBean = new AttributeBean();
            subjectIDAttributeBean.setQualifiedName("urn:oasis:names:tc:xspa:1.0:subject:subject-id");
            subjectId = UISupport.prompt("Enter Subject ID", "CONNECT SAML Parameters", "");
            subjectIDAttributeBean.setCustomAttributeValues(Collections.singletonList(subjectId));
            subjectAttrBean.setSamlAttributes(Collections.singletonList(subjectIDAttributeBean));
            attributeStatementBeans.add(subjectAttrBean);
            
            //organization name
            AttributeStatementBean orgAttrBean = new AttributeStatementBean();
            AttributeBean organizationAttributeBean = new AttributeBean();
            organization = UISupport.prompt("Enter Organization Name", "CONNECT SAML Parameters", "");
            organizationAttributeBean.setQualifiedName("urn:oasis:names:tc:xspa:1.0:subject:organization");
            organizationAttributeBean.setCustomAttributeValues(Collections.singletonList(organization));
            orgAttrBean.setSamlAttributes(Collections.singletonList(organizationAttributeBean));
            attributeStatementBeans.add(orgAttrBean);
            
            //organization oid
            AttributeStatementBean orgIDAttrBean = new AttributeStatementBean();
            AttributeBean organizationIDAttributeBean = new AttributeBean();
            organizationIDAttributeBean.setQualifiedName("urn:oasis:names:tc:xspa:1.0:subject:organization-id");
            organizationId = UISupport.prompt("Enter Organization OID", "CONNECT SAML Parameters", "");
            organizationIDAttributeBean.setCustomAttributeValues(Collections.singletonList(organizationId));
            orgIDAttrBean.setSamlAttributes(Collections.singletonList(organizationIDAttributeBean));
            attributeStatementBeans.add(orgIDAttrBean);
            
            //home community oid
            AttributeStatementBean homeCommunityAttrBean = new AttributeStatementBean();
            AttributeBean homeCommunityIDAttributeBean = new AttributeBean();
            homeCommunityIDAttributeBean.setQualifiedName("urn:nhin:names:saml:homeCommunityId");
            homeCommunityId = UISupport.prompt("Enter Org. Home Community Id", "CONNECT SAML Parameters", "");
            homeCommunityIDAttributeBean.setCustomAttributeValues(Collections.singletonList(homeCommunityId));
            homeCommunityAttrBean.setSamlAttributes(Collections.singletonList(homeCommunityIDAttributeBean));
            attributeStatementBeans.add(homeCommunityAttrBean);
            
            //patient id
            AttributeStatementBean resourceIDAttrBean = new AttributeStatementBean();
            AttributeBean resourceIDAttributeBean = new AttributeBean();
            resourceIDAttributeBean.setQualifiedName("urn:oasis:names:tc:xacml:2.0:resource:resource-id");
            resourceId = UISupport.prompt("Enter Patient ID", "CONNECT SAML Parameters", "");
            resourceIDAttributeBean.setCustomAttributeValues(Collections.singletonList(resourceId));
            resourceIDAttrBean.setSamlAttributes(Collections.singletonList(resourceIDAttributeBean));
            attributeStatementBeans.add(resourceIDAttrBean);
            
            //role
            roleName = UISupport.prompt("Enter Role Name", "CONNECT SAML Parameters", "");
            roleCode = UISupport.prompt("Enter Role Code", "CONNECT SAML Parameters", "");
            roleCodeSystem = UISupport.prompt("Enter Role Code System", "CONNECT SAML Parameters", "");
            roleCodeSystemName = UISupport.prompt("Enter Role Code System Name", "CONNECT SAML Parameters", "SNOMED_CT");
            roleDisplayName = UISupport.prompt("Enter Role Display Name", "CONNECT SAML Parameters", "");
            XSAny roleCoded = createHL7Attribute(roleName, roleCode, roleCodeSystem, roleCodeSystemName, roleDisplayName);
            AttributeStatementBean roleAttrBean = new AttributeStatementBean();
            AttributeBean roleAttributeBean = new AttributeBean();
            roleAttributeBean.setQualifiedName("urn:oasis:names:tc:xacml:2.0:subject:role");
            roleAttributeBean.setCustomAttributeValues(Collections.singletonList(roleCoded));
            roleAttrBean.setSamlAttributes(Collections.singletonList(roleAttributeBean));
            attributeStatementBeans.add(roleAttrBean);
            
            //purpose of use
            XSAny purpose = createPurposeOfUseAttribute(purposeCode, purposeSystem, purposeSystemName, purposeDisplay, "purposeOfUse");
            AttributeStatementBean purposeAttrBean = new AttributeStatementBean();
            AttributeBean purposeOfUseAttributeBean = new AttributeBean();
            purposeOfUseAttributeBean.setQualifiedName("urn:oasis:names:tc:xspa:1.0:subject:purposeofuse");
            purposeOfUseAttributeBean.setCustomAttributeValues(Collections.singletonList(purpose));
            purposeAttrBean.setSamlAttributes(Collections.singletonList(purposeOfUseAttributeBean));
            attributeStatementBeans.add(purposeAttrBean);

            callback.setAttributeStatementData(attributeStatementBeans);
            
            //set AuthZDecision (optional)
            
        } else {
            AuthDecisionStatementBean authzBean = new AuthDecisionStatementBean();
            if (subjectBean != null) {
                authzBean.setSubject(subjectBean);
            }
            ActionBean actionBean = new ActionBean();
            actionBean.setContents("Read");
            authzBean.setActions(Collections.singletonList(actionBean));
            authzBean.setResource("endpoint");
            authzBean.setDecision(AuthDecisionStatementBean.Decision.PERMIT);
            authzBean.setResource(resource);
            callback.setAuthDecisionStatementData(Collections.singletonList(authzBean));
        }
    }
    
    @Override
    protected KeyInfoBean createKeyInfo() throws Exception {
        KeyInfoBean keyInfo = new KeyInfoBean();
        if (statement == AbstractCONNECTSAMLCallbackHandler.Statement.AUTHN) {
            keyInfo.setCertificate(certs[0]);
            keyInfo.setCertIdentifer(certIdentifier);
        } else if(statement == AbstractCONNECTSAMLCallbackHandler.Statement.CONNECT) {
            keyInfo.setCertificate(certs[0]);
            keyInfo.setCertIdentifer(KeyInfoBean.CERT_IDENTIFIER.KEY_VALUE);
        } else if (statement == AbstractCONNECTSAMLCallbackHandler.Statement.ATTR) {
            // Build a new Document
            DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
            docBuilderFactory.setNamespaceAware(true);
            DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
            Document doc = docBuilder.newDocument();

            // Create an Encrypted Key
            WSSecEncryptedKey encrKey = new WSSecEncryptedKey();
            encrKey.setKeyIdentifierType(WSConstants.X509_KEY_IDENTIFIER);
            encrKey.setUseThisCert(certs[0]);
            encrKey.prepare(doc, null);
            ephemeralKey = encrKey.getEphemeralKey();
            Element encryptedKeyElement = encrKey.getEncryptedKeyElement();

            // Append the EncryptedKey to a KeyInfo element
            Element keyInfoElement = doc.createElementNS(WSConstants.SIG_NS, WSConstants.SIG_PREFIX + ":"
                    + WSConstants.KEYINFO_LN);
            keyInfoElement.setAttributeNS(WSConstants.XMLNS_NS, "xmlns:" + WSConstants.SIG_PREFIX, WSConstants.SIG_NS);
            keyInfoElement.appendChild(encryptedKeyElement);

            keyInfo.setElement(keyInfoElement);
        }
        return keyInfo;
    }

    private XSAny createPurposeOfUseAttribute(String purposeCode, String purposeSystem, String purposeSystemName,
            String purposeDisplay, String attributeName) {
        Map<QName, String> purposeOfUseAttributes = new HashMap<QName, String>();

        if (purposeCode != null) {
            purposeOfUseAttributes.put(new QName(CONNECTSamlConstants.CE_CODE_ID), purposeCode);
        }

        if (purposeSystem != null) {
            purposeOfUseAttributes.put(new QName(CONNECTSamlConstants.CE_CODESYS_ID), purposeSystem);
        }

        if (purposeSystemName != null) {
            purposeOfUseAttributes.put(new QName(CONNECTSamlConstants.CE_CODESYSNAME_ID), purposeSystemName);
        }

        if (purposeDisplay != null) {
            purposeOfUseAttributes.put(new QName(CONNECTSamlConstants.CE_DISPLAYNAME_ID), purposeDisplay);
        }

        XSAny attributeValue = createAttributeValue("urn:hl7-org:v3", attributeName, "hl7", purposeOfUseAttributes);
        return attributeValue;
    }
    
    private XSAny createHL7Attribute(String name, String code, String codeSystem, String codeSystemName, String displayName) {
        Map<QName, String> userRoleAttributes = new HashMap<QName, String>();

        if (code != null) {
            userRoleAttributes.put(new QName(CONNECTSamlConstants.CE_CODE_ID), code);
        }

        if (codeSystem != null) {
            userRoleAttributes.put(new QName(CONNECTSamlConstants.CE_CODESYS_ID), codeSystem);
        }

        if (codeSystemName != null) {
            userRoleAttributes.put(new QName(CONNECTSamlConstants.CE_CODESYSNAME_ID), codeSystemName);
        }

        if (displayName != null) {
            userRoleAttributes.put(new QName(CONNECTSamlConstants.CE_DISPLAYNAME_ID), displayName);
        }

        userRoleAttributes.put(new QName("type"), "hl7:CE");

        XSAny attributeValue = createAttributeValue("urn:hl7-org:v3", name, "hl7", userRoleAttributes);
        return attributeValue;
    }
    
    private XSAny createAttributeValue(List<XSAny> values) {
        XSAny attributeValue = xsAnyBuilder.buildObject(AttributeValue.DEFAULT_ELEMENT_NAME);
        attributeValue.getUnknownXMLObjects().addAll(values);

        return attributeValue;
    }
    
    private XSAny createAttributeValue(final String namespace, final String name, final String prefix,
            Map<QName, String> attributes) {

        XSAny attribute = createAny(namespace, name, prefix, attributes);
        return createAttributeValue(Arrays.asList(attribute));
    }
    
    private XSAny createAny(final String namespace, final String name, final String prefix, Map<QName, String> attributes) {

        XSAny any = createAny(namespace, name, prefix);

        for (Map.Entry<QName, String> keyValue : attributes.entrySet()) {
            any.getUnknownAttributes().put(keyValue.getKey(), keyValue.getValue());
        }
        return any;

    }
    
    private XSAny createAny(final String namespace, final String name, final String prefix) {
        XSAny any = xsAnyBuilder.buildObject(namespace, name, prefix);
        return any;
    }
}