package gov.va.med.nhin.adapter.datamanager.translators;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

import gov.va.med.nhin.adapter.datamanager.DataDate;
import gov.va.med.nhin.adapter.datamanager.DataManagerException;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.datamanager.DataTranslator;
import gov.va.med.nhin.adapter.datamanager.Reference;
import gov.va.med.nhin.adapter.utils.NullChecker;
import java.text.ParseException;

/**
 * Converts dates returned from VistA into the passed in format. This is a
 * modified version of FileManDateDataTranslater. This new class will not reject
 * date formats and will not limit the scope of the date pattern.
 * 
 * Time formats must be simple formats limited to: Begining with hour (H, h, K,
 * k) ending in either: minutes (m), Seconds (s), MilliSeconds (S), or AM/PM
 * designator (a). All in between characters will be removed.
 * 
 * Date formats should also be simple, as no in between characters will be
 * removed. If the need arises those should be added. There are special handlers
 * for using the "pretty" format.
 * 
 * @author vhaiswchmarc
 *
 */
public class FileManDateUnfilteredDataTranslator implements DataTranslator<DataDate>
{
	static private final String DATE_FORMAT = "yyyyMMddHHmmss";
	static private final String PRETTY_FORMAT = "MMM dd, yyyy, hh:mm a";
	static private final int YEAR_LENGTH = 4;
	static private final int MONTH_LENGTH = 6;
	static private final int DAY_LENGTH = 8;

	@Override
	public DataDate translate(Object input, Object result, Reference translation, DataQuery dataQuery)
	{
		DataDate ret = null;

		if(input != null)
		{
			if(!(input instanceof java.math.BigDecimal))
			{
				try
				{
					input = java.math.BigDecimal.valueOf(Double.parseDouble((String) input));
				}
				catch(NumberFormatException e)
				{
					throw new DataManagerException("input must be of type java.math.BigDecimal");
				}
			}

			String targetDateFormat = translation.getProperty("targetDateFormat");

			if(NullChecker.isNullOrEmpty(targetDateFormat))
			{
				targetDateFormat = PRETTY_FORMAT;
			}

			String s = fixFileManDate((BigDecimal) input);

			String datePattern = DATE_FORMAT.substring(0, s.length());

			SimpleDateFormat formatter = new SimpleDateFormat(datePattern);
			try
			{
				Date d = formatter.parse(s);
				ret = new DataDate(d.getTime(), filterFormat(s.length(), targetDateFormat));
			}
			catch(ParseException t)
			{
				throw new DataManagerException("There was an error parsing the date.", t);
			}
		}

		return ret;
	}

	private String fixFileManDate(BigDecimal fileManDate)
	{
		String ret;
		double d = fileManDate.doubleValue() + 17000000;
		DecimalFormat formatter = (DecimalFormat) DecimalFormat.getNumberInstance();
		formatter.applyPattern("00000000.######");
		ret = formatter.format(d);

		int dotIndex = ret.indexOf(".");
		if(dotIndex == -1)
		{
			while(ret.endsWith("00"))
			{
				ret = ret.substring(0, ret.length() - 2);
			}
		}
		else
		{
			if((ret.length() - 9) % 2 != 0)
			{
				ret += "0";
			}

			ret = ret.substring(0, dotIndex) + ret.substring(dotIndex + 1);
		}

		return ret;
	}

	/**
	 * This method will remove unused portions of the date format. For example
	 * if there is no time do not apply time. If there is no day ...
	 * 
	 * @param length
	 * @param format
	 * @return
	 */
	private String filterFormat(final int length, final String format)
	{
		String ret;
		switch(length)
		{
		case YEAR_LENGTH:
			ret = removeMonth(format);
			break;
		case MONTH_LENGTH:
			ret = removeDay(format);
			break;
		case DAY_LENGTH:
			ret = removeTime(format);
			break;
		default:
			ret = format;
		}

		return ret.trim();
	}

	private String removeMonth(final String format)
	{
		String ret = removeDay(format);
		return ret.replace("M", "");
	}

	private String removeDay(final String format)
	{
		String ret = removeTime(format);
		// Remove "pretty" formatted day.
		ret = ret.replace("dd, ", "");

		ret = ret.replace("d", "");
		ret = ret.replace("D", "");
		ret = ret.replace("F", "");
		ret = ret.replace("E", "");
		ret = ret.replace("w", "");
		return ret;
	}

	private String removeTime(final String format)
	{
		String ret = format;

		// Remove "pretty" formatted time.
		ret = ret.replaceAll(", h.+a", "");

		// Remove Times starting with Hour 0-23
		ret = ret.replaceAll("H.+a", "");
		ret = ret.replaceAll("H.+S", "");
		ret = ret.replaceAll("H.+s", "");
		ret = ret.replaceAll("H.+m", "");

		// Remove Times starting with hour 1-24
		ret = ret.replaceAll("k.+a", "");
		ret = ret.replaceAll("k.+S", "");
		ret = ret.replaceAll("k.+s", "");
		ret = ret.replaceAll("k.+m", "");

		// Remove times starting with hour 0-11
		ret = ret.replaceAll("K.+a", "");
		ret = ret.replaceAll("K.+S", "");
		ret = ret.replaceAll("K.+s", "");
		ret = ret.replaceAll("K.+m", "");

		// Remove times starting with hour 1-12
		ret = ret.replaceAll("h.+a", "");
		ret = ret.replaceAll("h.+S", "");
		ret = ret.replaceAll("h.+s", "");
		ret = ret.replaceAll("h.+m", "");

		return ret;
	}

}
