package gov.va.med.nhin.adapter.datamanager.adapters;

import com.sun.xml.ws.client.BindingProviderProperties;
import gov.hhs.fha.nhinc.common.nhinccommon.AssertionType;
import gov.va.med.nhin.adapter.datamanager.DataAdapter;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.utils.NullChecker;
import gov.va.oit.oed.vaww.VAIdMPort;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.xml.bind.JAXBElement;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.WebServiceException;
import javax.xml.ws.soap.SOAPFaultException;
import org.hl7.v3.AD;
import org.hl7.v3.ADExplicit;
import org.hl7.v3.ActClassControlAct;
import org.hl7.v3.AdxpExplicitCity;
import org.hl7.v3.AdxpExplicitPostalCode;
import org.hl7.v3.AdxpExplicitState;
import org.hl7.v3.AdxpExplicitStreetAddressLine;
import org.hl7.v3.CE;
import org.hl7.v3.COCTMT090003UV01AssignedEntity;
import org.hl7.v3.COCTMT090003UV01Organization;
import org.hl7.v3.COCTMT150002UV01Organization;
import org.hl7.v3.COCTMT150003UV03ContactParty;
import org.hl7.v3.COCTMT150003UV03Organization;
import org.hl7.v3.CS;
import org.hl7.v3.EnExplicitFamily;
import org.hl7.v3.EnExplicitGiven;
import org.hl7.v3.II;
import org.hl7.v3.MCCIIN000002UV01;
import org.hl7.v3.MFMIMT700701UV01Custodian;
import org.hl7.v3.PNExplicit;
import org.hl7.v3.PRPAIN201301UV02;
import org.hl7.v3.PRPAIN201301UV02MFMIMT700701UV01ControlActProcess;
import org.hl7.v3.PRPAIN201301UV02MFMIMT700701UV01RegistrationEvent;
import org.hl7.v3.PRPAIN201301UV02MFMIMT700701UV01Subject1;
import org.hl7.v3.PRPAIN201301UV02MFMIMT700701UV01Subject2;
import org.hl7.v3.PRPAMT201301UV02BirthPlace;
import org.hl7.v3.PRPAMT201301UV02OtherIDs;
import org.hl7.v3.PRPAMT201301UV02Patient;
import org.hl7.v3.PRPAMT201301UV02Person;
import org.hl7.v3.ParticipationTargetSubject;
import org.hl7.v3.RoleClassContact;
import org.hl7.v3.TELExplicit;
import org.hl7.v3.TSExplicit;
import org.hl7.v3.XActMoodIntentEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author DNS  RAYD1
 */
public class MVI1301DataAdapter extends MVIDataAdapter<Map<String,String>> implements DataAdapter<Map<String,String>>
{

    private static final Logger logger = LoggerFactory.getLogger(MVI1301DataAdapter.class.getName());

    /*
	 * Input parameters from the AdapterPatientCorrelation.java:
	 * 
	 * patientId ICN correlatedPatientId correlatedAssigningAuthority
	 * correlatedAssigningFacility familyName firstName middleName
	 * space-separated names, like "Juan Maria Garcia" gender dob
	 * homeAddressStreet1 homeAddressStreet2 homeAddressStreet3 homeAddressCity
	 * homeAddressState homeAddressPostal homePhone ssn
     */
    @Override
    public List<Map<String,String>> getData(DataQuery dataQuery)
    {
        initializeProperties(dataQuery);
        List<Map<String,String>> result = new ArrayList<>();
        // Implement a requirement to retry the operation on failure, up to a
        // retry limit
        int retriesRemaining = mviRetryLimit;
        while (true) {
            try {
                // build request 1301
                PRPAIN201301UV02 request1301 = buildRequest1301(dataQuery);
                VAIdMPort port = getVAIdMPort(wsdlURL);
                Map<String, Object> requestContext = ((BindingProvider) port).getRequestContext();
                requestContext.put(BindingProviderProperties.REQUEST_TIMEOUT, mviRequestTimeout);
                requestContext.put(BindingProviderProperties.CONNECT_TIMEOUT, mviConnectTimeout);
                MCCIIN000002UV01 ack = (MCCIIN000002UV01) port.prpaIN201301UV02(request1301);
                if (NullChecker.isNotNullOrEmpty(ack.getAcknowledgement()))
                {
                    if (ack.getAcknowledgement().get(0).getTypeCode().getCode().equalsIgnoreCase("AE")) {
                        String mviAckText = (String) ack.getAcknowledgement().get(0).getAcknowledgementDetail().get(0).getText().getContent().get(0);
                        String errorMsg = "MVI returned with error code of 'AE'.  Details: '" + mviAckText + "'";

                        // {}-CCR 177986
                        logger.error(errorMsg);
                        // meaningful exception
                        throw new DataAdapterException(errorMsg);
                    }
                     // parse 1301 response
                     result = mapAckToResult(ack);
                     
                }

               if (result == null || result.isEmpty()) {
                    result = createDefaultResult("<UNKNOWN>");
                }
                break;
            } catch (SOAPFaultException sfe) {
                String msg = sfe.getMessage();
                String sfStr = sfe.getFault().getFaultString();
                String sfCode = sfe.getFault().getFaultCode();

                // CCR 179231
                throw new DataAdapterException("SOAPFaultException occurred :" + "Message: " + msg + "; " + "FaultString: " + sfStr + ";" + "FaultCode: " + sfCode, sfe);
            } catch (WebServiceException ste) {
                logger.error("WebServiceException connecting to MVI.  Message={}", ste.getMessage());

                // only execute retries on timeout
                if (ste.getCause() instanceof java.net.SocketTimeoutException) {
                    if (--retriesRemaining == 0) {
                        throw new DataAdapterException(ste);
                    } else {
                        logger.debug("Retrying connection to MVI due to SocketTimeoutException. {} retries remaining.", retriesRemaining);
                    }
                } else {
                    throw new DataAdapterException(ste);
                }
            } catch (Throwable t) {
                logger.error("There was an error getting data from MVI.  Message={}", t.getMessage());

                throw new DataAdapterException("There was an error getting data from MVI", t);
            }
        }
        return result;

    }

    private List<Map<String,String>> createDefaultResult(String defaultValue)
    {
        Map<String,String> defaultInfo = new HashMap<>();
        ArrayList<Map<String,String>> defaultResult = new ArrayList<>();

        defaultInfo.put("result", defaultValue);

        defaultResult.add(defaultInfo);

        return defaultResult;
    }

    private PRPAIN201301UV02 buildRequest1301(DataQuery dataQuery)
    {
        org.hl7.v3.ObjectFactory of = new org.hl7.v3.ObjectFactory();
        PRPAIN201301UV02 request1301 = of.createPRPAIN201301UV02();
        AssertionType assertion = (AssertionType) dataQuery.getParameter("assertion");

        String sendingFacilityOID = (String) dataQuery.getParameter("sendingFacilityOID");
        String sendingFacilityNumber = (String) dataQuery.getParameter("sendingFacilityNumber");
        String assigningAuthorityID = (String) dataQuery.getParameter("correlatedAssigningAuthority");

        // Set up message header fields
        request1301.setITSVersion(request1301.getITSVersion()); // "XML_1.0"

        request1301.setId(createMessageId(request1301.getId()));

        // Set Creation Time
        request1301.setCreationTime(createCreationTime());

        II interactionId = new II();
        interactionId.setRoot("2.16.840.1.113883.1.6"); // HL7Constants.INTERACTION_ID_ROOT
        interactionId.setExtension("PRPA_IN201301UV02");
        request1301.setInteractionId(interactionId);

        // Set Processing Code
        CS processingCode = new CS();
        processingCode.setCode(processingCodeStr);
        request1301.setProcessingCode(processingCode);

        CS processingModeCode = new CS();
        // processingModeCode.setCode("R");
        processingModeCode.setCode("T");
        request1301.setProcessingModeCode(processingModeCode);

        // Set Acknowlegment Code
        CS acceptAckCode = new CS();
        acceptAckCode.setCode("AL");
        request1301.setAcceptAckCode(acceptAckCode);

        // Set Receiver
        request1301.getReceiver().add(createReceiver());

        // Set Sender
        request1301.setSender(createSender(mviSiteKey1301, sendingFacilityOID, sendingFacilityNumber));

        // set Control Act Process block
        PRPAIN201301UV02MFMIMT700701UV01ControlActProcess controlActProcess = new PRPAIN201301UV02MFMIMT700701UV01ControlActProcess();
        controlActProcess.setMoodCode(XActMoodIntentEvent.EVN);
        controlActProcess.setClassCode(ActClassControlAct.CACT);

        // Set DataEnterer
        controlActProcess.getDataEnterer().add(createDataEnterer1(assertion));

        // set up Subject
        PRPAIN201301UV02MFMIMT700701UV01Subject1 subject = new PRPAIN201301UV02MFMIMT700701UV01Subject1();
        subject.getTypeCode().add("SUBJ");

        // set up RegistrationEvent
        PRPAIN201301UV02MFMIMT700701UV01RegistrationEvent event = new PRPAIN201301UV02MFMIMT700701UV01RegistrationEvent();
        event.getClassCode().add("REG");
        event.getMoodCode().add("EVN");

        II nullId = new II();
        nullId.getNullFlavor().add("NA");
        event.getId().add(nullId);

        // set Registration Event Status Code
        CS eventStatusCode = new CS();
        eventStatusCode.setCode("active");
        event.setStatusCode(eventStatusCode);

        PRPAIN201301UV02MFMIMT700701UV01Subject2 subject1 = new PRPAIN201301UV02MFMIMT700701UV01Subject2();
        subject1.setTypeCode(ParticipationTargetSubject.SBJ);

        // set up Patient
        PRPAMT201301UV02Patient patient = new PRPAMT201301UV02Patient();
        patient.getClassCode().add("PAT");

        // set up ICN
        II patientId = new II();
        patientId.setRoot(VA_OID);
        patientId.setExtension((String) dataQuery.getParameter("patientId"));
        patient.getId().add(patientId);

        // set Patient Status Code
        CS patientStatusCode = new CS();
        patientStatusCode.setCode("active");
        patient.setStatusCode(patientStatusCode);

        // set up person
        PRPAMT201301UV02Person person = new PRPAMT201301UV02Person();
        PNExplicit personName = new PNExplicit();
        personName.getUse().add("L");
        person.getName().add(personName);

        // set up names
        if (NullChecker.isNotNullOrEmpty((String) dataQuery.getParameter("firstName"))) {
            EnExplicitGiven firstName = new EnExplicitGiven();
            firstName.setContent((String) dataQuery.getParameter("firstName"));
            JAXBElement<EnExplicitGiven> firstNameElem = of.createENExplicitGiven(firstName);
            personName.getContent().add(firstNameElem);
        }

        if (NullChecker.isNotNullOrEmpty((String) dataQuery.getParameter("middleName"))) {
            EnExplicitGiven middleName = new EnExplicitGiven();
            middleName.setContent((String) dataQuery.getParameter("middleName"));
            JAXBElement<EnExplicitGiven> middleNameElem = of.createENExplicitGiven(middleName);
            personName.getContent().add(middleNameElem);
        }

        if (NullChecker.isNotNullOrEmpty((String) dataQuery.getParameter("familyName"))) {
            EnExplicitFamily familyName = new EnExplicitFamily();
            familyName.setContent((String) dataQuery.getParameter("familyName"));
            JAXBElement<EnExplicitFamily> familyNameElem = of.createENExplicitFamily(familyName);
            personName.getContent().add(familyNameElem);
        }

        // Mothers Maiden Name
        String mmnStr = (String) dataQuery.getParameter("mothersMaidenName");
        if (NullChecker.isNotNullOrEmpty(mmnStr)) {
            PNExplicit mmnName = new PNExplicit();
            mmnName.getUse().add("C");
            EnExplicitFamily mmnFamily = new EnExplicitFamily();
            mmnFamily.setContent(mmnStr);
            JAXBElement<EnExplicitFamily> mmnFamilyElem = of.createENExplicitFamily(mmnFamily);
            mmnName.getContent().add(mmnFamilyElem);
            person.getName().add(mmnName);
        }

        // set up Telecom
        String homePhone = (String) dataQuery.getParameter("homePhone");
        if (NullChecker.isNotNullOrEmpty(homePhone)) {
            TELExplicit telecom = new TELExplicit();
            telecom.setValue(homePhone);
            telecom.getUse().add("HP");
            person.getTelecom().add(telecom);
        }

        // set up gender
        String genderStr = (String) dataQuery.getParameter("gender");
        if (NullChecker.isNotNullOrEmpty(genderStr)) {
            CE gender = new CE();
            gender.setCode(genderStr);
            person.setAdministrativeGenderCode(gender);
        }

        // set up BirthTime
        if (NullChecker.isNotNullOrEmpty((String) dataQuery.getParameter("dob"))) {
            TSExplicit birthTime = new TSExplicit();
            birthTime.setValue((String) dataQuery.getParameter("dob"));
            person.setBirthTime(birthTime);
        }

        // set up Address
        ADExplicit addr = null;

        if (NullChecker.isNotNullOrEmpty((String) dataQuery.getParameter("homeAddressStreet1"))) {
            AdxpExplicitStreetAddressLine streetAddressLine = new AdxpExplicitStreetAddressLine();
            streetAddressLine.setContent((String) dataQuery.getParameter("homeAddressStreet1") + " " + (String) dataQuery.getParameter("homeAddressStreet2") + " " + (String) dataQuery.getParameter("homeAddressStreet3"));
            JAXBElement<AdxpExplicitStreetAddressLine> streetAddressLineElem = of.createADExplicitStreetAddressLine(streetAddressLine);
            if (addr == null) {
                addr = new ADExplicit();
            }
            addr.getContent().add(streetAddressLineElem);
        }

        if (NullChecker.isNotNullOrEmpty((String) dataQuery.getParameter("homeAddressCity"))) {
            AdxpExplicitCity city = new AdxpExplicitCity();
            city.setContent((String) dataQuery.getParameter("homeAddressCity"));
            JAXBElement<AdxpExplicitCity> cityElem = of.createADExplicitCity(city);
            if (addr == null) {
                addr = new ADExplicit();
            }
            addr.getContent().add(cityElem);
        }

        if (NullChecker.isNotNullOrEmpty((String) dataQuery.getParameter("homeAddressState"))) {
            AdxpExplicitState state = new AdxpExplicitState();
            state.setContent((String) dataQuery.getParameter("homeAddressState"));
            JAXBElement<AdxpExplicitState> stateElem = of.createADExplicitState(state);
            if (addr == null) {
                addr = new ADExplicit();
            }
            addr.getContent().add(stateElem);
        }

        if (NullChecker.isNotNullOrEmpty((String) dataQuery.getParameter("homeAddressPostal"))) {
            AdxpExplicitPostalCode postalCode = new AdxpExplicitPostalCode();
            postalCode.setContent((String) dataQuery.getParameter("homeAddressPostal"));
            if (addr == null) {
                addr = new ADExplicit();
            }
            addr.getContent().add(of.createADExplicitPostalCode(postalCode));
        }

        if (addr != null) {
            person.getAddr().add(addr);
        }

        // BEGIN - Birth Place Address
        PRPAMT201301UV02BirthPlace bp = null;
        // ADExplicit pobAddr = null;
        AD pobAddr = null;
        String pobCity = (String) dataQuery.getParameter("pobCity");
        String pobState = (String) dataQuery.getParameter("pobState");

        // pob city
        if (NullChecker.isNotNullOrEmpty(pobCity)) {
            if (bp == null) {
                bp = of.createPRPAMT201301UV02BirthPlace();
            }

            AdxpExplicitCity city = new AdxpExplicitCity();
            city.setContent(pobCity);
            JAXBElement<AdxpExplicitCity> cityElem = of.createADExplicitCity(city);
            if (pobAddr == null) {
                pobAddr = new AD();
            }
            pobAddr.getContent().add(cityElem);
            bp.setAddr(pobAddr);
        }
        // pob state
        if (NullChecker.isNotNullOrEmpty(pobState)) {
            if (bp == null) {
                bp = of.createPRPAMT201301UV02BirthPlace();
            }

            AdxpExplicitState state = new AdxpExplicitState();
            state.setContent(pobState);
            JAXBElement<AdxpExplicitState> stateElem = of.createADExplicitState(state);
            if (pobAddr == null) {
                pobAddr = new AD();
            }
            pobAddr.getContent().add(stateElem);
            bp.setAddr(pobAddr);
        }

        if (bp != null) {
            javax.xml.namespace.QName xmlqname = new javax.xml.namespace.QName("urn:hl7-org:v3", "birthPlace");
            JAXBElement<PRPAMT201301UV02BirthPlace> bpElement = new JAXBElement<PRPAMT201301UV02BirthPlace>(xmlqname, PRPAMT201301UV02BirthPlace.class, bp);
            person.setBirthPlace(bpElement);
        }
        // END - Birth Place Address

        // set up remote patient ID and scoping organization
        PRPAMT201301UV02OtherIDs correlatedId = new PRPAMT201301UV02OtherIDs();
        correlatedId.getClassCode().add("PAT");
        II correlatedIdId = new II();

        correlatedIdId.setRoot(assigningAuthorityID);
        correlatedIdId.setExtension((String) dataQuery.getParameter("correlatedPatientId") + "^NI^" + (String) dataQuery.getParameter("correlatedAssigningFacility") + "^" + assigningAuthorityID);
        correlatedId.getId().add(correlatedIdId);

        COCTMT150002UV01Organization correlatedIdScopingOrg = new COCTMT150002UV01Organization();
        correlatedIdScopingOrg.setClassCode("ORG");
        correlatedIdScopingOrg.setDeterminerCode("INSTANCE");
        II correlatedIdScopingOrgId = new II();
        correlatedIdScopingOrgId.setRoot(VA_OID);
        correlatedIdScopingOrg.getId().add(correlatedIdScopingOrgId);
        correlatedId.setScopingOrganization(correlatedIdScopingOrg);

        person.getAsOtherIDs().add(correlatedId);

        // set up SSN and scoping org
        PRPAMT201301UV02OtherIDs ssn = new PRPAMT201301UV02OtherIDs();
        ssn.getClassCode().add("SSN");
        II ssnId = new II();
        ssnId.setRoot("2.16.840.1.113883.4.1");
        ssnId.setExtension((String) dataQuery.getParameter("ssn"));
        ssn.getId().add(ssnId);

        COCTMT150002UV01Organization ssnScopingOrg = new COCTMT150002UV01Organization();
        ssnScopingOrg.setClassCode("ORG");
        ssnScopingOrg.setDeterminerCode("INSTANCE");
        II ssnScopingOrgId = new II();
        ssnScopingOrgId.setRoot("2.16.840.1.113883.4.1");
        ssnScopingOrg.getId().add(ssnScopingOrgId);
        ssn.setScopingOrganization(ssnScopingOrg);
        person.getAsOtherIDs().add(ssn);

        JAXBElement<PRPAMT201301UV02Person> patientPerson = of.createPRPAMT201301UV02PatientPatientPerson(person);
        patient.setPatientPerson(patientPerson);

        // Set Provider Organization
        COCTMT150003UV03Organization providerOrg = new COCTMT150003UV03Organization();
        providerOrg.setClassCode("ORG");
        providerOrg.setDeterminerCode("INSTANCE");

        II providerOrgId = new II();
        providerOrgId.setRoot("1.2.840.114350.1.13.99998.8734");
        providerOrg.getId().add(providerOrgId);

        providerOrg.getName().add("Good Health Clinic");

        COCTMT150003UV03ContactParty contactParty = new COCTMT150003UV03ContactParty();
        contactParty.setClassCode(RoleClassContact.CON);
        TELExplicit contactTelecom = new TELExplicit();
        contactTelecom.setValue("tel:+1-342-555-8394");
        contactParty.getTelecom().add(contactTelecom);
        providerOrg.getContactParty().add(contactParty);

        patient.setProviderOrganization(providerOrg);

        MFMIMT700701UV01Custodian custodian = new MFMIMT700701UV01Custodian();
        custodian.getTypeCode().add("CST");

        COCTMT090003UV01AssignedEntity assignedEntity = new COCTMT090003UV01AssignedEntity();
        assignedEntity.setClassCode("ASSIGNED");
        II assignedEntityId = new II();
        assignedEntityId.setExtension("MCID-123456");
        assignedEntityId.setRoot("1.2.840.114350.1.13.99998.8734");
        assignedEntity.getId().add(assignedEntityId);

        COCTMT090003UV01Organization assignedOrg = new COCTMT090003UV01Organization();
        assignedOrg.setClassCode("ORG");
        assignedOrg.setDeterminerCode("INSTANCE");
        assignedOrg.getName().add("Good Health Clinic");

        assignedEntity.setAssignedOrganization(of.createCOCTMT090003UV01AssignedEntityAssignedOrganization(assignedOrg));

        custodian.setAssignedEntity(assignedEntity);

        subject1.setPatient(patient);
        event.setSubject1(subject1);
        event.setCustodian(custodian);
        subject.setRegistrationEvent(event);
        controlActProcess.getSubject().add(subject);
        request1301.setControlActProcess(controlActProcess);

        return request1301;
    }

    private ArrayList<Map<String,String>> mapAckToResult(MCCIIN000002UV01 ack)
    {
        String resultText = ack.getAcknowledgement().get(0).getAcknowledgementDetail().get(0).getText().getRepresentation().value();

        Map<String,String> info = new HashMap<>();
        ArrayList<Map<String,String>> result = new ArrayList<>();

        info.put("result", resultText);

        result.add(info);

        return result;
    }
}
