package gov.va.med.nhin.adapter.datamanager.adapters;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.beanutils.PropertyUtils;

import gov.va.med.nhin.adapter.datamanager.DataAdapter;
import gov.va.med.nhin.adapter.datamanager.DataManagerException;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.datamanager.QueryDoesNotExistException;
import gov.va.med.nhin.adapter.utils.NullChecker;
import java.util.Map.Entry;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.slf4j.MDC;

/**
 *
 * @author David Vazquez
 */
public class AggregatorDataAdapter implements DataAdapter<Map<String, List>>
{
    private class AsyncVistACallable implements Callable<HashMap<String, List>>
    {
        private final Object driverResult;
        private final DataQuery driverQuery;
        private final DataQuery dataQuery;
        private final Map<String, String> mdc;

        public AsyncVistACallable(Object driverResult, DataQuery driverQuery,
                                  DataQuery dataQuery, final Map<String, String> mdc)
        {
            this.driverResult = driverResult;
            this.driverQuery = driverQuery;
            this.dataQuery = dataQuery;
            this.mdc = mdc;
        }

        @Override
        public HashMap<String, List> call() throws Exception
        {
            MDC.setContextMap(this.mdc);
            try {
                HashMap<String, List> result = new HashMap<>();
                String setProperties = dataQuery.getProperty("setProperties");

                for (String resultName : dataQuery.getResultBySourceNames()) {
                    DataQuery resultQuery = setUpQuery(dataQuery, resultName);

                    if (resultQuery != null) {
                        for (String rn : driverQuery.getResultNames()) {
                            if (resultQuery.isParameter(rn)) {
                                resultQuery.setParameter(rn, getValue(driverResult, rn));
                            }
                        }

                        if (!NullChecker.isNullOrEmpty(setProperties)) {
                            for (String setProperty : setProperties.split(",")) {
                                String key = dataQuery.getProperty(setProperty);
                                if (!NullChecker.isNullOrEmpty(key)) {
                                    resultQuery.setProperty(setProperty, getValue(driverResult, key).toString());
                                }
                            }
                        }

                        List resultResults = resultQuery.getResults();
                        if (!NullChecker.isNullOrEmpty(resultResults)) {
                            List storedResults = result.get(resultName);
                            if (storedResults != null) {
                                storedResults.addAll(resultResults);
                            }
                            else {
                                result.put(resultName, resultResults);
                            }
                        }
                    }
                }

                return result;
            }
            finally {
                // MDC.clear();
            }
        }
    }

    private class AsyncVistASingleQueryCallable implements Callable<List>
    {
        private final Object driverResult;
        private final DataQuery driverQuery;
        private final DataQuery dataQuery;
        private final Map<String, String> mdc;

        public AsyncVistASingleQueryCallable(Object driverResult,
                                             DataQuery driverQuery, DataQuery dataQuery, final Map<String, String> mdc)
        {
            this.driverResult = driverResult;
            this.driverQuery = driverQuery;
            this.dataQuery = dataQuery;
            this.mdc = mdc;
        }

        @Override
        public List call() throws Exception
        {
            MDC.setContextMap(mdc);
            try {
                String singleQueryName = dataQuery.getProperty("singleQueryName");
                String setProperties = dataQuery.getProperty("setProperties");

                DataQuery resultQuery = setUpQuery(dataQuery, singleQueryName);

                for (String rn : driverQuery.getResultNames()) {
                    if (resultQuery.isParameter(rn)) {
                        resultQuery.setParameter(rn, getValue(driverResult, rn));
                    }
                }

                if (!NullChecker.isNullOrEmpty(setProperties)) {
                    for (String setProperty : setProperties.split(",")) {
                        String key = dataQuery.getProperty(setProperty);
                        if (!NullChecker.isNullOrEmpty(key)) {
                            resultQuery.setProperty(setProperty, getValue(driverResult, key).toString());
                        }
                    }
                }

                return resultQuery.getResults();
            }
            finally {
                // MDC.clear();
            }
        }
    }

    @Override
    public List<Map<String, List>> getData(DataQuery dataQuery) throws Exception
    {
        ExecutorService executorService = Executors.newCachedThreadPool();
        int numSubmitted = 0;

        List<Map<String, List>> ret = new ArrayList<>();
        String driverQueryName = dataQuery.getProperty("driverQueryName");
        String singleQueryName = dataQuery.getProperty("singleQueryName");

        DataQuery driverQuery = setUpQuery(dataQuery, driverQueryName);

        List driverResults = driverQuery.getResults();

        if (!NullChecker.isNullOrEmpty(driverResults)) {
            if (NullChecker.isNullOrEmpty(singleQueryName)) {
                ExecutorCompletionService<HashMap<String, List>> completionService
                    = new ExecutorCompletionService<>(executorService);

                HashMap<String, List> result = new HashMap<>();
                ret.add(result);
                for (Object driverResult : driverResults) {
                    AsyncVistACallable callable
                        = new AsyncVistACallable(driverResult, driverQuery,
                            dataQuery, MDC.getCopyOfContextMap());

                    completionService.submit(callable);
                    numSubmitted++;
                }
                for (int i = 0; i < numSubmitted; i++) {
                    HashMap<String, List> newResults = null;
                    try {
                        newResults = completionService.take().get();
                    }
                    catch (InterruptedException | ExecutionException e) {
                        // Results are missing from at least one VistA
                        // Throw exception to return Doc Query error
                        throw new DataManagerException("An error occured while retrieving VistA results.", e);
                    }

                    if (newResults != null) {
                        for (Entry<String, List> entry : newResults.entrySet()) {
                            String resultName = entry.getKey();
                            List storedResults = result.get(resultName);
                            if (storedResults != null) {
                                storedResults.addAll(entry.getValue());
                            }
                            else {
                                result.put(resultName, entry.getValue());
                            }
                        }
                    }
                }
            }
            else {
                ExecutorCompletionService<List> completionService
                    = new ExecutorCompletionService<>(executorService);

                for (Object driverResult : driverResults) {
                    AsyncVistASingleQueryCallable callable
                        = new AsyncVistASingleQueryCallable(driverResult,
                            driverQuery, dataQuery, MDC.getCopyOfContextMap());
                    completionService.submit(callable);
                    numSubmitted++;
                }
                for (int i = 0; i < numSubmitted; i++) {
                    try {
                        List resultResults = completionService.take().get();
                        if (!NullChecker.isNullOrEmpty(resultResults)) {
                            ret.addAll(resultResults);
                        }
                    }
                    catch (InterruptedException | ExecutionException e) {
                        // Results are missing from at least one VistA
                        // Throw exception to return Doc Query error
                        throw new DataManagerException("An error occured while retrieving VistA results.", e);
                    }
                }
            }
        }

        return ret;
    }

    private DataQuery setUpQuery(DataQuery dataQuery, String queryName)
    {
        DataQuery ret = null;

        try {
            ret = dataQuery.getDataManager().getQuery(queryName);
            for (String parameterName : dataQuery.getParameterNames()) {
                if (ret.isParameter(parameterName)) {
                    ret.setParameter(parameterName, dataQuery.getParameter(parameterName));
                }
            }
        }
        catch (QueryDoesNotExistException e) {
            ret = null;
        }

        return ret;
    }

    private Object getValue(Object bean, String property)
    {
        Object ret;

        if (bean instanceof Map) {
            ret = ((Map)bean).get(property);
        }
        else {
            try {
                ret = PropertyUtils.getProperty(bean, property);
            }
            catch (Exception e) {
                ret = null;
            }
        }

        return ret;
    }
}
