/* ----------------------------------------------------------------
 * File Name 	: DateTimeUtility.java
 * Authored By	: spawaradmin
 * Created on	: Jan 26, 2009
 * 
 * This is copyrighted software of the United States Federal Government.   
 * Any use must be authorized by the
 * 
 * 	     Department of Veterans Affairs
 * 	     National Health Information Network - (NHIN)
 * 	     
 *
 * Any un-authorized use is strictly prohibited and subject to 
 * legal action.
 * 
 * Purpose	:
 * 
 * 
 * --------------------------------------------------------------------
 *                 M o d i f i c a t i o n   H i s t o r y
 * 
 * Date 	:
 * Author 	:
 * Purpose 	: 
 * 
 * 
 * -------------------------------------------------------------------- 
 */

package gov.va.med.nhin.adapter.utils.datasharing.system;

import java.text.DateFormat;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Utility class on date time. Provide static methods for date time related
 * functions. i.e. conversion between COAS format: YYYY-MM-DDTHH:MM:SS.ddd, with
 * a timezone on the end. The timezone is + or - HH:MM from GMT or just Z for
 * GMT. and ISO 8824-1987(E) format:
 * YYYY[MM[DD[HHmm[SS[.S[S[S[S]]]]]]]][+/-ZZZZ].
 * 
 */
public class DateTimeUtility {

    private final static String[] tzIndicators = {"-", "+"};

    private final static String formatFull = "yyyyMMddHHmmss.SSS";

    private final static String formatTZ = "Z";

    /**
     * Returns a TimeStamp for the current time on the server. This can be
     * useful for a client which resides in another timezone or which has
     * questionable date/time settings (like a PC). A client can base a query on
     * the servers time rather than the clients time.
     * <p>
     * The time format is: YYYYMMDDHHMMSS.SSS, with a timezone on the end. The
     * timezone is + or - ZZZZ(HHMM) from GMT or just Z for GMT.
     * 
     */
    public static String getCurrentTime() {
        SimpleDateFormat df = new SimpleDateFormat(formatFull + formatTZ);
        return df.format(Calendar.getInstance().getTime());
    }

    // From GMT time with default server timezone
    public static Date fromGMT(Date d) {
        return new Date(d.getTime() + getOffset(d, TimeZone.getDefault()));
    }
    
    //  To GMT time with default server timezone
    public static Date toGMT(Date d) {
        // get GMT time 
        return new Date(d.getTime() - getOffset(d, TimeZone.getDefault()));
    }
    // Offset based on timezone
    public static long getOffset(Date d, TimeZone thisTz) {
        long timeOffset = thisTz.getRawOffset();
        if (thisTz.inDaylightTime(d)) 
            timeOffset += thisTz.getDSTSavings();
        return timeOffset;
    }
    
    public static Date getGMTTime() {
        // get current GMT time 
        return toGMT(new Date());
    }

    /**
     * 
     * This method converts the given time from milliseconds to ISO format
     * string.
     * 
     * @param long:
     *            the time, in milliseconds
     * @param TimeZone
     *            if null, use default.
     * @return String. This is the ISO date format that corresponds to the given
     *         date in milliseconds.
     */
    public static String getISOTime(long milliseconds) {
        return getISOTime(milliseconds, null);
    } // End - getISOTime

    /**
     * 
     * This method converts the given time from milliseconds to ISO format
     * string.
     * 
     * @param long:
     *            the time, in milliseconds
     * @param TimeZone
     *            if null, use default.
     * @return String. This is the ISO date format that corresponds to the given
     *         date in milliseconds.
     * 
     */
    public static String getISOTime(long milliseconds, TimeZone timezone) {
        String format = formatFull + formatTZ;
        DateFormat df = new SimpleDateFormat(format);
        if (null != timezone) df.setTimeZone(timezone);
        return df.format(new Date(milliseconds));
    } // End - getISOTime

    /**
     * Determines if the input String contains anyone of the Strings in the vals
     * param. It returns the position of the first String in vals it finds.
     * 
     * @param str
     *            The String to scan.
     * @param vals
     *            The Strings to scan for in str.
     * @return the position of the first String in vals it finds.
     */
    private static int contains(String str, String[] vals) {
        int idx = -1;
        for (int i = 0; i < vals.length; i++) {
            if (-1 != (idx = str.indexOf(vals[i]))) return idx;
        }
        return idx;
    }

    /**
     * Method to convert ISO time to millisecond.
     * 
     * @param String
     *            ISO time String which format is
     *            YYYY[MM[DD[HHmm[SS[.S[S[S[S]]]]]]]][+/-ZZZZ]
     * @return long time in milliSecond.
     * @exception Exception
     * thrown when parse time
     */
    public static long getMilliSecond(String isoTime) {
        final String methodName = "getMilliSecond";
        int tzOffset = contains(isoTime, tzIndicators);

        String dateTime = 0 > tzOffset ? isoTime : isoTime.substring(0,
                tzOffset);
        StringBuffer format = new StringBuffer(formatFull.substring(0, dateTime
                .length()));
        if (-1 < tzOffset) format.append("Z");
        SimpleDateFormat df = new SimpleDateFormat(format.toString());
        Date date = df.parse(isoTime, new ParsePosition(0));
        if (null == date)
            throw new IllegalArgumentException("Invalid timeStr: " + isoTime);
        return date.getTime();
    }

    /**
     * Method to check if the passed in timeStamp is in the correct ISO
     * 8824-1987 time format as: YYYY[MM[DD[HHmm[SS[.S[S[S]][+|-ZZZZ]]]]]].
     * 
     * @param String
     *            passed in time stamp
     * @return boolean: true/false
     */
    public static boolean validateISOTime(String timeStamp) {
        final int TIMEZONE = 8;
        // The maximum length of the full ISO time format is 23 characters
        // the minimum length is 4 characters (year only).
        int tsLength = timeStamp.length();
        if (!(23 < tsLength || 4 > tsLength)) {
            // this regex pattern describes:
            // yyyy[MM[dd[HHmm[ss[.S[S[S]]]]]]][+/-ZHZM]
            Pattern iso8824_1987 = Pattern
                    .compile("^(\\d{4})(?:(?<=\\d{4})(\\d{2})(?:(?<=\\d{6})(\\d{2})"
                            + "(?:(?<=\\d{8})(?:(\\d{2})(\\d{2}))(?:(?<=\\d{12})(\\d{2})(?:\\.(\\d{1,3}))?)?)?)?)?"
                            + "((?:\\+|-)(\\d{2})(?<=\\d{2})(\\d{2}))?$");
            Matcher match = iso8824_1987.matcher(timeStamp);
            if (match.matches()) {
                TimeZone tz = null;
                String strTZ = match.group(TIMEZONE);
                if ("-0000".equals(strTZ))
                    timeStamp = timeStamp.replace('-', '+');
                if (null != strTZ) tz = TimeZone.getTimeZone("GMT" + strTZ);
                String ts = getISOTime(getMilliSecond(timeStamp), tz);
                if (null != strTZ) {
                    int tzOffset = contains(timeStamp, tzIndicators);
                    int tstzOff = contains(ts, tzIndicators);
                    ts = ts.substring(0, tzOffset) + ts.substring(tstzOff);
                }
                else
                    ts = ts.substring(0, timeStamp.length());
                return (ts.equals(timeStamp));
            }
        }
        return false;
    }

    /**
     * This method is to convert ISO 8601:1988 format into ISO 8824-1987 format.
     * 
     * If the input datetime is not old ISO 8601:1988 format (4th & 7th char !=
     * '-'), then the input string will be returned.
     * 
     * @param String:
     *            ISO 8601:1988 time format as: YYYY-MM-DDTHH:MM:SS.dddTZD, with
     *            a timezone TZD on the end. The timezone is +/-HH:MM from GMT
     *            or just Z for GMT.
     * @return String: ISO 8824-1987 time format as: YYYYMMDDHHmmSS.SSS+/-ZZZZ,
     *         Timezone(+/-ZZZZ) in HHMM format.
     * 
     */
    public static String convertISOTimeFormat(String oldISO) {
        int length = oldISO.length();
        // this regex pattern describes:
        // yyyy[-MM[-dd[THH:mm[:ss[.S[S[S]]]]]]][+/-ZH:ZM]
        Pattern iso8601_1988 = Pattern.compile("((\\d{4})-(\\d{2})-(\\d{2})"
                + "(?:T(\\d{2}):(\\d{2})(?::(\\d{2})(\\.\\d{1,3})??)??)??)"
                + "((?:((?:\\+|-)\\d{2}):(\\d{2}))|Z)??");
        Matcher match = iso8601_1988.matcher(oldISO);
        if (!match.matches())
            throw new IllegalArgumentException(
                    "Invalid ISO 8601-1988 time format: " + oldISO);

        String dateTime = match.group(1);
        String tzone = match.group(9);
        DateFormat dfIn = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS");
        if (null != tzone) {
            String tzoneStr = "GMT" + ("Z".equals(tzone) ? "+0000" : tzone);
            dfIn.setTimeZone(TimeZone.getTimeZone(tzone));
        }
        Date dt = dfIn.parse(dateTime, new ParsePosition(0));

        DateFormat dfOut = new SimpleDateFormat("yyyyMMddHHmmss.SSS"
                + (null != tzone ? "Z" : ""));
        return dfOut.format(dt);
    }

    /**
     * formatDBTime converts string timestamp into DB date format
     * 
     * @param oTime
     *            input time string to format
     * @return time string in database format
     */
    public static String formatDBTime(String oTime) { // format the input
                                                        // oTime
        // value into
        // the following date/time format: YYYYMMDDhhmmss
        String nTime;
        try {
            nTime = convertISOTimeFormat(oTime);
        }
        catch (Exception e) {
            nTime = getISOTime(getMilliSecond(oTime));
        }
        nTime = nTime.substring(0, 14);
        return nTime;
    }

    /**
     * formatDBLongTime
     * 
     * converts string timestamp into DB date format with millisecond
     * 
     * @param input
     *            time string to format
     * @return time string in database format
     */
    public static String formatDBLongTime(String oTime) throws Exception { // format
        // the
        // input
        // oTime
        // value
        // into
        // the following date/time format: YYYYMMDDhhmmssddd
        String nTime;
        try {
            nTime = convertISOTimeFormat(oTime);
        }
        catch (Exception e) {
            nTime = getISOTime(getMilliSecond(oTime));
        }
        nTime = nTime.substring(0, 14) + nTime.substring(15, 18);
        return nTime.toString();
    }

    /**
     * formatTimeStamp
     * 
     * converts DB format (YYYYMMDDHHMISS[ddd]) into string format
     * 
     * @param input
     *            database time string to format
     * @return new time string format
     */
    public static String formatTimeStamp(String oTime) { // format the input
        // oTime into ISO
        // format
        // YYYYMMDDHHmmSS.SSS+/-ZZZZ
        StringBuffer nTime = new StringBuffer(oTime.substring(0, 14));
        if (oTime.length() > 14)
            nTime.append(".").append(oTime.substring(14)).append("0");
        else
            nTime.append(".0000");
        String ct = getCurrentTime();
        nTime.append(ct.substring(ct.length() - 5));

        return nTime.toString();
    }

    public static void usage() {
        System.out.println("Usage: ");
        System.out.println("\tDateTimeUtility YYYYMMDDHH24MISS.SSS[-0500]  or");
        System.out.println("\tDateTimeUtility 1051765200000");
    }

    /*
     * COnvert time unit to days A string with format D90W10 is converted to
     * days D160 Checks for days, minutes, seconds, hours, weeks, months TODO:
     * Interval and Indefinite More information at RXE-22 of ICD document
     * Depends on REGEX to succeed @param supplyDaysValue - the time value (ex.
     * RXE.22 in HL7 Pharmacy message) @return String - The converted number of
     * days (prefixed with D) @exception Throws number format exception
     */
    public static String convertTimeUnitToDays(String supplyDaysValue)
            throws NumberFormatException {

        String patternSplitter = "(?i)([DMSHWLT][0-9\\.]*[^A-Za-z])|(INDEF)";
        // Initialize to -1
        double resultDays = 0;
        // Supply days value cannot be null
        if (supplyDaysValue != null) {
            // Compile patterns
            Pattern pattern = Pattern.compile(patternSplitter);
            // Match the pattern
            Matcher m = pattern.matcher(supplyDaysValue);
            // Initialize the variables
            // TODO: INDEF and Interval
            boolean isINDEF = false;
            double interval = 0;
            // The found group
            while (m.find()) {
                // Get the matching group
                String val = m.group();
                // Check if the length is greater than 1 - after D, M, S, H, W,
                // L, T
                if (val != null && val.length() > 1 && !val.equals("INDEF")) {
                    // Get the value
                    String valStr = val.substring(1);
                    if (valStr != null && valStr.length() > 1) {
                        // Days
                        if (val.charAt(0) == 'D')
                            resultDays += Double.parseDouble(valStr);
                        // Minutes - 1 minute = 0.000694444444 days
                        else if (val.charAt(0) == 'M')
                            resultDays += Double.parseDouble(valStr) * 0.000694444444;
                        // Seconds - 1 second = 0.0000115740741 days
                        else if (val.charAt(0) == 'S')
                            resultDays += Double.parseDouble(valStr) * 0.0000115740741;
                        // Hours - 1 hour = 0.0416666667 days
                        else if (val.charAt(0) == 'H')
                            resultDays += Double.parseDouble(valStr) * 0.0416666667;
                        // Weeks - 1 Week = 7 days
                        else if (val.charAt(0) == 'W')
                            resultDays += Double.parseDouble(valStr) * 7;
                        // Months - 1 Month = 30.4368499 days (might need to be
                        // changed)
                        else if (val.charAt(0) == 'L')
                            resultDays += Double.parseDouble(valStr) * 30.4368499;
                        // Interval - TODO: To implement interval
                        else if (val.charAt(0) == 'T')
                            interval = Double.parseDouble(valStr);
                    }
                    // INDEF - Indefinite - TODO: To handle INDEF
                    else if (val != null && val.equalsIgnoreCase("INDEF")) {
                        // TODO: How to process INDEF??
                        isINDEF = true;
                    }
                }
            }
            // If we get a good value, return it
            if (!Double.isNaN(resultDays) && resultDays > 0) {
                int result = (int) resultDays;
                return "D" + Integer.toString(result);
            }
        }
        // Error Occured
        return null;
    }

    // for unit test only.
    public static void main(String args[]) {

        boolean ok = validateISOTime("20041215100000.000-0000");

        String currentTime = getCurrentTime() + " "
                + TimeZone.getDefault().getDisplayName(true,TimeZone.LONG);
        SimpleDateFormat df = new SimpleDateFormat(formatFull);
        String gmtTime = df.format(getGMTTime()) + " "
                + TimeZone.getTimeZone("GMT").getDisplayName(true,TimeZone.LONG);
        System.out.println();
        System.out.println(" Your local time is " + currentTime);
        System.out.println("Current GMT time is " + gmtTime);
        System.out.println();

        // the following code can be used to convert ISO time to a long and long
        // to ISO time.
        if (args.length > 0) {
            // System.out.println("args[0]="+args[0]);
            if (args[0].equalsIgnoreCase("-d")) {
                String inDate = args[1];
                String inTime = inDate;
                if (inTime.indexOf("-") < 0 && inTime.indexOf("+") < 0)
                    inTime = inTime + "-0500";
                // from time to long
                if (validateISOTime(inTime))
                    System.out.println("Long number for ISO time " + inTime
                            + " is " + getMilliSecond(inTime));
                else {
                    System.out.println("Input " + inDate
                            + " is not a valid ISO time format");
                    System.out
                            .println("Correct ISO format is YYYYMMDDHHmmSS.SSS+/-ZZZZ");
                }
            }
            else if (args[0].equalsIgnoreCase("-t")) {
                // from long to time
                String inTime = args[1];
                String result = getISOTime(Long.parseLong(inTime));
                System.out.println("Time for the input " + inTime + " is "
                        + result);
            }
            else
                usage();
        }
        else
            usage();
    }
}