package gov.va.med.nhin.adapter.datamanager.adapters;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.security.auth.login.LoginContext;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.exception.FoundationsException;
import gov.va.med.nhin.adapter.datamanager.DataAdapter;
import gov.va.med.nhin.adapter.datamanager.DataManagerException;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.utils.NullChecker;
import gov.va.med.nhin.adapter.utils.PropertiesCollectionFactory;
import gov.va.med.vistalink.adapter.cci.VistaLinkAppProxyConnectionSpec;
import gov.va.med.vistalink.adapter.cci.VistaLinkConnection;
import gov.va.med.vistalink.adapter.cci.VistaLinkConnectionFactory;
import gov.va.med.vistalink.adapter.cci.VistaLinkConnectionSpec;
import gov.va.med.vistalink.adapter.cci.VistaLinkDuzConnectionSpec;
import gov.va.med.vistalink.adapter.record.VistaLinkFaultException;
import gov.va.med.vistalink.rpc.RpcRequest;
import gov.va.med.vistalink.rpc.RpcRequestFactory;
import gov.va.med.vistalink.rpc.RpcResponse;
import gov.va.med.vistalink.security.CallbackHandlerUnitTest;
import gov.va.med.vistalink.security.VistaKernelPrincipalImpl;

/**
 * 
 * @author David Vazquez
 */
public class VistADataAdapter implements DataAdapter
{
	public static final Logger logger = LoggerFactory.getLogger(VistADataAdapter.class.getName());

	public List getData(DataQuery dataQuery)
	{
		List ret = null;
		VistaLinkConnection connection = null;

		// CCR 177986
		logger.info("getData() invoked");

		try
		{
			connection = getConnection(dataQuery);
			setConnectionAttributes(connection);
			RpcRequest rpcRequest = buildRpcRequest(dataQuery);
			RpcResponse rpcResponse = executeRpcRequest(connection, rpcRequest);
			ret = processRpcResponse(rpcResponse);
		}
		catch(Exception e)
		{
			throw new DataAdapterException("An error occurred during data retrieval.", e);
		}
		finally
		{
			if(connection != null)
			{
				try
				{
					closeConnection(dataQuery, connection);
				}
				catch(Throwable t)
				{
				}
			}

			// CCR 177986
			logger.debug("getData() exited");
		}

		return ret;
	}

	private VistaLinkConnection getConnection(DataQuery dataQuery) throws Exception
	{
		VistaLinkConnection ret;

		// CCR 177986
		logger.info("getConnection() invoked");

		String connectionFilename = dataQuery.getProperty("connectionFilename");
		logger.debug("connection file {} ", connectionFilename);

		String connection = dataQuery.getProperty("connection");
		logger.debug("connection info {} ", connection);

		Properties propertiesCollection = PropertiesCollectionFactory.getPropertiesCollection(connectionFilename);
		Properties connectionProperties = (Properties) propertiesCollection.get(connection);
		int connectionPropertiesSize = connectionProperties.size();

		try
		{
			if(!NullChecker.isNullOrEmpty(connectionProperties.getProperty("JNDIName")))
			{
				ret = getJNDIConnection(connectionProperties);
				if(connectionProperties.size() != connectionPropertiesSize)
				{
					PropertiesCollectionFactory.storeProperties(connectionFilename, propertiesCollection);
				}
			}
			else
			{
				ret = getJAASConnection(connectionProperties);
			}
		}
		// CCR 179231-Exceptions
		catch(Exception e)
		{
			throw new DataAdapterException("An error occurred while trying to obtain VistaLink connection.", e);
		}

		return ret;
	}

	private VistaLinkConnection getJNDIConnection(Properties connectionProperties) throws Exception
	{
		VistaLinkConnectionSpec connectionSpec;

		if(NullChecker.isNullOrEmpty(connectionProperties.getProperty("username")))
		{
			// this is a hack to get a DUZ until our RPCs are tagged as
			// ok for use by APPLICATION PROXY.
			logger.debug("vista link user name {} : ", connectionProperties.getProperty("username"));

			connectionSpec = getDUZConnectionSpec(connectionProperties);
		}
		else
		{
			connectionSpec = getAppProxyConnectionSpec(connectionProperties);
		}

		Context ctx = new InitialContext();
		VistaLinkConnectionFactory connectionFactory = (VistaLinkConnectionFactory) ctx.lookup(connectionProperties.getProperty("JNDIName"));
		return (VistaLinkConnection) connectionFactory.getConnection(connectionSpec);
	}

	private VistaLinkConnectionSpec getDUZConnectionSpec(Properties connectionProperties) throws Exception
	{
		String duz = connectionProperties.getProperty("DUZ");
		if(NullChecker.isNullOrEmpty(duz))
		{
			duz = getDUZ(connectionProperties);
			connectionProperties.setProperty("DUZ", duz);
		}
		return new VistaLinkDuzConnectionSpec(connectionProperties.getProperty("division"), duz);
	}

	private synchronized String getDUZ(Properties connectionProperties) throws Exception
	{
		String jaasConfigName = connectionProperties.getProperty("JAASConfigName");
		String accessCode = connectionProperties.getProperty("accessCode");
		String verifyCode = connectionProperties.getProperty("verifyCode");
		String division = connectionProperties.getProperty("division");

		CallbackHandlerUnitTest chut = new CallbackHandlerUnitTest(accessCode, verifyCode, division);
		LoginContext loginContext = new LoginContext(jaasConfigName, chut);
		loginContext.login();
		VistaKernelPrincipalImpl userPrincipal = VistaKernelPrincipalImpl.getKernelPrincipal(loginContext.getSubject());
		String ret = userPrincipal.getUserDemographicValue(VistaKernelPrincipalImpl.KEY_DUZ);
		loginContext.logout();

		return ret;
	}

	private VistaLinkConnectionSpec getAppProxyConnectionSpec(Properties connectionProperties) throws Exception
	{
		return new VistaLinkAppProxyConnectionSpec(connectionProperties.getProperty("division"), connectionProperties.getProperty("username"));
	}

	static private final Map<String, List<VistaLinkConnection>> connectionCache = new HashMap<String, List<VistaLinkConnection>>();

	private VistaLinkConnection getJAASConnection(Properties connectionProperties) throws Exception
	{
		VistaLinkConnection ret;
		String JAASConfigName = connectionProperties.getProperty("JAASConfigName");
		String accessCode = connectionProperties.getProperty("accessCode");
		String verifyCode = connectionProperties.getProperty("verifyCode");
		String division = connectionProperties.getProperty("division");
		String key = JAASConfigName + accessCode + verifyCode + division;

		synchronized(connectionCache)
		{
			if(!NullChecker.isNullOrEmpty(connectionCache.get(key)))
			{
				ret = connectionCache.get(key).remove(0);
			}
			else
			{
				CallbackHandlerUnitTest chut = new CallbackHandlerUnitTest(accessCode, verifyCode, division);
				LoginContext loginContext = new LoginContext(JAASConfigName, chut);
				loginContext.login();
				VistaKernelPrincipalImpl userPrincipal = VistaKernelPrincipalImpl.getKernelPrincipal(loginContext.getSubject());
				ret = userPrincipal.getAuthenticatedConnection();
			}
		}

		return ret;
	}

	private void setConnectionAttributes(VistaLinkConnection connection)
	{
		connection.setTimeOut(60 * 10000);
	}

	private RpcRequest buildRpcRequest(DataQuery dataQuery) throws FoundationsException
	{
		String rpcContext = dataQuery.getProperty("rpcContext");
		String rpcName = dataQuery.getProperty("rpcName");
		RpcRequest rpcRequest = null;
		try
		{ // CCR 1779231- added the try catch block
			rpcRequest = RpcRequestFactory.getRpcRequest(rpcContext, rpcName);
			ArrayList params = new ArrayList();

			// CCR 177986
			logger.debug("RPC Context {} : ", rpcContext);
			logger.debug("RPC Name {}: ", rpcName);

			for(String paramName : dataQuery.getParameterNames())
			{
				Object parameter = dataQuery.getParameter(paramName);
				params.add(parameter != null ? parameter : "");
				// fix for fortify issue - Privacy Violation: RTC ticket #
				// 162996
				// logger.finest("Rpc adding " + paramName + " with value: " +
				// parameter);
			}

			rpcRequest.setParams(params);
            try{
                logger.debug("Sending request: {}", rpcRequest.getRequestString());
            }catch(Exception ex){
                logger.debug("Error getting request String: {}" + rpcRequest.toString());
            }
		}
		catch(Exception e)
		{
			throw new DataAdapterException("Error getting request String", e);
		}

		return rpcRequest;
	}

	private RpcResponse executeRpcRequest(VistaLinkConnection connection, RpcRequest rpcRequest) throws FoundationsException, VistaLinkFaultException
	{
		RpcResponse rpcResponse;

		// CCR 177986
		logger.debug(getClass().getName() + "{}", "executeRpcRequest");

		try
		{
			rpcResponse = connection.executeRPC(rpcRequest);

			// {} if enabled CCR 177986
			logger.debug("rpc response {} ", rpcResponse.toString());
			logger.debug("Request returned: {}", rpcResponse.getRawResponse());
		}
		catch(Exception e)
		{
			logger.debug("Error getting raw response {} ", rpcRequest.toString());
			throw new DataAdapterException("Error getting raw response from request", e);
		}

		return rpcResponse;
	}

	private List processRpcResponse(RpcResponse rpcResponse) throws DataManagerException
	{
		List ret = new ArrayList<String>();
		ret.add(rpcResponse.getResults());
		return ret;
	}

	private void closeConnection(DataQuery dataQuery, VistaLinkConnection vistaLinkConnection) throws Exception
	{
		String connectionFilename = dataQuery.getProperty("connectionFilename");
		String connection = dataQuery.getProperty("connection");

		// CCR177986
		logger.debug("connection file name {} ", connectionFilename);
		logger.debug("connection {} ", connection);

		Properties propertiesCollection = PropertiesCollectionFactory.getPropertiesCollection(connectionFilename);
		Properties connectionProperties = (Properties) propertiesCollection.get(connection);

		if(!NullChecker.isNullOrEmpty(connectionProperties.getProperty("JNDIName")))
		{
			logger.debug("connection props {} ", connectionProperties.getProperty("JNDIName"));

			vistaLinkConnection.close();
		}
		else
		{
			closeJAASConnection(connectionProperties, vistaLinkConnection);
		}
	}

	private void closeJAASConnection(Properties connectionProperties, VistaLinkConnection vistaLinkConnection)
	{
		String JAASConfigName = connectionProperties.getProperty("JAASConfigName");
		String accessCode = connectionProperties.getProperty("accessCode");
		String verifyCode = connectionProperties.getProperty("verifyCode");
		String division = connectionProperties.getProperty("division");
		String key = JAASConfigName + accessCode + verifyCode + division;

		logger.debug("key info {} ", key);

		synchronized(connectionCache)
		{
			List<VistaLinkConnection> connectionsList = connectionCache.get(key);
			if(connectionsList == null)
			{
				connectionsList = new ArrayList<VistaLinkConnection>();
				connectionCache.put(key, connectionsList);
			}
			connectionsList.add(vistaLinkConnection);
		}
	}
}
