package gov.va.med.nhin.adapter.adaptergateway.docquery;

import java.math.BigInteger;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.TimeZone;
import java.util.UUID;

import javax.ejb.EJB;
import javax.ejb.*;
import javax.jws.WebService;
import javax.xml.bind.JAXBElement;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.hhs.fha.nhinc.adapterpolicyengine.AdapterPolicyEnginePortType;
import gov.hhs.fha.nhinc.common.nhinccommon.AssertionType;
import gov.hhs.fha.nhinc.common.nhinccommon.CeType;
import gov.hhs.fha.nhinc.common.nhinccommon.HomeCommunityType;
import gov.hhs.fha.nhinc.common.nhinccommon.NhinTargetCommunitiesType;
import gov.hhs.fha.nhinc.common.nhinccommon.NhinTargetCommunityType;
import gov.hhs.fha.nhinc.common.nhinccommon.PersonNameType;
import gov.hhs.fha.nhinc.common.nhinccommon.SamlIssuerType;
import gov.hhs.fha.nhinc.common.nhinccommon.UserType;
import gov.hhs.fha.nhinc.common.nhinccommonadapter.CheckPolicyRequestType;
import gov.hhs.fha.nhinc.common.nhinccommonadapter.CheckPolicyResponseType;
import gov.hhs.fha.nhinc.common.nhinccommonentity.RespondingGatewayCrossGatewayQueryRequestType;
import gov.hhs.fha.nhinc.entitydocquery.EntityDocQueryPortType;
import gov.va.med.nhin.adapter.adaptergateway.util.PrefetchConstants;
import gov.va.med.nhin.adapter.audit.requests.RequestActions;
import gov.va.med.nhin.adapter.audit.requests.RequestAudit;
import gov.va.med.nhin.adapter.audit.requests.RequestAuditEntity;
import gov.va.med.nhin.adapter.datamanager.DataManager;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.datamanager.ejb.DataManagerLocal;
import gov.va.med.nhin.adapter.documentrepository.Document;
import gov.va.med.nhin.adapter.documentrepository.DocumentRepository;
import gov.va.med.nhin.adapter.documentrepository.DocumentRepositoryLocal;
import gov.va.med.nhin.adapter.facilitymanager.Facility;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManager;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManagerLocal;
import gov.va.med.nhin.adapter.logging.CheckPolicy;
import gov.va.med.nhin.adapter.logging.ErrorMessage;
import gov.va.med.nhin.adapter.logging.EventAuditingFactory;
import gov.va.med.nhin.adapter.logging.EventAuditingFactoryImpl;
import gov.va.med.nhin.adapter.logging.LogConstants.AuditingEvent;
import gov.va.med.nhin.adapter.logging.MaintLog;
import gov.va.med.nhin.adapter.logging.MessagingHelper;
import gov.va.med.nhin.adapter.logging.MessagingHelper.QueryType;
import gov.va.med.nhin.adapter.mvi.hl7parsers.HL7Parser201306;
import gov.va.med.nhin.adapter.policyengine.AdapterPolicyEnginePortTypeLocal;
import gov.va.med.nhin.adapter.policyengine.CheckPolicyRequestBuilder;
import gov.va.med.nhin.adapter.prefetch.PrefetchQueryResponse;
import gov.va.med.nhin.adapter.propertylookup.PropertyLookup;
import gov.va.med.nhin.adapter.propertylookup.PropertyLookupLocal;
import gov.va.med.nhin.adapter.utils.NullChecker;
import gov.va.med.nhin.adapter.utils.prefetch.PrefetchHelper;
import javax.jws.HandlerChain;
import oasis.names.tc.ebxml_regrep.xsd.query._3.AdhocQueryRequest;
import oasis.names.tc.ebxml_regrep.xsd.query._3.AdhocQueryResponse;
import oasis.names.tc.ebxml_regrep.xsd.query._3.ResponseOptionType;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.AdhocQueryType;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.ClassificationType;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.ExternalIdentifierType;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.ExtrinsicObjectType;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.IdentifiableType;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.LocalizedStringType;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.RegistryObjectListType;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.SlotType1;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.ValueListType;
import oasis.names.tc.ebxml_regrep.xsd.rs._3.RegistryError;
import oasis.names.tc.ebxml_regrep.xsd.rs._3.RegistryErrorList;
import oasis.names.tc.xacml._2_0.context.schema.os.RequestType;
import org.hl7.fhir.dstu3.model.AuditEvent;

/**
 *
 * @author spawaradmin
 */
@WebService(serviceName = "EntityDocQuery",
            portName = "EntityDocQueryPortSoap11",
            endpointInterface = "gov.hhs.fha.nhinc.entitydocquery.EntityDocQueryPortType",
            targetNamespace = "urn:gov:hhs:fha:nhinc:entitydocquery"/*,
            wsdlLocation = "META-INF/wsdl/EntityDocQuery.wsdl"*/)
@HandlerChain(file = "SOAPHandlerChain.xml")
@TransactionAttribute(value = TransactionAttributeType.SUPPORTS)
@Stateless(name = "EntityDocQueryBean")
public class EntityDocQueryBean implements EntityDocQueryPortTypeLocal
{
    private static final Logger logger = LoggerFactory.getLogger(EntityDocQueryBean.class.getName());

    private static final String EBXML_DOCENTRY_PATIENT_ID = "$XDSDocumentEntryPatientId";
    private static final String EBXML_DOCENTRY_CLASS_CODE = "$XDSDocumentEntryClassCode";
    private static final String EBXML_DOCENTRY_SERVICE_START_TIME_FROM = "$XDSDocumentEntryServiceStartTimeFrom";
    private static final String EBXML_DOCENTRY_SERVICE_START_TIME_TO = "$XDSDocumentEntryServiceStartTimeTo";
    private static final String EBXML_DOCENTRY_SERVICE_STOP_TIME_FROM = "$XDSDocumentEntryServiceStopTimeFrom";
    private static final String EBXML_DOCENTRY_SERVICE_STOP_TIME_TO = "$XDSDocumentEntryServiceStopTimeTo";
    private static final String EBXML_DOCENTRY_STATUS = "$XDSDocumentEntryStatus";
    private static final String EBXML_DOCENTRY_FORMAT_CODE = "$XDSDocumentEntryFormatCode";

    private static final String EBXML_RESPONSE_REPOSITORY_UNIQUE_ID_SLOTNAME = "repositoryUniqueId";
    private static final String EBXML_RESPONSE_DOCID_IDENTIFICATION_SCHEME = "urn:uuid:2e82c1f6-a085-4c72-9da3-8640a32e42ab";
    private static final String EBXML_RESPONSE_DOCID_NAME = "XDSDocumentEntry.uniqueId";
    private static final String EBXML_RESPONSE_PATIENTID_IDENTIFICATION_SCHEME = "urn:uuid:58a6f841-87b3-4a3e-92fd-a8ffeff98427";
    private static final String EBXML_RESPONSE_CLASSCODE_CLASS_SCHEME = "urn:uuid:41a5887f-8865-4c09-adf7-e362475b143a";
    private static final String EBXML_RESPONSE_FORMATCODE_CLASS_SCHEME = "urn:uuid:a09d5840-386c-46f2-b5ad-9c3699a4309d";
    private static final String EBXML_RESPONSE_CREATIONTIME_SLOTNAME = "creationTime";
    private static final String EBXML_RESPONSE_HASH_SLOTNAME = "hash";
    private static final String EBXML_RESPONSE_SERVICESTARTTIME_SLOTNAME = "serviceStartTime";
    private static final String EBXML_RESPONSE_SERVICESTOPTIME_SLOTNAME = "serviceStopTime";
    private static final String EBXML_RESPONSE_SIZE_SLOTNAME = "size";
    private static final String DATE_FORMAT_FULL = "yyyyMMddhhmmssZ";
    private static final String DATE_FORMAT_SERVICE = "yyyyMMdd";
    private static final String PREFETCH_DOCUMENT_QUERY_URL = "PREFETCH_DOCUMENT_QUERY_URL";
    private static final String PREFETCH_CONNECTION_TIMEOUT_SECONDS = "PREFETCH_CONNECTION_TIMEOUT_SECONDS";
    private static final String PREFETCH_READ_TIMEOUT_SECONDS = "PREFETCH_READ_TIMEOUT_SECONDS";

    private PropertyLookup propertyLookup;
    private FacilityManager facilityManager;
    private AdapterPolicyEnginePortType adapterPolicyEngine;
    private DocumentRepository documentRepository;
    private DataManager dataManager;
    private EntityDocQueryPortType entityDocQuery;
    private RequestAudit requestAudit;

    @EJB(beanInterface = PropertyLookupLocal.class, beanName = "PropertyFileLookup")
    public void setPropertyLookup(PropertyLookup propertyLookup)
    {
        this.propertyLookup = propertyLookup;
    }

    @EJB(beanInterface = FacilityManagerLocal.class, beanName = "FacilityManager")
    public void setFacilityManager(FacilityManager facilityManager)
    {
        this.facilityManager = facilityManager;
    }

    @EJB(beanInterface = DocumentRepositoryLocal.class, beanName = "DocumentRepository")
    public void setDocumentRepository(DocumentRepository documentRepository)
    {
        this.documentRepository = documentRepository;
    }

    @EJB(beanInterface = AdapterPolicyEnginePortTypeLocal.class, beanName = "AdapterPolicyEngine")
    public void setAdapterPolicyEngine(AdapterPolicyEnginePortType adapterPolicyEngine)
    {
        this.adapterPolicyEngine = adapterPolicyEngine;
    }

    @EJB(beanInterface = EntityDocQueryPortTypeLocal.class, beanName = "EntityDocQueryOrch")
    public void setEntityDocQuery(EntityDocQueryPortType entityDocQuery)
    {
        this.entityDocQuery = entityDocQuery;
    }

    @EJB(beanInterface = DataManagerLocal.class, beanName = "DataManager")
    public void setDataManager(DataManager dataManager)
    {
        this.dataManager = dataManager;
    }

    @EJB(beanInterface = RequestAudit.class, beanName = "RequestAudit")
    public void setRequestAudit(RequestAudit requestAudit)
    {
        this.requestAudit = requestAudit;
    }

    @Override
    public AdhocQueryResponse respondingGatewayCrossGatewayQuery(RespondingGatewayCrossGatewayQueryRequestType respondingGatewayCrossGatewayQueryRequest)
    {
        EventAuditingFactory<AuditEvent> afac
            = EventAuditingFactoryImpl.getFactory(AuditEvent.class);
        afac.info(afac.newEvent(respondingGatewayCrossGatewayQueryRequest.getAssertion(),
            AuditingEvent.QDOUT_BEGIN, getClass()));

        AdhocQueryResponse ret = new AdhocQueryResponse();
        AdhocQueryRequest body = respondingGatewayCrossGatewayQueryRequest.getAdhocQueryRequest();
        AssertionType assertion = respondingGatewayCrossGatewayQueryRequest.getAssertion();
        NhinTargetCommunitiesType nhinTargetCommunities = respondingGatewayCrossGatewayQueryRequest.getNhinTargetCommunities();

        String partnerCommunityId = respondingGatewayCrossGatewayQueryRequest.getAssertion().getHomeCommunity().getHomeCommunityId();
        logger.debug("Entering Outbound DocQuery call and Partner OID is {} ", partnerCommunityId); // CCR
        // 177986

        boolean callFromPrefetch = false;
        String checkPrefetchCache = propertyLookup.getProperty(PrefetchConstants.PROPERTY_CHECK_PREFETCH_CACHE);
        boolean checkPrefetchCacheBool = Boolean.parseBoolean(checkPrefetchCache);
        if (checkPrefetchCacheBool) {
            logger.trace("checking if the call is for prefetch " + partnerCommunityId);
            String prefetchHCID = propertyLookup.getProperty(PrefetchConstants.PROPERTY_PREFETCH_HOME_COMMUNITY_ID);
            String prefetchUserId = propertyLookup.getProperty(PrefetchConstants.PROPERTY_PREFETCH_USER_ID);
            String userIdInRequest = assertion.getUserInfo().getUserName();
            if (prefetchHCID != null && partnerCommunityId.equalsIgnoreCase(prefetchHCID) && userIdInRequest != null && userIdInRequest.equalsIgnoreCase(prefetchUserId)) {
                callFromPrefetch = true;
            }
        }

        AuditEvent event = afac.newEvent(AuditingEvent.QDOUT_PARTNERAUTH, getClass());
        try {
            if (checkPolicy(assertion, event)) {
                event.setOutcome(AuditEvent.AuditEventOutcome._0);
                afac.info(event);

                Map<String, List<String>> slotMap = getMapFromSlots(body.getAdhocQuery().getSlot());
                logger.trace("slotMap:" + slotMap);

                String patientId = slotMap.get(EBXML_DOCENTRY_PATIENT_ID).get(0);
                boolean prefetched = false;
                AdhocQueryResponse result = null;

                if (!callFromPrefetch && checkPrefetchCacheBool) {
                    // check if cached query exist in prefetch cache.
                    logger.trace("checking if the prefetch data exist" + partnerCommunityId);
                    String patientICN = HL7Parser201306.extractICNValue(patientId);
                    String loinc = formatCodesForPrefetch(slotMap.get(EBXML_DOCENTRY_CLASS_CODE));
                    logger.trace("patientICN:" + patientICN + ",loinc:" + loinc);
                    String url = propertyLookup.getProperty(PREFETCH_DOCUMENT_QUERY_URL);
                    String connectTimeout = propertyLookup.getProperty(PREFETCH_CONNECTION_TIMEOUT_SECONDS);
                    String readTimeout = propertyLookup.getProperty(PREFETCH_READ_TIMEOUT_SECONDS);
                    Properties props = new Properties();
                    if (NullChecker.isNotNullOrEmpty(url)) {
                        props.setProperty("prefetch.service.url", url);
                    }
                    if (NullChecker.isNotNullOrEmpty(connectTimeout)) {
                        props.setProperty("prefetch.connect.timeout", connectTimeout);
                    }
                    if (NullChecker.isNotNullOrEmpty(readTimeout)) {
                        props.setProperty("prefetch.read.timeout", readTimeout);
                    }
                    PrefetchQueryResponse prefetchResponse = PrefetchHelper.queryDocuments(props, patientICN, loinc);
                    if (prefetchResponse != null) {
                        prefetched = true;
                        ret = createAdhocQueryResponse(body, prefetchResponse);
                    }

                }
                if (!prefetched) {
                    gov.hhs.fha.nhinc.common.nhinccommonentity.ObjectFactory objFactory = new gov.hhs.fha.nhinc.common.nhinccommonentity.ObjectFactory();
                    RespondingGatewayCrossGatewayQueryRequestType request = objFactory.createRespondingGatewayCrossGatewayQueryRequestType();
                    request.setAdhocQueryRequest(createAdhocQueryRequest(slotMap));
                    request.setAssertion(createAssertion(assertion, patientId));
                    request.setNhinTargetCommunities(createNhinTargetCommunities(nhinTargetCommunities));
                    result = entityDocQuery.respondingGatewayCrossGatewayQuery(request);
                    ret = createAdhocQueryResponse(body, result, patientId);
                }
            }
            else {
                event.setOutcome(AuditEvent.AuditEventOutcome._4);
                event.setOutcomeDesc("Failed policy check");
                afac.info(event);

                ret = createAdhocQueryResponseError(body);
                auditError(respondingGatewayCrossGatewayQueryRequest, "Error: Failed Policy Check");
            }
        }
        catch (Throwable t) {
            event.setOutcome(AuditEvent.AuditEventOutcome._8);
            event.setOutcomeDesc(t.getLocalizedMessage());
            afac.error(event);

            logger.error(getClass().getName(), "respondingGatewayCrossGatewayQuery", t);
            ret = createAdhocQueryResponseError(body);
            MaintLog.queryError(respondingGatewayCrossGatewayQueryRequest, ErrorMessage.OUT_DQ_UNKNOWN, t.getMessage(), logger);
            auditError(respondingGatewayCrossGatewayQueryRequest, "Error: " + t.getMessage());
        }

        afac.info(AuditingEvent.QDOUT_END, getClass());
        return ret;
    }

    private void auditError(final RespondingGatewayCrossGatewayQueryRequestType request, final String errorMsg)
    {
        RequestAuditEntity auditEntity = new RequestAuditEntity();
        auditEntity.setStartTime(new Date());
        auditEntity.generateUUID();
        String partnerCommunityId;
        if (request != null && request.getAssertion() != null && request.getAssertion().getHomeCommunity() != null) {
            partnerCommunityId = request.getAssertion().getHomeCommunity().getHomeCommunityId();
        }
        else {
            partnerCommunityId = "UNKNOWN";
        }

        auditEntity.setHcid(partnerCommunityId);

        AdhocQueryRequest body = request.getAdhocQueryRequest();
        if (body != null) {
            Map<String, List<String>> slotMap = getMapFromSlots(body.getAdhocQuery().getSlot());
            String patientId = slotMap.get(EBXML_DOCENTRY_PATIENT_ID).get(0);

            auditEntity.setPid(patientId);
        }
        else {
            auditEntity.setPid("UNKNOWN");
        }

        auditEntity.setAction(RequestActions.PARENT_REQUEST.getValue());

        auditEntity.setStopTime(new Date());
        auditEntity.setStatus(errorMsg);
        requestAudit.storeAudit(auditEntity);
    }

    private AdhocQueryRequest createAdhocQueryRequest(Map<String, List<String>> params)
    {
        AdhocQueryRequest ret = new AdhocQueryRequest();
        ret.setFederated(false);
        ret.setStartIndex(BigInteger.valueOf(0));
        ret.setMaxResults(BigInteger.valueOf(-1));
        ret.setAdhocQuery(createAdhocQuery(params));
        ret.setResponseOption(createResponseOption());
        return ret;
    }

    private AdhocQueryType createAdhocQuery(Map<String, List<String>> params)
    {
        AdhocQueryType ret = new AdhocQueryType();
        ret.setHome("urn:oid:" + propertyLookup.getProperty("HomeCommunityId"));
        ret.setId("urn:uuid:14d4debf-8f97-4251-9a74-a90016b0af0d");
        ret.getSlot().addAll(createSlots(params));
        return ret;
    }

    private List<SlotType1> createSlots(Map<String, List<String>> params)
    {
        ArrayList<SlotType1> ret = new ArrayList<>();

        String patientId = params.get(EBXML_DOCENTRY_PATIENT_ID).get(0);
        SlotType1 xdsEntryPatientId = new SlotType1();
        xdsEntryPatientId.setName(EBXML_DOCENTRY_PATIENT_ID);
        xdsEntryPatientId.setValueList(createValueList("'" + patientId + "'"));
        ret.add(xdsEntryPatientId);

        List<String> classCodes = params.get(EBXML_DOCENTRY_CLASS_CODE);
        if (classCodes != null) {
            SlotType1 xdsEntryClassCode = new SlotType1();
            xdsEntryClassCode.setName(EBXML_DOCENTRY_CLASS_CODE);
            xdsEntryClassCode.setValueList(createValueList(qualifyClassCodes(classCodes)));
            ret.add(xdsEntryClassCode);
        }

        if (params.get(EBXML_DOCENTRY_SERVICE_START_TIME_FROM) != null) {
            String startTime = params.get(EBXML_DOCENTRY_SERVICE_START_TIME_FROM).get(0);
            SlotType1 xdsEntryServiceStartTimeFrom = new SlotType1();
            xdsEntryServiceStartTimeFrom.setName(EBXML_DOCENTRY_SERVICE_START_TIME_FROM);
            xdsEntryServiceStartTimeFrom.setValueList(createValueList(startTime));
            ret.add(xdsEntryServiceStartTimeFrom);
        }

        if (params.get(EBXML_DOCENTRY_SERVICE_START_TIME_TO) != null) {
            String endTime = params.get(EBXML_DOCENTRY_SERVICE_START_TIME_TO).get(0);
            SlotType1 xdsEntryServiceStopTimeTo = new SlotType1();
            xdsEntryServiceStopTimeTo.setName(EBXML_DOCENTRY_SERVICE_START_TIME_TO);
            xdsEntryServiceStopTimeTo.setValueList(createValueList(endTime));
            ret.add(xdsEntryServiceStopTimeTo);
        }

        if (params.get(EBXML_DOCENTRY_SERVICE_STOP_TIME_FROM) != null) {
            String startTime = params.get(EBXML_DOCENTRY_SERVICE_STOP_TIME_FROM).get(0);
            SlotType1 xdsEntryServiceStartTimeFrom = new SlotType1();
            xdsEntryServiceStartTimeFrom.setName(EBXML_DOCENTRY_SERVICE_STOP_TIME_FROM);
            xdsEntryServiceStartTimeFrom.setValueList(createValueList(startTime));
            ret.add(xdsEntryServiceStartTimeFrom);
        }

        if (params.get(EBXML_DOCENTRY_SERVICE_STOP_TIME_TO) != null) {
            String endTime = params.get(EBXML_DOCENTRY_SERVICE_STOP_TIME_TO).get(0);
            SlotType1 xdsEntryServiceStopTimeTo = new SlotType1();
            xdsEntryServiceStopTimeTo.setName(EBXML_DOCENTRY_SERVICE_STOP_TIME_TO);
            xdsEntryServiceStopTimeTo.setValueList(createValueList(endTime));
            ret.add(xdsEntryServiceStopTimeTo);
        }

        if (params.get(EBXML_DOCENTRY_FORMAT_CODE) != null) {
            List<String> formatCode = params.get(EBXML_DOCENTRY_FORMAT_CODE);
            SlotType1 xdsEntryFormatCode = new SlotType1();
            xdsEntryFormatCode.setName(EBXML_DOCENTRY_FORMAT_CODE);
            xdsEntryFormatCode.setValueList(createValueList(qualifyFormatCodes(formatCode)));
            ret.add(xdsEntryFormatCode);
        }

        SlotType1 xdsDocEntryStatus = new SlotType1();
        xdsDocEntryStatus.setName(EBXML_DOCENTRY_STATUS);
        xdsDocEntryStatus.setValueList(createValueList("('urn:oasis:names:tc:ebxml-regrep:StatusType:Approved','urn:ihe:iti:2010:StatusType:DeferredCreation')"));
        ret.add(xdsDocEntryStatus);

        return ret;
    }

    private ValueListType createValueList(String fromString)
    {
        ValueListType ret = new ValueListType();
        ret.getValue().add(fromString);
        return ret;
    }

    private ValueListType createValueList(List<String> fromStrings)
    {
        ValueListType ret = new ValueListType();
        for (String s : fromStrings) {
            ret.getValue().add(s);
        }
        return ret;
    }

    private List<String> qualifyClassCodes(List<String> classCodes)
    {
        List<String> ret = new ArrayList<>();
        String docQueryClassCodeScheme = propertyLookup.getProperty("docQueryClassCodeScheme");
        for (String classCode : classCodes) {
            if (!NullChecker.isNullOrEmpty(docQueryClassCodeScheme) && !classCode.endsWith("^^" + docQueryClassCodeScheme)) {
                ret.add("'" + classCode + "^^" + docQueryClassCodeScheme + "'");
            }
            else {
                ret.add("'" + classCode + "'");
            }
        }
        return ret;
    }

    private List<String> qualifyFormatCodes(List<String> formatCodes)
    {
        List<String> ret = new ArrayList<>();
        String docQueryFormatCodeScheme = propertyLookup.getProperty("docQueryFormatCodeScheme");
        for (String formatCode : formatCodes) {
            if (!NullChecker.isNullOrEmpty(docQueryFormatCodeScheme) && !formatCode.endsWith("^^" + docQueryFormatCodeScheme)) {
                ret.add("'" + formatCode + "^^" + docQueryFormatCodeScheme + "'");
            }
            else {
                ret.add("'" + formatCode + "'");
            }
        }
        return ret;
    }

    private String formatCodesForPrefetch(List<String> formatCodes)
    {
        String ret = "";
        String docQueryClassCodeScheme = propertyLookup.getProperty("docQueryClassCodeScheme");
        int counter = 0;
        for (String formatCode : formatCodes) {
            String loincCode = "";
            if (!NullChecker.isNullOrEmpty(docQueryClassCodeScheme) && formatCode.endsWith("^^" + docQueryClassCodeScheme)) {
                int index = formatCode.indexOf("^^");
                loincCode = formatCode.substring(0, index);
                if (counter == 0) {
                    ret = loincCode;
                }
                else {
                    ret += "+" + loincCode;
                }
                counter++;
            }
        }
        return ret;
    }

    private ResponseOptionType createResponseOption()
    {
        ResponseOptionType ret = new ResponseOptionType();
        ret.setReturnComposedObjects(true);
        ret.setReturnType("LeafClass");
        return ret;
    }

    private AssertionType createAssertion(AssertionType fromAssertion, String patientId)
    {
        AssertionType ret = new AssertionType();
        Facility homeFacility = getHomeFacility();

        ret.setAuthorized(true);

        HomeCommunityType homeCommunity = new HomeCommunityType();
        homeCommunity.setHomeCommunityId(homeFacility.getFullHomeCommunityId());
        homeCommunity.setName(homeFacility.getFacilityName());
        ret.setHomeCommunity(homeCommunity);

        UserType user = new UserType();
        CeType roleCoded = new CeType();
        roleCoded.setCode("224608005");
        if (fromAssertion.getUserInfo() != null) {
            user.setUserName(fromAssertion.getUserInfo().getUserName());
            user.setPersonName(fromAssertion.getUserInfo().getPersonName());
            user.setOrg(fromAssertion.getUserInfo().getOrg());
            if (fromAssertion.getUserInfo().getRoleCoded() != null) {
                roleCoded.setCode(fromAssertion.getUserInfo().getRoleCoded().getCode());
            }
        }
        else {
            user.setUserName("VA_USER");
            PersonNameType personName = new PersonNameType();
            personName.setGivenName("VA");
            personName.setFamilyName("User");
            user.setPersonName(personName);
        }

        if (user.getPersonName() != null && (NullChecker.isNotNullOrEmpty(user.getPersonName().getFamilyName()) || NullChecker.isNotNullOrEmpty(user.getPersonName().getSecondNameOrInitials()) || NullChecker.isNotNullOrEmpty(user.getPersonName().getGivenName()))) {
            user.setUserName(user.getUserName() + ", CN=" + createFullName(user.getPersonName()) + ", O=" + homeFacility.getFacilityName());
        }
        else {
            user.setUserName(user.getUserName() + ", CN=" + user.getUserName() + ", O=" + homeFacility.getFacilityName());
        }

        roleCoded.setCodeSystem("2.16.840.1.113883.6.96");
        roleCoded.setCodeSystemName("SNOMED_CT");

        DataQuery query = dataManager.getQuery("STS.lookupSNOMED");
        query.setParameter("code", roleCoded.getCode());
        List<Map> results = query.getResults();
        roleCoded.setDisplayName((String)results.get(0).get("name"));
        user.setRoleCoded(roleCoded);
        // if (user.getOrg() == null) {
        user.setOrg(homeCommunity);
        // }
        ret.setUserInfo(user);

        ret.getUniquePatientId().add(patientId);

        CeType p = new CeType();
        p.setCodeSystem("2.16.840.1.113883.3.18.7.1");
        p.setCodeSystemName("nhin-purpose");
        if (fromAssertion.getPurposeOfDisclosureCoded() != null) {
            p.setCode(fromAssertion.getPurposeOfDisclosureCoded().getCode());
            if (NullChecker.isNotNullOrEmpty(fromAssertion.getPurposeOfDisclosureCoded().getDisplayName())) {
                p.setDisplayName(fromAssertion.getPurposeOfDisclosureCoded().getDisplayName());
            }
        }
        else {
            p.setCode("TREATMENT");
            p.setDisplayName("Treatment");
        }
        ret.setPurposeOfDisclosureCoded(p);

        SamlIssuerType sit = new SamlIssuerType();
        sit.setIssuer(propertyLookup.getProperty("AssertionIssuer"));
        sit.setIssuerFormat(propertyLookup.getProperty("AssertionIssuerFormat"));
        ret.setSamlIssuer(sit);

        ret.setMessageId("urn:uuid:" + UUID.randomUUID().toString());

        return ret;
    }

    private NhinTargetCommunitiesType createNhinTargetCommunities(NhinTargetCommunitiesType fromNhinTargetCommunities)
    {
        NhinTargetCommunitiesType ret = null;

        if (fromNhinTargetCommunities != null
            && NullChecker.isNotNullOrEmpty(fromNhinTargetCommunities.getNhinTargetCommunity())) {
            ret = new NhinTargetCommunitiesType();
            ret.setUseSpecVersion(fromNhinTargetCommunities.getUseSpecVersion());
            for (NhinTargetCommunityType fromNhinTargetCommunity : fromNhinTargetCommunities.getNhinTargetCommunity()) {
                NhinTargetCommunityType nhinTargetCommunity = new NhinTargetCommunityType();
                HomeCommunityType homeCommunity = new HomeCommunityType();
                HomeCommunityType fromHomeCommunity = fromNhinTargetCommunity.getHomeCommunity();
                homeCommunity.setDescription(fromHomeCommunity.getDescription());
                homeCommunity.setHomeCommunityId(fromHomeCommunity.getHomeCommunityId());
                homeCommunity.setName(fromHomeCommunity.getName());
                nhinTargetCommunity.setHomeCommunity(homeCommunity);;
                nhinTargetCommunity.setList(fromNhinTargetCommunity.getList());
                nhinTargetCommunity.setRegion(fromNhinTargetCommunity.getRegion());
                ret.getNhinTargetCommunity().add(nhinTargetCommunity);
            }
        }

        return ret;
    }

    private Map<String, List<String>> getMapFromSlots(List<SlotType1> slots)
    {
        HashMap<String, List<String>> ret = new HashMap<>();

        for (SlotType1 slot : slots) {
            if (!NullChecker.isNullOrEmpty(slot.getName()) && !NullChecker.isNullOrEmpty(slot.getValueList()) && !NullChecker.isNullOrEmpty(slot.getValueList().getValue())) {
                List<String> values = ret.get(slot.getName());
                if (values == null) {
                    values = new ArrayList<>();
                    ret.put(slot.getName(), values);
                }
                values.addAll(parseParamFormattedStrings(slot.getValueList().getValue()));
            }
        }

        return ret;
    }

    private AdhocQueryResponse createAdhocQueryResponse(AdhocQueryRequest fromAdhocQueryRequest, AdhocQueryResponse fromAdhocQueryResponse, String patientId)
    {
        oasis.names.tc.ebxml_regrep.xsd.query._3.ObjectFactory objFactory = new oasis.names.tc.ebxml_regrep.xsd.query._3.ObjectFactory();
        AdhocQueryResponse ret = objFactory.createAdhocQueryResponse();
        ret.setRequestId(fromAdhocQueryRequest.getId());
        ret.setStatus(fromAdhocQueryResponse.getStatus());
        ret.setRegistryErrorList(fromAdhocQueryResponse.getRegistryErrorList());
        ret.setResponseSlotList(fromAdhocQueryResponse.getResponseSlotList());
        ret.setStartIndex(fromAdhocQueryResponse.getStartIndex());
        ret.setTotalResultCount(fromAdhocQueryResponse.getTotalResultCount());
        ret.setRegistryObjectList(createRegistryObjectList(fromAdhocQueryResponse.getRegistryObjectList(), patientId));
        ret.setRegistryErrorList(createRegistryErrorList(fromAdhocQueryResponse.getRegistryErrorList()));
        return ret;
    }

    private AdhocQueryResponse createAdhocQueryResponse(AdhocQueryRequest fromAdhocQueryRequest, PrefetchQueryResponse fromPrefetchQueryResponse)
    {
        oasis.names.tc.ebxml_regrep.xsd.query._3.ObjectFactory objFactory = new oasis.names.tc.ebxml_regrep.xsd.query._3.ObjectFactory();
        AdhocQueryResponse ret = objFactory.createAdhocQueryResponse();
        ret.setRequestId(fromAdhocQueryRequest.getId());
        ret.setStatus("urn:oasis:names:tc:ebxml-regrep:ResponseStatusType:Success");
        ret.setStartIndex(BigInteger.ZERO);
        if (fromPrefetchQueryResponse != null && NullChecker.isNotNullOrEmpty(fromPrefetchQueryResponse.getIdentifiable())) {
            ret.setTotalResultCount(BigInteger.valueOf(fromPrefetchQueryResponse.getIdentifiable().size()));
        }
        else {
            ret.setTotalResultCount(BigInteger.ZERO);
        }

        ret.setRegistryObjectList(createRegistryObjectList(fromPrefetchQueryResponse.getIdentifiable()));
        return ret;
    }

    private RegistryObjectListType createRegistryObjectList(RegistryObjectListType fromRegistryObjectList, String patientId)
    {
        RegistryObjectListType ret = null;

        if (fromRegistryObjectList != null) {
            ret = new RegistryObjectListType();
            ret.getIdentifiable().addAll(createIdentifiables(fromRegistryObjectList.getIdentifiable(), patientId));
        }

        return ret;
    }

    private RegistryObjectListType createRegistryObjectList(List<JAXBElement<? extends IdentifiableType>> extrinsicObjects)
    {
        RegistryObjectListType ret = null;

        if (extrinsicObjects != null && NullChecker.isNotNullOrEmpty(extrinsicObjects)) {
            ret = new RegistryObjectListType();
            ret.getIdentifiable().addAll(extrinsicObjects);
        }

        return ret;
    }

    private List<JAXBElement<? extends IdentifiableType>> createIdentifiables(List<JAXBElement<? extends IdentifiableType>> fromIdentifiables, String patientId)
    {
        List<JAXBElement<? extends IdentifiableType>> ret = new ArrayList<>();

        for (JAXBElement<? extends IdentifiableType> identifiable : fromIdentifiables) {
            if (identifiable.getValue() instanceof ExtrinsicObjectType) {
                ret.add(createExtrinsicObject((ExtrinsicObjectType)identifiable.getValue(), patientId));
            }
            else {
                ret.add(identifiable);
            }
        }

        return ret;
    }

    private JAXBElement<? extends IdentifiableType> createExtrinsicObject(ExtrinsicObjectType fromExtrinsicObject, String patientId)
    {
        oasis.names.tc.ebxml_regrep.xsd.rim._3.ObjectFactory objFactory = new oasis.names.tc.ebxml_regrep.xsd.rim._3.ObjectFactory();
        ExtrinsicObjectType ret = new ExtrinsicObjectType();
        ret.setContentVersionInfo(fromExtrinsicObject.getContentVersionInfo());
        ret.setDescription(fromExtrinsicObject.getDescription());
        ret.setHome(fromExtrinsicObject.getHome());
        ret.setId(fromExtrinsicObject.getId());
        ret.setIsOpaque(fromExtrinsicObject.isIsOpaque());
        ret.setLid(fromExtrinsicObject.getLid());
        ret.setMimeType(fromExtrinsicObject.getMimeType());
        ret.setName(fromExtrinsicObject.getName());
        ret.setObjectType(fromExtrinsicObject.getObjectType());
        ret.setStatus(fromExtrinsicObject.getStatus());
        ret.setVersionInfo(fromExtrinsicObject.getVersionInfo());
        ret.getClassification().addAll(fromExtrinsicObject.getClassification());
        ret.getSlot().addAll(fromExtrinsicObject.getSlot());
        Document doc = createDocument(fromExtrinsicObject, patientId);
        ret.getExternalIdentifier().addAll(createExternalIdentifiers(fromExtrinsicObject.getExternalIdentifier(), doc));
        return objFactory.createExtrinsicObject(ret);
    }

    private Document createDocument(ExtrinsicObjectType fromExtrinsicObject, String patientId)
    {
        Document ret = new Document();
        Map<String, List<String>> slots = getMapFromSlots(fromExtrinsicObject.getSlot());

        DataQuery query = dataManager.getQuery("Composite.findDemographics2");
        query.setParameter("icn", patientId);
        List<Map> results = query.getResults();
        if (!NullChecker.isNullOrEmpty(results)) {
            Map result = results.get(0);
            Map demographics = (Map)result.get("demographics");
            ret.setPatientLastName((String)demographics.get("nameFamily"));
            ret.setPatientGivenName((String)demographics.get("nameGiven"));
            ret.setPatientSSN((String)demographics.get("SSN"));
            ret.setPatientPreferredFacilityNumber((String)result.get("patientPreferredFacilityNumber"));
            ret.setPatientPreferredFacilityName((String)result.get("patientPreferredFacilityName"));
        }

        ret.setPatientId(patientId.substring(0, 17));
        ret.setSourcePatientId(getExternalIdentifier(fromExtrinsicObject.getExternalIdentifier(), EBXML_RESPONSE_PATIENTID_IDENTIFICATION_SCHEME).getValue());

        if (!NullChecker.isNullOrEmpty(slots.get(EBXML_RESPONSE_SIZE_SLOTNAME))) {
            ret.setSize(Integer.parseInt(slots.get(EBXML_RESPONSE_SIZE_SLOTNAME).get(0)));
        }

        if (!NullChecker.isNullOrEmpty(slots.get(EBXML_RESPONSE_HASH_SLOTNAME))) {
            ret.setHash(slots.get(EBXML_RESPONSE_HASH_SLOTNAME).get(0));
        }

        if (!NullChecker.isNullOrEmpty(slots.get(EBXML_RESPONSE_SERVICESTARTTIME_SLOTNAME))) {
            ret.setBeginDate(parseDate(slots.get(EBXML_RESPONSE_SERVICESTARTTIME_SLOTNAME).get(0), DATE_FORMAT_SERVICE));
        }

        if (!NullChecker.isNullOrEmpty(slots.get(EBXML_RESPONSE_SERVICESTOPTIME_SLOTNAME))) {
            ret.setEndDate(parseDate(slots.get(EBXML_RESPONSE_SERVICESTOPTIME_SLOTNAME).get(0), DATE_FORMAT_SERVICE));
        }

        if (!NullChecker.isNullOrEmpty(slots.get(EBXML_RESPONSE_CREATIONTIME_SLOTNAME))) {
            ret.setCreationTime(new Date());
        }

        ret.setMimeType(fromExtrinsicObject.getMimeType());
        ret.setAvailabilityStatus("urn:ihe:iti:2010:StatusType:DeferredCreation");

        if (fromExtrinsicObject.getName() != null && !NullChecker.isNullOrEmpty(fromExtrinsicObject.getName().getLocalizedString())) {
            ret.setTitle(fromExtrinsicObject.getName().getLocalizedString().get(0).getValue());
        }

        ret.setDocumentUniqueId(UUID.randomUUID().toString());

        ClassificationType classification = getClassification(fromExtrinsicObject.getClassification(), EBXML_RESPONSE_CLASSCODE_CLASS_SCHEME);
        if (classification != null) {
            ret.setClassCode(classification.getNodeRepresentation());
            ret.setClassCodeScheme(getMapFromSlots(classification.getSlot()).get("codingScheme").get(0));
            ret.setClassCodeDisplayName(classification.getName().getLocalizedString().get(0).getValue());
        }

        classification = getClassification(fromExtrinsicObject.getClassification(), EBXML_RESPONSE_FORMATCODE_CLASS_SCHEME);
        if (classification != null) {
            ret.setFormatCode(classification.getNodeRepresentation());
            ret.setFormatCodeScheme(getMapFromSlots(classification.getSlot()).get("codingScheme").get(0));
            ret.setFormatCodeDisplayName(classification.getName().getLocalizedString().get(0).getValue());
        }

        String homeCommunityId = fromExtrinsicObject.getHome();
        String documentRepositoryId = slots.get(EBXML_RESPONSE_REPOSITORY_UNIQUE_ID_SLOTNAME).get(0);
        String documentUniqueId = getExternalIdentifier(fromExtrinsicObject.getExternalIdentifier(), EBXML_RESPONSE_DOCID_IDENTIFICATION_SCHEME).getValue();
        ret.setDocGenQueryName("Remote");
        ret.setDocGenQueryParams("homeCommunityId=" + homeCommunityId + ",documentRepositoryId=" + documentRepositoryId + ",documentUniqueId=" + documentUniqueId);

        documentRepository.storeDocument(ret);

        return ret;
    }

    private Date parseDate(String dateString, String formatString)
    {
        Date ret = null;
        SimpleDateFormat formatter = (SimpleDateFormat)DateFormat.getDateTimeInstance();
        formatter.applyPattern(formatString);
        formatter.setTimeZone(TimeZone.getTimeZone("UTC"));
        try {
            /* ret = */
            formatter.parse(dateString);
        }
        catch (ParseException pe) {
            ret = null;
        }

        return ret;
    }

    private ExternalIdentifierType getExternalIdentifier(List<ExternalIdentifierType> externalIdentifiers, String identificationScheme)
    {
        ExternalIdentifierType ret = null;

        for (ExternalIdentifierType id : externalIdentifiers) {
            if (id.getIdentificationScheme().equals(identificationScheme)) {
                ret = id;
                break;
            }
        }

        return ret;
    }

    private List<ExternalIdentifierType> createExternalIdentifiers(List<ExternalIdentifierType> fromExternalIdentifiers, Document document)
    {
        List<ExternalIdentifierType> ret = new ArrayList<>();
        for (ExternalIdentifierType id : fromExternalIdentifiers) {
            for (LocalizedStringType name : id.getName().getLocalizedString()) {
                if (name.getValue().equals(EBXML_RESPONSE_DOCID_NAME)) {
                    ret.add(createExternalIdentifier(id, document));
                }
                else {
                    ret.add(id);
                }
            }
        }

        return ret;
    }

    private ClassificationType getClassification(List<ClassificationType> fromClassifications, String classScheme)
    {
        ClassificationType ret = null;

        for (ClassificationType cls : fromClassifications) {
            if (cls.getClassificationScheme().equals(classScheme)) {
                ret = cls;
                break;
            }
        }

        return ret;
    }

    private ExternalIdentifierType createExternalIdentifier(ExternalIdentifierType fromId, Document document)
    {
        ExternalIdentifierType ret = new ExternalIdentifierType();
        ret.setDescription(fromId.getDescription());
        ret.setHome(fromId.getHome());
        ret.setId(fromId.getId());
        ret.setIdentificationScheme(fromId.getIdentificationScheme());
        ret.setLid(fromId.getLid());
        ret.setName(fromId.getName());
        ret.setObjectType(fromId.getObjectType());
        ret.setRegistryObject(fromId.getRegistryObject());
        ret.setStatus(fromId.getStatus());
        ret.setValue(document.getDocumentUniqueId());
        ret.setVersionInfo(fromId.getVersionInfo());
        return ret;
    }

    private RegistryErrorList createRegistryErrorList(RegistryErrorList fromRegistryErrorList)
    {
        RegistryErrorList ret = null;

        if (fromRegistryErrorList != null) {
            ret = new RegistryErrorList();
            ret.setHighestSeverity(fromRegistryErrorList.getHighestSeverity());
            ret.getRegistryError().addAll(createRegistryErrors(fromRegistryErrorList.getRegistryError()));
        }

        return ret;
    }

    private List<RegistryError> createRegistryErrors(List<RegistryError> fromRegistryErrors)
    {
        List<RegistryError> ret = new ArrayList<>();

        for (RegistryError registryError : fromRegistryErrors) {
            ret.add(createRegistryError(registryError));
        }

        return ret;
    }

    private RegistryError createRegistryError(RegistryError fromRegistryError)
    {
        RegistryError ret = new RegistryError();
        ret.setCodeContext(fromRegistryError.getCodeContext());
        ret.setErrorCode(fromRegistryError.getErrorCode());
        ret.setLocation(fromRegistryError.getLocation());
        ret.setSeverity(fromRegistryError.getSeverity());
        ret.setValue(fromRegistryError.getValue());
        return ret;
    }

    private AdhocQueryResponse createAdhocQueryResponseError(AdhocQueryRequest fromAdhocQueryRequest)
    {
        oasis.names.tc.ebxml_regrep.xsd.query._3.ObjectFactory objFactory = new oasis.names.tc.ebxml_regrep.xsd.query._3.ObjectFactory();
        AdhocQueryResponse ret = objFactory.createAdhocQueryResponse();
        ret.setStatus("urn:oasis:names:tc:ebxml-regrep:ResponseStatusType:Failure");
        ret.setRegistryErrorList(createDefaultRegistryErrorList());
        ret.setRequestId(fromAdhocQueryRequest.getId());
        return ret;
    }

    private RegistryErrorList createDefaultRegistryErrorList()
    {
        RegistryErrorList ret = new RegistryErrorList();
        ret.getRegistryError().add(createDefaultRegistryError());
        ret.setHighestSeverity("urn:oasis:names:tc:ebxml-regrep:ErrorSeverityType:Error");
        return ret;
    }

    private RegistryError createDefaultRegistryError()
    {
        RegistryError ret = new RegistryError();
        ret.setErrorCode("XDSRegistryError");
        ret.setCodeContext("Internal Registry/Repository Error");
        ret.setSeverity("urn:oasis:names:tc:ebxml-regrep:ErrorSeverityType:Error");
        return ret;
    }

    public List<String> parseParamFormattedStrings(List<String> paramFormattedStrings)
    {
        List<String> ret = new ArrayList<>();

        for (String paramFormattedString : paramFormattedStrings) {
            parseParamFormattedString(paramFormattedString, ret);
        }

        return ret;
    }

    public void parseParamFormattedString(String paramFormattedString, List<String> resultCollection)
    {
        if ((paramFormattedString != null) && (resultCollection != null)) {
            String working = paramFormattedString;
            if (paramFormattedString.startsWith("(")) {
                working = paramFormattedString.substring(1);
                int endIndex = working.indexOf(")");
                if (endIndex != -1) {
                    working = working.substring(0, endIndex);
                }
            }
            String[] multiValueString = working.split(",");
            if (multiValueString != null) {
                for (String singleValue : multiValueString) {
                    if (singleValue != null) {
                        singleValue = singleValue.trim();

                        if (singleValue.startsWith("'")) {
                            singleValue = singleValue.substring(1);
                            int endTickIndex = singleValue.indexOf("'");
                            if (endTickIndex != -1) {
                                singleValue = singleValue.substring(0, endTickIndex);
                            }
                        }
                        resultCollection.add(singleValue);
                    }
                }
            }
        }
    }

    private boolean checkPolicy(AssertionType assertion, AuditEvent ae)
    {
        boolean ret = false;

        if (assertion.getUserInfo() != null && !NullChecker.isNullOrEmpty(assertion.getUserInfo().getUserName())) {
            RequestType request = CheckPolicyRequestBuilder.buildNHINOutCheckPolicyRequest(assertion.getUserInfo().getUserName(), getHomeFacility().getFullHomeCommunityId());
            CheckPolicyRequestType checkPolicyRequest = new CheckPolicyRequestType();
            checkPolicyRequest.setRequest(request);

            MessagingHelper<AuditEvent> msg = EventAuditingFactoryImpl.getFactory(AuditEvent.class).messaging();
            msg.addQuery(ae, checkPolicyRequest, CheckPolicyRequestType.class,
                         QueryType.REQUEST, null);

            CheckPolicyResponseType checkPolicyResponse = adapterPolicyEngine.checkPolicy(checkPolicyRequest);
            msg.addQuery(ae, checkPolicyResponse, CheckPolicyResponseType.class,
                         QueryType.RESPONSE, null);

            return CheckPolicy.checkPolicy(checkPolicyRequest, checkPolicyResponse);
        }

        return ret;
    }

    private Facility getHomeFacility()
    {
        return facilityManager.getFacilityByFacilityNumber("VA");
    }

    private String createFullName(PersonNameType personName)
    {
        StringBuilder ret = new StringBuilder();

        if (NullChecker.isNotNullOrEmpty(personName.getGivenName())) {
            ret.append(personName.getGivenName());
        }

        if (NullChecker.isNotNullOrEmpty(personName.getSecondNameOrInitials())) {
            if (ret.length() > 0) {
                ret.append(' ');
            }
            ret.append(personName.getSecondNameOrInitials());
        }

        if (NullChecker.isNotNullOrEmpty(personName.getFamilyName())) {
            if (ret.length() > 0) {
                ret.append(' ');
            }
            ret.append(personName.getFamilyName());
        }

        return ret.toString();
    }
}
