/**
 * 
 * 
 */
package gov.va.med.nhin.adapter.dataquality.transmissionlog;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import javax.ejb.Stateless;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
//import javax.persistence.TemporalType;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.nhin.adapter.dataquality.common.NullChecker;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.Map;
import javax.persistence.EntityExistsException;
import javax.persistence.TemporalType;
import javax.persistence.TransactionRequiredException;


/**
 *  DAO for DQ_TRANSMISSION_LOG table/ DqTransmissionLog entity. 
 *
 */
@TransactionAttribute(value = TransactionAttributeType.SUPPORTS)
@Stateless(name = "DqTransmissionLogManager", mappedName = "DqTransmissionLogManager")
public class DqTransmissionLogManagerBean implements DqTransmissionLogManagerRemote, DqTransmissionLogManagerLocal {

    private static final Logger LOG = LoggerFactory.getLogger(DqTransmissionLogManagerBean.class.getName());
    
    @PersistenceContext()
    private EntityManager em;

    public void setEntityManager(EntityManager entityManager) {
        this.em = entityManager;
    }
    
    @TransactionAttribute(value = TransactionAttributeType.REQUIRED)
    @Override  
    public void logDHUpload(final String batchId, final Date startTime, final Date endTime, final String hostname, final String localDocId, final String filename, final String filetype, final double filesize, final String destination, final String status, final String remoteDocId, final String error) throws EntityExistsException , IllegalArgumentException , TransactionRequiredException {
        this.log(DqTransmissionLogManager.DHUPLOAD_EVENTTYPE, batchId, startTime, endTime, hostname, localDocId, filename, filetype, filesize, destination, status, remoteDocId, error);        
    }
    
    @TransactionAttribute(value = TransactionAttributeType.REQUIRED)
    @Override  
    public void logDQZipStore(final String batchId, final Date startTime, final Date endTime, final String hostname, final String localDocId, final String filename, final String filetype, final double filesize, final String destination, final String status, final String remoteDocId, final String error) throws EntityExistsException , IllegalArgumentException , TransactionRequiredException {
        this.log(DqTransmissionLogManager.DQZIPSTORE_EVENTTYPE, batchId, startTime, endTime, hostname, localDocId, filename, filetype, filesize, destination, status, remoteDocId, error);
    }
    
    @TransactionAttribute(value = TransactionAttributeType.REQUIRED)
    @Override      
    public void logDQZipMove(final String batchId, final Date startTime, final Date endTime, final String hostname, final String localDocId, final String filename, final String filetype, final double filesize, final String destination, final String status, final String remoteDocId, final String error) throws EntityExistsException , IllegalArgumentException , TransactionRequiredException {
        this.log(DqTransmissionLogManager.DQZIPMOVE_EVENTTYPE, batchId, startTime, endTime, hostname, localDocId, filename, filetype, filesize, destination, status, remoteDocId, error);  
    } 
    
    /**
     * Store new entry into DQ_TRANSMISSION_LOG table for an individual log event.
     * 
     */
    @TransactionAttribute(value = TransactionAttributeType.REQUIRED)    
    private void log(final String eventType, final String batchId, final Date startTime, final Date endTime, final String hostname, final String localDocId, final String filename, final String filetype, final double filesize, final String destination, final String status, final String remoteDocId, final String error) throws EntityExistsException , IllegalArgumentException , TransactionRequiredException  {
        LOG.debug("logging a new DqTransmissionLog entry...");
        if(null != startTime) {
            DqTransmissionLog dqtl = new DqTransmissionLog();
            try {

                // required
                dqtl.setStartTime(startTime);
                
                dqtl.setEndTime(endTime);
                
                dqtl.setBatchId(batchId);
                
                dqtl.setEventType(eventType);
                
                dqtl.setHostname(hostname);
                
                dqtl.setLocalDocId(localDocId);
                
                dqtl.setFilename(filename);
                
                dqtl.setFiletype(filetype);
                
                dqtl.setFilesize(filesize);
                
                dqtl.setDestination(destination);
                
                dqtl.setStatus(status);
                
                dqtl.setRemoteDocId(remoteDocId);
                
                dqtl.setError(error);

                // store a new one
                this.em.persist(dqtl);
                
                LOG.debug("returning new DqTransmissionLog entry TransmissionId...");
            } catch (EntityExistsException | IllegalArgumentException | TransactionRequiredException ex) {
                LOG.error("log() threw an Exception!: ",ex);
                throw ex;
            }
        } else {
             LOG.debug("returning without storing: required log parameter missing...");
        } 
    }
    
    @Override
    public int getDHUploadEventCount(final Date beginStartDate, final Date finishStartDate, final String batchId, final String hostname, final String filename, final String destination, final String status) {//, final boolean useFilenameDistinct) {
        int result = this.getEventCount(DqTransmissionLogManager.DHUPLOAD_EVENTTYPE, beginStartDate, finishStartDate, batchId, hostname, filename, destination, status);//, useFilenameDistinct);
        return result;
    } 

    @Override
    public int getDQZipStoreEventCount(final Date beginStartDate, final Date finishStartDate, final String batchId, final String hostname, final String filename, final String destination, final String status) {//, final boolean useFilenameDistinct) {
        int result = this.getEventCount(DqTransmissionLogManager.DQZIPSTORE_EVENTTYPE, beginStartDate, finishStartDate, batchId, hostname, filename, destination, status);//, useFilenameDistinct);
        return result;
    }  
   
    @Override    
    public int getDQZipSendEventCount(final Date beginStartDate, final Date finishStartDate, final String batchId, final String hostname, final String filename, final String destination, final String status) {//, final boolean useFilenameDistinct) {
        int result = this.getEventCount(DqTransmissionLogManager.DQZIPMOVE_EVENTTYPE, beginStartDate, finishStartDate, batchId, hostname, filename, destination, status); //, useFilenameDistinct);
        return result;
    }   
       
    @Override    
    public int getEventCount(final String eventType, final Date beginStartDate, final Date finishStartDate, final String batchId, final String hostname, final String filename, final String destination, final String status) { //, final boolean useFilenameDistinct) {
        Map<String, Object> setParams = new HashMap<String, Object>();
        
        String whereClause = this.buildWhereClause(eventType, beginStartDate, finishStartDate, null, null, batchId, hostname, null, filename, null, destination, status, null, false, setParams);
        
        String queryTemplate = null;
        // TODO: Can't do in Java Persistence? 
        //if(useFilenameDistinct) {
        //    queryTemplate = "SELECT COUNT(d.dqTransmissionId) FROM DqTransmissionLog d WHERE" + whereClause;
        //} else {
            queryTemplate = "SELECT COUNT(d.dqTransmissionId) FROM DqTransmissionLog d WHERE" + whereClause;  
        //}

        LOG.debug("getEventCount using queryTemplate: "+queryTemplate);
        Query query = em.createQuery(queryTemplate);       
        
        for (Map.Entry<String, Object> entry : setParams.entrySet()) {
            // TODO: need TemporalType.TIMESTAMP parameter for dates only?
            query.setParameter(entry.getKey(), entry.getValue());
        } 
        
        Object countObj = query.getSingleResult();
        int count = 0;
        if(countObj instanceof Long)
        {
            count = ((Long) countObj).intValue();
            //LOG.trace("long count {}:", count);
        }
        if(countObj instanceof Integer)
        {
            count = ((Integer) countObj);
            //LOG.trace("int count {}:", count);
        }

        return count;  
    } 
    
    @Override    
    public int getUniqueFilenameStatusCountNative(final String eventType, final Date beginStartDate, final Date finishStartDate, final String status) {
        
        DateFormat formatter = new SimpleDateFormat("MM/dd/yyyy HH:mm:ss.SSS");
        String beginStartDateTimestamp = formatter.format(beginStartDate);
        String finishStartDateTimestamp = formatter.format(finishStartDate);
        
        Query q = em.createNativeQuery("SELECT count(*) FROM ADAPTER.DQ_TRANSMISSION_LOG I " +
           "where I.DQ_TRANSMISSION_ID = " +
           "(select max(I2.DQ_TRANSMISSION_ID) from ADAPTER.DQ_TRANSMISSION_LOG I2 where I2.FILENAME1 = I.FILENAME1) " +
           "and I.EVENT_TYPE=? " +
           "and I.STATUS1=? " +
           "and I.START_TIME >= ? " +
           "and I.START_TIME <= ?");
         
       q.setParameter(1, eventType);
       q.setParameter(2, status);
       q.setParameter(3, beginStartDate, TemporalType.TIMESTAMP);
       q.setParameter(4, finishStartDate, TemporalType.TIMESTAMP);
        
        Object countObj = q.getSingleResult();
        int count = 0;
        if(countObj instanceof Long)
        {
            count = ((Long) countObj).intValue();
            //LOG.trace("long count {}:", count);
        }
        if(countObj instanceof Integer)
        {
            count = ((Integer) countObj);
            //LOG.trace("int count {}:", count);
        }

        return count;        
    }
    
    @Override    
    public int getUniqueFilenameEventCount(final String eventType, final Date beginStartDate, final Date finishStartDate, final String batchId, final String filename, final String status) {
        Map<String, Object> setParams = new HashMap<String, Object>();
        
        String whereClause = this.buildWhereClause(eventType, beginStartDate, finishStartDate, null, null, batchId, null, null, filename, null, null, status, null, false, setParams);
        
        String queryTemplate = "SELECT COUNT(DISTINCT d.filename) FROM DqTransmissionLog d WHERE"+whereClause;
        
        LOG.debug("getUniqueFilenameEventCount using queryTemplate: "+queryTemplate);
        Query query = em.createQuery(queryTemplate);       
        
        for (Map.Entry<String, Object> entry : setParams.entrySet()) {
            // TODO: need TemporalType.TIMESTAMP parameter for dates only?
            query.setParameter(entry.getKey(), entry.getValue());
        } 
        
        Object countObj = query.getSingleResult();
        int count = 0;
        if(countObj instanceof Long)
        {
            count = ((Long) countObj).intValue();
            //LOG.trace("long count {}:", count);
        }
        if(countObj instanceof Integer)
        {
            count = ((Integer) countObj);
            //LOG.trace("int count {}:", count);
        }

        return count;
    }
    
    @Override    
    public boolean isAlreadySuccessfulEventByFilename(final String eventType, final String filename, final Date beginStartDate, final Date finishStartDate) {
        Map<String, Object> setParams = new HashMap<String, Object>();
                
        String whereClause = this.buildWhereClause(eventType, beginStartDate, finishStartDate, null, null, null, null, null, filename, null, null, "success", null, false, setParams);
        
        String queryTemplate = "SELECT d FROM DqTransmissionLog d WHERE" + whereClause + " ORDER BY d.dqTransmissionId DESC"; 
        //System.out.println("queryTemplate: "+queryTemplate);
        Query query = em.createQuery(queryTemplate);       
        query.setMaxResults(1);
        
        for (Map.Entry<String, Object> entry : setParams.entrySet()) {
            // TODO: need timestamp parameter for dates only?
            query.setParameter(entry.getKey(), entry.getValue());
        }
        
        List<DqTransmissionLog> dqlList = (List<DqTransmissionLog>) query.getResultList();
        
        boolean exists = false;
        if(dqlList.size() > 0) {
            exists = true;
        }
        
        return exists;
    }
    
    public boolean isAlreadySuccessfulEventByLocalDocId(String eventType, String localDocId, Date beginStartDate, Date finishStartDate) {
        Map<String, Object> setParams = new HashMap<String, Object>();
                
        String whereClause = this.buildWhereClause(eventType, beginStartDate, finishStartDate, null, null, null, null, localDocId, null, null, null, "success", null, false, setParams);
        
        String queryTemplate = "SELECT d FROM DqTransmissionLog d WHERE" + whereClause + " ORDER BY d.dqTransmissionId DESC"; 
        //System.out.println("queryTemplate: "+queryTemplate);
        Query query = em.createQuery(queryTemplate);       
        query.setMaxResults(1);
        
        for (Map.Entry<String, Object> entry : setParams.entrySet()) {
            // TODO: need timestamp parameter for dates only?
            query.setParameter(entry.getKey(), entry.getValue());
        }
        
        List<DqTransmissionLog> dqlList = (List<DqTransmissionLog>) query.getResultList();
        
        boolean exists = false;
        if(dqlList.size() > 0) {
            exists = true;
        }
        
        return exists;
    }
    
    @Override    
    public List<DqTransmissionLog> getErrorsByMostRecentFirstForEventType(final String eventType, final Date beginStartDate, final Date finishStartDate, final String batchId, final String hostname, final boolean isMostRecentOnly) {
        // for isMostRecentOnly true: get the most recent error only if there is no other success logged later in same batch for same filename, 
        // i. e. for repeat filenames given from the audit table, which later succeeded 

        Map<String, Object> setParams = new HashMap<String, Object>();
                
        String whereClause = this.buildWhereClause(eventType, beginStartDate, finishStartDate, null, null, batchId, hostname, null, null, null, null, "failure", null, false, setParams);
        
        String queryTemplate = "SELECT d FROM DqTransmissionLog d WHERE" + whereClause + " ORDER BY d.dqTransmissionId DESC"; 
        //System.out.println("queryTemplate: "+queryTemplate);
        Query query = em.createQuery(queryTemplate); 
        if(isMostRecentOnly) {
            query.setMaxResults(1);
        }
        
        for (Map.Entry<String, Object> entry : setParams.entrySet()) {
            // TODO: need timestamp parameter for dates only?
            query.setParameter(entry.getKey(), entry.getValue());
        }
        
        List<DqTransmissionLog> errorList = (List<DqTransmissionLog>) query.getResultList();
        
        return errorList;
    }
    
    @Override   
    public DqTransmissionLog findDqTransmissionLog(final Long id) {
        if(null != id) {
            return this.em.find(DqTransmissionLog.class, id);
        } else {
            return null;
        }
    }

    @Override
    public List<DqTransmissionLog> findAll() {
        try {
            final Query q = this.em
                    .createNamedQuery("DqTransmissionLog.findAll");           
            return (List<DqTransmissionLog>) q.getResultList();
        } catch (final NoResultException nre) {
            return new ArrayList<DqTransmissionLog>();
        }
    }
        
    @Override
    public List<DqTransmissionLog> findByDqTransmissionId(final Long dqTransmissionId) {
        try {
            if(null != dqTransmissionId) {
                final Query q = this.em
                        .createNamedQuery("DqTransmissionLog.findByDqTransmissionId"); 
                q.setParameter("dqTransmissionId", dqTransmissionId);
                return (List<DqTransmissionLog>) q.getResultList();
            } else {
                return new ArrayList<DqTransmissionLog>();
            }
        } catch (final NoResultException nre) {
            return new ArrayList<DqTransmissionLog>();
        }
    }  
    
    @Override
    public List<DqTransmissionLog> findByBatchId(final String batchId) {
        try {
            final Query q = this.em
                    .createNamedQuery("DqTransmissionLog.findByBatchId");   
            q.setParameter("batchId", batchId);
            return (List<DqTransmissionLog>) q.getResultList();
        } catch (final NoResultException nre) {
            return new ArrayList<DqTransmissionLog>();
        }
    }    
    
    @Override
    public List<DqTransmissionLog> findByFilename(final String filename) {
        try {
            final Query q = this.em
                    .createNamedQuery("DqTransmissionLog.findByFilename");   
            q.setParameter("filename", filename);
            return (List<DqTransmissionLog>) q.getResultList();
        } catch (final NoResultException nre) {
            return new ArrayList<DqTransmissionLog>();
        }
    }
    
    @Override
    public List<DqTransmissionLog> findByLocalDocId(final String localDocId) {
        try {
            final Query q = this.em
                    .createNamedQuery("DqTransmissionLog.findByLocalDocId");   
            q.setParameter("localDocId", localDocId);
            return (List<DqTransmissionLog>) q.getResultList();
        } catch (final NoResultException nre) {
            return new ArrayList<DqTransmissionLog>();
        }
    }
    
    @Override
    public List<DqTransmissionLog> findByRemoteDocId(final String remoteDocId) {
        try {
            final Query q = this.em
                    .createNamedQuery("DqTransmissionLog.findByRemoteDocId");   
            q.setParameter("remoteDocId", remoteDocId);
            return (List<DqTransmissionLog>) q.getResultList();
        } catch (final NoResultException nre) {
            return new ArrayList<DqTransmissionLog>();
        }
    } 
    
    @Override
    public List<DqTransmissionLog> findByVarious(final String eventType, final Date beginStartDate, final Date finishStartDate, final Date beginEndDate, final Date finishEndDate, final String batchId, final String hostname, final String localDocId, final String filename, final String destination, final String status, final String remoteDocId) {
                
        if(NullChecker.isNullOrEmpty(eventType) && NullChecker.isNullOrEmpty(beginStartDate) 
                && NullChecker.isNullOrEmpty(finishStartDate) && NullChecker.isNullOrEmpty(beginEndDate) 
                && NullChecker.isNullOrEmpty(finishEndDate) && NullChecker.isNullOrEmpty(batchId) 
                && NullChecker.isNullOrEmpty(hostname) && NullChecker.isNullOrEmpty(localDocId) 
                && NullChecker.isNullOrEmpty(filename) && NullChecker.isNullOrEmpty(destination) 
                && NullChecker.isNullOrEmpty(status) && NullChecker.isNullOrEmpty(remoteDocId)) {
            // return an empty list if all the parameters are empty
            return new ArrayList<DqTransmissionLog>();
        }
        
        Map<String, Object> setParams = new HashMap<String, Object>();
        
        String whereClause = this.buildWhereClause(eventType, beginStartDate, finishStartDate, beginEndDate, finishEndDate, batchId, hostname, localDocId, filename, null, destination, status,remoteDocId, false, setParams);
        
        String queryTemplate = "SELECT d FROM DqTransmissionLog d WHERE" + whereClause + " ORDER BY d.dqTransmissionId DESC"; 
        //System.out.println("queryTemplate: "+queryTemplate);
        Query query = em.createQuery(queryTemplate); 
//        if(isMostRecentOnly) {
//            query.setMaxResults(1);
//        }
      
        for (Map.Entry<String, Object> entry : setParams.entrySet()) {
            // TODO: need timestamp parameter for dates only?
            query.setParameter(entry.getKey(), entry.getValue());
        }
        
        List<DqTransmissionLog> responseList = (List<DqTransmissionLog>) query.getResultList();
        
        return responseList;
    }    
    /*
    private Date setTimeInDate(final Date date, final int hourOfDay, final int minute, final int second, final int ms)
    {
        final GregorianCalendar gc = new GregorianCalendar();
        gc.setTime( date );
        gc.set( Calendar.HOUR_OF_DAY, hourOfDay );
        gc.set( Calendar.MINUTE, minute );
        gc.set( Calendar.SECOND, second );
        gc.set( Calendar.MILLISECOND, ms );
        return gc.getTime();
    }
    
    private Date setSecondMillisecInDateTime(final Date date, final int second, final int ms)
    {
        final GregorianCalendar gc = new GregorianCalendar();
        gc.setTime( date );
        gc.set( Calendar.SECOND, second );
        gc.set( Calendar.MILLISECOND, ms );
        return gc.getTime();
    }
    */
    private String buildWhereClause(final String eventType, final Date beginStartTime, final Date finishStartTime, 
            final Date beginEndTime, final Date finishEndTime, final String batchId, final String hostname, 
            final String localDocId, final String filename, final String filetype, final String destination, 
            final String status, final String remoteDocId, final boolean hasError, 
            Map<String, Object> setParams)
    {
        StringBuilder whereClauseSb = new StringBuilder();
        
        List<String> expressions = new ArrayList<String>();
        
        if (!NullChecker.isNullOrEmpty(eventType)) {
            expressions.add(" d.eventType = :eventType");
            setParams.put("eventType", eventType);
        }          
        
        if (!NullChecker.isNullOrEmpty(beginStartTime)) {  
            expressions.add(" d.startTime >= :beginStartTime");
            //Date beginStartDateZeroedToSecs = setSecondMillisecInDateTime(beginStartDate, 0, 0);            
            //setParams.put("beginStartTime", beginStartDateZeroedToSecs);
            setParams.put("beginStartTime", beginStartTime);
        } 
        
        if (!NullChecker.isNullOrEmpty(finishStartTime)) { 
            expressions.add(" d.startTime <= :finishStartTime");
            //Date finishStartDateMaxSecs = setSecondMillisecInDateTime(finishStartDate, 59, 999);
            //setParams.put("finishStartTime", finishStartDateMaxSecs);
            setParams.put("finishStartTime", finishStartTime);
        }
        
        if (!NullChecker.isNullOrEmpty(beginEndTime)) {
            expressions.add(" d.endTime >= :beginEndTime");
            //Date beginEndDateZeroedToSecs = setSecondMillisecInDateTime(beginEndDate, 0, 0);
            //setParams.put("beginEndTime", beginEndDateZeroedToSecs);
            setParams.put("beginEndTime", beginEndTime);
        }       
        
        if (!NullChecker.isNullOrEmpty(finishEndTime)) {
            expressions.add(" d.endTime <= :finishEndTime");
            //Date finishEndDateMaxSecs = setSecondMillisecInDateTime(finishEndDate, 59, 999);
            //setParams.put("finishEndTime", finishEndDateMaxSecs);
            setParams.put("finishEndTime", finishEndTime);
        } 
        
        if (!NullChecker.isNullOrEmpty(batchId)) {
            expressions.add(" d.batchId = :batchId");
            setParams.put("batchId", batchId);
        }         
        
        if (!NullChecker.isNullOrEmpty(hostname)) {
            expressions.add(" d.hostname = :hostname");
            setParams.put("hostname", hostname);
        } 
        
        if (!NullChecker.isNullOrEmpty(localDocId)) {
            expressions.add(" d.localDocId = :localDocId");
            setParams.put("localDocId", localDocId);
        } 
        
        if (!NullChecker.isNullOrEmpty(filename)) {
            expressions.add(" d.filename = :filename");
            setParams.put("filename", filename); 
        }  
        
        if (!NullChecker.isNullOrEmpty(filetype)) {
            expressions.add(" d.filetype = :filetype");
            setParams.put("filetype", filetype); 
        } 
        
        if (!NullChecker.isNullOrEmpty(destination)) {
            expressions.add(" d.destination = :destination");
            setParams.put("destination", destination);
        }     
                
        if (!NullChecker.isNullOrEmpty(status)) {
            expressions.add(" d.status = :status");
            setParams.put("status", status);
        } 
    
        if (!NullChecker.isNullOrEmpty(remoteDocId)) {
            expressions.add(" d.remoteDocId = :remoteDocId");
            setParams.put("remoteDocId", remoteDocId);           
        } 

//        if (hasError) {
//            expressions.add(" d.error != NULL");
//        }      

        int length = expressions.size();
        for (int i = 0; i < length; i++) {
            if(0 == i) {
                whereClauseSb.append(expressions.get(i));
            } else {
                whereClauseSb.append(" AND");
                whereClauseSb.append(expressions.get(i));
            }
        }
        
        return whereClauseSb.toString();
    }

}
