
package gov.va.med.nhin.adapter.dataquality.export;

import gov.va.med.nhin.adapter.dataquality.common.DatabaseUtils;
import gov.va.med.nhin.adapter.dataquality.common.Emailer;
import gov.va.med.nhin.adapter.dataquality.common.LogUtil;
import gov.va.med.nhin.adapter.dataquality.common.NullChecker;
import gov.va.med.nhin.adapter.dataquality.common.RestClientException;
import gov.va.med.nhin.adapter.dataquality.common.Utils;
import gov.va.med.nhin.adapter.dataquality.dh.DHUpload;
import static gov.va.med.nhin.adapter.dataquality.export.DQExportProperties.DQ_EXPORT_PROPS;
import static gov.va.med.nhin.adapter.dataquality.export.DQExportProperties.PROP_CONNECTION_TIMEOUT;
import static gov.va.med.nhin.adapter.dataquality.export.DQExportProperties.PROP_DHUPLOAD_MAXFAILURES;
import static gov.va.med.nhin.adapter.dataquality.export.DQExportProperties.PROP_DHUPLOAD_PERFORM;
import static gov.va.med.nhin.adapter.dataquality.export.DQExportProperties.PROP_DH_KEY;
import static gov.va.med.nhin.adapter.dataquality.export.DQExportProperties.PROP_DH_URL;
import static gov.va.med.nhin.adapter.dataquality.export.DQExportProperties.PROP_DH_URL_PATH;
import static gov.va.med.nhin.adapter.dataquality.export.DQExportProperties.PROP_DH_USER;
import static gov.va.med.nhin.adapter.dataquality.export.DQExportProperties.PROP_DQZIP_SEND_PERFORM;
import static gov.va.med.nhin.adapter.dataquality.export.DQExportProperties.PROP_DQZIP_STORE_PERFORM;
import static gov.va.med.nhin.adapter.dataquality.export.DQExportProperties.PROP_DQ_ZIP_DIR;
import static gov.va.med.nhin.adapter.dataquality.export.DQExportProperties.PROP_EMAIL_SEND;
import static gov.va.med.nhin.adapter.dataquality.export.DQExportProperties.PROP_END_AUDIT_DATETIME;
import static gov.va.med.nhin.adapter.dataquality.export.DQExportProperties.PROP_SOCKET_TIMEOUT;
import static gov.va.med.nhin.adapter.dataquality.export.DQExportProperties.PROP_START_AUDIT_DATETIME;
import static gov.va.med.nhin.adapter.dataquality.export.DQExportProperties.PROP_START_AUDIT_INCREMENT_DAYSAGO;
import static gov.va.med.nhin.adapter.dataquality.export.DQExportProperties.PROP_WATCH_DIR;
import gov.va.med.nhin.adapter.dataquality.reporting.TransmissionReport;
import gov.va.med.nhin.adapter.dataquality.transmissionlog.DqTransmissionLogManager;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import javax.sql.DataSource;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 */
public class Orchestrator {
    private static final Logger LOGGER = LoggerFactory.getLogger(Orchestrator.class);
    
   private DqTransmissionLogManager dqlm;
    
   private TransmissionReport trep;
   
   private DataSource dataSource;

    public void setTransmissionReport(TransmissionReport trep) {
        this.trep = trep;
    }

    public void setDqTransmissionLogManager(DqTransmissionLogManager dqlm) {
        this.dqlm = dqlm;
    }

   public void setDataSource(DataSource dataSource)
   {
      this.dataSource = dataSource;
   }
   
      public void orchestrate() {
        final String batchId = String.valueOf(System.currentTimeMillis());
        LOGGER.info("DQExportProperties batchId: "+batchId+" beginning execution...");
        
        final DQExportProperties dqExportInfo = new DQExportProperties();
        
        InputStream propInputStream = null; 
        try {
             final Date timeoutStartTime = new Date();              
             
             // get properties file 
             propInputStream = Utils.getResourceAsStream(DQ_EXPORT_PROPS);
             Properties properties = new Properties();
             properties.load(propInputStream);
             
            dqExportInfo.diameterHealthUrl = properties.getProperty(PROP_DH_URL);
            dqExportInfo.diameterHealthUrlPath = properties.getProperty(PROP_DH_URL_PATH);
            dqExportInfo.diameterHealthUser = properties.getProperty(PROP_DH_USER);
            dqExportInfo.diameterHealthKey = properties.getProperty(PROP_DH_KEY);
            dqExportInfo.connectionTimeout = new Integer(properties.getProperty(PROP_CONNECTION_TIMEOUT));
            dqExportInfo.socketTimeout = new Integer(properties.getProperty(PROP_SOCKET_TIMEOUT));
             
             // get step flags
             dqExportInfo.performDHUpload = "true".equals(properties.getProperty(PROP_DHUPLOAD_PERFORM)); 
             dqExportInfo.performDQZipStore = "true".equals(properties.getProperty(PROP_DQZIP_STORE_PERFORM));
             dqExportInfo.performDQZipSend = "true".equals(properties.getProperty(PROP_DQZIP_SEND_PERFORM));
             dqExportInfo.performEmailSend = "true".equals(properties.getProperty(PROP_EMAIL_SEND));
             
             dqExportInfo.dqZipDir = properties.getProperty(PROP_DQ_ZIP_DIR);
             
             final String dhuploadMaxFailureCountStr = properties.getProperty(PROP_DHUPLOAD_MAXFAILURES);
             LOGGER.debug("dhuploadMaxFailureCountStr: "+dhuploadMaxFailureCountStr);
             //final int dhuploadMaxFailureCount = (NullChecker.isNotNullOrEmpty(dhuploadMaxFailureCountStr)) ? new Integer(dhuploadMaxFailureCountStr) : 10;
             int dhuploadMaxFailureCountLocal = 10;
             if(NullChecker.isNotNullOrEmpty(dhuploadMaxFailureCountStr)) {
                 dhuploadMaxFailureCountLocal = new Integer(dhuploadMaxFailureCountStr).intValue();
             } 
             final int dhuploadMaxFailureCount = dhuploadMaxFailureCountLocal;
             LOGGER.debug("dhuploadMaxFailureCount: "+dhuploadMaxFailureCount);
             
             DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");             

             // set audit start time and audit end time based on properties values               
             final String initialAuditStartTimeStr = properties.getProperty(PROP_START_AUDIT_DATETIME);
             Date initialAuditStartTime = null;
             if(!NullChecker.isNullOrEmpty(initialAuditStartTimeStr)) {
                 initialAuditStartTime = dateFormat.parse(initialAuditStartTimeStr);
             }
             final String initialAuditEndTimeStr = properties.getProperty(PROP_END_AUDIT_DATETIME);
             Date initialAuditEndTime = null;
             if(!NullChecker.isNullOrEmpty(initialAuditEndTimeStr)) {
                 initialAuditEndTime = dateFormat.parse(initialAuditEndTimeStr);
             }
             final String daysAgoIncrementStr = properties.getProperty(PROP_START_AUDIT_INCREMENT_DAYSAGO);
             int daysAgoIncrement = 0;
             if(!NullChecker.isNullOrEmpty(daysAgoIncrementStr)) {
                 daysAgoIncrement = new Integer(daysAgoIncrementStr).intValue();
             }             
             Date auditStartTime;
             Date auditEndTime;                       
             if (!NullChecker.isNullOrEmpty(initialAuditStartTime) && !NullChecker.isNullOrEmpty(initialAuditEndTime)) {
                 // have both start date and end date given - use them
                 auditStartTime = initialAuditStartTime;
                 auditEndTime = initialAuditEndTime;
             } else if (!NullChecker.isNullOrEmpty(initialAuditStartTime) && NullChecker.isNullOrEmpty(initialAuditEndTime)) {
                 // have only start date given - use execution time as the end date
                 auditStartTime = initialAuditStartTime;
                 auditEndTime = new Date();
             } else {
                 // have neither given - use (current time - increment in days) for start date, use current time as end date
                 Date currentTime = new Date();
                 Calendar calendar = new GregorianCalendar();
                 calendar.setTime(currentTime);
                 calendar.add(Calendar.DAY_OF_YEAR, -daysAgoIncrement);
                 auditStartTime = calendar.getTime();       
                 auditEndTime = currentTime;
             }
             // set into BulkExportParams
             final BulkExportLib.BulkExportParams params = new BulkExportLib.BulkExportParams();
             params.startAuditedTime = auditStartTime; 
             params.endAuditedTime = auditEndTime; 

             // setup for zip file packing
             String zipDir = properties.getProperty(PROP_WATCH_DIR);
             // TODO: we need to verify the directory exists - or at least handle error gracefully - this dir is not in the dev VM!
             SimpleDateFormat sdf = new SimpleDateFormat(DQExportProperties.YEAR_MONTH_DAY);
             String auditDateStr = sdf.format(timeoutStartTime);
             final String zipFilename = FilenameUtils.normalize(zipDir + "/" + "DQUploadFiles" + "-"+ auditDateStr + "-" + batchId + ".zip");
             // create zip stream
             final FileOutputStream zipFileStream = new FileOutputStream(zipFilename);
             final ZipOutputStream zipOutputStream = new ZipOutputStream(zipFileStream);
             
             // execute the adapter data file download per audit date range
             final DHUpload uploader = new DHUpload();
             // set JDBC data source for adapter database
             BulkExportLib exporter = new BulkExportLib(dataSource);             
             exporter.exportReceivedDocs(params, new BulkFileProcessor() {                
                Date startAuditedTime;
                Date endAuditedTime;
                String hostname;
                boolean filesStoredInZipFile;
                int uploadFailureCount;
                boolean firstErrorConditionSet;
                
                @Override
                public void start() {                    
                    filesStoredInZipFile = false;
                    hostname = getHostname();
                    startAuditedTime = params.startAuditedTime; 
                    endAuditedTime = params.endAuditedTime; 
                    uploadFailureCount = 0;   
                    firstErrorConditionSet = false;
                } // end start()

                @Override
                public void processFile(String uniqueDocID, long rawFileSize, String filename, byte[] fileContents) { 
                //public void processFile(String uniqueDocID, long rawFileSize, String filename, InputStream fileContents) {   //Added  JMC 7/20/17
                    boolean errorCondition = false; 
                    
                    LOGGER.debug(LogUtil.cleanLogMessage("DQExportProperties processing filename ... " + filename));                   
                    
                    // Uses Raw_Data_Size from Adapter.Documents, pulled from EXPORT_SQL in BulkExportLib.java
                    long fileSizeBytes;
                    if (rawFileSize == 0 || NullChecker.isNullOrEmpty(rawFileSize)) {
                        fileSizeBytes = -1;
                    } else {
                        fileSizeBytes = rawFileSize;
                    }
                    if (fileContents == null) {
                        // no file content to upload
                        LOGGER.warn(LogUtil.cleanLogMessage("File content is null for filename " + filename + " - skipping processing."));
                        return;
                    }
                    // Get Transmisstion Log entry success values for docId - NOTE: eventType text must match that in DqTransmissionLogManagerBean
                    boolean alreadyDHUploaded = dqlm.isAlreadySuccessfulEventByLocalDocId(DqTransmissionLogManager.DHUPLOAD_EVENTTYPE, uniqueDocID, null, null);
                    long aday = 24L * 60L * 60L * 1000L;
                    Date adayago = new Date(System.currentTimeMillis() - aday);
                    boolean alreadyDQZipped = dqlm.isAlreadySuccessfulEventByLocalDocId(DqTransmissionLogManager.DQZIPSTORE_EVENTTYPE, uniqueDocID, adayago, null);

		    LOGGER.debug("uploadFailureCount: "+uploadFailureCount);
                    
                    if(uploadFailureCount >= dhuploadMaxFailureCount) { 
                        errorCondition = true;
                        if(!firstErrorConditionSet) {
                            firstErrorConditionSet = true;
                            LOGGER.warn("Aborting further upload attempts! - maximum failure count for upload attempts reached for current batch.");
                        }                        
                        LOGGER.debug("No more processing for this batch: maximum failure count reached.");
                    } else {
                        LOGGER.debug(LogUtil.cleanLogMessage("For filename: " + filename + " log has alreadyDHUploaded value of: " + alreadyDHUploaded + " alreadyDQZipped value of: "+alreadyDQZipped));
                    }
                     
		    if(!errorCondition) { 
                        // Only upload file with filename to DH if it hasn't already been uploaded
                        if(dqExportInfo.performDHUpload && !alreadyDHUploaded) {
                            final Date dhUploadStartTime = new Date();
                            try {                        
                                // upload file via REST to external DH site
                                // NOTE: Throws RestClientException when HTTP Response returns an error code, and also IOException
                                String dhDocId = uploader.uploadDoc(dqExportInfo.diameterHealthUrl, dqExportInfo.diameterHealthUrlPath, 
                                        dqExportInfo.diameterHealthUser, dqExportInfo.diameterHealthKey, 
                                    filename, fileContents, dqExportInfo.connectionTimeout, dqExportInfo.socketTimeout);                             

                                try {
                                    dqlm.logDHUpload(batchId, dhUploadStartTime, new Date(), hostname, uniqueDocID, filename, "xml", fileSizeBytes, "Diameter Health", "success", dhDocId, null);
                                    // reset the failure count from this success
                                    uploadFailureCount = 0;
                                } catch(Exception ex) {
                                    LOGGER.error("logDHUpload exception thrown: ",ex);
                                }
                            } catch (RestClientException e) {
                                // catch DH file upload exception, log it, and quit
                                errorCondition = true;
                                // increment the failure count for next attempt
                                uploadFailureCount = uploadFailureCount + 1;
                                LOGGER.error(LogUtil.cleanLogMessage("Diameter Health File Upload Exception for filename: " + filename + ": "),e);  
                                // set Transmission log
                                try {
                                    String errorText;
                                    if(NullChecker.isNotNullOrEmpty(e.getStatusCode())) {
                                        // NOTE: response body is not stored DqTransmissionLog, 
                                        // so that it isn't returned in Web-service response and display in a web-browser, for security reasons                                    
                                        errorText= "DHUpload HTTP request returned HTTP error response: "+ e.getStatusCode() + " / " + e.getMessage();
                                        LOGGER.error(errorText+" with HTTP response body: "+e.getHttpResponse(), e);
                                    } else {
                                        errorText = e.getMessage();
                                        LOGGER.error("DHUpload class returned error: "+errorText, e);                                    
                                    }                                
                                    dqlm.logDHUpload(batchId, dhUploadStartTime, new Date(), hostname, uniqueDocID, filename, "xml", fileSizeBytes, "Diameter Health", "failure", null, errorText);
                                } catch(Exception ex) {
                                    LOGGER.error("logForDataFile exception thrown: ",ex);
                                }                            
                            } catch (Exception e) {
                                // catch DH file upload exception, log it, and quit
                                errorCondition = true;
                                // increment the failure count for next attempt
                                uploadFailureCount = uploadFailureCount + 1;
                                LOGGER.error(LogUtil.cleanLogMessage("Diameter Health File Upload Exception for filename: " + filename + ": "),e);  
                                // set Transmission log
                                try {                                
                                    String errorText = e.getMessage();                 
                                    LOGGER.error("DHUpload class returned error: "+errorText, e); 
                                    dqlm.logDHUpload(batchId, dhUploadStartTime, new Date(), hostname, uniqueDocID, filename, "xml", fileSizeBytes, "Diameter Health", "failure", null, errorText);
                                } catch(Exception ex) {
                                    LOGGER.error("logForDataFile exception thrown: ",ex);
                                }   
                            }
                        } // end if attemptDHUpload && !alreadyDHUploaded
                    } // end if !errorCondition) 
                    
                    //if(!errorCondition) { 
                        // Only store file with filename into DQ zip file if it hasn't already been stored previously in an earlier run
                        if(dqExportInfo.performDQZipStore && !alreadyDQZipped) {   
                            final Date dqZipStoreStartTime = new Date();
                            // add current file to zip
                            ZipEntry entry = new ZipEntry(filename);
                            try {
                                zipOutputStream.putNextEntry(entry);
                                // NOTE: InputStream for Blob is always at end of file suddenly, which is expected, so changed to byte array implementation
                                 // how much of the blob to read/write at a time
//                                final int blobChunkSizeToWriteBytes = 4096;
//                                byte[] buff = new byte[blobChunkSizeToWriteBytes]; 
//                                int len = 0;
//                                
//                                while ((len = fileContents.read(buff)) != -1) {
//                                   zipOutputStream.write(buff, 0, len);
//                                }
                                if(NullChecker.isNotNullOrEmpty(fileContents)) {
                                    zipOutputStream.write(fileContents);
                                }
                                
                                // uncompressed file size - doesn't work, sets -1
                                //fileSizeBytes = entry.getSize();
                                
                                zipOutputStream.closeEntry();
                                if(!filesStoredInZipFile) {  
                                    filesStoredInZipFile = true;
                                }
                                try {
                                    dqlm.logDQZipStore(batchId, dqZipStoreStartTime, new Date(), hostname, uniqueDocID, filename, "xml", fileSizeBytes, "zip file on eHX Adapter server", "success", null, null);
                                } catch(Exception ex) {
                                    LOGGER.error("logDQZipStore exception thrown: ",ex);
                                }                                
                             } catch (Exception e) {
                                 //errorCondition = true;
                                 // update log entry for filename 
                                 LOGGER.error("Error storing into zip file for filename: " + LogUtil.cleanLogMessage(filename) + ": ", e);
                                 // set Transmission log
                                try {
                                    // Convert e stackTrace String array to String
                                    String errorText = getStacktraceString(e);                                    
                                    dqlm.logDQZipStore(batchId, dqZipStoreStartTime, new Date(), hostname, uniqueDocID, filename, "xml", fileSizeBytes, "zip file on eHX Adapter server", "failure", null, errorText);
                                } catch(Exception ex) {
                                    LOGGER.error("logDQZipStore exception thrown: ",ex);
                                }                                  
                            }
                        } // end if attemptDQZipStoreAndSend && !alreadyDQZipped
                    //} // end if !errorCondition

                } // end processFile()

                @Override
                public void finish() {
                    
                    boolean errorCondition = false;  
                    uploadFailureCount = 0;             
                    
                    if(!filesStoredInZipFile) {
                        long zipFileSize = 0; // can be double also
                        File zipFile = new File(zipFilename); //File instantiated with normalized 'zipFilename'                        
                        zipFileSize = zipFile.length();
                        if(zipFileSize == 0) {
                            zipFile.delete();
                        }
                    } else if(dqExportInfo.performDQZipSend && filesStoredInZipFile) {
                        final Date zipFileSendStartTime = new Date();
                        Exception zipFileException = null;
                        long zipFileSizeBytes = 0; // can be double also
                        // close zip stream
                        try {                        
                            // send the zip file
                            zipOutputStream.finish();
                            zipOutputStream.close();
                            zipFileStream.flush();
                            zipFileStream.close();
                            // set zip file size
                            File zipFile = new File(zipFilename);  //File instantiated with normalized 'zipFilename'                        

                            zipFileSizeBytes = zipFile.length();
                            if(zipFileSizeBytes == 0) {
                                zipFile.delete();
                            }
                        } catch (Exception e) {                               
                            errorCondition = true;
                            LOGGER.error(LogUtil.cleanLogMessage("Error closing zip file stream for zip filename " + zipFilename + ": "), e);
                            zipFileException = e;
                        } // TODO: finally which deletes file on disk if error?
                        
                        if(!errorCondition && dqExportInfo.performDQZipSend)
                        {
                            File origZipFile = new File(zipFilename);
                            File shareZipDir = new File(FilenameUtils.normalize(dqExportInfo.dqZipDir));
                            
                             try 
                                {
                                 if(origZipFile.exists() && shareZipDir.exists())
                                  {
                                    FileUtils.copyFileToDirectory(origZipFile, shareZipDir); // Copy ZipFile to Network Share
                                    origZipFile.delete();                 // Delete original on Server
                                  }
                                 else
                                  {
                                     errorCondition = true;
                                     LOGGER.error("Missing zip file or zip transmission Path for DQ ");
                                  }
                                }
                            catch(IOException ex) 
                            {
                                errorCondition = true;
                                zipFileException = ex;
                                LOGGER.error("The exception in moving zip file to network share is: "+LogUtil.cleanLogMessage(getStacktraceString(ex)));
                            }
                        } // end if !errorCondition 

                        if(errorCondition) {
                            // log error for zip file send
                            try {
//                                String errorText = null;
//                                if(null != zipFileException) {
//                                    StringBuilder sbe = new StringBuilder();
//                                    sbe.append(zipFileException.getMessage()); 
//                                    errorText = sbe.toString();
//                                } 
                                String errorText = getStacktraceString(zipFileException);
                                dqlm.logDQZipMove(batchId, zipFileSendStartTime, new Date(), hostname, null, zipFilename, "zip", zipFileSizeBytes, "Data Quality file share", "failure", null, errorText);
                            } catch(Exception ex) {
                                LOGGER.error("logDQZipMove exception thrown: ",ex);
                            }                              
                        } else {
                            try {
                                dqlm.logDQZipMove(batchId, zipFileSendStartTime, new Date(), hostname, null, zipFilename, "zip", zipFileSizeBytes, "Data Quality file share", "success", null, null);
                            } catch(Exception ex) {
                                LOGGER.error("logDQZipMove exception thrown: ",ex);
                            }                            
                        } // end else
                    } // end performDQZipSend && filesStoredInZipFile
                    
                    if (dqExportInfo.performEmailSend) {
                        // generate report and send in email
                        // TODO: need to add a second to end date?
                        String report = trep.generateReportText(timeoutStartTime, new Date(), startAuditedTime, endAuditedTime, dqExportInfo.performDHUpload, dqExportInfo.performDQZipSend);
                        LOGGER.debug(LogUtil.cleanLogMessage("Data Quality DQExportProperties Summary Email Notification Report: "+report));                        

                        Map<String, String> messageAttributes = new HashMap<>();
                        messageAttributes.put("body", report);
                        Emailer emailer = new Emailer();
                        boolean emailSuccess = emailer.emailNotificationDQUpload(messageAttributes);
                        if(!emailSuccess) {
                            LOGGER.warn(LogUtil.cleanLogMessage("Email notification sending failed!"));
                        }
                    } // end if performEmailSend
                    
                } // end finish()
                
            }); // end exporter.exportReceivedDocs()
        } catch (Exception ex) {
            LOGGER.error("DQExportProperties timeout method exception: ", ex);
        } finally {
            DatabaseUtils.closeIO(propInputStream, LOGGER);
            LOGGER.info("DQExportProperties batchId: "+batchId+" completed execution.");
        }
   } // end timeout()

   /**
    * Simple structure to hold the information needed to process a timed iteration of
    * the Data Quality Export.
    */

   
    // Can't use this version, due to security concerns, display in website page, etc.
//    public String getStacktraceString(Exception e) {
//        StringWriter sw = new StringWriter();
//        PrintWriter pw = new PrintWriter(sw);
//        e.printStackTrace(pw);
//        //return sw.toString(); // stack trace as a string        
//        // truncate to 2999 bytes for VARCHAR2 length of 3000 max size in DB
//        String errorStr = null;
//        try {
//            errorStr = Utils.getTruncatedString(sw.toString(), 2999, "UTF-8");
//        } catch (UnsupportedEncodingException ex) {
//            LOGGER.error("getStacktraceString() threw error: ",ex);             
//        }
//        return errorStr;
//    }
    
    public String getStacktraceString(Exception e) {
        String errorStr = null;
        if(NullChecker.isNotNullOrEmpty(e)) {
            errorStr = e.getMessage();
        } 
        return errorStr;
    }
   
    /**
     * Retrieve the system hostname.
     * 
     * @return 
     */
    public static String getHostname() {
        String hostname = null;
        //if (System.getProperty("os.name").startsWith("Windows")) {
            // Windows will always set the 'COMPUTERNAME' variable
        //    return System.getenv("COMPUTERNAME");
        //} else {
            // If it is not Windows then it is most likely a Unix-like operating system
            // such as Solaris, AIX, HP-UX, Linux or MacOS.
            // Most modern shells (such as Bash or derivatives) sets the 
            // HOSTNAME variable so lets try that first.
            hostname = System.getenv("HOSTNAME");
            if (null == hostname) {
                // some other option
                hostname = "Unknown";                
            }
        //}
        return hostname;
    }
}
