package gov.va.med.nhin.adapter.audit;

import gov.va.med.nhin.adapter.logging.LogConstants;

import java.util.Arrays;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;

import javax.ejb.*;
import javax.jws.WebService;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import javax.xml.datatype.DatatypeFactory;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.nhin.adapter.utils.NullChecker;
import javax.jws.HandlerChain;
import org.slf4j.MDC;

/**
 * @author David Vazquez
 */
@WebService(serviceName = "AuditManagerService",
            wsdlLocation = "META-INF/wsdl/AuditManager.wsdl",
            portName = "AuditManagerPort",
            endpointInterface = "gov.va.med.nhin.adapter.audit.AuditManagerPortType",
            targetNamespace = "urn:gov:va:med:nhin:adapter:audit")
@HandlerChain(file = "SOAPHandlerChain.xml")
@TransactionAttribute(value = TransactionAttributeType.SUPPORTS)
@Stateless(name = "AuditManager", mappedName = "AuditManager")
public class AuditManagerBean implements AuditManagerRemote, AuditManagerLocal, AuditManagerPortType
{
    private static final Logger logger = LoggerFactory.getLogger(AuditManagerBean.class.getName());

    private EntityManager entityManager;

    @PersistenceContext
    public void setEntityManager(EntityManager entityManager)
    {
        this.entityManager = entityManager;
    }

    public List<Audit> getAuditsByICN(String icn, Date fromDate, Date toDate)
    {
        Query query;

        if (fromDate != null && toDate != null) {
            query = entityManager.createNamedQuery("Audit.findByIcnAndDate");
            query.setParameter("fromDate", fromDate);
            query.setParameter("toDate", toDate);
        }
        else {
            query = entityManager.createNamedQuery("Audit.findByIcn");
        }

        query.setParameter("icn", icn);

        return query.getResultList();
    }

    public List<Audit> getAuditsByICN(String icn, Date fromDate, Date toDate, String action)
    {
        Query query;

        if (fromDate != null && toDate != null) {
            query = entityManager.createNamedQuery("Audit.findByIcnAndActionAndDate");
            query.setParameter("fromDate", fromDate);
            query.setParameter("toDate", toDate);
        }
        else {
            query = entityManager.createNamedQuery("Audit.findByIcnAndAction");
        }

        query.setParameter("icn", icn);
        query.setParameter("action", action);

        return query.getResultList();
    }

    public List<AuditsReport> getAuditsByICN(String icn, Date fromDate, Date toDate, String[] actions)
    {
        return getAudits(fromDate, toDate, new String[]{icn}, false, null, false, null, false, actions, false, -1, -1);
    }

    public List<Audit> getAuditsByUserId(String userId, Date fromDate, Date toDate)
    {
        Query query;

        if (fromDate != null && toDate != null) {
            query = entityManager.createNamedQuery("Audit.findByUserIdAndDate");
            query.setParameter("fromDate", fromDate);
            query.setParameter("toDate", toDate);
        }
        else {
            query = entityManager.createNamedQuery("Audit.findByUserId");
        }

        query.setParameter("userId", userId);

        return query.getResultList();
    }
    
    public List<Audit> getAuditsBySystemId(String systemId, Date fromDate, Date toDate)
    {
        Query query;

        if (fromDate != null && toDate != null) {
            query = entityManager.createNamedQuery("Audit.findBySystemIdAndDate");
            query.setParameter("fromDate", fromDate);
            query.setParameter("toDate", toDate);
        }
        else {
            query = entityManager.createNamedQuery("Audit.findBySystemId");
        }

        query.setParameter("systemId", systemId);

        return query.getResultList();
    }

    public GetAuditsResponseType getAudits(GetAuditsType body)
    {
        GetAuditsResponseType ret = new GetAuditsResponseType();
        Date fromDate = null, toDate = null;
        String[] patientIds = null;
        boolean notPatientIds = false;
        String[] patientSSNs = null;
        boolean notPatientSSNs = false;
        String[] patientGivenNames = null;
        boolean notPatientGivenNames = false;
        String[] patientLastNames = null;
        boolean notPatientLastNames = false;
        String[] patientFacilityNumbers = null;
        boolean notPatientFacilityNumbers = false;
        String[] organizations = null;
        boolean notOrganizations = false;
        String[] remoteOrganizations = null;
        boolean notRemoteOrganizations = false;
        String[] userIds = null;
        boolean notUserIds = false;
        String[] systemIds = null;
        boolean notSystemIds = false;
        String[] userNames = null;
        boolean notUserNames = false;
        String[] userFacilityNumbers = null;
        boolean notUserFacilityNumbers = false;
        String[] actions = null;
        boolean notActions = false;
        String details = null;
        int pageSize = -1, pageNumber = 0;
        SortFieldsType sortFields = null;
        int patientTypes = 2;
        String[] purposeForUses = null;
        boolean notPurposeForUses = false;

        try {
            if (body.getFromDate() != null) {
                fromDate = body.getFromDate().toGregorianCalendar().getTime();
            }

            if (body.getToDate() != null) {
                toDate = body.getToDate().toGregorianCalendar().getTime();
            }

            if (body.getPatientIds() != null) {
                patientIds = body.getPatientIds().getValue().toArray(new String[body.getPatientIds().getValue().size()]);
                notPatientIds = body.getPatientIds().isNotIn();
            }

            if (body.getPatientSSNs() != null) {
                patientSSNs = body.getPatientSSNs().getValue().toArray(new String[body.getPatientSSNs().getValue().size()]);
                notPatientSSNs = body.getPatientSSNs().isNotIn();
            }
            if (body.getPatientGivenNames() != null) {
                patientGivenNames = body.getPatientGivenNames().getValue().toArray(new String[body.getPatientGivenNames().getValue().size()]);
                notPatientGivenNames = body.getPatientGivenNames().isNotIn();
            }
            if (body.getPatientLastNames() != null) {
                patientLastNames = body.getPatientLastNames().getValue().toArray(new String[body.getPatientLastNames().getValue().size()]);
                notPatientLastNames = body.getPatientLastNames().isNotIn();
            }

            if (body.getPurposeForUses() != null) {
                purposeForUses = body.getPurposeForUses().getValue().toArray(new String[body.getPurposeForUses().getValue().size()]);
                notPurposeForUses = body.getPurposeForUses().isNotIn();
            }
            if (body.getPatientFacilityNumbers() != null) {
                patientFacilityNumbers = body.getPatientFacilityNumbers().getValue().toArray(new String[body.getPatientFacilityNumbers().getValue().size()]);
                notPatientFacilityNumbers = body.getPatientFacilityNumbers().isNotIn();
            }

            if (body.getOrganizationIds() != null) {
                organizations = body.getOrganizationIds().getValue().toArray(new String[body.getOrganizationIds().getValue().size()]);
                notOrganizations = body.getOrganizationIds().isNotIn();
            }

            if (body.getRemoteOrganizationIds() != null) {
                remoteOrganizations = body.getRemoteOrganizationIds().getValue().toArray(new String[body.getRemoteOrganizationIds().getValue().size()]);
                notRemoteOrganizations = body.getRemoteOrganizationIds().isNotIn();
            }

            if (body.getUserIds() != null) {
                userIds = body.getUserIds().getValue().toArray(new String[body.getUserIds().getValue().size()]);
                notUserIds = body.getUserIds().isNotIn();
            }
            
            if (body.getSystemIds() != null) {
                systemIds = body.getSystemIds().getValue().toArray(new String[body.getSystemIds().getValue().size()]);
                notSystemIds = body.getSystemIds().isNotIn();
            }

            if (body.getUserNames() != null) {
                userNames = body.getUserNames().getValue().toArray(new String[body.getUserNames().getValue().size()]);
                notUserNames = body.getUserNames().isNotIn();
            }

            if (body.getUserFacilityNumbers() != null) {
                userFacilityNumbers = body.getUserFacilityNumbers().getValue().toArray(new String[body.getUserFacilityNumbers().getValue().size()]);
                notUserIds = body.getUserFacilityNumbers().isNotIn();
            }

            if (body.getActions() != null && body.getActions().getValue().size() > 0) {
                actions = new String[body.getActions().getValue().size()];
                for (int i = 0; i < actions.length; ++i) {
                    try {
                        actions[i] = body.getActions().getValue().get(i).value();
                    }
                    catch (Exception ex) {
                        ex.printStackTrace();
                    }
                }
                notActions = body.getActions().isNotIn();
            }

            details = body.getDetails();

            if (body.getPatientTypes() != null) {
                patientTypes = body.getPatientTypes();
            }

            if (body.getPageInfo() != null) {
                PageInfoType pageInfo = body.getPageInfo();
                pageSize = pageInfo.getPageSize();
                pageNumber = pageInfo.getPageNumber();
            }
            else {
                pageSize = -1;
                pageNumber = 0;
            }

            if (body.getSortFields() != null && !NullChecker.isNullOrEmpty(body.getSortFields().getSortField())) {
                sortFields = body.getSortFields();
            }

            List<AuditsReport> audits = getAudits(fromDate, toDate, patientIds, notPatientIds, patientSSNs, notPatientSSNs, patientGivenNames, notPatientGivenNames, patientLastNames, notPatientLastNames, purposeForUses, notPurposeForUses, patientFacilityNumbers, notPatientFacilityNumbers, organizations, notOrganizations, remoteOrganizations, notRemoteOrganizations, userIds, notUserIds, systemIds, notSystemIds, userNames, notUserNames, userFacilityNumbers, notUserFacilityNumbers, actions, notActions, details, sortFields, pageSize, pageNumber, patientTypes);

            PageInfoType pageInfo = new PageInfoType();
            pageInfo.setPageSize(pageSize != -1 ? Math.min(pageSize, audits.size()) : audits.size());
            pageInfo.setPageNumber(pageNumber);
            pageInfo.setTotalSize(getAuditsCount(fromDate, toDate, patientIds, notPatientIds, patientSSNs, notPatientSSNs, patientGivenNames, notPatientGivenNames, patientLastNames, notPatientLastNames, purposeForUses, notPurposeForUses, patientFacilityNumbers, notPatientFacilityNumbers, organizations, notOrganizations, remoteOrganizations, notRemoteOrganizations, userIds, notUserIds, systemIds, notSystemIds, userNames, notUserNames, userFacilityNumbers, notUserFacilityNumbers, actions, notActions, details, patientTypes));
            ret.setPageInfo(pageInfo);

            if (!NullChecker.isNullOrEmpty(audits)) {
                DatatypeFactory df = DatatypeFactory.newInstance();
                AuditsType auditsType = new AuditsType();
                ret.setAudits(auditsType);
                for (AuditsReport audit : audits) {
                    AuditType auditType = new AuditType();

                    auditType.setAuditId(audit.getAuditId().longValue());
                    auditType.setAction(ActionType.fromValue(audit.getActionName()));
                    Calendar auditTime = GregorianCalendar.getInstance(TimeZone.getTimeZone("UTC"));
                    auditTime.setTime(audit.getAuditTime());
                    auditType.setAuditTime(df.newXMLGregorianCalendar((GregorianCalendar) auditTime));
                    auditType.setDocumentId(audit.getDocumentId());
                    auditType.setDocumentTitle(audit.getDocumentTitle());
                    auditType.setSourcePatientId(audit.getDocumentSourcePatientId());
                    auditType.setOptOutReason(audit.getOptOutReasonText());
                    auditType.setOrganizationId(audit.getOrganizationId());
                    auditType.setOrganizationName(audit.getOrganizationName());
                    auditType.setPatientId(audit.getPatientId());
                    auditType.setPatientSSN(audit.getPatientSSN());
                    auditType.setPatientGivenName(audit.getPatientGivenName());
                    auditType.setPatientLastName(audit.getPatientLastName());
                    auditType.setPatientMiddleName(audit.getPatientMiddleName());
                    auditType.setPatientFacilityNumber(audit.getPatientFacilityNumber());
                    auditType.setPatientFacilityName(audit.getPatientFacilityName());
                    auditType.setPurposeForUse(audit.getPurposeForUse());
                    auditType.setRemoteDocumentId(audit.getRemoteDocId());
                    auditType.setRemoteDocumentRepositoryId(audit.getRemoteDocRepositoryId());
                    auditType.setRemoteOrganizationId(audit.getRemoteOrganizationId());
                    auditType.setRemoteOrganizationName(audit.getRemoteOrganizationName());
                    auditType.setUserId(audit.getUserId());
                    auditType.setSystemId(audit.getSystemId());
                    auditType.setUserName(audit.getUserName());
                    auditType.setUserFacilityNumber(audit.getUserFacilityNumber());
                    auditType.setUserFacilityName(audit.getUserFacilityName());
                    auditType.setUserRole(audit.getUserRole());
                    auditType.setHieTransactionId(audit.getHieTransactionId());
                    if(audit.getOccupation() != null){
                        auditType.setRoleName(audit.getOccupation().getRoleName());
                    }
                    else{
                        auditType.setRoleName("Unknown");
                    }
                    auditType.setDetails(audit.getDetails());

                    auditType.setDocSpecType(audit.getDocSpecType());

                    auditsType.getAudit().add(auditType);
                }
            }
        }
        catch (Throwable t) {
            throw new UnsupportedOperationException("There was an error process this request.", t);
        }

        return ret;
    }

    public List<AuditsReport> getAudits(Date fromDate, Date toDate, String[] patientIds, boolean notPatientIds, String[] organizations, boolean notOrganizations, String[] remoteOrganizations, boolean notRemoteOrganizations, String[] actions, boolean notActions, int pageSize, int pageNumber)
    {
        return getAudits(fromDate, toDate, patientIds, notPatientIds, null, false, null, false, null, false, null, false, null, false, organizations, notOrganizations, remoteOrganizations, notRemoteOrganizations, null, false, null, false, null, false, null, false, actions, notActions, null, null, pageSize, pageNumber, 3);
    }

    public List<AuditsReport> getAudits(Date fromDate, Date toDate, String[] patientIds, boolean notPatientIds, String[] organizations, boolean notOrganizations, String[] remoteOrganizations, boolean notRemoteOrganizations, String[] actions, boolean notActions, String detailsSearch, int pageSize, int pageNumber)
    {
        return getAudits(fromDate, toDate, patientIds, notPatientIds, null, false, null, false, null, false, null, false, null, false, organizations, notOrganizations, remoteOrganizations, notRemoteOrganizations, null, false, null, false, null, false, null, false, actions, notActions, detailsSearch, null, pageSize, pageNumber, 3);
    }

    static private final Map<String, String> fields = new HashMap<String, String>();

    static {
        fields.put("auditTime", "a.auditTime");
        fields.put("action", "a.actionName");
        fields.put("userId", "a.userId");
        fields.put("systemId", "a.systemId");
        fields.put("userName", "a.userName");
        fields.put("userFacilityNumber", "a.userFacilityNumber");
        fields.put("organizationId", "a.organizationId");
        fields.put("organizationName", "a.organizationName");
        fields.put("patientId", "a.patientId");
        fields.put("patientSSN", "a.patientSSN");
        fields.put("patientLastName", "a.patientLastName");
        fields.put("patientGivenName", "a.patientGivenName");
        fields.put("patientMiddleName", "a.patientMiddleName");
        fields.put("patientFacilityNumber", "a.patientFacilityNumber");
        fields.put("patientFacilityName", "a.patientFacilityName");
        fields.put("purposeForUse", "a.purposeForUse");
        fields.put("remoteOrganizationId", "a.remoteOrganizationId");
        fields.put("remoteOrganizationName", "a.remoteOrganizationName");
        fields.put("optOutReason", "a.optOutReasonText");
        fields.put("documentId", "a.documentId");
        fields.put("documentTitle", "a.documentTitle");
    }

    public List<AuditsReport> getAudits(Date fromDate, Date toDate, String[] patientIds, boolean notPatientIds, String[] patientSSNs, boolean notPatientSSNs, String[] patientGivenNames, boolean notPatientGivenNames, String[] patientLastNames, boolean notPatientLastNames, String[] purposeForUses, boolean notPurposeForUses, String[] patientFacilityNumbers, boolean notPatientFacilityNumbers, String[] organizations, boolean notOrganizations, String[] remoteOrganizations, boolean notRemoteOrganizations, String[] userIds, boolean notUserIds, String[] systemIds, boolean notSystemIds, String[] userNames, boolean notUserNames, String[] userFacilityNumbers, boolean notUserFacilityNumbers, String[] actions, boolean notActions, String detailsSearch, SortFieldsType sortFields, int pageSize, int pageNumber, int patientTypes)
    {
        HashMap<String, Object> setParams = new HashMap<String, Object>();
        String whereClause = buildWhereClause(fromDate, toDate, patientIds, notPatientIds, patientSSNs, notPatientSSNs, patientGivenNames, notPatientGivenNames, patientLastNames, notPatientLastNames, purposeForUses, notPurposeForUses, patientFacilityNumbers, notPatientFacilityNumbers, organizations, notOrganizations, remoteOrganizations, notRemoteOrganizations, userIds, notUserIds, systemIds, notSystemIds, userNames, notUserIds, userFacilityNumbers, notUserFacilityNumbers, actions, notActions, detailsSearch, patientTypes, setParams);

        String orderByClause = null;
        if (sortFields != null && !NullChecker.isNullOrEmpty(sortFields.getSortField())) {
            for (SortFieldType sortField : sortFields.getSortField()) {
                String field = null;
                if (sortField.getField() != null && (field = fields.get(sortField.getField().value())) != null) {
                    if (orderByClause == null) {
                        orderByClause = "ORDER BY ";
                    }
                    else {
                        orderByClause += ",";
                    }

                    orderByClause += field + (sortField.getDirection() == SortDirection.DESC ? " DESC" : " ASC");
                }
                else {
                    throw new RuntimeException("Invalid field name was used as sortField.");
                }
            }
        }
        else {
            orderByClause = "ORDER BY a.auditTime DESC";
        }

        String q = "SELECT a FROM AuditsReport a " + whereClause + " " + orderByClause;
        logger.trace("Query: {}", q);

        Query query = entityManager.createQuery(q);

        // use a query hint to specify the number of records at a time to fetch
        // from database in order to improve performance.  Fetch size is set to
        // pageSize when given so we don't waste time fetching more records
        // than needed.  Sufficiently large fetch size is used when pageSize
        // is not given so we minimize number of trips to the database.
        // WARNING: query hints tend to be specific to JPA provider that's being
        // used.  The hint we use to set the JDBC fetch size is specific to
        // eclipselink and will have to change if the provider ever changes.
        if (pageSize > 0) {
            if (pageNumber >= 0) {
                query.setFirstResult(pageNumber * pageSize);
                query.setMaxResults(pageSize);
            }
            query.setHint("eclipselink.jdbc.fetch-size", pageSize);
        }
        else {
            query.setHint("eclipselink.jdbc.fetch-size", 1000);
        }

        for (Map.Entry<String, Object> entry : setParams.entrySet()) {
            query.setParameter(entry.getKey(), entry.getValue());
        }

        return query.getResultList();
    }

    private int getAuditsCount(Date fromDate, Date toDate, String[] patientIds, boolean notPatientIds, String[] patientSSNs, boolean notPatientSSNs, String[] patientGivenNames, boolean notPatientGivenNames, String[] patientLastNames, boolean notPatientLastNames, String[] purposeForUses, boolean notPurposeForUses, String[] patientFacilityNumbers, boolean notPatientFacilityNumbers, String[] organizations, boolean notOrganizations, String[] remoteOrganizations, boolean notRemoteOrganizations, String[] userIds, boolean notUserIds, String[] systemIds, boolean notSystemIds, String[] userNames, boolean notUserNames, String[] userFacilityNumbers, boolean notUserFacilityNumbers, String[] actions, boolean notActions, String detailsSearch, int patientTypes)
    {
        HashMap<String, Object> setParams = new HashMap<>();
        String whereClause = buildWhereClause(fromDate, toDate, patientIds, notPatientIds, patientSSNs, notPatientSSNs, patientGivenNames, notPatientGivenNames, patientLastNames, notPatientLastNames, purposeForUses, notPurposeForUses, patientFacilityNumbers, notPatientFacilityNumbers, organizations, notOrganizations, remoteOrganizations, notRemoteOrganizations, userIds, notUserIds, systemIds, notSystemIds, userNames, notUserIds, userFacilityNumbers, notUserFacilityNumbers, actions, notActions, detailsSearch, patientTypes, setParams);

        String qCount = "SELECT COUNT(a) FROM AuditsReport a " + whereClause;
        logger.trace("Query: {}", qCount);

        Query countQuery = entityManager.createQuery(qCount);

        for(Map.Entry<String, Object> entry : setParams.entrySet())
        {
            countQuery.setParameter(entry.getKey(), entry.getValue());
        }

        Object countObj = countQuery.getSingleResult();
        int count = 0;
		if(countObj instanceof Long)
		{
			count = ((Long) countObj).intValue();
			logger.trace("long count {}:", count);
		}
		if(countObj instanceof Integer)
		{
			count = ((Integer) countObj);
			logger.trace("int count {}:", count);
		}

        return count;
    }

    private String buildWhereClause(Date fromDate, Date toDate, String[] patientIds, boolean notPatientIds, String[] patientSSNs, boolean notPatientSSNs, String[] patientGivenNames, boolean notPatientGivenNames, String[] patientLastNames, boolean notPatientLastNames, String[] purposeForUses, boolean notPurposeForUses, String[] patientFacilityNumbers, boolean notPatientFacilityNumbers, String[] organizations, boolean notOrganizations, String[] remoteOrganizations, boolean notRemoteOrganizations, String[] userIds, boolean notUserIds, String[] systemIds, boolean notSystemIds, String[] userNames, boolean notUserNames, String[] userFacilityNumbers, boolean notUserFacilityNumbers, String[] actions, boolean notActions, String detailsSearch, int patientTypes, HashMap<String, Object> setParams)
    {
        StringBuffer whereClause = new StringBuffer();

        if (fromDate != null) {
            appendWhereClause(whereClause, "a.auditTime >= :fromDate");
            setParams.put("fromDate", fromDate);
        }

        if (toDate != null) {
            appendWhereClause(whereClause, "a.auditTime <= :toDate");
            setParams.put("toDate", toDate);
        }

        if (!NullChecker.isNullOrEmpty(patientIds)) {
            if (!notPatientIds) {
                appendWhereClause(whereClause, "a.patientId in (:icn0");
            }
            else {
                appendWhereClause(whereClause, "a.patientId not in (:icn0");
            }
            setParams.put("icn0", patientIds[0]);

            for (int i = 1; i < patientIds.length; ++i) {
                whereClause.append(",:icn").append(i);
                setParams.put("icn" + i, patientIds[i]);
            }

            whereClause.append(')');
        }

        if (!NullChecker.isNullOrEmpty(patientSSNs)) {
            if (!notPatientSSNs) {
                appendWhereClause(whereClause, "a.patientSSN in (:patientSSN0");
            }
            else {
                appendWhereClause(whereClause, "a.patientSSN not in (:patientSSN0");
            }
            setParams.put("patientSSN0", patientSSNs[0]);

            for (int i = 1; i < patientSSNs.length; ++i) {
                whereClause.append(",:patientSSN").append(i);
                setParams.put("patientSSN" + i, patientSSNs[i]);
            }

            whereClause.append(')');
        }

        if (!NullChecker.isNullOrEmpty(patientGivenNames)) {
            if (!notPatientGivenNames) {
                appendWhereClause(whereClause, "UPPER(a.patientGivenName) in (:patientGivenName0");
            }
            else {
                appendWhereClause(whereClause, "UPPER(a.patientGivenName) not in (:patientGivenName0");
            }
            setParams.put("patientGivenName0", patientGivenNames[0].toUpperCase());

            for (int i = 1; i < patientGivenNames.length; ++i) {
                whereClause.append(",:patientGivenName").append(i);
                setParams.put("patientGivenName" + i, patientGivenNames[i].toUpperCase());
            }

            whereClause.append(')');
        }

        if (!NullChecker.isNullOrEmpty(patientLastNames)) {
            if (!notPatientLastNames) {
                appendWhereClause(whereClause, "UPPER(a.patientLastName) in (:patientLastName0");
            }
            else {
                appendWhereClause(whereClause, "UPPER(a.patientLastName) not in (:patientLastName0");
            }
            setParams.put("patientLastName0", patientLastNames[0].toUpperCase());

            for (int i = 1; i < patientLastNames.length; ++i) {
                whereClause.append(",:patientLastName").append(i);
                setParams.put("patientLastName" + i, patientLastNames[i].toUpperCase());
            }

            whereClause.append(')');
        }

        if (!NullChecker.isNullOrEmpty(purposeForUses)) {
            if (!notPurposeForUses) {
                appendWhereClause(whereClause, "UPPER(a.purposeForUse) in (:purposeForUse0");
            }
            else {
                appendWhereClause(whereClause, "UPPER(a.purposeForUse) not in (:purposeForUse0");
            }
            setParams.put("purposeForUse0", purposeForUses[0].toUpperCase());

            for (int i = 1; i < purposeForUses.length; ++i) {
                whereClause.append(",:purposeForUse").append(i);
                setParams.put("purposeForUse" + i, purposeForUses[i].toUpperCase());
            }

            whereClause.append(')');
        }

        if (!NullChecker.isNullOrEmpty(patientFacilityNumbers)) {
            if (!notPatientFacilityNumbers) {
                appendWhereClause(whereClause, "UPPER(a.patientFacilityNumber) in (:patientFacilityNumber0");
            }
            else {
                appendWhereClause(whereClause, "UPPER(a.patientFacilityNumber) not in (:patientFacilityNumber0");
            }
            setParams.put("patientFacilityNumber0", patientFacilityNumbers[0].toUpperCase());

            for (int i = 1; i < patientFacilityNumbers.length; ++i) {
                whereClause.append(",:patientFacilityNumber").append(i);
                setParams.put("patientFacilityNumber" + i, patientFacilityNumbers[i].toUpperCase());
            }

            whereClause.append(')');
        }

        if (!NullChecker.isNullOrEmpty(organizations) || !NullChecker.isNullOrEmpty(remoteOrganizations)) {
            appendWhereClause(whereClause, "(");
            StringBuffer inClause = new StringBuffer();

            if (!NullChecker.isNullOrEmpty(organizations)) {
                if (!notOrganizations) {
                    appendWhereClause(inClause, "a.organizationId like :organization0", "", " OR ");
                }
                else {
                    appendWhereClause(inClause, "not (a.organizationId like :organization0", "", " OR ");
                }
                setParams.put("organization0", organizations[0]);

                for (int i = 1; i < organizations.length; ++i) {
                    inClause.append(" OR a.organizationId like :organization").append(i);
                    setParams.put("organization" + i, organizations[i]);
                }

                if (notOrganizations) {
                    inClause.append(')');
                }
            }

            if (!NullChecker.isNullOrEmpty(remoteOrganizations)) {
                if (!notRemoteOrganizations) {
                    appendWhereClause(inClause, "a.remoteOrganizationId like :remoteOrganization0", "", " OR ");
                }
                else {
                    appendWhereClause(inClause, "not (a.remoteOrganizationId like :remoteOrganization0", "", " OR ");
                }
                setParams.put("remoteOrganization0", remoteOrganizations[0]);

                for (int i = 1; i < remoteOrganizations.length; ++i) {
                    inClause.append(" OR a.remoteOrganizationId LIKE :remoteOrganization").append(i);
                    setParams.put("remoteOrganization" + i, remoteOrganizations[i]);
                }

                if (notRemoteOrganizations) {
                    inClause.append(')');
                }
            }

            whereClause.append(inClause).append(")");
        }

        if (!NullChecker.isNullOrEmpty(userIds)) {
            if (!notUserIds) {
                if (userIds.length == 1) {
                    appendWhereClause(whereClause, "UPPER(a.userId) like (:userId0");
                } else {
                    appendWhereClause(whereClause, "UPPER(a.userId) in (:userId0");
                }
            }
            else {
                appendWhereClause(whereClause, "UPPER(a.userId) not in (:userId0");
            }
            if (userIds.length == 1) {
                setParams.put("userId0", "%" + userIds[0].toUpperCase() + "%");
            } else {
                setParams.put("userId0", userIds[0].toUpperCase());
            }

            for (int i = 1; i < userIds.length; ++i) {
                whereClause.append(",:userId").append(i);
                setParams.put("userId" + i, userIds[i].toUpperCase());
            }

            whereClause.append(")");
        }
        
        if (!NullChecker.isNullOrEmpty(systemIds)) {
            if (!notSystemIds) {
                if (systemIds.length == 1) {
                    appendWhereClause(whereClause, "UPPER(a.systemId) like (:systemId0");
                } else {
                    appendWhereClause(whereClause, "UPPER(a.systemId) in (:systemId0");
                }
            }
            else {
                appendWhereClause(whereClause, "UPPER(a.systemId) not in (:systemId0");
            }
            if (systemIds.length == 1) {
                setParams.put("systemId0", "%" + systemIds[0].toUpperCase() + "%");
            } else {
                setParams.put("systemId0", systemIds[0].toUpperCase());
            }

            for (int i = 1; i < systemIds.length; ++i) {
                whereClause.append(",:systemId").append(i);
                setParams.put("systemId" + i, systemIds[i].toUpperCase());
            }

            whereClause.append(")");
        }

        if(!NullChecker.isNullOrEmpty(userNames)) {
            if (!notUserNames) {
                if (userNames.length == 1) {
                    appendWhereClause(whereClause, "UPPER(a.userName) like (:userName0");
                } else {
                    appendWhereClause(whereClause, "UPPER(a.userName) in (:userName0");
                }
            }
            else {
                appendWhereClause(whereClause, "UPPER(a.userName) not in (:userName0");
            }
            if (userNames.length == 1) {
                setParams.put("userName0", "%" + userNames[0].toUpperCase() + "%");
            } else {
                setParams.put("userName0", userNames[0].toUpperCase());
            }

            for (int i = 1; i < userNames.length; ++i) {
                whereClause.append(",:userName").append(i);
                setParams.put("userName" + i, userNames[i].toUpperCase());
            }

            whereClause.append(")");
        }

        if (!NullChecker.isNullOrEmpty(userFacilityNumbers)) {
            if (!notUserFacilityNumbers) {
                appendWhereClause(whereClause, "UPPER(a.userFacilityNumber) in (:userFacilityNumber0");
            }
            else {
                appendWhereClause(whereClause, "UPPER(a.useFacilityNumber) not in (:userFacilityNumber0");
            }
            setParams.put("userFacilityNumber0", userFacilityNumbers[0].toUpperCase());

            for (int i = 1; i < userFacilityNumbers.length; ++i) {
                whereClause.append(",:userFacilityNumber").append(i);
                setParams.put("userFacilityNumber" + i, userFacilityNumbers[i].toUpperCase());
            }

            whereClause.append(")");
        }

        if (!NullChecker.isNullOrEmpty(actions)) {
            if (!notActions) {
                appendWhereClause(whereClause, "a.actionName in (:action0");
            }
            else {
                appendWhereClause(whereClause, "a.actionName not in (:action0");
            }
            setParams.put("action0", actions[0]);

            for (int i = 1; i < actions.length; ++i) {
                whereClause.append(",:action").append(i);
                setParams.put("action" + i, actions[i]);
            }

            whereClause.append(")");
        }

        if (patientTypes == 1) {
            // Show real patients only, exclude test patients
            appendWhereClause(whereClause, " (a.patientSsnEx is null or"
                    + // Ignore
                    // null
                    // entries
                    "  (substring(a.patientSsnEx, 1, 3) != '000' and"
                    + // Area number
                    // cannot be 000
                    "   substring(a.patientSsnEx, 1, 3) != '666' and"
                    + // Area number
                    // cannot be 666
                    "   substring(a.patientSsnEx, 1, 1) != '9'   and"
                    + // Area number
                    // cannot be in
                    // 900-999 range
                    "   substring(a.patientSsnEx, 4, 2) != '00'  and"
                    + // Group number
                    // cannot be 00
                    "   substring(a.patientSsnEx, 6, 4) != '0000')"
                    + // Serial number
                    // cannot be
                    // 0000
                    " )");
        } 
        else if (patientTypes == 2) {
            // Show test patients only, exclude real patients
            appendWhereClause(whereClause, " (a.patientSsnEx is not null and" + //null is real patient
            "  (substring(a.patientSsnEx, 1, 3) = '000' or" + // Area number
                                                                // can be 000
            "   substring(a.patientSsnEx, 1, 3) = '666' or" + // Area number
                                                                // can be 666
            "   substring(a.patientSsnEx, 1, 1) = '9'   or" + // Area number
                                                                // can be in
                                                                // 900-999 range
            "   substring(a.patientSsnEx, 4, 2) = '00'  or" + // Group number
                                                                // can be 00
            "   substring(a.patientSsnEx, 6, 4) = '0000')" + // Serial number
                                                                // can be
                                                                // 0000
            " )");
        }

        if (detailsSearch != null) {
            String search = "%" + detailsSearch.trim() + "%";
            appendWhereClause(whereClause, "a.details LIKE :search");
            setParams.put("search", search);
        }

        return whereClause.toString();
    }

    private void appendWhereClause(StringBuffer buffer, String str)
    {
        appendWhereClause(buffer, str, "WHERE ", " AND ");
    }

    private void appendWhereClause(StringBuffer buffer, String str, String empty, String notEmpty)
    {
        if (buffer.length() > 0) {
            buffer.append(notEmpty);
        }
        else {
            buffer.append(empty);
        }
        buffer.append(str);
    }

    @TransactionAttribute(value = TransactionAttributeType.REQUIRED)
    public void storeAudit(Audit audit)
    {
        audit.setAuditTime(new Date());
				if( null == audit.getHieTransactionId() ){
					audit.setHieTransactionId( MDC.get( LogConstants.CORRELATION_MDCKEY ) );
				}


        if (audit.getAuditId() == null) {
            entityManager.persist(audit);
        }
        else {
            entityManager.merge(audit);
        }
    }

    public GetAuditsSummaryResponseType getAuditsSummary(GetAuditsSummaryType body)
    {
        GetAuditsSummaryResponseType ret = new GetAuditsSummaryResponseType();
        Date fromDate = null, toDate = null;
        String[] patientIds = null;
        boolean notPatientIds = false;
        String[] patientSSNs = null;
        boolean notPatientSSNs = false;
        String[] patientGivenNames = null;
        boolean notPatientGivenNames = false;
        String[] patientLastNames = null;
        boolean notPatientLastNames = false;
        String[] patientFacilityNumbers = null;
        boolean notPatientFacilityNumbers = false;
        String[] organizations = null;
        boolean notOrganizations = false;
        String[] remoteOrganizations = null;
        boolean notRemoteOrganizations = false;
        String[] userIds = null;
        boolean notUserIds = false;
        String[] systemIds = null;
        boolean notSystemIds = false;
        String[] userNames = null;
        boolean notUserNames = false;
        String[] userFacilityNumbers = null;
        boolean notUserFacilityNumbers = false;
        String[] actions = null;
        boolean notActions = false;
        String details = null;
        GroupByFieldsType groupByFields = null;
        String[] purposeForUses = null;
        boolean notPurposeForUse = false;

        int patientType = 3;
        try {
            if (body.getFromDate() != null) {
                fromDate = body.getFromDate().toGregorianCalendar().getTime();
            }

            if (body.getToDate() != null) {
                toDate = body.getToDate().toGregorianCalendar().getTime();
            }

            if (body.getPatientIds() != null) {
                patientIds = body.getPatientIds().getValue().toArray(new String[body.getPatientIds().getValue().size()]);
                notPatientIds = body.getPatientIds().isNotIn();
            }

            if (body.getPatientSSNs() != null) {
                patientSSNs = body.getPatientSSNs().getValue().toArray(new String[body.getPatientSSNs().getValue().size()]);
                notPatientSSNs = body.getPatientSSNs().isNotIn();
            }
            if (body.getPatientGivenNames() != null) {
                patientGivenNames = body.getPatientGivenNames().getValue().toArray(new String[body.getPatientGivenNames().getValue().size()]);
                notPatientGivenNames = body.getPatientGivenNames().isNotIn();
            }
            if (body.getPatientLastNames() != null) {
                patientLastNames = body.getPatientLastNames().getValue().toArray(new String[body.getPatientLastNames().getValue().size()]);
                notPatientLastNames = body.getPatientLastNames().isNotIn();
            }

            if (body.getPurposeForUses() != null) {
                purposeForUses = body.getPurposeForUses().getValue().toArray(new String[body.getPurposeForUses().getValue().size()]);
                notPurposeForUse = body.getPurposeForUses().isNotIn();
            }

            if (body.getPatientFacilityNumbers() != null) {
                patientFacilityNumbers = body.getPatientFacilityNumbers().getValue().toArray(new String[body.getPatientFacilityNumbers().getValue().size()]);
                notPatientFacilityNumbers = body.getPatientFacilityNumbers().isNotIn();
            }

            if (body.getOrganizationIds() != null) {
                organizations = body.getOrganizationIds().getValue().toArray(new String[body.getOrganizationIds().getValue().size()]);
                notOrganizations = body.getOrganizationIds().isNotIn();
            }

            if (body.getRemoteOrganizationIds() != null) {
                remoteOrganizations = body.getRemoteOrganizationIds().getValue().toArray(new String[body.getRemoteOrganizationIds().getValue().size()]);
                notRemoteOrganizations = body.getRemoteOrganizationIds().isNotIn();
            }

            if (body.getUserIds() != null) {
                userIds = body.getUserIds().getValue().toArray(new String[body.getUserIds().getValue().size()]);
                notUserIds = body.getUserIds().isNotIn();
            }
            
            if (body.getSystemIds() != null) {
                systemIds = body.getSystemIds().getValue().toArray(new String[body.getSystemIds().getValue().size()]);
                notSystemIds = body.getSystemIds().isNotIn();
            }

            if (body.getUserNames() != null) {
                userNames = body.getUserNames().getValue().toArray(new String[body.getUserNames().getValue().size()]);
                notUserNames = body.getUserNames().isNotIn();
            }

            if (body.getUserFacilityNumbers() != null) {
                userFacilityNumbers = body.getUserFacilityNumbers().getValue().toArray(new String[body.getUserFacilityNumbers().getValue().size()]);
                notUserIds = body.getUserFacilityNumbers().isNotIn();
            }

            if (body.getActions() != null) {
                actions = new String[body.getActions().getValue().size()];
                for (int i = 0; i < actions.length; ++i) {
                    actions[i] = body.getActions().getValue().get(i).value();
                }
                notActions = body.getActions().isNotIn();
            }
            details = body.getDetails();

            if (body.getGroupByFields() != null && !NullChecker.isNullOrEmpty(body.getGroupByFields().getGroupByField())) {
                groupByFields = body.getGroupByFields();
            }
            if(body.getPatientTypes() != null){
                patientType = body.getPatientTypes();
            }
            List<Object> results = getAuditsSummary(fromDate, toDate, patientIds, notPatientIds, patientSSNs, notPatientSSNs, patientGivenNames, notPatientGivenNames, patientLastNames, notPatientLastNames, purposeForUses, notPurposeForUse, patientFacilityNumbers, notPatientFacilityNumbers, organizations, notOrganizations, remoteOrganizations, notRemoteOrganizations, userIds, notUserIds, systemIds, notSystemIds, userNames, notUserNames, userFacilityNumbers, notUserFacilityNumbers, actions, notActions, details, groupByFields, patientType);

            ret.setGroupByFields(groupByFields);

            if (!NullChecker.isNullOrEmpty(results)) {
                AuditSummariesType auditSummaries = new AuditSummariesType();
                ret.setAuditSummaries(auditSummaries);
                for (Object result : results) {
                    AuditSummaryType auditSummary = new AuditSummaryType();
                    if (groupByFields != null) {
                        Object[] r = (Object[]) result;
                        SummaryFieldsType g = new SummaryFieldsType();
                        auditSummary.setSummaryFields(g);
                        for (int i = 0; i < r.length - 1; ++i) {
                            if (r[i] != null) {
                                g.getSummaryField().add(r[i].toString());
                            }
                            else {
                                g.getSummaryField().add("NULL");
                            }
                        }
                        auditSummary.setCount((Long) r[r.length - 1]);
                    }
                    else {
                        auditSummary.setCount((Long) result);
                    }

                    auditSummaries.getAuditSummary().add(auditSummary);
                }
            }
        }
        catch (Throwable t) {
            throw new UnsupportedOperationException("There was an error process this request.", t);
        }

        return ret;
    }

    public List<Object> getAuditsSummary(Date fromDate, Date toDate, String[] patientIds, boolean notPatientIds, String[] patientSSNs, boolean notPatientSSNs, String[] patientGivenNames, boolean notPatientGivenNames, String[] patientLastNames, boolean notPatientLastNames, String[] purposeForUses, boolean notPurposeForUse, String[] patientFacilityNumbers, boolean notPatientFacilityNumbers, String[] organizations, boolean notOrganizations, String[] remoteOrganizations, boolean notRemoteOrganizations, String[] userIds, boolean notUserIds, String[] systemIds, boolean notSystemIds, String[] userNames, boolean notUserNames, String[] userFacilityNumbers, boolean notUserFacilityNumbers, String[] actions, boolean notActions, String detailsSearch, GroupByFieldsType groupByFields, int patientType)
    {
        HashMap<String, Object> setParams = new HashMap<String, Object>();
        String whereClause = buildWhereClause(fromDate, toDate, patientIds, notPatientIds, patientSSNs, notPatientSSNs, patientGivenNames, notPatientGivenNames, patientLastNames, notPatientLastNames, purposeForUses, notPurposeForUse, patientFacilityNumbers, notPatientFacilityNumbers, organizations, notOrganizations, remoteOrganizations, notRemoteOrganizations, userIds, notUserIds, systemIds, notSystemIds, userNames, notUserNames, userFacilityNumbers, notUserFacilityNumbers, actions, notActions, detailsSearch, patientType, setParams);
        String selectClause = null;
        String groupByClause = null;

        if (groupByFields != null && !NullChecker.isNullOrEmpty(groupByFields.getGroupByField())) {
            for (FieldType groupByField : groupByFields.getGroupByField()) {
                String field = null;
                if (groupByField != null && (field = fields.get(groupByField.value())) != null) {
                    if (selectClause == null) {
                        selectClause = field;
                        groupByClause = "GROUP BY " + field;
                    }
                    else {
                        selectClause += "," + field;
                        groupByClause += "," + field;
                    }
                }
                else {
                    throw new RuntimeException("Invalid field name was used as groupByField.");
                }
            }
            selectClause += ",";
        }
        else {
            groupByClause = "";
        }

        selectClause += "count(distinct case when (a.isTestPatient = 0) then a.patientSsnEx else '' end) realPatient,"
                + // Count
                // of
                // unique
                // real
                // patients
                "count(distinct case when (a.isTestPatient = 1) then a.patientSsnEx else '' end) matchFoundReal,"
                + // Count
                // of
                // unique
                // test
                // patients
                "sum(case when (a.matchFound = 1 and a.isTestPatient = 0) then 1 else 0 end) matchFailedReal,"
                + // Matches
                // found
                // for
                // real
                // patients
                "sum(case when (a.matchFound = 1 and a.isTestPatient = 1) then 1 else 0 end) testPatient,"
                + // Matches
                // found
                // for
                // test
                // patients
                "sum(case when (a.matchFailed = 1 and a.isTestPatient = 0) then 1 else 0 end) matchFoundTest,"
                + // Match
                // fails
                // for
                // real
                // patients
                "sum(case when (a.matchFailed = 1 and a.isTestPatient = 1) then 1 else 0 end) matchFailedTest,"
                + // Match
                // fails
                // for
                // test
                // patients
                "count(a.auditId) AUDITS_COUNT";

        String q = "SELECT " + selectClause + " FROM AuditsReport a " + whereClause + " " + groupByClause;
        logger.trace("Query: {0}", q);

        Query query = entityManager.createQuery(q);

        for (Map.Entry<String, Object> entry : setParams.entrySet()) {
            query.setParameter(entry.getKey(), entry.getValue());
        }

        return query.getResultList();
    }

    @Override
    public GetDocumentAuditsResponseType getDocumentAudits(GetDocumentAuditsType body)
    {
        GetDocumentAuditsResponseType ret = new GetDocumentAuditsResponseType();
        Date fromDate = null;
        Date toDate = null;
        String[] organizations = null;
        boolean notOrganizations = false;
        List<String> actions = new ArrayList<String>();
        boolean notActions = false;
        StringBuffer queryString = new StringBuffer();

        try
        {
            if (body.getFromDate() != null)
            {
                fromDate = body.getFromDate().toGregorianCalendar().getTime();
}
            if (body.getToDate() != null)
            {
                toDate = body.getToDate().toGregorianCalendar().getTime();
            }
            if (body.getOrganizationIds() != null)
            {
                organizations = body.getOrganizationIds().getValue().toArray(new String[body.getOrganizationIds().getValue().size()]);
                notOrganizations = body.getOrganizationIds().isNotIn();
            }
            if (body.getActions() != null && body.getActions().getValue().size() > 0)
            {
                for (int i = 0; i < body.getActions().getValue().size(); ++i)
                {
                    if (body.getActions().getValue().get(i) == ActionType.RETRIEVE_DOCUMENT || body.getActions().getValue().get(i) == ActionType.RETRIEVE_DOCUMENT_OUT)
                    {
                        actions.add(body.getActions().getValue().get(i).value());
                    }
                }
                notActions = body.getActions().isNotIn();
            }
            else
            {
                actions.add(ActionType.RETRIEVE_DOCUMENT.value());
                actions.add(ActionType.RETRIEVE_DOCUMENT_OUT.value());
            }

            Map<String, Object> params = new HashMap<String, Object>();
            queryString.append("SELECT a.organizationId, a.organizationName, a.remoteOrganizationId, a.remoteOrganizationName, a.classCode, a.title, count(a.rawDataSize), avg(a.rawDataSize), min(a.rawDataSize), max(a.rawDataSize), a.actionName FROM AuditsReport a ");

            StringBuffer whereClause = new StringBuffer();
            whereClause.append("WHERE ");
            // Case where the organization is specified and wants all inbound and outbound documents
            if (NullChecker.isNotNullOrEmpty(organizations) && actions.size() > 1)
            {
                whereClause.append("(");
                whereClause.append(buildDocumentWhereClause(fromDate, toDate, organizations, notOrganizations, Arrays.asList(actions.get(0)), notActions, params, 0));
                whereClause.append(") OR (");
                whereClause.append(buildDocumentWhereClause(fromDate, toDate, organizations, notOrganizations, Arrays.asList(actions.get(1)), notActions, params, 1));
                whereClause.append(")");
            }
            else
            {
                 whereClause.append(buildDocumentWhereClause(fromDate, toDate, organizations, notOrganizations, actions, notActions, params, 0));
            }

            queryString.append(whereClause.toString());
            queryString.append(" GROUP BY a.organizationId, a.organizationName, a.remoteOrganizationId, a.remoteOrganizationName, a.classCode, a.title, a.actionName ");
            queryString.append(" ORDER BY a.organizationName, a.actionName ");

            logger.trace("Query: {0}", queryString.toString());

            Query query = entityManager.createQuery(queryString.toString());
            
            for (Map.Entry<String, Object> entry : params.entrySet())
            {
                query.setParameter(entry.getKey(), entry.getValue());
            }

            List<Object> results = query.getResultList();
            if (results != null)
            {
                DocumentsAuditType audits = new DocumentsAuditType();
                ret.setAudits(audits);
                for (Object result : results)
                {
                    Object[] r = (Object[]) result;
                    DocumentAuditType docAudit = new DocumentAuditType();
                    docAudit.setOrganizationId(r[0] != null ? (String) r[0] : "");
                    docAudit.setOrganizationName(r[1] != null ? (String) r[1] : "");
                    docAudit.setClassCode(r[4] != null ? (String) r[4] : "");
                    docAudit.setTitle(r[5] != null ? (String) r[5] : "");
                    docAudit.setNumberOfDocs(r[6] != null ? (Long) r[6] : 0);
                    docAudit.setAverageSize(r[7] != null ? (Double) r[7] : 0.0);
                    docAudit.setMinimumSize(r[8] != null ? String.valueOf(r[8]) : "0.00");
                    docAudit.setMaximumSize(r[9] != null ? String.valueOf(r[9]) : "0.00");
                    docAudit.setAction(r[10] != null ? ActionType.fromValue((String) r[10]) : null);
                    
                    if (docAudit.getAction() != null && ActionType.RETRIEVE_DOCUMENT_OUT.value().equals(docAudit.getAction().value()))
                    {
                        docAudit.setOrganizationId(r[2] != null ? (String) r[2] : "");
                        docAudit.setOrganizationName(r[3] != null ? (String) r[3] : "");
                    }

                    audits.getAudit().add(docAudit);
                }
            }
        }
        catch (Throwable t)
        {
            throw new UnsupportedOperationException("There was an error process this request.", t);
        }
        
        return ret;
    }

    private String buildDocumentWhereClause(Date fromDate, Date toDate, String[] organizations, boolean notOrganizations, List<String> actions, boolean notActions, Map<String, Object> params, int id)
    {
        StringBuffer sb = new StringBuffer();

        if (fromDate != null)
        {
            sb.append("a.auditTime >= :fromDate").append(id).append(" AND ");
            params.put("fromDate" + id, fromDate);
        }

        if (toDate != null)
        {
            sb.append("a.auditTime <= :toDate").append(id).append(" AND ");
            params.put("toDate" + id, toDate);
        }
        
        if (!NullChecker.isNullOrEmpty(organizations) && !NullChecker.isNullOrEmpty(organizations[0]))
        {
            // outbound document use organization_id field
            if (!NullChecker.isNullOrEmpty(actions) && ActionType.RETRIEVE_DOCUMENT.value().equals(actions.get(0)))
            {
                if (!notOrganizations)
                {
                    sb.append("a.organizationId like :organization").append(id).append(" AND ");
                }
                else
                {
                    sb.append("not a.organizationId like :organization").append(id).append(" AND ");
                }
                params.put("organization" + id, organizations[0]);
            }
            // inbound documents use remote_organization_id field
            else if (!NullChecker.isNullOrEmpty(actions) && ActionType.RETRIEVE_DOCUMENT_OUT.value().equals(actions.get(0)))
            {
                if (!notOrganizations)
                {
                    sb.append("a.remoteOrganizationId like :remoteOrganization").append(id).append(" AND ");
                }
                else
                {
                    sb.append("not a.remoteOrganizationId like :remoteOrganization").append(id).append(" AND ");
                }
                params.put("remoteOrganization" + id, organizations[0]);
            }
        }
        
        if (!NullChecker.isNullOrEmpty(actions))
        {
            if (!notActions)
            {
                sb.append("a.actionName in (:action").append(id);
            }
            else
            {
                sb.append("a.actionName not in (:action").append(id);
            }
            params.put("action" + id, actions.get(0));

            for (int i = 1; i < actions.size(); ++i)
            {
                sb.append(",:action").append(i);
                params.put("action" + i, actions.get(i));
            }

            sb.append(") AND ");
        }
        
        //filter to only grab real patients
        sb.append("a.isTestPatient = 0 ");
        
        return sb.toString();
    }
}
