package gov.va.ccd.service.util;

import gov.va.ccd.components.transform.Transformer;
import gov.va.ccd.service.value.object.UnitValue;
import gov.va.ccd.service.value.objects.impl.ItemObject;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Stack;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.URIConverter;
import org.eclipse.emf.ecore.util.FeatureMap;
import org.eclipse.emf.ecore.util.FeatureMap.Entry;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.emf.ecore.xmi.impl.GenericXMLResourceFactoryImpl;
import org.eclipse.emf.ecore.xml.type.AnyType;
import org.eclipse.emf.ecore.xml.type.XMLTypeDocumentRoot;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.openhealthtools.mdht.uml.cda.Act;
import org.openhealthtools.mdht.uml.cda.AssignedAuthor;
import org.openhealthtools.mdht.uml.cda.AssignedEntity;
import org.openhealthtools.mdht.uml.cda.Author;
import org.openhealthtools.mdht.uml.cda.CDAFactory;
import org.openhealthtools.mdht.uml.cda.Entity;
import org.openhealthtools.mdht.uml.cda.EntryRelationship;
import org.openhealthtools.mdht.uml.cda.Informant12;
import org.openhealthtools.mdht.uml.cda.Observation;
import org.openhealthtools.mdht.uml.cda.Organization;
import org.openhealthtools.mdht.uml.cda.Organizer;
import org.openhealthtools.mdht.uml.cda.Participant2;
import org.openhealthtools.mdht.uml.cda.ParticipantRole;
import org.openhealthtools.mdht.uml.cda.Performer2;
import org.openhealthtools.mdht.uml.cda.PlayingEntity;
import org.openhealthtools.mdht.uml.cda.Procedure;
import org.openhealthtools.mdht.uml.cda.StrucDocText;
import org.openhealthtools.mdht.uml.cda.SubstanceAdministration;
import org.openhealthtools.mdht.uml.cda.ccd.ContinuityOfCareDocument;
import org.openhealthtools.mdht.uml.hl7.datatypes.ANY;
import org.openhealthtools.mdht.uml.hl7.datatypes.CD;
import org.openhealthtools.mdht.uml.hl7.datatypes.ED;
import org.openhealthtools.mdht.uml.hl7.datatypes.II;
import org.openhealthtools.mdht.uml.hl7.datatypes.IVL_TS;
import org.openhealthtools.mdht.uml.hl7.datatypes.PQ;
import org.openhealthtools.mdht.uml.hl7.datatypes.TEL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Utils
{
	private static final Logger logger = LoggerFactory.getLogger(Utils.class);

	/**
	 * prepare narrative text value
	 * 
	 * @param narrativeBlock
	 * @return
	 */
	public static StrucDocText setValueToNarrativeBlock(String narrativeBlock) throws IOException {
		StrucDocText text = null;
		URIConverter.ReadableInputStream is = null;
		try
		{

			Resource.Factory factory = new GenericXMLResourceFactoryImpl();
			XMLResource resource = (XMLResource) factory.createResource(null);
			is = new URIConverter.ReadableInputStream(narrativeBlock);
			resource.load(is, null);
			XMLTypeDocumentRoot root = (XMLTypeDocumentRoot) resource.getContents().get(0);
			AnyType value = (AnyType) root.getMixed().getValue(0);
			text = CDAFactory.eINSTANCE.createStrucDocText();
			text.getMixed().addAll(value.getMixed());

		}
		catch(Exception e)
		{
			e.printStackTrace();
		}
		finally
			{
			if(is != null)
				{
				is.close();
				}
			}

		return text;

	}

	/**
	 * Special parser for Vitals but may be needed elsewhere. If a value has
	 * more than one row, it will return a table instead of the Value Unit.
	 * 
	 * @param values
	 * @return
	 */
	public static String valueOrDefault(final List<UnitValue> values)
	{
		if(CollectionUtils.isNotEmpty(values))
		{
			if(values.size() > 1)
			{
				StringBuilder ret = new StringBuilder();
				// ret.append("<table>");
				for(UnitValue uv : values)
				{
					// ret.append("<tr><td>");
					if(ret.length() > 1)
					{
						ret.append("//");
					}
					ret.append(uv.getValue());

					if(StringUtils.isNotBlank(uv.getUnit()))
					{
						ret.append("  ").append(uv.getUnit());
					}

					if(StringUtils.isNotBlank(uv.getTime()))
					{
						DateTime dt = EffTimeUtils.convertDate(uv.getTime());
						String format = "h:mm a";
						DateTimeFormatter dtf = DateTimeFormat.forPattern(format).withLocale(Locale.US);
						String time = dtf.print(dt);
						if(StringUtils.isNotBlank(time))
						{
							ret.append("  (").append(time).append(")");
						}
					}

					// ret.append("</td></tr>");
				}
				// ret.append("</table>");
				if(ret.length() > 1)
				{
					return ret.toString();
				}
			}
			else
			{
				UnitValue uv = values.get(0);
				return uv.getValue() + " " + uv.getUnit();
			}
		}
		return "--";
	}

	public static String valueOrDefault(final String value)
	{
		if(StringUtils.isNotBlank(value))
		{
			return value;
		}

		return "--";
	}

	public static String getNonNullValue(String dn, String rn, String ot)
	{
		if(StringUtils.isNotBlank(dn))
		{
			return dn;
		}
		else if(StringUtils.isNotBlank(rn))
		{
			return rn;
		}
		else if(StringUtils.isNotBlank(ot))
		{
			return ot;
		}
		return "--";
	}

	/**
	 * Return non Null Value.
	 * 
	 * @param dn
	 * @param rn
	 * @return
	 */
	public static String getNonNullValue(String dn, String rn)
	{
		if(StringUtils.isNotBlank(dn) && !"null".equalsIgnoreCase(dn))
		{
			return dn;
		}
		else
		{
			if(StringUtils.isNotBlank(rn))
			{
				return rn;
			}
			else
			{
				return "--";
			}
		}
	}

	public static boolean containsTemplateId(List<II> templates, final String id)
	{
		if(CollectionUtils.isNotEmpty(templates) && StringUtils.isNotBlank(id))
		{
			for(II ii : templates)
			{
				if(id.equalsIgnoreCase(ii.getRoot()))
				{
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * return the value of the first available effective time. First value,
	 * Second low, Third High
	 * 
	 * @param time
	 * @return
	 */
	public static String getEffectiveTime(final IVL_TS time)
	{
		if(time != null)
		{
			if(StringUtils.isNotBlank(time.getValue()))
			{
				return time.getValue();
			}

			if(time.getLow() != null)
			{
				if(StringUtils.isNotBlank(time.getLow().getValue()))
				{
					return time.getLow().getValue();
				}
			}

			if(time.getHigh() != null)
			{
				if(StringUtils.isNotBlank(time.getHigh().getValue()))
				{
					return time.getHigh().getValue();
				}
			}
		}
		return StringUtils.EMPTY;
	}

	/**
	 * For BR rule purposes return code in the following manner: 1) displayName
	 * 2) Translation displayName 3) Original Text 4) Code
	 * 
	 * @param code
	 * @return
	 */
	public static ItemObject getFromCode(final CD code, final StrucDocText narrative)
	{
		if(code == null)
		{
			return new ItemObject();
		}

		// First get displayname
		if(StringUtils.isNotBlank(code.getDisplayName()))
		{
			return new ItemObject(code.getDisplayName());
		}

		ItemObject ret = null;
		// Second, get translation displayName.

		ret = getFromTranslations(code.getTranslations(), narrative);
		if(ret != null)
		{
			return ret;
		}
		// Third, see if there is Original Text to process.
		ret = getFromOriginalText(code.getOriginalText(), narrative);

		if(ret != null)
		{
			return ret;
		}

		if(StringUtils.isNotBlank(code.getCode()))
		{
			return new ItemObject(code.getCode());
		}
		return new ItemObject();
	}

	public static ItemObject getFromTranslations(final List<CD> trans, final StrucDocText narrative)
	{
		if(CollectionUtils.isEmpty(trans))
		{
			return null;
		}

		StringBuilder sb = new StringBuilder();

		for(CD tran : trans)
		{
			if(StringUtils.isNotBlank(tran.getDisplayName()))
			{
				sb.append(tran.getDisplayName()).append(",");
			}
		}

		if(sb.length() > 0)
		{
			return new ItemObject(sb.substring(0, sb.length() - 1));
		}

		return null;
	}

	public static ItemObject getFromOriginalText(final ED oText, final StrucDocText narrative)
	{
		if(oText == null)
			return null;

		String text = oText.getText();

		if(StringUtils.isNotBlank(text))
			return new ItemObject(text);

		TEL ref = oText.getReference();
		if(ref != null)
		{
			String val = getReferenceValue(narrative, ref.getValue());
			if(val != null)
			{
				return new ItemObject(val, ref.getValue());
			}
		}
		return null;
	}

	/**
	 * Looks for a component where ID="{id}" returns a map containing the
	 * element name a list of Attributes stored in ItemObjects and the value.
	 * 
	 * @param narrative
	 * @param id
	 * @return
	 */
	public static Map<String, Object> getTagMapFromId(final StrucDocText narrative, final String id)
	{
		if(StringUtils.isNotBlank(id))
		{
			String tid = id.startsWith("#") ? id.substring(1) : id;
			if(narrative != null)
			{
				FeatureMap mixed = narrative.getMixed();
				Map<String, Object> map = traverseForTag(mixed, tid);
				if(!map.isEmpty())
				{
					return map;
				}
			}
		}
		return Collections.emptyMap();
	}

	// To extract the reference value from the narrative block
	public static String getReferenceValue(StrucDocText narrativeText, String id)
	{
		if(StringUtils.isNotBlank(id) && narrativeText != null)
		{
			String tid = id.startsWith("#") ? id.substring(1) : id;
			return getAllRefernceValue(narrativeText.getMixed(), tid);
		}
		return StringUtils.EMPTY;
	}

	public static void walkFeatureMap(FeatureMap map, String id)
	{
		if(map != null && StringUtils.isNotBlank(id))
		{
			for(Entry entry : map)
			{
				logger.debug(entry.getValue().toString());
				if(entry.getEStructuralFeature() != null)
				{
					logger.debug(entry.getEStructuralFeature().getName() + ": " + entry.getValue());
					if(entry instanceof AnyType)
					{
						AnyType any = (AnyType) entry;
						walkFeatureMap(any.getAny(), id);
						walkFeatureMap(any.getMixed(), id);
					}
					else
					{
						logger.debug("Not AnyType: " + entry.getClass().toString());
					}
				}
			}
		}
	}

	public static String listToString(List<String> list)
	{
		StringBuffer sb = new StringBuffer();
		try
		{

			if(CollectionUtils.isNotEmpty(list))
			{
				int i = 1;

				for(String s : list)
				{
					if(StringUtils.isNotBlank(s))
					{
						if(i < list.size())
							sb.append(s + ", ");
						else
							sb.append(s);
					}
					i = i + 1;
				}

				return sb.toString();
			}
		}
		catch(Exception e)
		{
			return "";
		}
		return sb.toString();
	}

	public static String getSource(ContinuityOfCareDocument ccdDocument)
	{
		String source = null;
		if(ccdDocument != null)
		{
			if(ccdDocument.getAuthors() != null && ccdDocument.getAuthors().size() > 0)
			{

				if(ccdDocument.getAuthors().get(0).getAssignedAuthor() != null)
				{
					if(ccdDocument.getAuthors().get(0).getAssignedAuthor().getRepresentedOrganization() != null)
					{
						source = ccdDocument.getAuthors().get(0).getAssignedAuthor().getRepresentedOrganization().getNames().get(0).getText();
						return source;
					}
				}
			}

		}
		return null;

	}

	public static String removeLastChar(String str)
	{
		return str.substring(0, str.length() - 2);
	}

	public static String getCommaSeperateString(List<String> list)
	{
		StringBuffer sb = new StringBuffer();
		if(CollectionUtils.isNotEmpty(list))
		{
			for(String value : list)
			{
				if(StringUtils.isNotBlank(value))
				{
					if(sb.length() > 0)
					{
						sb.append(", ");
					}
					sb.append(value);
				}
			}
			return sb.toString();
		}
		return "--";
	}

	public static String getNotNullValue(String value1, String value2)
	{

		if(StringUtils.isNotBlank(value1))
		{
			return value1.trim();
		}
		else if(StringUtils.isNotBlank(value2))
		{
			return value2.trim();
		}
		else
		{
			return "--";
		}
	}

	public static String getNotNullValue(String value1, String value2, String value3)
	{

		if(StringUtils.isNotBlank(value1))
		{
			return value1;
		}
		else if(StringUtils.isNotBlank(value2))
		{

			return value2;
		}
		else if(StringUtils.isNotBlank(value3))
		{

			return value3;
		}
		else
		{
			return "--";
		}
	}

	public static String getRefList(Map<String, String> mp)
	{
		StringBuffer sb = new StringBuffer();
		sb.append("<list>");

		for(Map.Entry<String, String> pairs : mp.entrySet())
		{
			sb.append("<item>");
			sb.append("<content ID=\"" + pairs.getKey() + "\">" + pairs.getValue() + "</content>");
			sb.append("</item>");
		}

		sb.append("</list>");

		return sb.toString();

	}

	public static String getCommaSeperateString(Map<String, String> map)
	{
		StringBuffer sb = new StringBuffer();
		for(Map.Entry<String, String> pairs : map.entrySet())
		{
			sb.append(pairs.getValue() + ", ");
		}
		if(sb.length() > 0)
		{
			return removeLastChar(sb.toString());
		}
		return "--";
	}

	private static Map<String, Object> traverseForTag(final FeatureMap root, final String id)
	{
		Stack<FeatureMap> stack = new Stack<FeatureMap>();
		Map<String, Object> ret = new HashMap<String, Object>();
		stack.push(root);
		while(!stack.isEmpty())
		{
			FeatureMap featureMap = stack.pop();
			for(Entry entry : featureMap)
			{
				if(entry.getEStructuralFeature() instanceof EReference)
				{
					AnyType anyType = (AnyType) entry.getValue();
					if(traverseForID(anyType.getAnyAttribute(), id))
					{
						ret.put("name", entry.getEStructuralFeature().getName());
						List<ItemObject> attrs = traverseMyAttributes(anyType.getAnyAttribute());
						if(CollectionUtils.isNotEmpty(attrs))
						{
							ret.put("attributes", attrs);
						}
						StringBuilder sb = new StringBuilder();
						for(Entry foundEntry : anyType.getMixed())
						{
							sb.append(rebuildEntry(foundEntry)); // getValuesFromMixed(anyType.getMixed());
						}
						if(StringUtils.isNotBlank(sb.toString()))
						{
							ret.put("value", sb.toString());
						}
						return ret;
					}
					else
					{
						stack.push(anyType.getMixed());
					}
				}
			}
		}
		return Collections.emptyMap();
	}

	private static boolean traverseForID(final FeatureMap attributes, final String id)
	{
		if(StringUtils.isEmpty(id))
		{
			return false;
		}

		for(Entry entry : attributes)
		{
			if("ID".equals(entry.getEStructuralFeature().getName()) && id.equals(entry.getValue().toString()))
			{
				return true;
			}
		}

		return false;
	}

	private static List<ItemObject> traverseMyAttributes(FeatureMap anyAttribute)
	{
		List<ItemObject> ret = new ArrayList<ItemObject>();
		for(Entry entry : anyAttribute)
		{
			ret.add(new ItemObject(entry.getValue().toString(), entry.getEStructuralFeature().getName()));
		}
		return ret;
	}

	/**
	 * Quick check to see if the value should be returned or if the default
	 * should be returned.
	 * 
	 * @param value
	 * @param defaultValue
	 * @return
	 */
	public static String returnDefaultOrBlank(final String value, final String defaultValue)
	{
		if(StringUtils.isNotBlank(value))
		{
			return value;
		}

		return returnDefaultOrBlank(defaultValue);
	}

	/**
	 * if default value exists return it, else return empty string.
	 * 
	 * @param defaultValue
	 * @return
	 */
	public static String returnDefaultOrBlank(final String defaultValue)
	{
		if(StringUtils.isNotBlank(defaultValue))
		{
			return defaultValue;
		}
		return StringUtils.EMPTY;
	}

	/**
	 * For legacy code to check if there is a reference. If there is a reference
	 * return it in a content tag. If not return the default value.
	 * 
	 * @param reference
	 * @param value
	 * @param defaultValue
	 * @return
	 */
	public static String createReferenceWDefault(final String reference, final List<String> value, final String defaultValue)
	{

		if(StringUtils.isBlank(reference))
		{
			return returnDefaultOrBlank(listToString(value), defaultValue);
		}

		return "<content ID=\"" + reference + "\">" + returnDefaultOrBlank(listToString(value), defaultValue) + "</content>";
	}

	/**
	 * For legacy code to check if there is a reference. If there is a reference
	 * return it in a content tag. If not return the default value.
	 * 
	 * @param reference
	 * @param value
	 * @param defaultValue
	 * @return
	 */
	public static String createReferenceWDefault(final String reference, final String value, final String defaultValue)
	{

		if(StringUtils.isBlank(reference))
		{
			return returnDefaultOrBlank(value, defaultValue);
		}

		return "<content ID=\"" + reference + "\">" + returnDefaultOrBlank(value, defaultValue) + "</content>";
	}

	public static List<String> getAllReferncesFromNarr(final FeatureMap map)
	{
		List<String> ret = new ArrayList<String>();

		for(Entry entry : map)
		{
			if(entry.getEStructuralFeature() instanceof EReference)
			{
				AnyType anyType = (AnyType) entry.getValue();
				if(hasId(anyType.getAnyAttribute()) && !"table".equalsIgnoreCase(entry.getEStructuralFeature().getName()))
				{
					// Found a reference, now build it.
					String component = rebuildEntry(entry);
					if(StringUtils.isNotBlank(component))
					{
						ret.add(component);
					}
				}
				else
				{
					ret.addAll(getAllReferncesFromNarr(anyType.getMixed()));
				}
			}
		}

		return ret;
	}

	public static String getAllRefernceValue(final FeatureMap map, final String id)
	{
		StringBuilder ret = new StringBuilder();

		for(Entry entry : map)
		{
			if(entry.getEStructuralFeature() instanceof EReference)
			{
				AnyType anyType = (AnyType) entry.getValue();
				if(hasId(anyType.getAnyAttribute(), id) && !"table".equalsIgnoreCase(entry.getEStructuralFeature().getName()))
				{
					// Found a reference, now build it.
					String component = rebuildItem(anyType.getMixed());
					if(StringUtils.isNotBlank(component))
					{
						ret.append(component);
					}
				}
				else
				{
					ret.append(getAllRefernceValue(anyType.getMixed(), id));
				}
			}
		}

		return ret.toString();
	}

	public static boolean hasId(final FeatureMap attributes, final String id)
	{
		if(StringUtils.isEmpty(id))
		{
			return false;
		}

		for(Entry entry : attributes)
		{
			if("ID".equals(entry.getEStructuralFeature().getName()) && id.equalsIgnoreCase(entry.getValue().toString()))
			{
				return true;
			}
		}

		return false;
	}

	public static boolean hasId(final FeatureMap attributes)
	{
		for(Entry entry : attributes)
		{
			if("ID".equals(entry.getEStructuralFeature().getName()))
			{
				return true;
			}
		}

		return false;
	}

	private static String rebuildItem(final FeatureMap map)
	{
		final StringBuilder sb = new StringBuilder();

		for(Entry entry : map)
		{
			sb.append(rebuildEntry(entry));
		}
		return sb.toString();
	}

	private static String rebuildEntry(final Entry entry)
	{
		StringBuilder sb = new StringBuilder();

		if(entry.getEStructuralFeature() instanceof EReference)
		{
			String tag = entry.getEStructuralFeature().getName();
			sb.append("<").append(tag).append(" ");
			AnyType any = (AnyType) entry.getValue();
			sb.append(recreateAttributes(any.getAnyAttribute()));
			sb.append(">");
			if(any.getMixed() != null)
			{
				sb.append(rebuildItem(any.getMixed()));
			}
			sb.append("</").append(tag).append(">");
		}
		else
		{
			if(entry.getValue() != null)
			{
				String value = StringUtils.trimToEmpty(entry.getValue().toString());
				if(StringUtils.isNotBlank(value))
				{
					sb.append(StringEscapeUtils.escapeXml(value));
				}
			}
			else
			{
				sb.append(">");
			}
		}

		return sb.toString();
	}

	private static String recreateAttributes(final FeatureMap anyAttribute)
	{
		final StringBuilder sb = new StringBuilder();
		for(Entry entry : anyAttribute)
		{
			sb.append(entry.getEStructuralFeature().getName()).append("=");
			sb.append("\"").append(entry.getValue().toString()).append("\"").append(" ");
		}

		return sb.toString();
	}

	/**
	 * Will convert the unit abbreviation to an actual display name.
	 * 
	 * This only process the 2.16.840.1.113883.11.20.9.21 (AgePQ_UCUM)
	 * 2012-06-02 value set.
	 * 
	 * @param value
	 * @return
	 */
	public static String convertPQUnit(final PQ value)
	{
		StringBuilder ret = new StringBuilder();

		if(value != null)
		{
			if(StringUtils.isNotBlank(value.getUnit()))
			{
				if("a".equalsIgnoreCase(value.getUnit()))
				{
					ret.append("Year");
				}
				else if("wk".equalsIgnoreCase(value.getUnit()))
				{
					ret.append("Week");
				}
				else if("mo".equalsIgnoreCase(value.getUnit()))
				{
					ret.append("Month");
				}
				else if("min".equalsIgnoreCase(value.getUnit()))
				{
					ret.append("Minute");
				}
				else if("h".equalsIgnoreCase(value.getUnit()))
				{
					ret.append("Hour");
				}
				else if("d".equalsIgnoreCase(value.getUnit()))
				{
					ret.append("Day");
				}
			}
                        
			if(value.getValue() != null && value.getValue().compareTo(BigDecimal.ONE) > 0)
			{
                            ret.append("s");
			}
		}

		return ret.toString();
	}

	public static PQ getAge(final EntryRelationship rel)
	{
		if(rel != null)
		{
			Observation obs = rel.getObservation();
			if(obs != null && containsTemplateId(obs.getTemplateIds(), "2.16.840.1.113883.10.20.1.38"))
			{
				// Return the first value.
				for(ANY any : obs.getValues())
				{
					if(any instanceof PQ)
					{
						return (PQ) any;
					}
				}
			}
		}
		return null;
	}

	public static String getSourceFromAct(final Act act)
	{
		if(act != null)
		{
			return getSource(act.getAuthors(), act.getPerformers(), act.getParticipants(), act.getInformants());
		}
		return StringUtils.EMPTY;
	}

	public static String getSourceFromObservation(final Observation obs)
	{
		if(obs != null)
		{
			return getSource(obs.getAuthors(), obs.getPerformers(), obs.getParticipants(), obs.getInformants());
		}
		return StringUtils.EMPTY;
	}

	public static String getSourceFromOrganizer(final Organizer org)
	{
		if(org != null)
		{
			return getSource(org.getAuthors(), org.getPerformers(), org.getParticipants(), org.getInformants());
		}
		return StringUtils.EMPTY;
	}

	public static String getSourceFromSubstanceAdmin(final SubstanceAdministration substance)
	{
		if(substance != null)
		{
			return getSource(substance.getAuthors(), substance.getPerformers(), substance.getParticipants(), substance.getInformants());
		}
		return StringUtils.EMPTY;
	}

	public static String getSourceFromProcedures(final Procedure proc)
	{
		if(proc != null)
		{
			return getSource(proc.getAuthors(), proc.getPerformers(), proc.getParticipants(), proc.getInformants());
		}
		return StringUtils.EMPTY;
	}

	public static String getSource(final EList<Author> authors)
	{
		return getSource(authors, null);
	}

	public static String getSource(final EList<Author> authors, final EList<Performer2> performers)
	{
		return getSource(authors, performers, null);
	}

	public static String getSource(final EList<Author> authors, final EList<Performer2> performers, final EList<Participant2> participants)
	{
		return getSource(authors, performers, participants, null);
	}

	public static String getSource(final EList<Author> authors, final EList<Performer2> performers, final EList<Participant2> participants, final EList<Informant12> informants)
	{
		String source = Utils.getSourceFromAuthors(authors);
		if(StringUtils.isBlank(source))
		{
			source = Utils.getSourceFromPerformer(performers);
			if(StringUtils.isBlank(source))
			{
				source = Utils.getSourceFromScoping(participants);
				if(StringUtils.isBlank(source))
				{
					source = Utils.getSourceFromPlayingEntity(participants);
					if(StringUtils.isBlank(source))
					{
						source = Utils.getSourceFromInformants(informants);
					}
				}
			}
		}
		return source;
	}

	public static String getSourceFromAuthors(final EList<Author> authors)
	{
		if(CollectionUtils.isNotEmpty(authors))
		{
			StringBuilder sb = new StringBuilder();
			for(Author author : authors)
			{
				AssignedAuthor aAuthor = author.getAssignedAuthor();
				if(aAuthor != null)
				{
					Organization org = aAuthor.getRepresentedOrganization();
					if(org != null)
					{
						sb.append(Transformer.createNamesON(org.getNames()));
					}
				}
			}
			if(sb.length() > 0)
			{
				return sb.toString();
			}
		}
		return StringUtils.EMPTY;
	}

	public static String getSourceFromInformants(final EList<Informant12> informants)
	{
		if(CollectionUtils.isNotEmpty(informants))
		{
			StringBuilder sb = new StringBuilder();
			for(Informant12 informant : informants)
			{
				AssignedEntity entity = informant.getAssignedEntity();
				if(entity != null)
				{

					EList<Organization> orgs = entity.getRepresentedOrganizations();
					if(CollectionUtils.isNotEmpty(orgs))
					{
						for(Organization org : orgs)
						{
							if(sb.length() > 0)
							{
								sb.append(", ");
							}
							sb.append(Transformer.createNamesON(org.getNames()));
						}
					}
				}
			}
			if(sb.length() > 0)
			{
				return sb.toString();
			}
		}
		return StringUtils.EMPTY;
	}

	public static String getSourceFromPerformer(final EList<Performer2> performers)
	{
		if(CollectionUtils.isNotEmpty(performers))
		{
			StringBuilder sb = new StringBuilder();
			for(Performer2 performer : performers)
			{
				AssignedEntity assignedEntity = performer.getAssignedEntity();
				if(assignedEntity != null)
				{
					if(CollectionUtils.isNotEmpty(assignedEntity.getRepresentedOrganizations()))
					{
						for(Organization org : assignedEntity.getRepresentedOrganizations())
						{
							if(org != null)
							{
								if(sb.length() > 0)
								{
									sb.append(", ");
								}
								sb.append(Transformer.createNamesON(org.getNames()));
							}
						}
					}
				}
			}
			if(sb.length() > 0)
			{
				return sb.toString();
			}
		}
		return StringUtils.EMPTY;
	}

	public static String getSourceFromScoping(final EList<Participant2> participants)
	{
		if(CollectionUtils.isNotEmpty(participants))
		{
			StringBuilder sb = new StringBuilder();
			for(Participant2 part : participants)
			{
				ParticipantRole role = part.getParticipantRole();
				if(role != null)
				{
					Entity scope = role.getScopingEntity();
					if(scope != null && scope.getDesc() != null && StringUtils.isNotBlank(scope.getDesc().getText()))
					{
						if(sb.length() > 1)
						{
							sb.append(", ");
						}
						sb.append(scope.getDesc().getText());
					}
				}
			}
			if(sb.length() > 0)
			{
				return sb.toString();
			}
		}
		return StringUtils.EMPTY;
	}

	public static String getSourceFromPlayingEntity(final EList<Participant2> participants)
	{
		if(CollectionUtils.isNotEmpty(participants))
		{
			StringBuilder sb = new StringBuilder();
			for(Participant2 part : participants)
			{
				ParticipantRole role = part.getParticipantRole();
				if(role != null)
				{
					PlayingEntity entity = role.getPlayingEntity();
					if(entity != null && entity.getDesc() != null && StringUtils.isNotBlank(entity.getDesc().getText()))
					{
						if(sb.length() > 1)
						{
							sb.append(", ");
						}
						sb.append(entity.getDesc().getText());
					}
				}
			}
			if(sb.length() > 0)
			{
				return sb.toString();
			}
		}
		return StringUtils.EMPTY;
	}
}
