package gov.va.ccd.consol;

import gov.va.ccd.comments.extractor.CommentsExtractor;
import gov.va.ccd.comments.extractor.CommentsExtractorFactory;
import gov.va.ccd.components.api.CcdComponent;
import gov.va.ccd.service.util.Utils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.openhealthtools.mdht.uml.cda.Section;
import org.openhealthtools.mdht.uml.cda.StrucDocText;
import org.openhealthtools.mdht.uml.cda.consol.ContinuityOfCareDocument;
import org.openhealthtools.mdht.uml.hl7.datatypes.impl.STImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Abstract Ccd Component. Taken from AbstractComponent and updated for
 * Consolidated.
 * 
 * @author CHMARAC
 * @since 1.0
 * 
 */

public abstract class AbstractConsolComponent<S extends Section> implements CcdComponent<S>
{
	private static final Logger logger = LoggerFactory.getLogger(AbstractConsolComponent.class);

	protected StrucDocText narrativeText = null;
	protected S section = null;
	protected Map<String, Object> values = new HashMap<String, Object>();

	/** MDHT CCD Document **/
	protected ContinuityOfCareDocument ccdDocument;

	/**
	 * Comments extractor for building the comments section
	 */
	private final CommentsExtractor commentsExtractor;

	/**
	 * Comments narrative Text This is not in the correct spot but until we get
	 * each section done the same way this is what we have for now.
	 */
	private List<String> comments = new ArrayList<String>();

	public AbstractConsolComponent(final S section, final ContinuityOfCareDocument ccd)
	{
		this.section = section;
		this.setCcdDocument(ccd);
		this.commentsExtractor = CommentsExtractorFactory.getInstance(ccd.getId());
		comments = this.getCommentsExtractor().extractSectionComments(section);
		if(CollectionUtils.isNotEmpty(comments))
		{
			values.put("comments", comments);
		}
		if(section != null && section.getText() != null)
		{
			narrativeText = section.getText();
		}
	}

	/**
	 * Abstract method that must be overriden in order to create the narrative
	 * block.
	 * 
	 * @return
	 */
	protected abstract String generateXml();

	/**
	 * Abstract method the must be overriden in order to create the objects that
	 * will be used by {@link #generateXml()}
	 */
	protected abstract void createObjects();

	/**
	 * Creates the objects from the implementing section. Also, creates the
	 * narrative block from the string returned by {@link #generateXml()}
	 */
	public void execute()
	{
		// Set the narrative text for the section.
		if(section != null)
		{
			createObjects();
			appendNarrative(generateXml());
		}
	}

	/**
	 * Returns the C-CDA being used.
	 */
	public ContinuityOfCareDocument getCcdDocument()
	{
		return ccdDocument;
	}

	/**
	 * Sets the C-CDA to use.
	 * 
	 * @param ccdDocument
	 */
	public void setCcdDocument(ContinuityOfCareDocument ccdDocument)
	{
		this.ccdDocument = ccdDocument;
	}

	/**
	 * Determines whether or not to add a narrative to the implementing section.
	 * Will also parse all values with references in the original file and
	 * append them to the new narrative block.
	 * 
	 * @param generated
	 */
	protected void appendNarrative(final String generated)
	{
		if(StringUtils.isNotBlank(generated))
		{
			StringBuilder narrativeBlockValue = new StringBuilder();
			narrativeBlockValue.append(generated);
			if(narrativeText != null)
			{
				for(String comp : Utils.getAllReferncesFromNarr(narrativeText.getMixed()))
				{
					narrativeBlockValue.append(comp).append("\n");
				}
			}
			if(narrativeBlockValue.length() > 0)
			{
				logger.debug("narrativeBlockValue: " + narrativeBlockValue);
				section.createStrucDocText(narrativeBlockValue.toString());
			}
		}
	}

	/**
	 * Returns the title of the implementing section.
	 * 
	 * @return
	 */
	public String getTitle()
	{
		if(section == null)
			return StringUtils.EMPTY;

		STImpl title = (STImpl) section.getTitle();
		return title.getText();
	}

	/**
	 * Convenience method to get the value located at id in the narrative block.
	 * 
	 * @param id
	 * @return
	 */
	public String getReferenceValue(String id)
	{
		if(section == null)
			return StringUtils.EMPTY;

		StrucDocText narrativeText = section.getText();
		return Utils.getReferenceValue(narrativeText, id);
	}

	public CommentsExtractor getCommentsExtractor()
	{
		return commentsExtractor;
	}

	public List<String> getComments()
	{
		return comments;
	}

	public void setComments(List<String> comments)
	{
		this.comments = comments;
	}
}
