package gov.va.ccd.components.impl;

import gov.va.ccd.service.util.Utils;
import gov.va.ccd.service.value.objects.impl.ProcedureVO;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.openhealthtools.mdht.uml.cda.Entry;
import org.openhealthtools.mdht.uml.cda.Performer2;
import org.openhealthtools.mdht.uml.cda.Procedure;
import org.openhealthtools.mdht.uml.cda.ccd.ContinuityOfCareDocument;
import org.openhealthtools.mdht.uml.cda.ccd.ProceduresSection;
import org.openhealthtools.mdht.uml.hl7.datatypes.CD;
import org.openhealthtools.mdht.uml.hl7.datatypes.CR;
import org.openhealthtools.mdht.uml.hl7.datatypes.ENXP;
import org.openhealthtools.mdht.uml.hl7.datatypes.PN;
import org.openhealthtools.mdht.uml.hl7.vocab.ActClass;

public class ProcedureComponent extends AbstractComponent<ProceduresSection>
{

	public ProcedureComponent(final ContinuityOfCareDocument ccdDocument)
	{
		super(ccdDocument.getProceduresSection(), ccdDocument);
	}

	@Override
	public void execute()
	{

		// Set the narrative text for the section.
		if(section != null)
		{
			List<ProcedureVO> rows = this.createProcedureHelperObjects();
			appendNarrative(this.generateXml(rows));
		}

	}

	private String generateXml(List<ProcedureVO> rows)
	{

		StringBuffer sb = new StringBuffer();
		if(CollectionUtils.isNotEmpty(rows))
		{
			if(CollectionUtils.isNotEmpty(this.getComments()))
			{
				sb.append("<table ID='_nbProceduresComments' border='1' width='100%'>");
				sb.append("<thead>");
				sb.append("<tr><th>Comments</th></tr>");
				sb.append("</thead>");
				sb.append("<tbody>");

				for(String comment : this.getComments())
				{
					sb.append("<tr width='100%'>");
					sb.append("<td width='100%'>");
					sb.append("<p>");
					sb.append(comment);
					sb.append("</p>");
					sb.append("</td></tr>");
				}

				sb.append("</tbody>");
				sb.append("</table>");
			}
			sb.append("<table ID='_nbProcedures' border='1' width='100%'>");

			sb.append("<thead>");
			sb.append("<tr>");
			sb.append("<th>Date/Time</th>");
			sb.append("<th>Procedure Type</th>");
			sb.append("<th>Procedure Description</th>");
			sb.append("<th>Procedure Type Details</th>");
			sb.append("<th>Provider</th>");
			sb.append("<th>Body Site</th>");
			sb.append("<th>Source</th>");
			sb.append("</tr>\n");
			sb.append("</thead>");
			sb.append("<tbody>");

			for(ProcedureVO helper : rows)
			{
				sb.append("<tr>");

				sb.append("<td>");
				sb.append(helper.getDateTime().getValueWithRefDefault());
				sb.append("</td>");

				sb.append("<td>");
				sb.append(helper.getProcedureType().getValueWithRefDefault());
				sb.append("</td>");

				sb.append("<td>");
				sb.append(helper.getProcedureDescription().getValueWithRefDefault());
				sb.append("</td>");

				sb.append("<td>");
				sb.append(helper.getProcedureTypeDetails().getValueWithRefDefault());
				sb.append("</td>");

				sb.append("<td>");
				sb.append(helper.getProvider().getValueWithRefDefault());
				sb.append("</td>");

				sb.append("<td>");
				sb.append(helper.getBodySite().getValueWithRefDefault());
				sb.append("</td>");

				sb.append("<td>");
				sb.append(helper.getSource().getValueWithRefDefault());
				sb.append("</td>");

				sb.append("</tr>");
			}

			sb.append("</tbody>");
			sb.append("</table>");
		}
		return sb.toString();
	}

	private List<ProcedureVO> createProcedureHelperObjects()
	{

		List<ProcedureVO> rows = new ArrayList<ProcedureVO>();

		for(Entry entry : this.ccdDocument.getProceduresSection().getEntries())
		{
			if(entry != null)
			{
				if(entry.getProcedure() != null)
				{
					final ProcedureVO helper = new ProcedureVO();

					this.getDateTime(entry, helper);
					this.getProcedureType(entry, helper);
					this.getProcedureTypeDetails(entry, helper);
					this.getProvider(entry, helper);
					this.getBodySite(entry, helper);
					helper.setSource(Utils.getSourceFromProcedures(entry.getProcedure()));
					rows.add(helper);
				}
			}
		}

		return rows;
	}

	private void getDateTime(final Entry entry, final ProcedureVO helper)
	{
		if(entry != null && entry.getProcedure() != null && entry.getProcedure().getEffectiveTime() != null)
		{
			helper.setDateTime(Utils.getEffectiveTime(entry.getProcedure().getEffectiveTime()));
		}
	}

	private void getProcedureType(final Entry entry, final ProcedureVO helper)
	{

		if(entry != null && entry.getProcedure() != null && ActClass.PROC.equals(entry.getProcedure().getClassCode()))
		{

			// checking for ref. first..
			if(entry.getProcedure().getCode() != null && entry.getProcedure().getCode().getOriginalText() != null && entry.getProcedure().getCode().getOriginalText().getReference() != null)
			{

				final String refId = entry.getProcedure().getCode().getOriginalText().getReference().getValue();

				if(refId != null)
				{

					final String refValue = Utils.getReferenceValue(narrativeText, entry.getProcedure().getCode().getOriginalText().getReference().getValue());

					if(refValue != null && !refValue.equals("null"))
					{
						helper.setProcedureDescription(refValue, refId);
					}
				}
			}

			// 1st check get from display name
			if(helper.getProcedureType() != null && helper.getProcedureType().isEmpty())
			{

				if(entry.getProcedure().getClassCode().equals(ActClass.PROC))
				{
					helper.setProcedureType(entry.getProcedure().getCode().getDisplayName());
				}
			}

			// 2nd check - translation display name
			if(helper.getProcedureType() != null && helper.getProcedureType().isEmpty())
			{
				if(entry.getProcedure().getClassCode().equals(ActClass.PROC))
				{
					if(!entry.getProcedure().getCode().getTranslations().isEmpty())
					{
						helper.setProcedureType(entry.getProcedure().getCode().getTranslations().get(0).getDisplayName());
					}
				}
			}

			// 3rd check - code original text
			if(helper.getProcedureType() != null && helper.getProcedureType().isEmpty())
			{

				if(entry != null && entry.getProcedure() != null && ActClass.PROC.equals(entry.getProcedure().getClassCode()) && entry.getProcedure().getCode() != null && entry.getProcedure().getCode().getOriginalText() != null && StringUtils.isNotBlank(entry.getProcedure().getCode().getOriginalText().getText()))
				{
					helper.setProcedureType(entry.getProcedure().getCode().getOriginalText().getText());
				}
			}
		}
	}

	private void getProcedureTypeDetails(final Entry entry, final ProcedureVO helper)
	{
		if(entry != null)
		{
			Procedure proc = entry.getProcedure();
			if(proc != null && ActClass.PROC.equals(proc.getClassCode()))
			{
				CD code = proc.getCode();
				if(code != null)
				{
					StringBuilder sb = new StringBuilder();
					boolean first = true;
					for(CR qual : code.getQualifiers())
					{
						CD qualVal = qual.getValue();
						if(qualVal != null && StringUtils.isNotBlank(qualVal.getDisplayName()))
						{
							if(!first)
							{
								sb.append(", ");
							}
							else
							{
								first = false;
							}
							sb.append(qualVal.getDisplayName());
						}
					}
					if(sb.length() > 0)
					{
						helper.setProcedureTypeDetails(sb.toString());
					}
				}
			}
		}

	}

	private void getProvider(final Entry entry, final ProcedureVO helper)
	{

		if(entry != null && entry.getProcedure() != null && CollectionUtils.isNotEmpty(entry.getProcedure().getPerformers()))
		{
			final Performer2 entity = entry.getProcedure().getPerformers().get(0);
			final PN name = entity.getAssignedEntity().getAssignedPerson().getNames().get(0);
			// Get the family name(Last Name)
			for(ENXP lastName : name.getFamilies())
			{
				helper.setProvider(lastName.getText());
			}

			// Get the given name (First Name)
			for(ENXP firstName : name.getGivens())
			{
				helper.setProvider(helper.getProvider() + ", " + firstName.getText());
			}

			if(helper.getProvider() != null && helper.getProvider().isEmpty())
			{
				helper.setProvider(name.getText());
			}
		}
	}

	private void getBodySite(final Entry entry, final ProcedureVO helper)
	{
		if(entry != null && entry.getProcedure() != null && CollectionUtils.isNotEmpty(entry.getProcedure().getTargetSiteCodes()))
		{
			CD siteCode = entry.getProcedure().getTargetSiteCodes().get(0);
			helper.setBodySite(siteCode.getDisplayName());
		}
	}

	/**
	 * No-op handled above.
	 */
	@Override
	protected String generateXml()
	{
		return null;
	}

	/**
	 * No-op handled above.
	 */
	@Override
	protected void createObjects()
	{

	}
}
