package gov.va.soapui;

import com.eviware.soapui.SoapUI;
import com.eviware.soapui.impl.support.AbstractHttpRequest;
import com.eviware.soapui.impl.support.AbstractHttpRequestInterface;
import com.eviware.soapui.impl.wsdl.WsdlRequest;
import com.eviware.soapui.model.ModelItem;
import com.eviware.soapui.model.iface.Attachment;
import com.eviware.soapui.model.settings.Settings;
import com.eviware.soapui.plugins.auto.PluginResponseEditorView;
import com.eviware.soapui.support.components.WebViewBasedBrowserComponent;
import com.eviware.soapui.support.editor.Editor;
import com.eviware.soapui.support.editor.views.AbstractXmlEditorView;
import com.eviware.soapui.support.editor.xml.XmlEditor;
import java.beans.PropertyChangeEvent;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.Iterator;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.xml.XMLConstants;
import javax.xml.namespace.NamespaceContext;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.IOUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

@PluginResponseEditorView(viewId = "HTMLView")
public class HTMLEditor extends AbstractXmlEditorView {

    private ModelItem modelItem;
    private AbstractHttpRequest<?> request;
    private JPanel panel;
    private HtmlBrowser browser;
    
    public HTMLEditor(Editor<?> editor, ModelItem modelItem) {
        super("C32/CCDA HTML", (XmlEditor) editor, "HTMLView");
        if (modelItem instanceof AbstractHttpRequestInterface<?>) {
            this.request = (AbstractHttpRequest<?>) modelItem;
        }
        this.modelItem = modelItem;
        this.browser = new HtmlBrowser(WebViewBasedBrowserComponent.PopupStrategy.EXTERNAL_BROWSER);
        this.panel = buildPanel();
    }
    
    private JPanel buildPanel() {
        browser.setContent(getContent());
        JPanel cmpPanel = browser.getComponent();
        return cmpPanel;
    }
    
    @Override
    public JComponent getComponent() {   
        return panel;
    }

    @Override
    public void setEditable(boolean enabled) {
    }

    @Override
    public boolean saveDocument(boolean validate) {
        return true;
    }
    
    @Override
    public void documentUpdated(){
        browser.setContent(getContent());
    }
    
    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        documentUpdated();
    }
        
    public String getContent() {
        if(this.request != null && this.request.getResponse() != null && this.request.getResponse().getAttachments() != null) {
            Attachment[] attachments = this.request.getResponse().getAttachments();
            if(attachments.length > 0) {
                for(Attachment attachment : attachments) {
                    try { 
                        InputStream attachStream = attachment.getInputStream();
                        String html = transformCDA(attachStream);
                        if(html != null) {
                            return html;
                        }
                    } catch (IOException ex) {
                        SoapUI.logError(ex);
                    } catch (Exception ex) {
                        SoapUI.logError(ex);
                    }
                }
            }
            else {
                SoapUI.log("No attachments in response, looking for Document element.");
                String soapResponseStr = this.request.getResponse().getContentAsXml();
                try {
                    DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
                    factory.setNamespaceAware(true);
                    Document doc = null;
                    doc = factory.newDocumentBuilder().parse(new ByteArrayInputStream(soapResponseStr.getBytes()));
                    XPath xpath = XPathFactory.newInstance().newXPath();
                    xpath.setNamespaceContext(new NamespaceContext() {
                        public String getNamespaceURI(String prefix) {
                            if (prefix == null) throw new NullPointerException("Null Xpath prefix.");
                            else if ("ns2".equals(prefix)) return "urn:ihe:iti:xds-b:2007";
                            else if ("S".equals(prefix)) return "http://schemas.xmlsoap.org/soap/envelope/";
                            else if ("ns6".equals(prefix)) return "urn:oasis:names:tc:ebxml-regrep:xsd:rs:3.0";
                            return XMLConstants.NULL_NS_URI;
                        }

                        // This method isn't necessary for XPath processing.
                        public String getPrefix(String uri) {
                            throw new UnsupportedOperationException();
                        }

                        // This method isn't necessary for XPath processing either.
                        public Iterator getPrefixes(String uri) {
                            throw new UnsupportedOperationException();
                        }
                    });
                    try {
                        XPathExpression expr = xpath.compile("/S:Envelope/S:Body/ns2:RetrieveDocumentSetResponse/ns2:DocumentResponse/ns2:Document");
                        Object result = expr.evaluate(doc, XPathConstants.NODESET);
                        NodeList nodes = (NodeList) result;
                        SoapUI.log("Found " + nodes.getLength() + " matches");
                        for (int index = 0; index < nodes.getLength(); index++) {
                            Node node = nodes.item(index);
                            String base64encoded = node.getTextContent();
                            byte[] decodedBytes = Base64.decodeBase64(base64encoded.getBytes());
                            InputStream attachStream = new ByteArrayInputStream(decodedBytes);
                            String html = transformCDA(attachStream);
                            if(html != null) {
                                return html;
                            }
                        }
                    } catch (XPathExpressionException ex) {
                        SoapUI.logError(ex);
                    } catch (TransformerException ex) {
                        SoapUI.logError(ex);
                    } 
                } catch (IOException ex) {
                    SoapUI.logError(ex);
                } catch (SAXException ex) {
                    SoapUI.logError(ex);
                } catch (ParserConfigurationException ex) {
                    SoapUI.logError(ex);
                }
            }
        }
        return "<html><body><h1>The attachment could not be loaded</h1></body></html>";
    }

    private String transformCDA(InputStream attachStream) throws TransformerException, IOException {
        Settings settings = SoapUI.getSettings();
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        System.setErr(new PrintStream(baos));
        String xmlStr = IOUtils.toString(attachStream);
        attachStream.reset();
        TransformerFactory factory = TransformerFactory.newInstance();
        File xsltFile;
        if(xmlStr.indexOf("2.16.840.1.113883.3.88.11.32.1") >= 0) {
            xsltFile = new File(settings.getString(CDASettings.C32_STYLESHEET, ""));
        }
        else {
            xsltFile = new File(settings.getString(CDASettings.CCDA_STYLESHEET, ""));
        }
        Source xslt = new StreamSource(xsltFile);
        xslt.setSystemId(xsltFile.toURI().toString());
        try {
            Transformer transformer = factory.newTransformer(xslt);
            Source text = new StreamSource(attachStream);
            ByteArrayOutputStream toStr = new ByteArrayOutputStream();
            StreamResult result = new StreamResult(toStr);
            transformer.transform(text, result);
            String html = new String(toStr.toByteArray());
            //if there is a css page, add a base href to make sure supporting pages load
            int cssLinkRef = html.indexOf("<link ");
            if(cssLinkRef >= 0) {
                String path = xsltFile.toURI().toString();
                html = new StringBuilder(html).insert(cssLinkRef, "<base href=\""+ (path.substring(0, path.lastIndexOf("/")+1)) + "\" />").toString();
            }
            return html;
        }
        catch(TransformerConfigurationException ex) {
            SoapUI.log(baos.toString());
        }
        return null;
    }
}
