package gov.va.med.nhin.adapter.audit;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Blob;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.sql.DataSource;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.nhin.adapter.utils.DatabaseUtils;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;


public class BulkDownloadLib  
{
	public static final String YEAR_MONTH_DAY = "yyyyMMdd";
	public static final String DISCLOSURE_ACTION = "Retrieve Document";
   public static final String RETRIEVE_FROM_PARTNER_ACTION = "RetrieveDocumentOut";

   public static final String ORG_URI_SCHEME = "urn:oid:";
   // this is the real org ID, but in SQA environments, .1 or .2 can be
   // appended to make a unique community ID for testing
   public static final String VA_ORG_NUMBER_PREFIX = "2.16.840.1.113883.4.349";
   public static final String VA_LONG_NAME = "DEPARTMENT OF VETERANS AFFAIRS";

   public static final String PROP_START_AUDIT_DATETIME = "start.audit.datetime";
   public static final String PROP_END_AUDIT_DATETIME = "end.audit.datetime";
   public static final String PROP_EXPORT_ROOT_DIR = "export.root.dir";
   public static final String PROP_CREATOR_ORG_ID = "creator.org.id";

   /**
    * The main query.
    * <table border="1" cellpadding="3px" cellspacing="0">
    *  <tr style="font-weight: bold">
    *   <td>Action Type</td>
    *   <td>Logical Action</td>
    *   <td>ORGANIZATION_ID</td>
    *   <td>REMOTE_ORGANIZATION_ID</td>
    *   <td>Doc Creator</td>
    *  </tr>
    * <tr>
    *   <td>RetrieveDocumentOut</td>
    *   <td>VA Downloads *from* Partner (Receipt)</td>
    *   <td>Dept. of Vet Affairs</td>
    *   <td>Partner</td>
    *   <td>REMOTE_ORGANIZATION_ID</td>
    *  </tr>
    *  <tr>
    *   <td>Retrieve Document</td>
    *   <td>Partner Downloads *from* VA (Disclosure)</td>
    *   <td>Partner (Maybe Internal VA)</td>
    *   <td>NULL</td>
    *   <td>Dept. of Vet Affairs</td>
    *  </tr>
    * </table>
    */
   private static final String EXPORT_SQL ="SELECT ACTION_NAME, ORGANIZATION_ID /* all same for RetrieveDocumentOut (same initiator, DoVA) */, " +
      "REMOTE_ORGANIZATION_ID /* use for RetrieveDocumentOut, null for Retrieve Document */,PATIENT_FACILITY_NUMBER/* the patient preferred VAMC as per PR.016*/," +
      "REMOTE_ORG_FAC.FACILITY_NAME AS REMOTE_ORG_FACILITY_NAME,REMOTE_ORG_FAC.FACILITY_NUMBER AS REMOTE_ORG_FACILITY_NUMBER," +
      "AUDIT_TIME, DOCUMENTS.DOCUMENT_ID INTERNAL_DOC_ID, CLASS_CODE, FORMAT_CODE, RAW_DATA " +
      "FROM AUDITS JOIN DOCUMENTS " +
      "ON DOCUMENTS.DOCUMENT_UNIQUE_ID = AUDITS.DOCUMENT_ID " +
      "LEFT JOIN FACILITIES REMOTE_ORG_FAC " +
      "ON AUDITS.REMOTE_ORGANIZATION_ID = REMOTE_ORG_FAC.FULL_HOME_COMMUNITY_ID " +      
      "WHERE AUDIT_TIME >= ? AND AUDIT_TIME < ? " +
      "AND ACTION_NAME IN ('RetrieveDocumentOut', 'Retrieve Document') ";
	
    private static final String OR_CREATOR_VA = " (ACTION_NAME = 'Retrieve Document' AND 1 = ?) "; // will be 1 iff they want documents created by VA";
    private static final String OR_CREATOR_NON_VA = " (ACTION_NAME = 'RetrieveDocumentOut' AND REMOTE_ORGANIZATION_ID = ?) ";
	
	private static final String AND_PATIENT_ID_SQL =
      "AND AUDITS.PATIENT_ID = ? ";
	
	private static final String AND_PATIENT_SSN_SQL =
      "AND AUDITS.PATIENT_SSN = ? ";
	
	private static final String AND_PATIENT_LAST_NAME_SQL =
      "AND UPPER(AUDITS.PATIENT_LAST_NAME) = UPPER(?) ";

	private static final String AND_PATIENT_GIVEN_NAME_SQL =
      "AND UPPER(AUDITS.PATIENT_GIVEN_NAME) = UPPER(?) ";

	private static final String ORDER_BY_SQL =
	"ORDER BY AUDIT_TIME";

   private static final Logger LOGGER = LoggerFactory.getLogger(BulkDownloadLib.class);

   private final Date exportDate = new Date();
   private final DataSource dataSource;
   private final boolean doZip;

   public BulkDownloadLib(DataSource dataSource)
   {
      this.dataSource = dataSource;
      doZip = true;
   }

   public BulkDownloadLib(DataSource dataSource, boolean doZip)
   {
      this.dataSource = dataSource;
      this.doZip = doZip;
   }
   
   /**
    * Exports a copy of all documents that were either disclosed ("Retrieve Document" audit record)
    * or retrieved from a partner ("RetrieveDocumentOut" audit record).
    * @param params query parameters
    * @param destinationRoot the root folder into which to export the document tree;
    * ignored when destinationStream is specified
    * @param destinationStream an open output stream to which the contents of a zip
    * file holding all of the documents will be written. This stream will be closed
    * at the end of this method. If destinationStream is null, destination
    * must be specified instead.
    * @throws SQLException 
    * @throws IOException 
    */
   public void exportRetrievedDocs(BulkExportParams params, File destinationRoot, OutputStream destinationStream)
      throws IOException, SQLException
   {
      Connection conn = null;
      PreparedStatement stmt = null;
      ZipOutputStream zipOutStream = null;
      Set<String> pathsWritten = null;
      
      boolean doStream = destinationStream != null;
      boolean doZipNow = doStream || doZip;
      Blob rawData = null;
      try
      {
      	conn = dataSource.getConnection();

            StringBuilder sqlBldr = new StringBuilder(EXPORT_SQL);
            List<String> creators = new ArrayList<>();
            if(!StringUtils.isBlank(params.creatorOrgId)) {
                sqlBldr.append(" AND ( ");
                String[] creatorsAry = params.creatorOrgId.split(",");
                boolean first = true;
                for (String creator : creatorsAry) {
                    if (!StringUtils.isBlank(creator)) {
                        if (first) {
                            first = false;
                        } else {
                            sqlBldr.append(" OR ");
                        }
                        if (creator.equals("1") || creator.startsWith(VA_ORG_NUMBER_PREFIX)) {
                            sqlBldr.append(OR_CREATOR_VA);
                        } else {
                            sqlBldr.append(OR_CREATOR_NON_VA);
                        }
                        creators.add(creator);
                    }
                }
                LOGGER.debug(creators.toString());
                sqlBldr.append(") ");
            }
            if(!StringUtils.isBlank(params.patientIdentifier)) {
                    sqlBldr.append(AND_PATIENT_ID_SQL);
            }
            if(!StringUtils.isBlank(params.ssn)) {
                    sqlBldr.append(AND_PATIENT_SSN_SQL);
            }
            if(!StringUtils.isBlank(params.lastName)) {
                    sqlBldr.append(AND_PATIENT_LAST_NAME_SQL);
            }
            if(!StringUtils.isBlank(params.firstName)) {
                    sqlBldr.append(AND_PATIENT_GIVEN_NAME_SQL);
            }
            sqlBldr.append(ORDER_BY_SQL);
            
            LOGGER.debug(sqlBldr.toString());

			// minimize locking via forward-only + read only ResultSet
                        stmt = conn.prepareStatement(sqlBldr.toString(), ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
                        int nextParamIdx = 1;
			stmt.setTimestamp(nextParamIdx++, new Timestamp(params.startAuditedTime.getTime()));
                        stmt.setTimestamp(nextParamIdx++, new Timestamp(params.endAuditedTime.getTime()));
         
			//***
			// The ordering here of conditionally setting parameters based on filter
			// criteria existing or not must follow the ordering above on conditionally
			// appending AND clauses to the SQL. The whole SQL has to be built before
			// we can prepare a statement object to set parameters on, so these same
			// conditional checks have to be repeated.
			//***
                        if(!creators.isEmpty())
                        {
                            for (String creator : creators) {
                                if ("1".equals(creator) || creator.startsWith(VA_ORG_NUMBER_PREFIX)) {
                                    stmt.setInt(nextParamIdx++, 1);
                                } else {
                                    stmt.setString(nextParamIdx++, ORG_URI_SCHEME + creator);                  
                                }
                            }
                        }
			if(!StringUtils.isBlank(params.patientIdentifier)) {
				stmt.setString(nextParamIdx++, params.patientIdentifier);             
			}
			if(!StringUtils.isBlank(params.ssn)) {
				stmt.setString(nextParamIdx++, params.ssn);             
			}
			if(!StringUtils.isBlank(params.lastName)) {
				stmt.setString(nextParamIdx++, params.lastName);             
			}
			if(!StringUtils.isBlank(params.firstName)) {
				stmt.setString(nextParamIdx++, params.firstName);             
			}

         ResultSet rst = stmt.executeQuery();
         
         // hold off on creating the zip file until the query runs OK
         if(doZipNow)
         {
            if(doStream)
            {
               zipOutStream = new ZipOutputStream(new BufferedOutputStream(destinationStream));
            }
            else
            {
               FileUtils.forceMkdir(destinationRoot);
               SimpleDateFormat sdf = new SimpleDateFormat(YEAR_MONTH_DAY);
               File zipFile = new File(destinationRoot, sdf.format(exportDate) + ".zip");
               zipOutStream = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(zipFile)));
            }
            pathsWritten = new HashSet<>();
         }
              
         while(rst.next())
         {
            String actionName = rst.getString("ACTION_NAME");
            Timestamp auditDateTime = rst.getTimestamp("AUDIT_TIME");
            long internalDocId = rst.getLong("INTERNAL_DOC_ID");
            String docClassCode = rst.getString("CLASS_CODE");
            String formatCode = rst.getString("FORMAT_CODE"); 
            rawData = rst.getBlob("RAW_DATA");
            // long auditId = rst.getLong("AUDIT_ID"); -- not required as per PR.016
            
            String docCreatorOrgId;
            String docCreatorName;
            String docOrgName;
            if(DISCLOSURE_ACTION.equals(actionName))
            {
                // Doesn't really matter as of now, because docCreatorName
                // is not null. But just being defensively consistent.
            	docCreatorOrgId = ORG_URI_SCHEME + VA_ORG_NUMBER_PREFIX;
            	docOrgName = VA_LONG_NAME;
                docCreatorName  = rst.getString("PATIENT_FACILITY_NUMBER"); // Change as per bug report PR.016 use preferred VAMC for DoVA
            }
            else
            {
            	docCreatorOrgId = rst.getString("REMOTE_ORGANIZATION_ID");
            	docOrgName = rst.getString("REMOTE_ORG_FACILITY_NAME");
                docCreatorName = rst.getString("REMOTE_ORG_FACILITY_NUMBER");// Change as per bug report PR.016 use partner ID instead of organization name
            }
            
            // make sure docCreatorName is populated
            if(docCreatorName == null) {
            	docCreatorName = docCreatorOrgId == null ? "UNKNOWN" : docCreatorOrgId;
            }
            
            if(docOrgName == null) {
            	docOrgName = docCreatorOrgId == null ? "UNKNOWN" : docCreatorOrgId;
            }
            String documentType = mapToDocumentTypeName(docClassCode, formatCode);
            String fileSubPath = buildFileSubPath(auditDateTime, docCreatorName, docOrgName,"" + internalDocId, documentType);

            if (rawData != null) { //null check on rawData object.
                try (InputStream stream = rawData.getBinaryStream()){
                            
                    if(doZipNow)
                    {
                        exportFile(zipOutStream, pathsWritten, fileSubPath, stream);
                    }
                    else
                    {
                        exportFile(destinationRoot, fileSubPath, stream);
                    }            
                } catch(IOException e){
                    LOGGER.warn(e.getMessage());
                }
            }           
         }      
         if(doZipNow)
         {
         	zipOutStream.flush();                
         }
      }
      finally
      { 
         //DatabaseUtils.closeBlob(rawData, LOGGER);// closes Blob object and releases resources it holds 
         DatabaseUtils.closeStatement(stmt, LOGGER);
         DatabaseUtils.closeConnection(conn, LOGGER);
         DatabaseUtils.closeIO(zipOutStream, LOGGER);
      }
   }

   public static Date parseRangeDateStr(String rangeDateStr) throws ParseException
   {
   	SimpleDateFormat dateOnly = new SimpleDateFormat("yyyy-MM-dd");
	   SimpleDateFormat dateTime = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm");

	   Date rangeDate;
      try
      {
         if(rangeDateStr.contains("T"))
         {
            rangeDate = dateTime.parse(rangeDateStr);
         }
         else
         {
            rangeDate = dateOnly.parse(rangeDateStr);
         }
      }
      catch(ParseException ex)
      {
         throw new ParseException("Start date and end date must be of the " +
           "format 'yyyy-MM-dd' or 'yyyy-MM-dd'T'HH:mm'.", ex.getErrorOffset());
      }
      return rangeDate;
   }

   /**
    * Wrapper around {@link #exportRetrievedDocs(Date, Date, File, OutputStream)} that takes parameters as Strings.
    * @param startAudit
    * @param endAudit
    * @param creatorOrgId hierarchal organization number, like {@link #VA_ORG_NUMBER}, without <code>urn:oid:</code> prefix
    * @param destinationRoot
    * @throws ParseException
    * @throws SQLException 
    * @throws IOException 
    */
   public void exportRetrievedDocs(String startAudit, String endAudit, String creatorOrgId, String destinationRoot) throws ParseException, IOException, SQLException
   {
   	if(StringUtils.isBlank(startAudit) || StringUtils.isBlank(endAudit))
	   {
	   	throw new ParseException("Start and end audit dates must both be specified.", 0);
	   }
	   Date startAuditDate = parseRangeDateStr(startAudit);
	   Date endAuditDate = parseRangeDateStr(endAudit);

		BulkExportParams params = new BulkExportParams();
		params.startAuditedTime = startAuditDate;
		params.endAuditedTime = endAuditDate;
		params.creatorOrgId = creatorOrgId;
	   exportRetrievedDocs(params, new File(FilenameUtils.normalize(destinationRoot)), null /* destinationStream */);
   }

   /**
    * Wrapper around {@link #exportRetrievedDocs(String, String, String)} that takes
    * parameters inside a properties file.
    * @param exportProps
    * @throws ParseException
    * @throws SQLException 
    * @throws IOException 
    */
   public void exportRetrievedDocs(Properties exportProps) throws ParseException, IOException, SQLException
   {
	   String startAuditDateTime = exportProps.getProperty(PROP_START_AUDIT_DATETIME);
	   String endAuditDateTime = exportProps.getProperty(PROP_END_AUDIT_DATETIME);
	   String exportRootDir = exportProps.getProperty(PROP_EXPORT_ROOT_DIR);
		String creatorOrgId = exportProps.getProperty(PROP_CREATOR_ORG_ID);
	   exportRetrievedDocs(startAuditDateTime, endAuditDateTime, creatorOrgId, exportRootDir);
   }
   
   /**
    * Maps to friendly name that business likes.
    * @param docClassCode LOINC
    * @param formatCode from FORMAT_CODE in Documents table
    * @return
    */
   static String mapToDocumentTypeName(String docClassCode, String formatCode)
   {
   	if(StringUtils.isBlank(docClassCode))
   	{
   		return "UNK-DOC-TYPE";
   	}
   	
   	// obviate null checks below
   	if(formatCode == null) {
   		formatCode = "";
   	}
   	
   	if(docClassCode.equals("34133-9"))
   	{
   		if(formatCode.equals("urn:ihe:pcc:xphr:2007"))
   		{
   			return "C32";
   		}
   		
   		if(
   			formatCode.equals("urn:hl7-org:sdwg:ccda-structuredBody:1.1") ||
   			formatCode.equals("urn:hl7-org:sdwg:CCDA:1.1") ||
   			formatCode.equals("urn:hl7-org:sdwg:ccda-structuredBody:2.1"))
   		{
   			return "CCDACCD";
   		}
   	}
   	
   	if(docClassCode.equals("11506-3"))
   	{
   		if(
   			formatCode.equals("urn:hl7-org:sdwg:ccda-structuredBody:1.1") ||
   			formatCode.equals("urn:hl7-org:sdwg:ccda-structuredBody:2.1"))
   		{
   			return "CCDAPN";
   		}
   	}
   	
		if(formatCode.equals("urn:ihe:iti:xds-sd:pdf:2008"))
		{
			return "C62PDF";
		}

		if(formatCode.equals("urn:ihe:iti:xds-sd:text:2008"))
		{
			return "C62TXT";
		}
		
		// no mapping...default to LOINC
		return docClassCode;
   }

   static void exportFile(File destinationRoot, String fileSubPath, InputStream input) throws IOException, SQLException
   {
   	File targetFile = new File(destinationRoot, FilenameUtils.normalize(fileSubPath));
   	if(targetFile.exists())
      {
         return;
      }
      
      FileUtils.forceMkdir(targetFile.getParentFile());
      FileOutputStream fileOut = null;
      
      try
      {
         fileOut = new FileOutputStream(targetFile);
         IOUtils.copy(input, fileOut);
         fileOut.flush();
      }
      finally
      {
         DatabaseUtils.closeIO(fileOut, LOGGER);
      }
   }

   static void exportFile(ZipOutputStream zipOut, Set<String> pathsWritten, String fileSubPath, InputStream input)
   		throws IOException, SQLException
   {
   	if(pathsWritten.contains(fileSubPath))
		{
			return;
		}
   	pathsWritten.add(fileSubPath);
   	
   	ZipEntry entry = new ZipEntry(fileSubPath);
   	zipOut.putNextEntry(entry);
      IOUtils.copy(input, zipOut);
   }

   /**
    * Naming Convention for Downloaded Patient Files    * 
    * The sub parent folders will be separated by document creator.
    * Naming convention for specific files: 
    * <creator><type><dateStored><auditsIndex><documentsIndex>.xml
    * e.g. EPIC/EPIC_C32_20161201_456_185.xml
    *
    * where dateStored = yyyyMMdd format of date generated or received
    */
   private String buildFileSubPath(Date auditDateTime, String docCreatorName,String docOrgName, String index, String docType)
   {
      SimpleDateFormat sdf = new SimpleDateFormat(YEAR_MONTH_DAY);
      StringBuilder path = new StringBuilder(StringUtils.EMPTY);
   	
      docCreatorName = docCreatorName.replace("urn:oid:", "oid_");
      docCreatorName = docCreatorName.replace("<", "");
      docCreatorName = docCreatorName.replace(">", "");
      docCreatorName = cleanFileName(docCreatorName);
      docOrgName     = cleanFileName(docOrgName);
      //path.append("/").append(docCreatorName); // NOTE: This creates an empty folder in the zip file in Windows.
      path.append(docOrgName);
      
      docType = cleanFileName(docType);
      String auditDateStr = sdf.format(auditDateTime);
      //String fileName = index + "_" + docType + "_" + auditDateStr + ".xml"; -- Original code 
      String fileName = docCreatorName + "_" + docType + "_" + index +"_" + auditDateStr + ".xml";
       if (fileName.length()>255)
       {
          int    diff = 255 -(fileName.length()); // This is the excess length to be trimmed off from the Partner name
          String truncatedFileName =  docCreatorName.substring(0, docCreatorName.length()-diff-1) + "_" + docType + "_" + index +"_" + auditDateStr +".xml";
          path.append("/").append(truncatedFileName);
       }
       else
       {
          path.append("/").append(fileName);
       }
      return path.toString();
   }
   
   static final String cleanFileName(String fileName)
   {
   	char[] cleaned = null;
   	int fileLen = fileName.length();
   	for(int i = 0; i < fileLen; i++)
   	{
   		char charAt = fileName.charAt(i);
   		boolean ok =
   				('A' <= charAt && charAt <= 'Z') ||
   				('a' <= charAt && charAt <= 'z') ||
   				('0' <= charAt && charAt <= '9') ||
   				charAt == '-' ||
   				charAt == '_' ||
   				charAt == '.' ||
   				charAt == ' ';
   		
   		if(ok) {
   			continue;
   		}
   		
   		// lazy create copy...most names will be fine in reality
   		if(cleaned == null) {
   			cleaned = fileName.toCharArray();
   		}
   		cleaned[i] = '-';
   	}
   	
   	if(cleaned == null) {
   		return fileName;
   	} else {
   		return new String(cleaned);
   	}
   }

	public static class BulkExportParams
	{
		public Date startAuditedTime;
		public Date endAuditedTime;
		public String creatorOrgId;
		public String patientIdentifier;
		public String ssn;
		public String lastName;
		public String firstName;
	}
}
