/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package gov.va.med.nhin.adapter.subscription.web.resource;

import ca.uhn.fhir.model.api.annotation.Block;
import ca.uhn.fhir.model.api.annotation.Child;
import ca.uhn.fhir.model.api.annotation.Description;
import ca.uhn.fhir.model.api.annotation.Extension;
import ca.uhn.fhir.model.api.annotation.ResourceDef;
import ca.uhn.fhir.util.ElementUtil;
import java.util.ArrayList;
import java.util.Formatter;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.hl7.fhir.dstu3.model.BackboneElement;
import org.hl7.fhir.dstu3.model.BooleanType;
import org.hl7.fhir.dstu3.model.EnumFactory;
import org.hl7.fhir.dstu3.model.Enumeration;
import org.hl7.fhir.dstu3.model.StringType;
import org.hl7.fhir.dstu3.model.Subscription;

/**
 *
 * @author david
 */
@ResourceDef(name = "Subscription", profile = "http://URL/StructureDefinition/EHXSubscription")
public class EHXSubscription extends Subscription
{
    @Child(name = "announcePatient")
    @Extension(url = "http://URL/StructureDefinition/EHXSubscription/announcePatient", definedLocally = true, isModifier = false)
    @Description(shortDefinition = "Specify if the patient should be announced to all eHX partners.")
    private BooleanType announcePatient;

    @Child(name = "autoDocRetrieve")
    @Extension(url = "http://URL/StructureDefinition/EHXSubscription/autoDocRetrieve", definedLocally = true, isModifier = false)
    @Description(shortDefinition = "Specify if documents should be automatically retrieved.  Set to true in prefetch situations.")
    private BooleanType autoDocRetrieve;

    @Child(name = "assertions")
    @Extension(url = "http://URL/StructureDefinition/EHXSubscription/assertions", definedLocally = true, isModifier = false)
    @Description(shortDefinition = "Assertions about entity that created the subscription.")
    private Assertions assertions;

    @Child(name = "nhioStatus", max = Child.MAX_UNLIMITED)
    @Extension(url = "http://URL/StructureDefinition/EHXSubscription/nhioStatus", definedLocally = true, isModifier = false)
    @Description(shortDefinition = "Status of NHIOs that were part of the processing of this subscription.")
    private List<NHIOStatus> nhioStatuses;

    /**
     * It is important to override the isEmpty() method, adding a check for any
     * newly added fields.
     */
    @Override
    public boolean isEmpty()
    {
        return super.isEmpty()
               && ElementUtil.isEmpty(announcePatient, autoDocRetrieve, assertions, nhioStatuses);
    }

    public BooleanType getAnnouncePatient()
    {
        return announcePatient;
    }

    public void setAnnouncePatient(BooleanType announcePatient)
    {
        this.announcePatient = announcePatient;
    }

    public BooleanType getAutoDocRetrieve()
    {
        return autoDocRetrieve;
    }

    public void setAutoDocRetrieve(BooleanType autoDocRetrieve)
    {
        this.autoDocRetrieve = autoDocRetrieve;
    }

    public Assertions getAssertions()
    {
        return assertions;
    }

    public void setAssertions(Assertions assertions)
    {
        this.assertions = assertions;
    }

    public boolean hasAllAssertions()
    {
        return !ElementUtil.isEmpty(assertions)
               && assertions.isFull();
    }

    public boolean hasPatientID()
    {
        return (null != getPatientID());
    }

    public String getPatientID()
    {
        final Pattern patientid = Pattern.compile("DocumentReference\\?patient=Patient/(\\d{10}V\\d{6})");
        Matcher m = patientid.matcher(getCriteria());
        return (m.matches() ? m.group(1) : null);
    }

    public boolean allDone()
    {
        boolean ret;
        
        switch (getStatus()) {
            case ERROR:
            case OFF:
                ret = true;
                break;

            case ACTIVE:
                ret = true;
                for (EHXSubscription.NHIOStatus s : getNHIOStatuses()) {
                    ret = ret &= s.getStatus() == NHIOStatus.StatusType.DONE;
                    if (!ret) {
                        break;
                    }
                }
                break;
                
            default:
                ret = false;
                break;
        }

        return ret;
    }

    public boolean isAnnounce()
    {
        return !ElementUtil.isEmpty(getAnnouncePatient())
               && getAnnouncePatient().booleanValue();
    }

    public boolean isAutoDocRetrieve()
    {
        return !ElementUtil.isEmpty(getAutoDocRetrieve())
               && getAutoDocRetrieve().booleanValue();
    }

    public void setNHIOStatuses(List<NHIOStatus> nhioStatuses)
    {
        this.nhioStatuses = nhioStatuses;
    }

    public List<NHIOStatus> getNHIOStatuses()
    {
        return nhioStatuses;
    }

    public void addNHIOStatus(String hcid, NHIOStatus.StatusType status, NHIOStatus.ProcessingStageType processingStage)
    {
        NHIOStatus nhioStatus = new NHIOStatus();
        nhioStatus.setHCID(hcid);
        nhioStatus.setStatus(status);
        nhioStatus.setProcessingStage(processingStage);
        if (nhioStatuses == null) {
            nhioStatuses = new ArrayList<>();
        }
        nhioStatuses.add(nhioStatus);
    }

    public void updateNHIOStatus(String hcid, NHIOStatus.StatusType status, NHIOStatus.ProcessingStageType processingStage, List<String> errors)
    {
        if (!ElementUtil.isEmpty(nhioStatuses)) {
            nhioStatuses.forEach((n) -> {
                if (n.getHCID().equals(hcid)) {
                    n.setProcessingStage(processingStage);
                    n.setStatus(status);
                    if (errors != null) {
                        errors.forEach((err) -> {
                            n.addError(err);
                        });
                    }
                }
            });
        }
    }

    @Block
    public static class Assertions extends BackboneElement
    {
        @Child(name = "userID")
        @Extension(url = "userID", definedLocally = true, isModifier = false)
        @Description(shortDefinition = "User/System ID that triggered subscription.")
        private StringType userID;

        @Child(name = "userName")
        @Extension(url = "userName", definedLocally = true, isModifier = false)
        @Description(shortDefinition = "User/System name that triggered subscription.")
        private StringType userName;

        @Child(name = "systemID")
        @Extension(url = "systemID", definedLocally = true, isModifier = false)
        @Description(shortDefinition = "System ID of system that triggered the request.")
        private StringType systemID;

        @Child(name = "organizationID")
        @Extension(url = "organizationID", definedLocally = true, isModifier = false)
        @Description(shortDefinition = "ID of organization that triggered subscription.")
        private StringType organizationID;

        @Child(name = "organizationName")
        @Extension(url = "organizationName", definedLocally = true, isModifier = false)
        @Description(shortDefinition = "Name of organization that triggered subscription.")
        private StringType organizationName;

        @Child(name = "purposeOfUse")
        @Extension(url = "purposeOfUse", definedLocally = true, isModifier = false)
        @Description(shortDefinition = "Purpose of subscription.")
        private StringType purposeOfUse;

        @Child(name = "role")
        @Extension(url = "role", definedLocally = true, isModifier = false)
        @Description(shortDefinition = "User role SNOMED code.")
        private StringType role;

        @Override
        public Assertions copy()
        {
            Assertions assertions = new Assertions();
            assertions.purposeOfUse = purposeOfUse;
            assertions.role = role;
            assertions.userID = userID;
            assertions.userName = userName;
            assertions.organizationID = organizationID;
            assertions.organizationName = organizationName;
            assertions.systemID = systemID;
            return assertions;
        }

        @Override
        public boolean isEmpty()
        {
            return super.isEmpty()
                   && ElementUtil.isEmpty(userID, userName, systemID, organizationID, organizationName, purposeOfUse, role);
        }

        public boolean isFull()
        {
            return !(ElementUtil.isEmpty(userID)
                     || ElementUtil.isEmpty(userName)
                     || ElementUtil.isEmpty(systemID)
                     || ElementUtil.isEmpty(organizationID)
                     || ElementUtil.isEmpty(organizationName)
                     || ElementUtil.isEmpty(purposeOfUse)
                     || ElementUtil.isEmpty(role));
        }

        public StringType getUserID()
        {
            return userID;
        }

        public void setUserID(StringType userID)
        {
            this.userID = userID;
        }

        public StringType getUserName()
        {
            return userName;
        }

        public void setUserName(StringType userName)
        {
            this.userName = userName;
        }

        public StringType getSystemID()
        {
            return systemID;
        }

        public void setSystemID(StringType systemID)
        {
            this.systemID = systemID;
        }

        public StringType getOrganizationID()
        {
            return organizationID;
        }

        public void setOrganizationID(StringType organizationID)
        {
            this.organizationID = organizationID;
        }

        public StringType getOrganizationName()
        {
            return organizationName;
        }

        public void setOrganizationName(StringType organizationName)
        {
            this.organizationName = organizationName;
        }

        public StringType getPurposeOfUse()
        {
            return purposeOfUse;
        }

        public void setPurposeOfUse(StringType purposeOfUse)
        {
            this.purposeOfUse = purposeOfUse;
        }

        public StringType getRole()
        {
            return role;
        }

        public void setRole(StringType role)
        {
            this.role = role;
        }

        public String toString()
        {
            StringBuilder sb = new StringBuilder();
            Formatter f = new Formatter(sb);
            f.format("{userID: %s, userName: %s, systemID: %s, organizationID: %s, organizationName: %s, purposeOfUse: %s, role: %s}",
                     userID, userName, systemID, organizationID, organizationName, purposeOfUse, role);
            return sb.toString();
        }
    }

    @Block
    public static class NHIOStatus extends BackboneElement
    {
        public enum StatusType
        {
            NEW,
            PROCESSING,
            DONE
        };

        public static class StatusTypeEnumFactory implements EnumFactory<StatusType>
        {
            @Override
            public StatusType fromCode(String string) throws IllegalArgumentException
            {
                return StatusType.valueOf(string);
            }

            @Override
            public String toCode(StatusType t)
            {
                return t.toString();
            }

            @Override
            public String toSystem(StatusType t)
            {
                return t.toString();
            }
        }
        
        public enum ProcessingStageType
        {
            STARTING,
            ANNOUNCE,
            ANNOUNCE_SUCCESS,
            ANNOUNCE_FAILURE,
            DOCUMENT_QUERY,
            DOCUMENT_QUERY_SUCCESS,
            DOCUMENT_QUERY_FAILURE,
            DOCUMENT_RETRIEVE,
            DOCUMENT_RETRIEVE_SUCCESS,
            DOCUMENT_RETRIEVE_FAILURE,
        };

        public static class ProcessingStageTypeEnumFactory implements EnumFactory<ProcessingStageType>
        {
            @Override
            public ProcessingStageType fromCode(String string) throws IllegalArgumentException
            {
                return ProcessingStageType.valueOf(string);
            }

            @Override
            public String toCode(ProcessingStageType t)
            {
                return t.toString();
            }

            @Override
            public String toSystem(ProcessingStageType t)
            {
                return t.toString();
            }
        }
        
        @Child(name = "hcid")
        @Extension(url = "hcid", definedLocally = true, isModifier = false)
        private StringType hcid;

        @Child(name = "status")
        @Extension(url = "status", definedLocally = true, isModifier = false)
        private Enumeration<StatusType> status;

        @Child(name = "type")
        @Extension(url = "type", definedLocally = true, isModifier = false)
        private Enumeration<ProcessingStageType> processingStage;

        @Child(name = "error")
        @Extension(url = "error", definedLocally = true, isModifier = false)
        private List<StringType> errors;
        
        @Override
        public boolean isEmpty()
        {
            return super.isEmpty()
                   && ElementUtil.isEmpty(hcid, status, processingStage);
        }

        @Override
        public NHIOStatus copy()
        {
            NHIOStatus nhioStatus = new NHIOStatus();
            nhioStatus.hcid = hcid;
            nhioStatus.status = this.status;
            nhioStatus.processingStage = processingStage;
            nhioStatus.errors = errors;
            return nhioStatus;
        }

        public String getHCID()
        {
            return hcid.getValue();
        }

        public void setHCID(String hcid)
        {
            this.hcid = new StringType(hcid);
        }

        public StatusType getStatus()
        {
            return status.getValue();
        }

        public void setStatus(StatusType status)
        {
            this.status = new Enumeration<>(new StatusTypeEnumFactory(), status);
        }

        public ProcessingStageType getProcessingStage()
        {
            return processingStage.getValue();
        }

        public void setProcessingStage(ProcessingStageType type)
        {
            this.processingStage = new Enumeration<>(new ProcessingStageTypeEnumFactory(), type);
        }

        public List<StringType> getError()
        {
            return errors;
        }

        public void setError(List<StringType> errors)
        {
            this.errors = errors;
        }
        
        public void addError(String error)
        {
            if (errors == null) {
                errors = new ArrayList<>();
            }
            errors.add(new StringType(error));
        }
    }
}