/*
 * DAO class for Subscription object.
 */
package gov.va.med.nhin.adapter.subscription.web.proxy.provider.subscription;

import ca.uhn.fhir.context.FhirContext;
import ca.uhn.fhir.parser.IParser;
import gov.va.med.nhin.adapter.subscription.web.dao.SubscriptionDAO;
import gov.va.med.nhin.adapter.subscription.web.entity.Subscription;
import gov.va.med.nhin.adapter.subscription.web.resource.EHXSubscription;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.util.Date;
import java.util.function.Function;
import javax.enterprise.context.ApplicationScoped;
import javax.inject.Named;
import javax.inject.Inject;
import javax.persistence.LockModeType;
import javax.transaction.Transactional;
import org.hl7.fhir.dstu3.model.IdType;

@ApplicationScoped
@Named("DAOSubscriptionProvider")
public class DAOSubscriptionProvider implements SubscriptionProvider
{
    @Inject
    private SubscriptionDAO subscriptionDAO;
    @Inject
    private FhirContext fhirContext;
    
    @Override
    @Transactional
    public Long create(EHXSubscription subscription)
    {
        SubscriptionProvider.super.create(subscription);
        Subscription entity = createEntityFromResource(subscription);
        entity.setCreatedTime(new Date());
        subscriptionDAO.store(entity);
        Long ret = entity.getSubscriptionId().longValue();
        subscription.setUserData("subscriptionID", ret);
        return ret;
    }

    @Override
    public EHXSubscription read(Long id)
    {
        Subscription entity = subscriptionDAO.findBySubscriptionID(BigDecimal.valueOf(id));
        return createResourceFromEntity(entity);
    }
    
    @Override
    public EHXSubscription readAndLock(Long id)
    {
        Subscription entity = subscriptionDAO.findBySubscriptionID(BigDecimal.valueOf(id), LockModeType.PESSIMISTIC_WRITE);
        return createResourceFromEntity(entity);
    }
    
    @Override
    public EHXSubscription read(IdType id)
    {
        Subscription entity = subscriptionDAO.findByResourceID(id.getIdPart());
        return createResourceFromEntity(entity);
    }

    @Override
    @Transactional
    public boolean readWithUpdate(Long id, Function<EHXSubscription, Boolean> function)
    {
        return SubscriptionProvider.super.readWithUpdate(id, function);
    }
    
    @Override
    @Transactional
    public void update(Long id, EHXSubscription resource)
    {
        SubscriptionProvider.super.update(id, resource);
        Subscription entity = subscriptionDAO.findBySubscriptionID(BigDecimal.valueOf(id));
        updateEntityFromResource(resource, entity);
        subscriptionDAO.update(entity);
    }
    
    @Override
    public EHXSubscription delete(Long id)
    {
        // TODO: map EHXSubscription to Subscription entity and call entityManager.delete();
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.        
    }
    
    private Subscription createEntityFromResource(EHXSubscription resource)
    {
        Subscription ret = new Subscription();
        updateEntityFromResource(resource, ret);
        return ret;
    }
    
    private void updateEntityFromResource(EHXSubscription resource, Subscription entity)
    {
        entity.setResourceId(resource.getIdElement().getIdPart());
        entity.setChannel(resource.getChannel().getEndpoint());
        entity.setLastUpdatedTime(resource.getMeta().getLastUpdated());
        entity.setCriteria(resource.getCriteria());
        entity.setOrganizationId(resource.getAssertions().getOrganizationID().asStringValue());
        entity.setOrganizationName(resource.getAssertions().getOrganizationName().asStringValue());
        entity.setPurposeOfUse(resource.getAssertions().getPurposeOfUse().asStringValue());
        entity.setStatus(resource.getStatus().getDisplay());
        entity.setUserId(resource.getAssertions().getUserID().toString());
        entity.setUserName(resource.getAssertions().getUserName().toString());
        IParser jsonParser = fhirContext.newJsonParser();
        try {
            entity.setFhirResource(jsonParser.encodeResourceToString(resource).getBytes("UTF-8"));
        }
        catch (UnsupportedEncodingException uee) {
            throw new RuntimeException("You're using an unsupported encoding.  FIX IT NOW!!!", uee);
        }
    }
    
    private EHXSubscription createResourceFromEntity(Subscription entity)
    {
        EHXSubscription ret = null;
        if (entity != null) {
            IParser jsonParser = fhirContext.newJsonParser();
            try {
                ret = jsonParser.parseResource(EHXSubscription.class, new String(entity.getFhirResource(), "UTF-8"));
                ret.setUserData("subscriptionID", entity.getSubscriptionId().longValue());
            }
            catch (UnsupportedEncodingException uee) {
                throw new RuntimeException("You're using an unsupported encoding.  FIX IT NOW!!!", uee);
            }
        }
        return ret;
    }
}
