/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package gov.va.med.nhin.adapter.subscription.web.process;

import gov.va.med.nhin.adapter.subscription.web.annotations.Created;
import gov.va.med.nhin.adapter.subscription.web.annotations.Property;
import gov.va.med.nhin.adapter.subscription.web.event.NHIOEvent;
import gov.va.med.nhin.adapter.subscription.web.event.NotificationEvent;
import gov.va.med.nhin.adapter.subscription.web.event.SubscriptionEvent;
import gov.va.med.nhin.adapter.subscription.web.proxy.aatohcidmapper.AssigningAuthorityToHomeCommunityMapper;
import gov.va.med.nhin.adapter.subscription.web.proxy.correlationretriever.CorrelationRetriever;
import gov.va.med.nhin.adapter.subscription.web.proxy.nhioretriever.NHIORetriever;
import gov.va.med.nhin.adapter.subscription.web.proxy.provider.subscription.SubscriptionProvider;
import gov.va.med.nhin.adapter.subscription.web.resource.EHXSubscription;
import gov.va.med.nhin.adapter.subscription.web.resource.EHXSubscription.NHIOStatus;
import gov.va.med.nhin.adapter.subscription.web.utils.logging.Log;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;

import javax.ejb.Asynchronous;
import javax.ejb.Stateless;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.enterprise.event.Event;
import javax.enterprise.event.Observes;
import javax.inject.Inject;

import org.apache.commons.lang3.StringUtils;
import org.hl7.fhir.dstu3.model.Identifier;
import org.hl7.fhir.dstu3.model.Subscription;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author david_vetsez
 */
@Stateless()
@TransactionAttribute(value = TransactionAttributeType.SUPPORTS)
public class SubscriptionProcessor
{
    private static final Logger logger = LoggerFactory.getLogger(SubscriptionProcessor.class);
    
    @Inject
    private NHIORetriever nhioRetriever;
    @Inject
    private CorrelationRetriever correlationRetriever;
    @Inject
    @Created
    private Event<NHIOEvent> nhioEvent;
    @Inject
    private Event<NotificationEvent> notificationEvent;
    @Inject
    private SubscriptionProvider subscriptionProvider;
    @Inject
    private AssigningAuthorityToHomeCommunityMapper assigningAuthorityToHomeCommunityMapper;
    @Inject
    @Property
    private boolean announceEnabled;
    @Inject
    @Property
    private String announcePattern;

    @Asynchronous()
    public void processSubscriptionCreated(@Observes @Created SubscriptionEvent event)
    {
        EHXSubscription subscription = subscriptionProvider.read(event.getSubscriptionID());
        String siteID = StringUtils.substringBefore(subscription.getAssertions().getUserID().asStringValue(),":");
        String patientID = subscription.getPatientID();
        List<String> nhios;

        try {
            // Get the list of NHIOs given the setting of announcePatient.
            if (announceEnabled && subscription.isAnnounce() && Pattern.matches(announcePattern, siteID)) {
                nhios = nhioRetriever.getNHIOs();
            }
            else {
                nhios = getNHIOsFromPatientCorrelations(patientID, subscription.getAssertions());
            }

            if (!nhios.isEmpty()) {
                // Update status to ACTIVE to show that subscription is being processed.
                subscription.setStatus(Subscription.SubscriptionStatus.ACTIVE);
                Log.debug(logger, "Subscription ID: " + subscription.getId() + " Has been updated to an 'ACTIVE' Status for patient: " + patientID);

                // Add NHIO status entry for each NHIO that we're going to process.
                for (String nhio : nhios) {
                    subscription.addNHIOStatus(nhio, EHXSubscription.NHIOStatus.StatusType.NEW, EHXSubscription.NHIOStatus.ProcessingStageType.STARTING);
                    Log.debug(logger, "NHIO Updated to status: 'NEW' and Processing Stage 'Starting': " + nhio + " for subscriptionID: " + subscription.getId() + " patient: " + patientID);
                }
            }
            else {
                subscription.setStatus(Subscription.SubscriptionStatus.OFF);
                Log.info(logger, "Subscription ID: " + subscription.getId() + " Has been updated to status: 'OFF' for patient: " + patientID);
            }
        }
        catch (Exception e) {
            subscription.setStatus(Subscription.SubscriptionStatus.ERROR);
            Log.warn(logger, "Subscription ID: " + subscription.getId() + " Subscription has been updated to status: 'Error' for patient: " + patientID);
            nhios = new ArrayList<>();
        }

        subscriptionProvider.update(event.getSubscriptionID(), subscription);

        // fire event for each NHIO.
        if (subscription.getStatus() == Subscription.SubscriptionStatus.ACTIVE) {
            for (NHIOStatus nhioStatus : subscription.getNHIOStatuses()) {
                nhioEvent.fire(new NHIOEvent(event.getSubscriptionID(), nhioStatus.getHCID(), nhioStatus.getStatus(), nhioStatus.getProcessingStage(), null));
                Log.debug(logger, "Firing nhioEvent, SubscriptionID: " + event.getSubscriptionID() + "  HCID: " + nhioStatus.getHCID() +" NHIO-status: " + nhioStatus.getStatus() +" Processing stage: "+ nhioStatus.getProcessingStage());
            }
        }

        // notify clients.
        notificationEvent.fire(new NotificationEvent(event.getSubscriptionID(), true));
        Log.debug(logger, "Notfiying clients of process Subscription created update. Subscription ID: " + subscription.getId());
    }

    private List<String> getNHIOsFromPatientCorrelations(String patientID, EHXSubscription.Assertions assertions)
    {
        List<String> ret = new ArrayList<>();
        Set<String> hcids = new HashSet<>();
        List<Identifier> correlatedPatientIDs = correlationRetriever.getCorrelations(patientID, assertions);
        for (Identifier i : correlatedPatientIDs) {
            List<String> correlatedHomeCommunities = assigningAuthorityToHomeCommunityMapper.getHomeCommunities(i.getSystem());
            for (String hcid : correlatedHomeCommunities) {
                hcids.add(hcid);
            }
        }

        for (String hcid : hcids) {
            ret.add(hcid);
        }
        Log.debug(logger, "Retreived NHIOs from patient Correlations for Patient ID: " + patientID);
        return ret;
    }

    public void setNHIORetriever(NHIORetriever nhioRetriever)
    {
        this.nhioRetriever = nhioRetriever;
    }

    public void setCorrelationRetriever(CorrelationRetriever correlationRetriever)
    {
        this.correlationRetriever = correlationRetriever;
    }

    public void setNHIOEvent(Event<NHIOEvent> event)
    {
        this.nhioEvent = event;
    }

    public void setSubscriptionProvier(SubscriptionProvider subscriptionProvier)
    {
        this.subscriptionProvider = subscriptionProvier;
    }

    public void setAssigningAuthorityToHomeCommunityMapper(AssigningAuthorityToHomeCommunityMapper assigningAuthorityToHomeCommunityMapper)
    {
        this.assigningAuthorityToHomeCommunityMapper = assigningAuthorityToHomeCommunityMapper;
    }
}
