/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package gov.va.med.nhin.adapter.subscription.web.process;

import gov.va.med.nhin.adapter.subscription.web.annotations.Created;
import gov.va.med.nhin.adapter.subscription.web.annotations.Property;
import gov.va.med.nhin.adapter.subscription.web.annotations.Updated;
import gov.va.med.nhin.adapter.subscription.web.event.NHIOEvent;
import gov.va.med.nhin.adapter.subscription.web.proxy.announce.PatientAnnouncer;
import gov.va.med.nhin.adapter.subscription.web.proxy.documentquery.DocumentQueryer;
import gov.va.med.nhin.adapter.subscription.web.proxy.documentquery.DocumentQueryerResponse;
import gov.va.med.nhin.adapter.subscription.web.proxy.documentretrieve.DocumentRetrieverResponse;
import gov.va.med.nhin.adapter.subscription.web.proxy.provider.documentreference.DocumentReferenceProvider;
import gov.va.med.nhin.adapter.subscription.web.proxy.provider.subscription.SubscriptionProvider;
import gov.va.med.nhin.adapter.subscription.web.resource.EHXDocumentReference;
import gov.va.med.nhin.adapter.subscription.web.resource.EHXSubscription;
import gov.va.med.nhin.adapter.subscription.web.resource.EHXSubscription.NHIOStatus;
import gov.va.med.nhin.adapter.subscription.web.utils.logging.Log;

import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.ejb.Asynchronous;
import javax.ejb.Stateless;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.enterprise.event.Event;
import javax.enterprise.event.Observes;
import javax.inject.Inject;
import javax.inject.Named;

import org.apache.commons.lang3.StringUtils;
import org.hl7.fhir.dstu3.model.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author david_vetsez
 */
@Stateless()
@TransactionAttribute(value = TransactionAttributeType.SUPPORTS)
@Named("NHIOProcessor")
public class NHIOProcessor
{
    @Inject
    private SubscriptionProvider subscriptionProvider;
    @Inject
    private DocumentReferenceProvider documentReferenceProvider;
    @Inject
    private PatientAnnouncer patientAnnouncer;
    @Inject
    private DocumentQueryer documentQueryer;
    @Inject
    private DocumentProcessor documentProcessor;
    @Inject
    @Updated
    private Event<NHIOEvent> nhioEvent;
    @Inject
    @Property
    private boolean announceEnabled;
    @Inject
    @Property
    private String announcePattern;

    private static final Logger logger = LoggerFactory.getLogger(NHIOProcessor.class);
    
    @Asynchronous()
    public void processNHIO(@Observes @Created NHIOEvent event)
    {
        EHXSubscription subscription = subscriptionProvider.read(event.getSubscriptionID());
        String siteID = StringUtils.substringBefore(subscription.getAssertions().getUserID().asStringValue(),":");
        String patientID = subscription.getPatientID();
        String hcid = event.getHCID();
        boolean announceSuccess;
        StringBuilder errors = new StringBuilder();

        if (announceEnabled && subscription.isAnnounce() && Pattern.matches(announcePattern, siteID)) {
            nhioEvent.fire(new NHIOEvent(event.getSubscriptionID(), event.getHCID(), NHIOStatus.StatusType.PROCESSING, NHIOStatus.ProcessingStageType.ANNOUNCE));
            announceSuccess = patientAnnouncer.sendAnnouncePatient(patientID, hcid, subscription.getAssertions()) > 0;
            if (announceSuccess) {
                nhioEvent.fire(new NHIOEvent(event.getSubscriptionID(), event.getHCID(), NHIOStatus.StatusType.PROCESSING, NHIOStatus.ProcessingStageType.ANNOUNCE_SUCCESS));
                Log.info(logger, "Announce has succeeded for Subscription ID "+ subscription.getId() + " Patient ID " +patientID + " at "+hcid);
            }
            else {
                nhioEvent.fire(new NHIOEvent(event.getSubscriptionID(), event.getHCID(), NHIOStatus.StatusType.DONE, NHIOStatus.ProcessingStageType.ANNOUNCE_FAILURE));
                Log.info(logger, "Announce has failed for Subscription ID "+ subscription.getId() + " Patient ID "+patientID + " at "+hcid);
            }
        }
        else {
            announceSuccess = true;
        }

        if (announceSuccess) {
            nhioEvent.fire(new NHIOEvent(event.getSubscriptionID(), event.getHCID(), NHIOStatus.StatusType.PROCESSING, NHIOStatus.ProcessingStageType.DOCUMENT_QUERY));
            Log.debug(logger, "Document Query being processed for subscription ID: " +  event.getSubscriptionID() + " Patient "+patientID+" EventHCID: " +event.getHCID());
            
            DocumentQueryerResponse dqr = documentQueryer.sendDocumentQuery(patientID, hcid, subscription.getAssertions());
            
            
            if (dqr.isSuccess()) {
                nhioEvent.fire(new NHIOEvent(event.getSubscriptionID(),
                                             event.getHCID(),
                                             !dqr.getDocumentReferences().isEmpty() ? NHIOStatus.StatusType.PROCESSING : NHIOStatus.StatusType.DONE,
                                             NHIOStatus.ProcessingStageType.DOCUMENT_QUERY_SUCCESS,
                                             dqr.getErrors()));
                Log.debug(logger, "Document Query was successful for subscriptionID: " +  event.getSubscriptionID() + " Patient "+patientID+" EventHCID: " + event.getHCID());
                if (!dqr.getDocumentReferences().isEmpty()) {
                    for (EHXDocumentReference dr : dqr.getDocumentReferences()) {
                        dr.setSubscription(new Reference(subscription.getId()));
                        documentReferenceProvider.create(dr);
                    }

                    if (subscription.isAutoDocRetrieve()) {
                        nhioEvent.fire(new NHIOEvent(event.getSubscriptionID(), event.getHCID(), NHIOStatus.StatusType.PROCESSING, NHIOStatus.ProcessingStageType.DOCUMENT_RETRIEVE));
                        Log.debug(logger, "Auto-Doc Retrieve is Processing for SubscriptionID: " + event.getSubscriptionID() + " Patient "+patientID+"  EventHCID: " + event.getHCID());
                        int i = 0;
                        for (EHXDocumentReference dr : dqr.getDocumentReferences()) {
                            DocumentRetrieverResponse drr = documentProcessor.getDocument(dr);
                            if (drr.isSuccess()) {
                                NHIOStatus.StatusType stat = (++i < dqr.getDocumentReferences().size()
                                    ? NHIOStatus.StatusType.PROCESSING
                                    : NHIOStatus.StatusType.DONE);
                                nhioEvent.fire(new NHIOEvent(event.getSubscriptionID(), event.getHCID(), stat, NHIOStatus.ProcessingStageType.DOCUMENT_RETRIEVE_SUCCESS, true, drr.getErrors()));
                                Log.info(logger, "Auto-Doc Retrieve Successful for SubscriptionID: " + event.getSubscriptionID() + " Patient "+patientID+" EventHCID: " + event.getHCID() + " Document ID:" + dr.getDocumentUniqueId());
                            }
                            else {
                                nhioEvent.fire(new NHIOEvent(event.getSubscriptionID(), event.getHCID(), NHIOStatus.StatusType.DONE, NHIOStatus.ProcessingStageType.DOCUMENT_RETRIEVE_FAILURE, drr.getErrors()));
                                //concatenate the list of errors into a Stringbuilder
                                dqr.getErrors().forEach(errors::append);
                                Log.warn(logger, "Auto-Doc Retrieve Failed for SubscriptionID: " + event.getSubscriptionID() + " Patient "+patientID+" EventHCID: " + event.getHCID() + " Document ID:" + dr.getDocumentUniqueId() + " With the following errors: " + errors.toString() );
                                //errors = new StringBuilder();
                                break;
                            }
                        }
                    }
                    else {
                        nhioEvent.fire(new NHIOEvent(event.getSubscriptionID(), event.getHCID(), NHIOStatus.StatusType.DONE, NHIOStatus.ProcessingStageType.DOCUMENT_QUERY_SUCCESS));
                        Log.info(logger, "Document Query was succesful for subscriptionID: " +  event.getSubscriptionID() + " Patient "+patientID+" EventHCID: " + event.getHCID());
                    }
                }
            }
            else {
                nhioEvent.fire(new NHIOEvent(event.getSubscriptionID(), event.getHCID(), NHIOStatus.StatusType.DONE, NHIOStatus.ProcessingStageType.DOCUMENT_QUERY_FAILURE, dqr.getErrors()));
                //concatenate the list of errors into a Stringbuilder
                dqr.getErrors().forEach(errors::append);
                Log.warn(logger, "Document Query Failed for subscriptionID: " +  event.getSubscriptionID() + " Patient "+patientID+" EventHCID: " + event.getHCID()+ " With the following errors: " + errors.toString());
            }
        }
    }

    public void setSubscriptionProvider(SubscriptionProvider subscriptionProvider)
    {
        this.subscriptionProvider = subscriptionProvider;
    }

    public void setDocumentReferenceProvider(DocumentReferenceProvider documentReferenceProvider)
    {
        this.documentReferenceProvider = documentReferenceProvider;
    }

    public void setPatientAnnouncer(PatientAnnouncer patientAnnouncer)
    {
        this.patientAnnouncer = patientAnnouncer;
    }

    public void setDocumentQueryer(DocumentQueryer documentQueryer)
    {
        this.documentQueryer = documentQueryer;
    }

    public void setNhioEvent(Event<NHIOEvent> nhioEvent)
    {
        this.nhioEvent = nhioEvent;
    }

    public void setDocumentProcessor(DocumentProcessor documentProcessor)
    {
        this.documentProcessor = documentProcessor;
    }
}
