package gov.va.med.nhin.adapter.xmlutils;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.xml.sax.*;
import org.w3c.dom.*;
//import org.apache.soap.Constants;

public class XMLParserUtils {

    private static DocumentBuilderFactory dbf = null;
    private static List allocatedBuildersList = new ArrayList(10);

    static {
        // Create a default instance.
        refreshDocumentBuilderFactory(null, true, false, false);
    }

    /**
     * Causes the private DocumentBuilderFactory reference to point
     * to a new instance of a DocumentBuilderFactory. This method
     * only needs to be called if you want to specify a different
     * DocumentBuilderFactory implementation then that specified
     * prior to this class being initialized. Or, if you want to
     * specify different configuration options.
     *
     * @param factoryClassName the fully-qualified name of a class
     * that implemements DocumentBuilderFactory. If this argument
     * is null, the default (platform-specific) implementation is
     * used. Basically, if this argument is not null, the
     * javax.xml.parsers.DocumentBuilderFactory system property
     * is set (with the specified value) before the
     * DocumentBuilderFactory.newInstance() method is invoked.
     * @param namespaceAware configure the new DocumentBuilderFactory
     * to produce namespace aware parsers (i.e. DocumentBuilders)
     * @param validating configure the new DocumentBuilderFactory to
     * produce validating parsers (i.e. DocumentBuilders)
     */
    synchronized public static void refreshDocumentBuilderFactory(
            String factoryClassName,
            boolean namespaceAware,
            boolean validating) {
        refreshDocumentBuilderFactory(factoryClassName,
                namespaceAware,
                validating,
                false);
    }

    /**
     * Causes the private DocumentBuilderFactory reference to point
     * to a new instance of a DocumentBuilderFactory. This method
     * only needs to be called if you want to specify a different
     * DocumentBuilderFactory implementation then that specified
     * prior to this class being initialized. Or, if you want to
     * specify different configuration options.
     *
     * @param factoryClassName the fully-qualified name of a class
     * that implemements DocumentBuilderFactory. If this argument
     * is null, the default (platform-specific) implementation is
     * used. Basically, if this argument is not null, the
     * javax.xml.parsers.DocumentBuilderFactory system property
     * is set (with the specified value) before the
     * DocumentBuilderFactory.newInstance() method is invoked.
     * @param namespaceAware configure the new DocumentBuilderFactory
     * to produce namespace aware parsers (i.e. DocumentBuilders)
     * @param validating configure the new DocumentBuilderFactory to
     * produce validating parsers (i.e. DocumentBuilders)
     * @param expandEntityReferences configure the new DocumentBuilderFactory
     * to produce parsers that expand entity references
     */
    synchronized public static void refreshDocumentBuilderFactory(
            String factoryClassName,
            boolean namespaceAware,
            boolean validating,
            boolean expandEntityReferences) {
        if (factoryClassName != null) {
            System.setProperty("javax.xml.parsers.DocumentBuilderFactory",
                    factoryClassName);
        }

        // Step 1: create a DocumentBuilderFactory and configure it
        dbf = DocumentBuilderFactory.newInstance();

        // Optional: set various configuration options
        dbf.setNamespaceAware(namespaceAware);
        dbf.setValidating(validating);

        // Add various options explicitly to prevent XXE attacks. add try/catch around every
        // setAttribute just in case a specific parser does not support it.
        dbf.setExpandEntityReferences(expandEntityReferences);
        try {
            dbf.setAttribute("http://xml.org/sax/features/external-general-entities",
                    Boolean.FALSE);
        } catch (Throwable t) { }
        try {
            dbf.setAttribute("http://xml.org/sax/features/external-parameter-entities",
                    Boolean.FALSE);
        } catch (Throwable t) { }
        try {
            dbf.setAttribute("http://apache.org/xml/features/disallow-doctype-decl",
                    Boolean.TRUE);
        } catch (Throwable t) { }
        try {
            dbf.setAttribute("http://javax.xml.XMLConstants/feature/secure-processing",
                    Boolean.TRUE);
        } catch (Throwable t) { }
        try {
            dbf.setAttribute("http://apache.org/xml/features/nonvalidating/load-external-dtd",
                    Boolean.FALSE);
        } catch (Throwable t) { }

        try {
            // Some parsers don't throw an exception here, but throw one when the
            // factory creates an instance instead, so try to only do this for
            // Xerces.
            if (dbf.getClass().getName().equals("org.apache.xerces.jaxp.DocumentBuilderFactory")) {
                // speed up processing by turning off deferred node expansion
                dbf.setAttribute("http://apache.org/xml/features/dom/defer-node-expansion",
                        Boolean.FALSE);
            }
        } catch (IllegalArgumentException e) {
            // parsers that do not support this option *should* throw this exception
        }

        try {
            // Some parsers don't throw an exception here, but throw one when the
            // factory creates an instance instead, so try to only do this for
            // Xerces.
            if (dbf.getClass().getName().equals("org.apache.xerces.jaxp.DocumentBuilderFactory")) {
                // future: protect against DOS attacks through DOCTYPE processing
                dbf.setAttribute("http://apache.org/xml/features/disallow-doctype-decl",
                        Boolean.TRUE);
            }
        } catch (IllegalArgumentException e) {
            // parsers that do not support this option *should* throw this exception
        }

    /*
      At this point the DocumentBuilderFactory instance can be saved
      and reused to create any number of DocumentBuilder instances
      with the same configuration options.
    */
    }

    /**
     * Use this method to get a JAXP document builder.
     * This method creates a namespace aware, nonvalidating
     * instance of the XML parser.
     *
     * @return DocumentBuilder an instance of a document builder,
     * or null if a ParserConfigurationException was thrown.
     */
    synchronized public static DocumentBuilder getXMLDocBuilder()
            throws IllegalArgumentException {
        try {
            return dbf.newDocumentBuilder();
        } catch (ParserConfigurationException pce) {
            throw new IllegalArgumentException(pce.toString());
        }
    }


    /**
     * Use this method to get a JAXP document builder.
     * This method either returns a previosly allocated DocumentBuilder
     * or creates a namespace aware, nonvalidating
     * instance of the XML parser.
     *
     * @return DocumentBuilder an instance of a document builder,
     * or null if a ParserConfigurationException was thrown.
     */
    synchronized public static DocumentBuilder getXMLDocBuilderFromPool() {
        // First check if we have DocBuider available
        DocumentBuilder  builder = null;
        int size = allocatedBuildersList.size();
        if ( size > 0) {
            builder = (DocumentBuilder)allocatedBuildersList.get(size-1);
            allocatedBuildersList.remove(size-1);
        } else
            // Not available - create a new DocumentBuilder
            builder = XMLParserUtils.getXMLDocBuilder();

        return builder;
    }

    /**
     * Return a JAXP document builder, previosly allocated by method
     * getXMLDocBuilderFromPool() for further reuse
     *
     * @param builder a DocumentBuilder to release
     */
    synchronized public static void returnDocumentBuilderToPool(DocumentBuilder builder) {
        if (builder != null)
            allocatedBuildersList.add(builder);
    }

    /**
     * Gets a new XML DOM.
     *
     * @return The new XML DOM.
     */
    public static Document newDocument() {
        DocumentBuilder xdb = getXMLDocBuilderFromPool();
        Document doc = null;
        try {
            doc = xdb.newDocument();
        } finally {
            XMLParserUtils.returnDocumentBuilderToPool(xdb);
        }
        return doc;
    }

    /**
     * Parses XML text from a Reader into an XML DOM.
     *
     * @param xmlReader XML reader.
     * @return The XML in DOM form.
     * @exception IOException For errors reading the XML.
     * @exception SAXException For errors parsing the XML.
     */
    public static Document parse(Reader xmlReader)
            throws IOException, SAXException {
        return parse(new InputSource(xmlReader), true);
    }

    /**
     * Parses XML text into an XML DOM.
     *
     * @param xmlText XML text.
     * @return The XML in DOM form.
     * @exception IOException For errors reading the XML.
     * @exception SAXException For errors parsing the XML.
     */
    public static Document parse(String xmlText)
            throws IOException, SAXException {
        return parse(new InputSource(xmlText), true);
    }

    /**
     * Parses XML text into an XML DOM.
     *
     * @param xmlText XML text.
     * @param useDocumentBuilder True to use a document builder, false to use
     *                           a transform.  If the transform classes cannot
     *                           be loaded, a document builder will be used.
     * @return The XML in DOM form.
     * @exception IOException For errors reading the XML.
     * @exception SAXException For errors parsing the XML.
     */
    public static Document parse(String xmlText, boolean useDocumentBuilder)
            throws IOException, SAXException {
        return parse(new InputSource(new StringReader(xmlText)), useDocumentBuilder);
    }

    /**
     * Parses an XML source into an XML DOM.
     *
     * @param xmlSource The source of the XML.
     * @param useDocumentBuilder True to use a document builder, false to use
     *                           a transform.  If the transform classes cannot
     *                           be loaded, a document builder will be used.
     * @return The XML in DOM form.
     * @exception IOException For errors reading the XML.
     * @exception SAXException For errors parsing the XML.
     */
    public static Document parse(InputSource xmlSource, boolean useDocumentBuilder)
            throws IOException, SAXException {
        if (!useDocumentBuilder) {
            try {
                // If classes required for transformation are all present, use that method
                Class domResultClass = Class.forName("javax.xml.transform.dom.DOMResult");
                Class transformerFactoryClass = Class.forName("javax.xml.transform.TransformerFactory");
                Class transformerClass = Class.forName("javax.xml.transform.Transformer");
                Class saxSourceClass = Class.forName("javax.xml.transform.sax.SAXSource");
                Class sourceClass = Class.forName("javax.xml.transform.Source");
                Class resultClass = Class.forName("javax.xml.transform.Result");

                Object output = domResultClass.newInstance();
                Object source = saxSourceClass.newInstance();

                // Create factory and use to create transformer
                Method method = transformerFactoryClass.getDeclaredMethod("newInstance", new Class[]{});
                Object xformFactory = method.invoke(null, new Object[]{});
                method = transformerFactoryClass.getDeclaredMethod("newTransformer", new Class[]{});
                Object idTransform = method.invoke(xformFactory, new Object[]{});

                // Set input source for SAX source
                method = saxSourceClass.getDeclaredMethod("setInputSource", new Class[]{InputSource.class});
                method.invoke(source, new Object[]{xmlSource});

                // Transform from SAX source to DOM result
                method = transformerClass.getDeclaredMethod("transform", new Class[]{sourceClass, resultClass});
                method.invoke(idTransform, new Object[]{source, output});

                // Grab document node from DOM result
                method = domResultClass.getDeclaredMethod("getNode", new Class[]{});

                // If all has worked, we return here; for exceptions,
                // we fall through to the DOM parser.
                return (Document) method.invoke(output, new Object[]{});
            } catch (ClassNotFoundException e) {
            } catch (NoSuchMethodException e) {
            } catch (InvocationTargetException e) {
            } catch (InstantiationException e) {
            } catch (IllegalAccessException e) {
            }
        }

        DocumentBuilder xdb = getXMLDocBuilderFromPool();
        Document doc = null;
        try {
            doc = xdb.parse(xmlSource);
        } finally {
            returnDocumentBuilderToPool(xdb);
        }
        return doc;
    }

}

