package gov.va.med.nhin.adapter.utils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Format utility for patient identifiers.
 * 
 * @author Neil Webb
 */
public class PatientIdFormatUtil
{
	// CCR 177986 -Logger updates
	private static final Logger logger = LoggerFactory.getLogger(PatientIdFormatUtil.class);

	/**
	 * Parse an optionally HL7 encoded patient identifier. If the patient
	 * identifier is not HL7 encoded, the original id will be returned. The
	 * format of an HL7 encoded patient id is "<id>^^^&<home coummunity id>&ISO"
	 * 
	 * @param receivedPatientId Optionally HL7 encoded patient identifier
	 * @return Parsed patient id
	 */
	public static String parsePatientId(String receivedPatientId)
	{
		logger.debug("Parsing patient id: " + receivedPatientId);
		String patientId = receivedPatientId;
		if((patientId != null) && (patientId.length() > 0))
		{
			// In some cases we see a quote - in others we do not. So lets strip
			// them off if we see them.
			// ---------------------------------------------------------------------------------------------
			if((patientId.startsWith("'")) && (patientId.length() > 1))
			{
				StringBuffer sbPatientId = new StringBuffer(patientId);
				if(patientId.endsWith("'"))
				{
					// strip off the ending quote
					sbPatientId.deleteCharAt(sbPatientId.length() - 1); 
				}
				
				// strip off the first char quote
				sbPatientId.deleteCharAt(0); 

				patientId = sbPatientId.toString();
			}

			int componentIndex = patientId.indexOf("^");
			logger.debug("Index: " + componentIndex);
			
			if(componentIndex != -1)
			{
				patientId = patientId.substring(0, componentIndex);
				logger.debug("Parsed patient id: " + patientId);
			}
		}
		
		return patientId;
	}

	/**
	 * Parse an optionally HL7 encoded community id. If the patient identifier
	 * is not HL7 encoded, null will be returned. The format of an HL7 encoded
	 * patient id is "<id>^^^&<home coummunity id>&ISO"
	 * 
	 * @param encodedPatientId Optionally HL7 encoded patient identifier
	 * @return Parsed community id
	 */
	public static String parseCommunityId(String encodedPatientId)
	{
		logger.debug("Parsing community id: " + encodedPatientId);
		String communityId = null;
		if((encodedPatientId != null) && (encodedPatientId.length() > 0))
		{
			String workingCommunityId = encodedPatientId;
			// In some cases we see a quote - in others we do not. So lets strip
			// them off if we see them.
			// ---------------------------------------------------------------------------------------------
			if((workingCommunityId.startsWith("'")) && (workingCommunityId.length() > 1))
			{
				StringBuffer sbCommunityId = new StringBuffer(workingCommunityId);
				if(workingCommunityId.endsWith("'"))
				{
					// strip off the ending quote
					sbCommunityId.deleteCharAt(sbCommunityId.length() - 1); 
				}
				// strip off the first char quote
				sbCommunityId.deleteCharAt(0); 

				workingCommunityId = sbCommunityId.toString();
			}

			// First remove the first components
			int componentIndex = workingCommunityId.lastIndexOf("^");
			logger.debug("Index: " + componentIndex);
			if((componentIndex != -1) && (workingCommunityId.length() > (componentIndex + 1)))
			{
				workingCommunityId = workingCommunityId.substring(componentIndex + 1);
				logger.debug("Working community id after first components removed: " + workingCommunityId);

				if(workingCommunityId.startsWith("&"))
				{
					workingCommunityId = workingCommunityId.substring(1);
				}
				int subComponentIndex = workingCommunityId.indexOf("&");
				if(subComponentIndex != -1)
				{
					workingCommunityId = workingCommunityId.substring(0, subComponentIndex);
				}
				communityId = workingCommunityId;
			}
		}
		return communityId;
	}

	/**
	 * HL7 encode a patient identifier. The resulting format will be:
	 * "<id>^^^&<home coummunity id>&ISO"
	 * 
	 * @param patientId Patient identifier
	 * @param homeCommunityId Home community id
	 * @return HL7 encoded patient id
	 */
	public static String hl7EncodePatientId(String patientId, String homeCommunityId)
	{
		// Sometimes the homeCommunityId is prepended with "urn:oid:" for
		// various reasons. We do not
		// want that included when putting together the Patient ID. If it is
		// there, we need to
		// strip it off.
		// ---------------------------------------------------------------------------------------------
		String sLocalHomeCommunityId = homeCommunityId;
		if(homeCommunityId.startsWith("urn:oid:"))
		{
			sLocalHomeCommunityId = sLocalHomeCommunityId.substring("urn:oid:".length());
		}
		String encodedPatientId = null;
		logger.debug("Creating HL7 encoded patient id for patient id: " + patientId + ", home community id: " + sLocalHomeCommunityId);
		if(patientId != null)
		{
			encodedPatientId = "'" + patientId + "^^^&" + sLocalHomeCommunityId + "&ISO" + "'";
			logger.debug("HL7 encoded patient id: " + encodedPatientId);
		}
		return encodedPatientId;
	}
}
