package gov.va.med.nhin.adapter.mvi.hl7parsers;

import java.io.Serializable;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.bind.JAXBElement;

import org.hl7.v3.AdxpExplicitCity;
import org.hl7.v3.AdxpExplicitCountry;
import org.hl7.v3.AdxpExplicitPostalCode;
import org.hl7.v3.AdxpExplicitState;
import org.hl7.v3.AdxpExplicitStreetAddressLine;
import org.hl7.v3.EnExplicitFamily;
import org.hl7.v3.EnExplicitGiven;
import org.hl7.v3.EnExplicitPrefix;
import org.hl7.v3.EnExplicitSuffix;
import org.hl7.v3.II;
import org.hl7.v3.PNExplicit;
import org.hl7.v3.PRPAIN201306UV02;
import org.hl7.v3.PRPAMT201301UV02Person;
import org.hl7.v3.PRPAMT201306UV02LivingSubjectName;
import org.hl7.v3.PRPAMT201306UV02PatientAddress;
import org.hl7.v3.PRPAMT201310UV02Patient;
import org.hl7.v3.PRPAMT201310UV02Person;
import org.hl7.v3.TELExplicit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.nhin.adapter.utils.NullChecker;

/**
 *
 * @author SRA International, Inc.
 */
public class HL7Parser201306
{
	private static final Logger logger = LoggerFactory.getLogger(HL7Parser201306.class.getName());

	public static boolean isValidICNValue(String icn)
	{
		return extractICNValue(icn) != null;
	}

	public static String extractICNValue(String icn)
	{
		String newICN = null;
		Pattern p = Pattern.compile("\\d{10}V\\d{6}");
		Matcher m = p.matcher(icn);

		if(m.find())
		{
			newICN = m.group();
		}

		return newICN;
	}

	public static II extractICNId(PRPAIN201306UV02 result1306)
	{
		II icn = null;
		List<II> ids = extractPatientIds(result1306);
		Object[] result = HL7Parser201306.extractICNArray(ids);
		if(result != null)
		{
			icn = (II) result[1];
		}
		return icn;
	}

	public static String extractICN(PRPAIN201306UV02 result1306)
	{
		String icn = null;
		List<II> ids = extractPatientIds(result1306);
		Object[] result = HL7Parser201306.extractICNArray(ids);
		if(result != null)
		{
			icn = (String) result[0];
		}
		return icn;
	}

	public static II extractICNId(List<II> ids)
	{
		II icn = null;
		Object[] result = HL7Parser201306.extractICNArray(ids);
		if(result != null)
		{
			icn = (II) result[1];
			logger.debug("extractICNId: ICN: {}:{}", new Object[] { icn.getRoot(), icn.getExtension() });
		}
		else
		{
			logger.debug("no ICN extracted");
		}
		return icn;
	}

	public static II extractMatchingId(PRPAIN201306UV02 result1306, String idValue, String idType, String facilityNumber, String aa)
	{
		II match = null;
		List<II> ids = extractPatientIds(result1306);
		if(NullChecker.isNotNullOrEmpty(ids))
		{
			Object[] result = extractMatchingIdArray(ids, idValue, idType, facilityNumber, aa);
			if(result != null)
			{
				match = (II) result[1];
			}
		}
		return match;
	}

	private static Object[] extractMatchingIdArray(List<II> ids, String idValue, String idType, String facilityNumber, String aa)
	{
		Object[] result = null;

		for(II id : ids)
		{
			logger.debug("extractMatchingIdArray: ID: {}:{}", new Object[] { id.getRoot(), id.getExtension() });

			String extension = id.getExtension();
			String[] pieces = extension.split("\\^", -1);
			logger.debug("pieces: {}:{}", new Object[] { pieces.length, Arrays.toString(pieces) });
			if(
                                pieces.length >= 4 
                                && (idValue == null || pieces[0].equals(idValue)) 
                                && pieces[1].equals(idType) 
                                && pieces[2].equals(facilityNumber) 
                                && pieces[3].equals(aa))
			{
				result = new Object[2];
				result[0] = pieces[0]; // this is just the ICN value.
				result[1] = id; // this is the II object with fully qualified
								// ICN
				break;
			}
		}
		return result;
	}

	/*
	 * @return - 0 index is the ICN String 1 index is the II object representing
	 * the ICN.
	 */
	private static Object[] extractICNArray(List<II> ids)
	{

		Object[] result = extractMatchingIdArray(ids, null, "NI", "200M", "USVHA");
		/*
		 * for (II id : ids) { String extension = id.getExtension(); String []
		 * pieces = extension.split("\\^", -1); if (pieces.length >= 4 &&
		 * pieces[1].equals("NI") && pieces[2].equals("200M") &&
		 * pieces[3].equals("USVHA")) { result = new Object[2]; result[0] =
		 * pieces[0]; //this is just the ICN value. result[1] = id; //this is
		 * the II object with fully qualified ICN break; } }
		 */

		return result;
	}

	public static List<II> extractPatientIds(PRPAIN201306UV02 result1306)
	{
		List<II> ids = null;

		if(result1306.getControlActProcess() != null || NullChecker.isNotNullOrEmpty(result1306.getControlActProcess().getSubject()) || result1306.getControlActProcess().getSubject().get(0).getRegistrationEvent() != null || result1306.getControlActProcess().getSubject().get(0).getRegistrationEvent().getSubject1() != null || result1306.getControlActProcess().getSubject().get(0).getRegistrationEvent().getSubject1().getPatient() != null || result1306.getControlActProcess().getSubject().get(0).getRegistrationEvent().getSubject1().getPatient().getId() != null)
		{
			ids = result1306.getControlActProcess().getSubject().get(0).getRegistrationEvent().getSubject1().getPatient().getId();
		}
		return ids;
	}

	public static String extractPatientGender(PRPAIN201306UV02 result1306)
	{
		String gender = null;
		PRPAMT201310UV02Person patientPerson = extractPatientPerson(result1306);

		if(patientPerson != null && patientPerson.getAdministrativeGenderCode() != null && patientPerson.getAdministrativeGenderCode().getCode() != null)
		{
			gender = patientPerson.getAdministrativeGenderCode().getCode();
		}

		return gender;
	}

	public static String extractSSN(PRPAIN201306UV02 result1306)
	{
		String ssn = null;
		List<II> ids = result1306.getControlActProcess().getSubject().get(0).getRegistrationEvent().getSubject1().getPatient().getId();

		for(II id : ids)
		{
			String extension = id.getExtension();
			String[] pieces = extension.split("\\^", -1);
			if(pieces.length >= 2)
			{
				if(pieces[1].equals("SS"))
				{
					ssn = pieces[0];
					break;
				}
			}
		}

		return ssn;
	}

	public static String extractBirthTime(PRPAIN201306UV02 result1306)
	{
		String birthTime = null;
		PRPAMT201310UV02Person patientPerson = extractPatientPerson(result1306);

		if(patientPerson != null && patientPerson.getBirthTime() != null && patientPerson.getBirthTime().getValue() != null)
		{
			birthTime = patientPerson.getBirthTime().getValue();
		}

		return birthTime;
	}

	public static String extractMultipleBirthInd(PRPAIN201306UV02 result1306)
	{
		String multipleBirthInd = null;
		PRPAMT201310UV02Person patientPerson = extractPatientPerson(result1306);

		if(patientPerson != null && patientPerson.getMultipleBirthInd() != null && patientPerson.getMultipleBirthInd().isValue() != null)
		{
			multipleBirthInd = patientPerson.getMultipleBirthInd().isValue().toString();
		}

		return multipleBirthInd;
	}

	public static String extractTelecom(PRPAIN201306UV02 result1306, String use)
	{
		String telecom = null;
		PRPAMT201310UV02Person patientPerson = extractPatientPerson(result1306);

		if(patientPerson != null && patientPerson.getTelecom() != null && patientPerson.getTelecom().size() > 0 && patientPerson.getTelecom().get(0) != null && patientPerson.getTelecom().get(0).getValue() != null)
		{
			Iterator<TELExplicit> it = patientPerson.getTelecom().iterator();

			while(it.hasNext())
			{
				TELExplicit tel = it.next();
				if(tel.getUse() != null && tel.getUse().get(0) != null && use.equalsIgnoreCase(tel.getUse().get(0)))
				{
					telecom = tel.getValue();
					break;
				}
			}
		}

		return telecom;
	}

	public static PRPAMT201310UV02Patient extractSubjectPatient(PRPAIN201306UV02 result1306)
	{
		PRPAMT201310UV02Patient subjectPatient = null;

		if(result1306 != null && result1306.getControlActProcess() != null && result1306.getControlActProcess().getSubject() != null && result1306.getControlActProcess().getSubject().size() > 0 && result1306.getControlActProcess().getSubject().get(0) != null && result1306.getControlActProcess().getSubject().get(0).getRegistrationEvent() != null && result1306.getControlActProcess().getSubject().get(0).getRegistrationEvent().getSubject1() != null && result1306.getControlActProcess().getSubject().get(0).getRegistrationEvent().getSubject1().getPatient() != null)
		{
			subjectPatient = result1306.getControlActProcess().getSubject().get(0).getRegistrationEvent().getSubject1().getPatient();
		}

		return subjectPatient;
	}

	public static PRPAMT201310UV02Person extractPatientPerson(PRPAIN201306UV02 result1306)
	{
		PRPAMT201310UV02Person patientPerson = null;

		PRPAMT201310UV02Patient subjectPatient = extractSubjectPatient(result1306);

		if(subjectPatient != null && subjectPatient.getPatientPerson() != null && subjectPatient.getPatientPerson().getValue() != null)
		{
			patientPerson = subjectPatient.getPatientPerson().getValue();
		}

		return patientPerson;
	}

	public static String extractMaritalStatusCode(PRPAIN201306UV02 result1306)
	{
		String maritalStatusCode = null;
		PRPAMT201310UV02Person patientPerson = extractPatientPerson(result1306);

		if(patientPerson != null && patientPerson.getMaritalStatusCode() != null && patientPerson.getMaritalStatusCode().getCode() != null)
		{
			maritalStatusCode = patientPerson.getMaritalStatusCode().getCode();
		}

		return maritalStatusCode;
	}

	/*
	 * @param PRPAIN201306UV02 - result of MVI person search
	 * 
	 * @return Map<String, Object> the demographics data
	 */
	public static Map<String, Object> extractDemographics(PRPAIN201306UV02 result1306)
	{
		Map demographics = extractPatientPersonNames(result1306);
		demographics.put("SSN", extractSSN(result1306));
		demographics.put("genderCode", extractPatientGender(result1306));
		demographics.put("birthDate", extractBirthTime(result1306));
		demographics.put("multipleBirthCode", extractMultipleBirthInd(result1306));
		demographics.put("mothersMaidenName", extractMothersMaidenName(result1306));
		demographics.put("phoneHome", extractTelecom(result1306, "HP"));
		demographics.put("maritalStatusCode", extractMaritalStatusCode(result1306));
		Map homeAddress = extractHomeAddressMap(result1306);
		if(homeAddress != null)
		{
			demographics.putAll(homeAddress);
		}
		Map pobAddress = extractPlaceOfBirth(result1306);
		if(pobAddress != null)
		{
			demographics.putAll(pobAddress);
		}
		/*
		 * SR345 - for the RPC to MVI web service migration, not sure what these
		 * two fields should map to. AnnouncePatient is the only place that uses
		 * it and it doesn't seem to impact Adapter otherwise.
		 **/
		// String patientId = (String) demographics.get("MRN"); ????
		/**
		 * String homeCommunityId = getHomeFacility().getHomeCommunityId();
		 */
		demographics.put("MRN", extractICN(result1306));

		return demographics;
	}

	public static Map extractAddress(List<Serializable> choice, String prefix)
	{
		Map<String, String> addressMap = new HashMap<String, String>();

		Iterator<Serializable> iterSerialObjects = choice.iterator();

		while(iterSerialObjects.hasNext())
		{
			Serializable contentItem = iterSerialObjects.next();

			if(contentItem instanceof JAXBElement)
			{
				JAXBElement oJAXBElement = (JAXBElement) contentItem;

				if(oJAXBElement.getValue() instanceof AdxpExplicitCity)
				{
					AdxpExplicitCity city = (AdxpExplicitCity) oJAXBElement.getValue();
					if(city != null)
					{
						addressMap.put(prefix + "City", city.getContent());
					}
				}
				else if(oJAXBElement.getValue() instanceof AdxpExplicitState)
				{
					AdxpExplicitState state = (AdxpExplicitState) oJAXBElement.getValue();
					if(state != null)
					{
						addressMap.put(prefix + "State", state.getContent());
					}
				}
				else if(oJAXBElement.getValue() instanceof AdxpExplicitCountry)
				{
					AdxpExplicitCountry country = (AdxpExplicitCountry) oJAXBElement.getValue();
					if(country != null)
					{
						addressMap.put(prefix + "Country", country.getContent());
					}
				}
				else if(oJAXBElement.getValue() instanceof AdxpExplicitPostalCode)
				{
					AdxpExplicitPostalCode postalCode = (AdxpExplicitPostalCode) oJAXBElement.getValue();
					if(postalCode != null)
					{
						addressMap.put(prefix + "Postal", postalCode.getContent());
					}
				}
				else if(oJAXBElement.getValue() instanceof AdxpExplicitStreetAddressLine)
				{

					AdxpExplicitStreetAddressLine streetAddress = (AdxpExplicitStreetAddressLine) oJAXBElement.getValue();
					if(streetAddress != null)
					{
						addressMap.put(prefix + "Street", streetAddress.getContent());
					}
				}
			}
		}
		return addressMap;
	}

	public static Map extractHomeAddressMap(List<Serializable> choice)
	{
		return extractAddress(choice, "homeAddress");
	}

	public static Map extractHomeAddressMap(PRPAIN201306UV02 result1306)
	{
		Map<String, String> addressMap = null;
		PRPAMT201310UV02Person patientPerson = extractPatientPerson(result1306);

		if(patientPerson != null && patientPerson.getAddr() != null && patientPerson.getAddr().size() > 0 && patientPerson.getAddr().get(0) != null && patientPerson.getAddr().get(0).getContent() != null)
		{
			addressMap = extractAddress(patientPerson.getAddr().get(0).getContent(), "homeAddress");
		}

		return addressMap;
	}

	public static Map<String, String> extractPatientPersonNames(PRPAMT201301UV02Person person)
	{
		return parseNames(person.getName().get(0).getContent());
	}

	public static Map<String, String> extractPatientPersonNames(PRPAIN201306UV02 result1306)
	{
		List<PNExplicit> pnName = result1306.getControlActProcess().getSubject().get(0).getRegistrationEvent().getSubject1().getPatient().getPatientPerson().getValue().getName();

		return parseNames(pnName.get(0).getContent());
	}

	public static Map<String, String> parseNames(List<Serializable> choice)
	{
		HashMap nameMap = new HashMap();
		Iterator<Serializable> iterSerialObjects = choice.iterator();

		while(iterSerialObjects.hasNext())
		{
			Serializable contentItem = iterSerialObjects.next();

			if(contentItem instanceof JAXBElement)
			{
				JAXBElement oJAXBElement = (JAXBElement) contentItem;

				if(oJAXBElement.getValue() instanceof EnExplicitPrefix)
				{
					EnExplicitPrefix prefixName = (EnExplicitPrefix) oJAXBElement.getValue();
					nameMap.put("namePrefix", prefixName.getContent());
				}
				else if(oJAXBElement.getValue() instanceof EnExplicitGiven)
				{
					EnExplicitGiven givenName = (EnExplicitGiven) oJAXBElement.getValue();
					if(nameMap.get("nameGiven") == null)
					{ // then set First Name
						nameMap.put("nameGiven", givenName.getContent());
					}
					else
					{ // set Middle Name
						nameMap.put("nameMiddle", givenName.getContent());
					}
				}
				else if(oJAXBElement.getValue() instanceof EnExplicitFamily)
				{
					EnExplicitFamily familyName = (EnExplicitFamily) oJAXBElement.getValue();
					nameMap.put("nameFamily", familyName.getContent());
				}
				else if(oJAXBElement.getValue() instanceof EnExplicitSuffix)
				{
					EnExplicitSuffix suffixName = (EnExplicitSuffix) oJAXBElement.getValue();
					nameMap.put("nameSuffix", suffixName.getContent());
				}
			}
		}
		return nameMap;
	}

	public static String extractMothersMaidenName(PRPAMT201310UV02Person person)
	{
		String result = null;

		if(person != null && NullChecker.isNotNullOrEmpty(person.getName()))
		{
			PNExplicit mmn = extractPatientPersonName(person.getName(), "C");
			if(mmn != null)
			{
				Map<String, String> nameMap = HL7Parser201306.parseNames(mmn.getContent());
				if(nameMap != null)
				{
					result = nameMap.get("nameFamily");
				}
			}
		}
		return result;
	}

	public static String extractMothersMaidenName(PRPAIN201306UV02 result1306)
	{
		return extractMothersMaidenName(HL7Parser201306.extractPatientPerson(result1306));
	}

	/*
	 * public static PRPAMT201306UV02QueryByParameter
	 * getQueryByParameter(PRPAIN201306UV02 result1306) {
	 * PRPAMT201306UV02QueryByParameter ret = null;
	 * 
	 * if (result1306 != null && result1306.getControlActProcess() != null &&
	 * result1306.getControlActProcess().getQueryByParameter() != null &&
	 * result1306.getControlActProcess().getQueryByParameter().getValue() !=
	 * null) { ret =
	 * result1306.getControlActProcess().getQueryByParameter().getValue(); }
	 * return ret; }
	 */

	public static PNExplicit extractPatientPersonName(List<PNExplicit> names, String useCodeParam)
	{

		PNExplicit result = null;

		for(PNExplicit name : names)
		{
			String useCode = null;
			if(NullChecker.isNotNullOrEmpty(name.getUse()) && NullChecker.isNotNullOrEmpty(name.getUse().get(0)))
			{
				useCode = name.getUse().get(0);
			}

			// if useCode passed in request is null then assume implied intent
			// was "L" for Legal Name.
			if((useCodeParam.equals("L") && useCode == null) || (useCode != null && useCode.equals(useCodeParam)))
			{
				result = name;
				break;
			}
		}
		return result;
	}

	public static Map extractPlaceOfBirth(List<Serializable> choice)
	{
		return extractAddress(choice, "pob");
	}

	public static Map extractPlaceOfBirth(PRPAIN201306UV02 result1306)
	{

		Map result = null;
		PRPAMT201310UV02Person pp = extractPatientPerson(result1306);

		if(pp != null && pp.getBirthPlace() != null && pp.getBirthPlace().getValue() != null && pp.getBirthPlace().getValue().getAddr() != null && NullChecker.isNotNullOrEmpty(pp.getBirthPlace().getValue().getAddr().getContent()))
		{
			result = extractPlaceOfBirth(pp.getBirthPlace().getValue().getAddr().getContent());
		}
		return result;
	}

	public static PRPAMT201306UV02PatientAddress extractPatientAddress(List<PRPAMT201306UV02PatientAddress> addresses, String useCodeParam)
	{

		PRPAMT201306UV02PatientAddress result = null;

		for(PRPAMT201306UV02PatientAddress address : addresses)
		{
			String useCode = null;
			if(address != null && address.getValue() != null && address.getValue().size() > 0 && address.getValue().get(0) != null && address.getValue().get(0).getUse() != null && address.getValue().get(0).getUse().size() > 0)
			{
				useCode = address.getValue().get(0).getUse().get(0);
			}

			// if useCode passed in request is null then assume implied intent
			// was "PHYS" for Physical Address.
			if((useCodeParam.equals("PHYS") && useCode == null) || (useCode != null && useCode.equals(useCodeParam)))
			{
				result = address;
				break;
			}
		}
		return result;
	}

	public static PRPAMT201306UV02LivingSubjectName extractSubjectName(List<PRPAMT201306UV02LivingSubjectName> names, String useCodeParam)
	{

		PRPAMT201306UV02LivingSubjectName result = null;

		for(PRPAMT201306UV02LivingSubjectName name : names)
		{
			String useCode = null;
			if(name.getValue() != null && name.getValue().size() > 0 && name.getValue().get(0) != null && name.getValue().get(0).getUse() != null && name.getValue().get(0).getUse().size() > 0)
			{
				useCode = name.getValue().get(0).getUse().get(0);
			}

			// if useCode passed in request is null then assume implied intent
			// was "L" for Legal Name.
			if((useCodeParam.equals("L") && useCode == null) || (useCode != null && useCode.equals(useCodeParam)))
			{
				result = name;
				break;
			}
		}
		return result;
	}

}