package gov.va.med.nhin.adapter.mvi.hl7parsers;

import java.io.Serializable;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.xml.bind.JAXBElement;

import org.hl7.v3.ADExplicit;
import org.hl7.v3.AdxpExplicitCity;
import org.hl7.v3.AdxpExplicitPostalCode;
import org.hl7.v3.AdxpExplicitState;
import org.hl7.v3.AdxpExplicitStreetAddressLine;
import org.hl7.v3.CE;
import org.hl7.v3.EnExplicitFamily;
import org.hl7.v3.EnExplicitGiven;
import org.hl7.v3.EnExplicitPrefix;
import org.hl7.v3.EnExplicitSuffix;
import org.hl7.v3.II;
import org.hl7.v3.IVLTSExplicit;
import org.hl7.v3.PRPAIN201305UV02;
import org.hl7.v3.PRPAIN201305UV02QUQIMT021001UV01ControlActProcess;
import org.hl7.v3.PRPAMT201306UV02LivingSubjectAdministrativeGender;
import org.hl7.v3.PRPAMT201306UV02LivingSubjectBirthTime;
import org.hl7.v3.PRPAMT201306UV02LivingSubjectId;
import org.hl7.v3.PRPAMT201306UV02LivingSubjectName;
import org.hl7.v3.PRPAMT201306UV02ParameterList;
import org.hl7.v3.PRPAMT201306UV02PatientAddress;
import org.hl7.v3.PRPAMT201306UV02PatientTelecom;
import org.hl7.v3.PRPAMT201306UV02QueryByParameter;
import org.hl7.v3.TELExplicit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.hhs.fha.nhinc.patientdb.model.Address;
import gov.hhs.fha.nhinc.patientdb.model.Identifier;
import gov.hhs.fha.nhinc.patientdb.model.Patient;
import gov.hhs.fha.nhinc.patientdb.model.Personname;
import gov.hhs.fha.nhinc.patientdb.model.Phonenumber;
import gov.hhs.fha.nhinc.util.format.UTCDateUtil;
import gov.va.med.nhin.adapter.datamanager.ParseErrorException;
import gov.va.med.nhin.adapter.utils.NullChecker;
import java.util.Map;
import org.hl7.v3.PRPAMT201306UV02LivingSubjectBirthPlaceAddress;

/**
 *
 * @author richard.ettema
 */
public class HL7Parser201305
{
	private static final Logger logger = LoggerFactory.getLogger(HL7Parser201305.class.getName());

	public static final Long POB_ADDRESS_ID = new Long(1);
	public static final Long HOME_ADDRESS_ID = new Long(2);
	public static final Long MOTHERS_MAIDEN_NAME_ID = new Long(3);
	public static final Long PATIENT_NAME_ID = new Long(4);

	/**
	 * Social Security Number Root Identifier.
	 */
	public static final String SSN_ROOT_IDENTIFIER = "2.16.840.1.113883.4.1";

	/**
	 * Method to extract Gender Code from a PRPAMT201306UV02ParameterList.
	 * 
	 * @param params the Paramater list from which to extract a Gender Code
	 * @return The Gender Code is returned
	 */
	public static String extractGender(PRPAMT201306UV02ParameterList params)
	{

		String genderCode = null;

		// Extract the gender from the query parameters - Assume only one was
		// specified
		if(params.getLivingSubjectAdministrativeGender() != null && params.getLivingSubjectAdministrativeGender().size() > 0 && params.getLivingSubjectAdministrativeGender().get(0) != null)
		{
			PRPAMT201306UV02LivingSubjectAdministrativeGender gender = params.getLivingSubjectAdministrativeGender().get(0);

			if(gender.getValue() != null && gender.getValue().size() > 0 && gender.getValue().get(0) != null)
			{
				CE administrativeGenderCode = gender.getValue().get(0);

				logger.debug("Found gender in query parameters {} = ", administrativeGenderCode.getCode());
				genderCode = administrativeGenderCode.getCode();
			}
			else
			{
				// CCR177986-loggin updates
				logger.debug("query does not contain a gender code");
			}
		}
		else
		{
			logger.debug("query does not contain a gender code");
		}

		return genderCode;
	}

	/**
	 * Method to extract birthdate from a PRPAMT201306UV02ParameterList.
	 * 
	 * @param params the parameterList from which to extract the birthdate
	 * @return a Timestamp object containing the birthdate.
	 */
	public static Timestamp extractBirthdate(PRPAMT201306UV02ParameterList params)
	{
		Timestamp birthDate = null;
		final int dateOnlyLength = 8;

		try
		{
			// Extract the birth time from the query parameters - Assume only
			// one was specified
			if(params.getLivingSubjectBirthTime() != null && params.getLivingSubjectBirthTime().size() > 0 && params.getLivingSubjectBirthTime().get(0) != null)
			{
				PRPAMT201306UV02LivingSubjectBirthTime birthTime = params.getLivingSubjectBirthTime().get(0);

				if(birthTime.getValue() != null && birthTime.getValue().size() > 0 && birthTime.getValue().get(0) != null)
				{
					IVLTSExplicit birthday = birthTime.getValue().get(0);
					logger.debug("Found birthTime in query parameters {} = ", birthday.getValue());
					UTCDateUtil utcDateUtil = new UTCDateUtil();
					// Check date string length
					if(birthday.getValue().length() == dateOnlyLength)
					{
						birthDate = new Timestamp(utcDateUtil.parseDate(birthday.getValue(), UTCDateUtil.DATE_ONLY_FORMAT, null).getTime());
					}
					else if(birthday.getValue().length() > dateOnlyLength)
					{
						birthDate = new Timestamp(utcDateUtil.parseDate(birthday.getValue(), UTCDateUtil.DATE_FORMAT_UTC, null).getTime());
					}
					else
					{
						logger.debug("message does not contain a valid formatted birthtime");
					}
				}
				else
				{
					logger.debug("message does not contain a birthtime ");
				}
			}
			else
			{
				logger.debug("message does not contain a birthtime ");
			}
		}
		catch(Exception e)
		{
			// CCR1779231
			throw new ParseErrorException("Exception parsing birth date: ", e);
		}

		return birthDate;
	}

	/**
	 * Method to extract a list of Person names from a
	 * PRPAMT201306UV02ParameterList.
	 * 
	 * @param params the ParamaterList from which to extract names.
	 * @return a list of names from the ParamaterList.
	 */
	public static List<Personname> extractPersonnames(PRPAMT201306UV02ParameterList params)
	{

		List<Personname> personnames = new ArrayList<Personname>();

		// Extract the person names from the query parameters
		if(params.getLivingSubjectName() != null && params.getLivingSubjectName().size() > 0 && params.getLivingSubjectName().get(0) != null)
		{

			for(PRPAMT201306UV02LivingSubjectName name : params.getLivingSubjectName())
			{
				if(name.getValue() != null && name.getValue().size() > 0 && name.getValue().get(0) != null)
				{

					// We only want to extract the Patient's Name (e.g. not
					// Mothers Maiden Name) so just allow
					// useCodes that are null or "L".
					if(NullChecker.isNotNullOrEmpty(name.getValue().get(0).getUse()))
					{
						String useCode = name.getValue().get(0).getUse().get(0);
						if(!useCode.equalsIgnoreCase("L"))
						{
							continue;
						}
					}

					List<Serializable> choice = name.getValue().get(0).getContent();

					logger.debug("choice.size() {}= ", choice.size());

					Iterator<Serializable> iterSerialObjects = choice.iterator();

					String nameString = "";
					EnExplicitFamily lastname = null;
					EnExplicitGiven firstname = null;
					EnExplicitGiven middlename = null;
					EnExplicitPrefix prefix = null;
					EnExplicitSuffix suffix = null;

					while(iterSerialObjects.hasNext())
					{
						logger.debug("in iterSerialObjects.hasNext() loop");

						Serializable contentItem = iterSerialObjects.next();

						if(contentItem instanceof String)
						{
							logger.debug("contentItem is an instance of string");
							String strValue = (String) contentItem;

							if(nameString != null)
							{
								nameString += strValue;
							}
							else
							{
								nameString = strValue;
							}
							logger.debug("nameString {} = ", nameString);
						}
						else if(contentItem instanceof JAXBElement)
						{
							logger.debug("contentItem is an instance of JAXBElement");

							JAXBElement oJAXBElement = (JAXBElement) contentItem;

							if(oJAXBElement.getValue() instanceof EnExplicitFamily)
							{
								//lastname = new EnExplicitFamily();
								lastname = (EnExplicitFamily) oJAXBElement.getValue();
								logger.debug("found lastname element {}; content= ", lastname.getContent());
							}
							else if(oJAXBElement.getValue() instanceof EnExplicitGiven)
							{
								if(firstname == null)
								{
									//firstname = new EnExplicitGiven();
									firstname = (EnExplicitGiven) oJAXBElement.getValue();
									logger.debug("found firstname element {} ; content= ", firstname.getContent());
								}
								else if(middlename == null)
								{
									//middlename = new EnExplicitGiven();
									middlename = (EnExplicitGiven) oJAXBElement.getValue();
									logger.debug("found middlename element {}; content= ", middlename.getContent());
								}
								else
								{
									// ignore all other given values
								}
							}
							else if(oJAXBElement.getValue() instanceof EnExplicitPrefix)
							{
								//prefix = new EnExplicitPrefix();
								prefix = (EnExplicitPrefix) oJAXBElement.getValue();
								logger.debug("found prefix element {}; content= ", prefix.getContent());
							}
							else if(oJAXBElement.getValue() instanceof EnExplicitSuffix)
							{
								//suffix = new EnExplicitSuffix();
								suffix = (EnExplicitSuffix) oJAXBElement.getValue();
								logger.debug("found suffix element {}; content= ", suffix.getContent());
							}
							else
							{
								logger.debug("other name part {}= ", oJAXBElement.getValue());
							}
						}
						else
						{
							logger.debug("contentItem is other type of item");
						}
					}

					// If text string in patient name, then set in name
					// else set in element.
					boolean namefound = false;
					Personname personname = new Personname();
					personname.setPersonnameId(HL7Parser201305.PATIENT_NAME_ID);
					if(lastname != null && lastname.getContent() != null)
					{
						personname.setLastName(lastname.getContent());
						logger.debug("FamilyName {}: ", personname.getLastName());
						namefound = true;
					}

					if(firstname != null && firstname.getContent() != null)
					{
						personname.setFirstName(firstname.getContent());
						logger.debug("GivenName(first) {}: ", personname.getFirstName());
						namefound = true;
					}

					if(middlename != null && middlename.getContent() != null)
					{
						personname.setMiddleName(middlename.getContent());
						logger.debug("GivenName(middle) {}: ", personname.getMiddleName());
						namefound = true;
					}

					if(prefix != null && prefix.getContent() != null)
					{
						personname.setPrefix(prefix.getContent());
						logger.debug("Prefix {}: ", personname.getPrefix());
						namefound = true;
					}

					if(suffix != null && suffix.getContent() != null)
					{
						personname.setSuffix(suffix.getContent());
						logger.debug("Suffix {}: ", personname.getSuffix());
						namefound = true;
					}

					if(!namefound && !nameString.trim().contentEquals(""))
					{
						logger.debug("setting name by nameString {} ", nameString);
						personname.setLastName(nameString);

					}

					personnames.add(personname);
				}
				else
				{
					logger.debug("message does not contain a subject name");
				}
			}
		}
		else
		{
			logger.debug("message does not contain a subject name");
		}

		return personnames;
	}

	/**
	 * Method to extract a list of Person Identifiers from a
	 * PRPAMT201306UV02ParameterList.
	 * 
	 * @param params the PRPAMT201306UV02ParameterList from which to extract a
	 *            list of Person Identifiers
	 * @return a List of Identifiers.
	 */
	public static List<Identifier> extractPersonIdentifiers(PRPAMT201306UV02ParameterList params)
	{

		List<Identifier> ids = new ArrayList<Identifier>();

		if(params.getLivingSubjectId() != null && params.getLivingSubjectId().size() > 0 && params.getLivingSubjectId().get(0) != null)
		{

			for(PRPAMT201306UV02LivingSubjectId livingSubjectId : params.getLivingSubjectId())
			{
				if(livingSubjectId.getValue() != null && livingSubjectId.getValue().size() > 0 && livingSubjectId.getValue().get(0) != null)
				{
					II subjectId = livingSubjectId.getValue().get(0);

					if(subjectId.getExtension() != null && subjectId.getExtension().length() > 0 && subjectId.getRoot() != null && subjectId.getRoot().length() > 0)
					{
						// Ignore SSN identifiers
						if(!subjectId.getRoot().equals(SSN_ROOT_IDENTIFIER))
						{
							Identifier id = new Identifier();
							id.setId(subjectId.getExtension());
							id.setOrganizationId(subjectId.getRoot());
							logger.debug("Created id from patient identifier with organization id {} and id {} ", id.getOrganizationId(), id.getId()); // CCR
																																						// 177986
							ids.add(id);
						}
					}
					else
					{
						logger.debug("message does not contain an identifier");
					}
				}
				else
				{
					logger.debug("message does not contain an identifier");
				}
			}
		}
		else
		{
			logger.debug("message does not contain an");
		}

		return ids;
	}

	/**
	 * Method to extract a Social Security Number from a
	 * PRPAMT201306UV02ParameterList.
	 * 
	 * @param params the PRPAMT201306UV02ParameterList from which to extract a
	 *            SSN
	 * @return a Social Security Number from the PRPAMT201306UV02ParameterList
	 */
	public static String extractSSNIdentifier(PRPAMT201306UV02ParameterList params)
	{

		String ssn = null;

		if(params.getLivingSubjectId() != null && params.getLivingSubjectId().size() > 0 && params.getLivingSubjectId().get(0) != null)
		{

			for(PRPAMT201306UV02LivingSubjectId livingSubjectId : params.getLivingSubjectId())
			{
				if(livingSubjectId.getValue() != null && livingSubjectId.getValue().size() > 0 && livingSubjectId.getValue().get(0) != null)
				{
					II subjectId = livingSubjectId.getValue().get(0);

					if(subjectId.getExtension() != null && subjectId.getExtension().length() > 0 && subjectId.getRoot() != null && subjectId.getRoot().length() > 0)
					{
						// Look for first SSN identifier
						if(subjectId.getRoot().equals(SSN_ROOT_IDENTIFIER))
						{
							Identifier ssnId = new Identifier();
							ssnId.setId(subjectId.getExtension());
							ssnId.setOrganizationId(subjectId.getRoot());
							logger.debug("Created id from ssn identifier with organization id {} and ssn id {}", ssnId.getOrganizationId(), ssnId.getId()); // CCR
																																							// 177986
							ssn = ssnId.getId();
						}
					}
					else
					{
						logger.debug("message does not contain an id");
					}
				}
				else
				{
					logger.debug("message does not contain an id");
				}
			}
		}
		else
		{
			logger.debug("message does not contain an id");
		}

		return ssn;
	}

	/**
	 * Method to extract a list of Addresses from a
	 * PRPAMT201306UV02ParameterList.
	 * 
	 * @param params the PRPAMT201306UV02ParameterList from which to extract the
	 *            Addresses
	 * @return a list of Addresses from PRPAMT201306UV02ParameterList.
	 */
	public static List<Address> extractPersonAddresses(PRPAMT201306UV02ParameterList params)
	{
		List<Address> addresses = new ArrayList<Address>();

		if(params.getPatientAddress() != null && params.getPatientAddress().size() > 0 && params.getPatientAddress().get(0) != null)
		{
			for(PRPAMT201306UV02PatientAddress patientAddress : params.getPatientAddress())
			{
				if(patientAddress.getValue() != null && patientAddress.getValue().size() > 0 && patientAddress.getValue().get(0) != null)
				{
					ADExplicit adExplicit = patientAddress.getValue().get(0);

					List<Serializable> choice = adExplicit.getContent();

					logger.debug("choice.size()= {} ", choice.size());

					Iterator<Serializable> iterSerialObjects = choice.iterator();

					int addressLineCounter = 0;
					AdxpExplicitStreetAddressLine addressLine1 = null;
					AdxpExplicitStreetAddressLine addressLine2 = null;
					AdxpExplicitCity city = null;
					AdxpExplicitState state = null;
					AdxpExplicitPostalCode postalCode = null;

					Address address = null;

					while(iterSerialObjects.hasNext())
					{
						logger.debug("in iterSerialObjects.hasNext() loop");

						Serializable contentItem = iterSerialObjects.next();

						if(contentItem instanceof JAXBElement)
						{
							logger.debug("contentItem is JAXBElement");

							JAXBElement oJAXBElement = (JAXBElement) contentItem;

							if(oJAXBElement.getValue() instanceof AdxpExplicitStreetAddressLine)
							{
								addressLineCounter++;
								if(addressLineCounter == 1)
								{
									//addressLine1 = new AdxpExplicitStreetAddressLine();
									addressLine1 = (AdxpExplicitStreetAddressLine) oJAXBElement.getValue();
									logger.debug("found addressLine1 element {}; content= ", addressLine1.getContent());
									if(address == null)
									{
										address = new Address();
									}
									address.setStreet1(addressLine1.getContent());
								}
								if(addressLineCounter == 2)
								{
									//addressLine2 = new AdxpExplicitStreetAddressLine();
									addressLine2 = (AdxpExplicitStreetAddressLine) oJAXBElement.getValue();
									logger.debug("found addressLine2 element {}; content= ", addressLine2.getContent());
									if(address == null)
									{
										address = new Address();
									}
									address.setStreet2(addressLine2.getContent());
								}
							}
							else if(oJAXBElement.getValue() instanceof AdxpExplicitCity)
							{
								//city = new AdxpExplicitCity();
								city = (AdxpExplicitCity) oJAXBElement.getValue();
								logger.debug("found city element {}; content= ", city.getContent());
								if(address == null)
								{
									address = new Address();
								}
								address.setCity(city.getContent());
							}
							else if(oJAXBElement.getValue() instanceof AdxpExplicitState)
							{
								//state = new AdxpExplicitState();
								state = (AdxpExplicitState) oJAXBElement.getValue();
								logger.debug("found state element {}; content= ", state.getContent());
								if(address == null)
								{
									address = new Address();
								}
								address.setState(state.getContent());
							}
							else if(oJAXBElement.getValue() instanceof AdxpExplicitPostalCode)
							{
								//postalCode = new AdxpExplicitPostalCode();
								postalCode = (AdxpExplicitPostalCode) oJAXBElement.getValue();
								logger.debug("found postalCode element {}; content= ", postalCode.getContent());
								if(address == null)
								{
									address = new Address();
								}
								address.setPostal(postalCode.getContent());
							}
							else
							{
								logger.debug("other address part {} = ", oJAXBElement.getValue());
							}
						}
						else
						{
							logger.debug("contentItem is other type");
						}
					}

					if(address != null)
					{
						address.setAddressId(HOME_ADDRESS_ID);
						addresses.add(address);
					}
				}
			}
		}

		return addresses;
	}

	/**
	 * Method to extract a list of Phone Numbers from a
	 * PRPAMT201306UV02ParameterList.
	 * 
	 * @param params the PRPAMT201306UV02ParameterList from which to extract
	 *            Phone Numbers.
	 * @return a List of Phone Numbers from the PRPAMT201306UV02ParameterList
	 */
	public static List<Phonenumber> extractTelecoms(PRPAMT201306UV02ParameterList params)
	{

		List<Phonenumber> phonenumbers = new ArrayList<Phonenumber>();

		Phonenumber phonenumber = null;

		// Extract the telecom (phone number) from the query parameters - Assume
		// only one was specified
		if(params.getPatientTelecom() != null && params.getPatientTelecom().size() > 0 && params.getPatientTelecom().get(0) != null)
		{

			int count = 0;
			for(PRPAMT201306UV02PatientTelecom patientTelecom : params.getPatientTelecom())
			{
				if(patientTelecom.getValue() != null && patientTelecom.getValue().size() > 0 && patientTelecom.getValue().get(0) != null)
				{
					TELExplicit telecomValue = patientTelecom.getValue().get(0);
					logger.debug("Found patientTelecom in query parameters {} = ", telecomValue.getValue());
					String telecom = telecomValue.getValue();
					if(telecom != null)
					{
						if(!telecom.startsWith("tel:"))
						{
							// telecom is not valid without tel: prefix
							logger.debug("Found patientTelecom with {} in query parameters is not in the correct uri format", telecom); // CCR
																																		// 177986
							//telecom = null;
						}
						else
						{
							phonenumber = new Phonenumber();
							phonenumber.setValue(telecom);
							phonenumbers.add(phonenumber);
						}
					}
				}
				else
				{
					logger.debug("patientTelecom with a count {} does not contain a value.", count); // CCR
																										// 177986
				}
				count++;
			}
		}
		else
		{
			logger.debug("message does not contain a patientTelecom");
		}

		return phonenumbers;
	}

	/**
	 * Method to extract a PRPAMT201306UV02ParameterList object from a
	 * PRPAIN201305UV02 message.
	 * 
	 * @param message the PRPAIN201305UV02 message from which to extract the
	 *            PRPAMT201306UV02ParameterList
	 * @return PRPAMT201306UV02ParameterList
	 */
	public static PRPAMT201306UV02ParameterList extractHL7QueryParamsFromMessage(PRPAIN201305UV02 message)
	{
		PRPAMT201306UV02ParameterList queryParamList = null;

		if(message == null)
		{
			logger.debug("input message was null, no query parameters present in message");
			return null;
		}

		PRPAIN201305UV02QUQIMT021001UV01ControlActProcess controlActProcess = message.getControlActProcess();
		if(controlActProcess == null)
		{
			logger.debug("controlActProcess is null - no query parameters present in message");
			return null;
		}

		if(controlActProcess.getQueryByParameter() != null && controlActProcess.getQueryByParameter().getValue() != null)
		{
			PRPAMT201306UV02QueryByParameter queryParams = controlActProcess.getQueryByParameter().getValue();

			if(queryParams.getParameterList() != null)
			{
				queryParamList = queryParams.getParameterList();
			}

		}

		return queryParamList;
	}

	public static Personname extractMothersMaidenName(PRPAMT201306UV02ParameterList params)
	{
		Personname mmName = null;

		String mothersMaidenName = extractMothersMaidenNameValue(params);
		if(NullChecker.isNotNullOrEmpty(mothersMaidenName))
		{
			mmName = new Personname();
			mmName.setPersonnameId(MOTHERS_MAIDEN_NAME_ID);
			mmName.setLastName(mothersMaidenName);
		}
		return mmName;

	}

	public static String extractMothersMaidenNameValue(PRPAMT201306UV02ParameterList params)
	{
		String name = null;
		PRPAMT201306UV02LivingSubjectName mmn = HL7Parser201306.extractSubjectName(params.getLivingSubjectName(), "C");
		if(mmn != null)
		{
			Map<String, String> nameMap = HL7Parser201306.parseNames(mmn.getValue().get(0).getContent());
			if(nameMap != null)
			{
				name = nameMap.get("nameFamily");
			}
		}
		return name;
	}

	public static Map extractPlaceOfBirthMap(PRPAMT201306UV02ParameterList params)
	{
		Map<String, String> pobMap = null;

		if(params != null && params.getLivingSubjectBirthPlaceAddress() != null && params.getLivingSubjectBirthPlaceAddress().size() > 0 && params.getLivingSubjectBirthPlaceAddress().get(0) != null && params.getLivingSubjectBirthPlaceAddress().get(0).getValue() != null && params.getLivingSubjectBirthPlaceAddress().get(0).getValue().get(0) != null && params.getLivingSubjectBirthPlaceAddress().get(0).getValue().get(0).getContent() != null)
		{
			PRPAMT201306UV02LivingSubjectBirthPlaceAddress pob = params.getLivingSubjectBirthPlaceAddress().get(0);
			pobMap = HL7Parser201306.extractAddress(pob.getValue().get(0).getContent(), "pob");
		}

		return pobMap;
	}

	public static Address extractPlaceOfBirth(PRPAMT201306UV02ParameterList params)
	{
		Address address = null;

		Map<String, String> pobCityStateCountryMap = HL7Parser201305.extractPlaceOfBirthMap(params);
		if(pobCityStateCountryMap != null && pobCityStateCountryMap.size() > 0)
		{
			address = new Address();
			address.setAddressId(POB_ADDRESS_ID);
			if(NullChecker.isNotNullOrEmpty(pobCityStateCountryMap.get("pobCity")))
			{
				address.setCity(pobCityStateCountryMap.get("pobCity"));
			}
			if(NullChecker.isNotNullOrEmpty(pobCityStateCountryMap.get("pobState")))
			{
				address.setState(pobCityStateCountryMap.get("pobState"));
			}
			// if
			// (NullChecker.isNotNullOrEmpty(pobCityStateCountryMap.get("pobCountry")))
			// {
			// address.set(pobCityStateCountryMap.get("pobCountry"));
			// }
		}
		return address;
	}

	public static String extractRemoteAssigningAuthorityID(PRPAIN201305UV02 message)
	{

		String aaID = null;

		if(message.getControlActProcess() != null && message.getControlActProcess().getAuthorOrPerformer() != null && message.getControlActProcess().getAuthorOrPerformer().size() > 0 && message.getControlActProcess().getAuthorOrPerformer().get(0) != null && message.getControlActProcess().getAuthorOrPerformer().get(0).getAssignedDevice() != null && message.getControlActProcess().getAuthorOrPerformer().get(0).getAssignedDevice().getValue() != null && message.getControlActProcess().getAuthorOrPerformer().get(0).getAssignedDevice().getValue().getId() != null && message.getControlActProcess().getAuthorOrPerformer().get(0).getAssignedDevice().getValue().getId().size() > 0 && message.getControlActProcess().getAuthorOrPerformer().get(0).getAssignedDevice().getValue().getId().get(0) != null && NullChecker.isNotNullOrEmpty(message.getControlActProcess().getAuthorOrPerformer().get(0).getAssignedDevice().getValue().getId().get(0).getRoot()))
		{
			aaID = message.getControlActProcess().getAuthorOrPerformer().get(0).getAssignedDevice().getValue().getId().get(0).getRoot();
		}
		return aaID;
	}

	/**
	 * Method to extract the Patient from a PRPAIN201305UV02 message.
	 * 
	 * @param message the incoming message which contains the patient to be
	 *            extracted.
	 * @return a Patient from the PRPAIN201305UV02 object
	 */
	public static Patient extractMpiPatientFromMessage(PRPAIN201305UV02 message)
	{

		PRPAMT201306UV02ParameterList queryParamList = extractHL7QueryParamsFromMessage(message);
		Patient mpipatient = extractMpiPatientFromQueryParams(queryParamList);

		return mpipatient;
	}

	/**
	 * Method to extract the Patient from a PRPAMT201306UV02ParameterList
	 * message.
	 * 
	 * @param params the incoming message which contains the patient to be
	 *            extracted.
	 * @return a Patient from the PRPAMT201306UV02ParameterList object
	 */
	public static Patient extractMpiPatientFromQueryParams(PRPAMT201306UV02ParameterList params)
	{

		Patient mpiDbPatient = new Patient();

		if(params != null)
		{

			mpiDbPatient.setPersonnames(extractPersonnames(params));

			Personname mmn = extractMothersMaidenName(params);
			if(NullChecker.isNotNullOrEmpty(mmn) && NullChecker.isNotNullOrEmpty(mmn.getLastName()))
			{
				mpiDbPatient.getPersonnames().add(mmn);
			}

			mpiDbPatient.setGender(extractGender(params));

			mpiDbPatient.setDateOfBirth(extractBirthdate(params));

			mpiDbPatient.setIdentifiers(extractPersonIdentifiers(params));

			mpiDbPatient.setSsn(extractSSNIdentifier(params));

			mpiDbPatient.setAddresses(extractPersonAddresses(params));

			Address pob = extractPlaceOfBirth(params);
			if(NullChecker.isNotNullOrEmpty(pob))
			{
				mpiDbPatient.getAddresses().add(pob);
			}

			mpiDbPatient.setPhonenumbers(extractTelecoms(params));

		}
		else
		{
			logger.debug("QueryParams empty or null...");
			mpiDbPatient = null;
		}

		return mpiDbPatient;
	}
}
