/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package gov.va.med.nhin.adapter.logging;

import gov.va.med.nhin.adapter.utils.NullChecker;
import javax.xml.namespace.QName;
import org.hl7.fhir.dstu3.model.AuditEvent.AuditEventAction;
import org.hl7.fhir.dstu3.model.CodeableConcept;
import org.hl7.fhir.dstu3.model.Coding;

/**
 * Provides some constants to standardize logging. At the moment, this is just
 * some keys for handling transaction ids for the {@link org.slf4j.MDC}
 *
 * @author ryan
 */
public final class LogConstants {

  public static final String CORRELATION_MDCKEY = "correlationId";
  public static final String REMOTE_ADDR = "remoteAddress";
  public static final String LOCAL_ADDR = "localAddress";
  public static final String ENDPOINT = "soapEndpoint";
  public static final String AUDIT_SUBTYPE_CODE = "auditSubtype";

  public static final QName CORRELATION_SOAPPROP
          = new QName("http://connectopensource.org/transaction/", "TransactionID", "txn");
  public static final QName MESSAGEID_SOAPPROP
          = new QName("http://schemas.xmlsoap.org/ws/2004/08/addressing", "MessageId", "wsa");
  public static final CodeableConcept CORRELATION_CONCEPT = new CodeableConcept().addCoding(
          new Coding("http://connectopensource.org/transaction", "txn", "Correlation ID"));
  public static final QName ROOTLESS_JAXB
          = new QName("http://vetsez.com/messaging/marshalling/", "Element", "ele");

  private static final String MESSAGING = "messaging";

  private static final Coding DCM_QUERY
          = new Coding("http://dicom.nema.org/resources/ontology/DCM", "110112", "Query");
  private static final Coding DCM_EXPORT
          = new Coding("http://dicom.nema.org/resources/ontology/DCM", "110106", "Export");
  private static final Coding DCM_IMPORT
          = new Coding("http://dicom.nema.org/resources/ontology/DCM", "110117", "Import");
  private static final Coding DCM_APPACTIVITY
          = new Coding("http://dicom.nema.org/resources/ontology/DCM", "110100", "Application Activity");

  public static final CodeableConcept SOURCE_CONCEPT = new CodeableConcept().addCoding(
          new Coding("http://dicom.nema.org/resources/ontology/DCM", "110153", "Source"));
  public static final CodeableConcept DEST_CONCEPT = new CodeableConcept().addCoding(
          new Coding("http://dicom.nema.org/resources/ontology/DCM", "110152", "Destination"));

  public static final String MESSAGING_URI = "urn:gov:va:med:nhin:adapter:audit";

  private static final String ENTITY_TYPE_URI = "http://hl7.org/fhir/audit-entity-type";
  private static final String OBJECT_ROLE_URI = "http://hl7.org/fhir/object-role";

  public static final Coding PERSON = new Coding(ENTITY_TYPE_URI, "1", "Person");
  public static final Coding PATIENT = new Coding(OBJECT_ROLE_URI, "2", "Patient");
  public static final Coding SYSTEM_OBJ = new Coding(ENTITY_TYPE_URI, "2", "System Object");
  public static final Coding QUERY = new Coding(OBJECT_ROLE_URI, "24", "Query");
  public static final Coding CORRELATION_CODING = new Coding(OBJECT_ROLE_URI, "20", "Job");
  public static final Coding RESPONSE_TIME = new Coding(MESSAGING_URI, "FunctionResponseTime", "Response Time - ms");
  public static final Coding QUERY_RESPONSE = new Coding(MESSAGING_URI, "QueryResponse", "Query Response");

  public static enum AuditingEvent {
    /**
     * General logging
     */
    INFO(MESSAGING, "", MESSAGING_URI, DCM_QUERY, "Informational", "Informational", AuditEventAction.R),
    /**
     * PD constants
     */
    PDIN_BEGIN(MESSAGING, "pd.in", MESSAGING_URI, DCM_QUERY, "PD_IN-Request", "Cross Gateway Patient Discovery - Request", AuditEventAction.E),
    PDIN_END(MESSAGING, "pd.in", MESSAGING_URI, DCM_QUERY, "PD_IN-Response", "Patient Discovery Inbound - Response", AuditEventAction.E),
    PDIN_PARTNERAUTH(MESSAGING, "pd.in", MESSAGING_URI, DCM_APPACTIVITY, "PD_IN-CheckPDPartnerAuthZ", "Patient Discovery Inbound - Check if PD from Partner is authorized", AuditEventAction.E),
    PDIN_PATIENTMATCH(MESSAGING, "pd.in", MESSAGING_URI, DCM_APPACTIVITY, "PD_IN-FindPatientMatch", "Patient Discovery Inbound - Find Patient Match", AuditEventAction.E),
    PDIN_PATIENTSHARE(MESSAGING, "pd.in", MESSAGING_URI, DCM_APPACTIVITY, "PD_IN-CheckPatientSharing", "Patient Discovery Inbound - Check if the Patient is Sharing", AuditEventAction.E),
    PDIN_CORRELATESTORED(MESSAGING, "pd.in", MESSAGING_URI, DCM_APPACTIVITY, "PD_IN-StorePatientCor", "Patient Discovery Inbound - Store Correlation Stored", AuditEventAction.E),
    PDOUT_BEGIN(MESSAGING, "pd.out", MESSAGING_URI, DCM_QUERY, "PD_OUT-Begin", "Patient Discovery Outbound begin", AuditEventAction.E),
    PDOUT_END(MESSAGING, "pd.out", MESSAGING_URI, DCM_QUERY, "PD_OUT-End", "Patient Discovery Outbound end", AuditEventAction.E),
    PDOUT_GETPARTNERS(MESSAGING, "pd.out", MESSAGING_URI, DCM_QUERY, "PD_OUT-GetPartners", "Patient Discovery Outbound Get Partners from UDDI", AuditEventAction.E),
    PDOUT_PARTNERAUTH(MESSAGING, "pd.out", MESSAGING_URI, DCM_QUERY, "PD_OUT-PartnerAuth", "Patient Discovery Outbound Partner Authorized", AuditEventAction.E),
    PDOUT_PD(MESSAGING, "pd.out", MESSAGING_URI, DCM_QUERY, "PD-OUT_pd", "Patient Discovery Outbound Patient Patient Discovery", AuditEventAction.E),
    PDOUT_VALIDMATCH(MESSAGING, "pd.out", MESSAGING_URI, DCM_QUERY, "PD_OUT-ValidateMatch", "Patient Discovery Outbound Validate Match", AuditEventAction.E),
    PDOUT_CORRELATESTORED(MESSAGING, "pd.out", MESSAGING_URI, DCM_QUERY, "PD_OUT-Cor", "Patient Discovery Outbound Correlation Stored", AuditEventAction.E),
    PDOUT_CORRELATENOTSTORED(MESSAGING, "pd.out", MESSAGING_URI, DCM_QUERY, "PD_OUT-CorNot", "Patient Discovery Outbound Correlation Not Stored: already exists", AuditEventAction.E),
    /**
     * QD constants
     */
    QDIN_BEGIN(MESSAGING, "qd.in", MESSAGING_URI, DCM_QUERY, "QD_IN-Begin", "Query Documents Inbound begin", AuditEventAction.E),
    QDIN_END(MESSAGING, "qd.in", MESSAGING_URI, DCM_QUERY, "QD_IN-End", "Query Documents Inbound end", AuditEventAction.E),
    QDIN_PARTNERAUTH(MESSAGING, "qd.in", MESSAGING_URI, DCM_QUERY, "QD_IN-PartnerAuth", "Query Documents Inbound Partner Authorized", AuditEventAction.E),
    QDIN_PATIENTSHARE(MESSAGING, "qd.in", MESSAGING_URI, DCM_QUERY, "QD_IN-PatientShare", "Query Documents Inbound Patient Share OK", AuditEventAction.E),
    QDIN_GATHER(MESSAGING, "qd.in", MESSAGING_URI, DCM_QUERY, "QD_IN-gather", "Query Documents Inbound Data Gathered/Document Generation", AuditEventAction.E),
    QDOUT_BEGIN(MESSAGING, "qd.out", MESSAGING_URI, DCM_QUERY, "QD_OUT-Begin", "Query Documents Outbound begin", AuditEventAction.E),
    QDOUT_END(MESSAGING, "qd.out", MESSAGING_URI, DCM_QUERY, "QD_OUT-End", "Query Documents Outbound end", AuditEventAction.E),
    QDOUT_GETCORRELATIONS(MESSAGING, "qd.out", MESSAGING_URI, DCM_QUERY, "QD_OUT-Cor", "Query Documents Outbound Get Correlations from MVI", AuditEventAction.E),
    QDOUT_PARTNERAUTH(MESSAGING, "qd.out", MESSAGING_URI, DCM_QUERY, "QD_OUT-PartnerAuth", "Query Documents Outbound Partner Authorized", AuditEventAction.E),
    QDOUT_QD(MESSAGING, "qd.out", MESSAGING_URI, DCM_QUERY, "QD_OUT-qd", "Query Documents Outbound Patient Query Documents", AuditEventAction.E),
    /**
     * RD constants
     */
    RDIN_BEGIN(MESSAGING, "rd.in", MESSAGING_URI, DCM_EXPORT, "RD_IN-Begin", "Retrieve Document Inbound begin", AuditEventAction.R),
    RDIN_END(MESSAGING, "rd.in", MESSAGING_URI, DCM_EXPORT, "RD_IN-End", "Retrieve Document Inbound end", AuditEventAction.R),
    RDIN_PARTNERAUTH(MESSAGING, "rd.in", MESSAGING_URI, DCM_EXPORT, "RD_IN-PartnerAuth", "Retrieve Document Inbound Partner Authorized", AuditEventAction.R),
    RDIN_DISCLOSEOK(MESSAGING, "rd.in", MESSAGING_URI, DCM_EXPORT, "RD_IN-DisclosureOk", "Retrieve Document Inbound Disclosure OK", AuditEventAction.R),
    RDIN_GATHER(MESSAGING, "rd.in", MESSAGING_URI, DCM_EXPORT, "RD_IN-Gather", "Retrieve Document Inbound Data Gathered/Document Generation", AuditEventAction.R),
    RDOUT_BEGIN(MESSAGING, "rd.out", MESSAGING_URI, DCM_IMPORT, "RD_OUT-Begin", "Retrieve Document Outbound begin", AuditEventAction.C),
    RDOUT_END(MESSAGING, "rd.out", MESSAGING_URI, DCM_IMPORT, "RD_OUT-End", "Retrieve Document Outbound end", AuditEventAction.C),
    RDOUT_PARTNERAUTH(MESSAGING, "rd.out", MESSAGING_URI, DCM_IMPORT, "RD_OUT-PartnerAuth", "Retrieve Document Outbound Partner Authorized", AuditEventAction.C),
    RDOUT_RD(MESSAGING, "rd.out", MESSAGING_URI, DCM_IMPORT, "RD_OUT-rd", "Retrieve Document Outbound Patient Retrieve Document", AuditEventAction.C),
    /**
     * Submit constants
     */
    SUBMITIN_BEGIN(MESSAGING, "submit.in", MESSAGING_URI, DCM_IMPORT, "SUB_IN-Begin", "Submit Document Inbound begin", AuditEventAction.C),
    SUBMITIN_END(MESSAGING, "submit.in", MESSAGING_URI, DCM_IMPORT, "SUB_IN-End", "Submit Document Inbound end", AuditEventAction.C),
    SUBMITIN_PARTNERAUTH(MESSAGING, "submit.in", MESSAGING_URI, DCM_IMPORT, "SUB_IN-PartnerAuth", "Submit Document Inbound Partner Authorized", AuditEventAction.C),
    SUBMITIN_STORE(MESSAGING, "submit.in", MESSAGING_URI, DCM_IMPORT, "SUB_IN-DocStore", "Submit Document Inbound Patient Store", AuditEventAction.C);

    public final Coding coding;
    public final String category;
    public final Coding type;
    public final AuditEventAction action;

    AuditingEvent(String cat, String subcat, String systemuri, Coding maintype,
            String codename, String desc, AuditEventAction at) {

      category = (NullChecker.isNullOrEmpty(subcat)
              ? cat
              : cat + "." + subcat);
      coding = new Coding(systemuri, codename, desc);
      type = maintype;
      action = at;
    }

    public static AuditingEvent fromCode(String codename) {
      for (AuditingEvent ae : values()) {
        if (ae.coding.getCode().equals(codename)) {
          return ae;
        }
      }

      throw new IllegalArgumentException("unknown code: " + codename);
    }
  }
}
