package gov.va.med.nhin.adapter.datamanager.translators;

import gov.va.med.nhin.adapter.datamanager.DataManagerException;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.datamanager.DataTranslator;
import gov.va.med.nhin.adapter.datamanager.Reference;
import gov.va.med.nhin.adapter.datamanager.SmartHashMap;
import gov.va.med.nhin.adapter.utils.NullChecker;
import java.io.IOException;
import java.util.HashSet;
import java.util.ListIterator;
import java.util.Set;

/**
 *
 * @author Ry Bobko
 */
public class PruningFilterDataTranslator implements DataTranslator<Object>
{
    private static final Logger logger = LoggerFactory.getLogger(PruningFilterDataTranslator.class.getName());

    @Override
    public Object translate(Object input, Object result,
                            Reference translation, DataQuery dataQuery) throws IOException
    {
        String listName = translation.getProperty("listName");
        if (NullChecker.isNullOrEmpty(listName)) {
            throw new DataManagerException("listName property is required.");
        }

        String subListName = translation.getProperty("subListName");

        String key = translation.getProperty("key");
        if (NullChecker.isNullOrEmpty(key)) {
            throw new DataManagerException("key propety is required.");
        }

        String[] pruneListNames = translation.getProperty("pruneListNames").split("\\s*,\\s*");
        if (NullChecker.isNullOrEmpty(pruneListNames)) {
            throw new DataManagerException("pruneListNames property is required.");
        }

        String pruneKey = translation.getProperty("pruneKey");
        if (NullChecker.isNullOrEmpty(pruneKey)) {
            throw new DataManagerException("pruneKey property is required.");
        }
        if (NullChecker.isNullOrEmpty(input)) {
            logger.debug("Trying to prune and filter data, and found no data.");
            return input;
        }

        SmartHashMap smh = getTopLevel(input);
        List<SmartHashMap> list = List.class.cast(smh.get(listName));

        // build the list of keys to prune.
        final Set<String> keys = new HashSet<>();
        if (NullChecker.isNotNullOrEmpty(list)) {
            for (SmartHashMap s : list) {
                // subListName is optional, so we default to list if it's not provided.
                List<SmartHashMap> l = list;
                if (NullChecker.isNotNullOrEmpty(subListName)) {
                    l = getList(s.get(subListName));
                }

                if (NullChecker.isNotNullOrEmpty(l)) {
                    for (SmartHashMap s2 : l) {
                        Object idstr = makeKey(s2, key);
                        if (NullChecker.isNotNullOrEmpty(idstr)) {
                            keys.add(idstr.toString());
                        }
                    }
                }
            }

            // prune objects with matching keys from the prune lists.
            for (String pln : pruneListNames) {
                List<SmartHashMap> pruneList = getList(smh.get(pln));
                if (NullChecker.isNotNullOrEmpty(pruneList)) {
                    ListIterator<SmartHashMap> it = getList(pruneList).listIterator();

                    while (it.hasNext()) {
                        SmartHashMap pp = it.next();
                        String idstr = makeKey(pp, pruneKey);
                        if (NullChecker.isNotNullOrEmpty(idstr)) {
                            if (keys.contains(idstr)) {
                                // we have a duplicate ID, so remove it from our child listName
                            logger.debug("pruning note id {} from {} (already in {} field)",
                                    idstr, pruneList, listName );
                                it.remove();
                            }
                        }
                    }
                }
            }
        }
        
        return input;
    }

    private SmartHashMap getTopLevel(Object o)
    {
        List<SmartHashMap> olist = getList(o);
        if (!olist.isEmpty()) {
            return olist.get(0);
        }
        throw new RuntimeException("empty input list");
    }

    private List<SmartHashMap> getList(Object o)
    {
        if (NullChecker.isNotNullOrEmpty(o)) {
            if (o instanceof List) {
                List l = List.class.cast(o);
                int smhcount = 0;
                if (!l.isEmpty()) {
                    for (Object oo : l) {
                        if (oo instanceof SmartHashMap) {
                            smhcount++;
                        }
                    }
                }

                if (smhcount == l.size()) {
                    return (List<SmartHashMap>) l;
                }
            }
        }
        else {
            return (List<SmartHashMap>) o;
        }
        
        throw new DataManagerException("object must be a List of SmartHashMaps");
    }

    private String makeKey(SmartHashMap smh, String key)
    {
        StringBuilder ret = new StringBuilder();

        for (String s : key.split("\\s*,\\s*")) {
            Object keyPart = smh.get(s);
            if (NullChecker.isNotNullOrEmpty(keyPart)) {
                ret.append(keyPart.toString());
            }
        }

        return ret.toString();
    }
}
