package gov.va.med.nhin.adapter.datamanager.translators;

import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.datamanager.DataTranslator;
import gov.va.med.nhin.adapter.datamanager.Reference;
import gov.va.med.nhin.adapter.datamanager.SmartHashMap;
import gov.va.med.nhin.adapter.utils.NullChecker;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.beanutils.PropertyUtils;
import org.joda.time.LocalDateTime;


public class ImmunizationsFilter implements DataTranslator<List<SmartHashMap>> {
    
    private String cvxField = "cvxCode";
    private String cvxGroupField = "cvxGroup";
    private String cvxShortDesc = "cvxDesc";
    private String filteredDesc = "unspecified formulation";
    private String dateField = "adminDate";
    private String dateTimeField = "adminDateTime";
    private String facilityField = "facilityNumber";
    private String preferredFacilityField = "patientPreferredFacilityNumber";
    private String preferredFacility;
    private SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd");
    private int cnt = 0;
    
    private LocalDateTime getDate( SmartHashMap smh, String datefield ) {
        try {
            Object date = PropertyUtils.getProperty(smh, datefield);
            if(date instanceof Date) {
                return LocalDateTime.fromDateFields( ((Date) date ) );
            }
        } catch (Exception ex) {
            Logger.getLogger(ImmunizationsFilter.class.getName()).log(Level.SEVERE, null, ex);
        }
        return null;
    }
    
    private HashMap<String, List<SmartHashMap>> createCVXMap(List<SmartHashMap> list) {
        HashMap<String, List<SmartHashMap>> mapByCVX = new HashMap<>();
        List<SmartHashMap> listForCVX;
        for(SmartHashMap shm: list) {
            Object f = null;
            try {
                f = PropertyUtils.getProperty(shm, cvxField);
            } catch (Exception ex) {
                Logger.getLogger(ImmunizationsFilter.class.getName()).log(Level.SEVERE, null, ex);
            }
            String key = (f instanceof String) ? (String) f : f == null ? "CVX"+Integer.toString(++cnt) : f.toString();
            if(mapByCVX.containsKey(key)) {
                listForCVX = mapByCVX.get(key);
            } else {
                listForCVX = new ArrayList<>();
                mapByCVX.put(key, listForCVX);
            }
            listForCVX.add(shm);
        }
        return mapByCVX;
    }
    
    private HashMap<String, List<SmartHashMap>> createMapByDate(List<SmartHashMap> list) {
        HashMap<String, List<SmartHashMap>> mapByDate = new HashMap<>();
        List<SmartHashMap> listForDate;
        for(SmartHashMap shm: list) {
            Object f = null;
            try {
                f = PropertyUtils.getProperty(shm, dateField);
            } catch (Exception ex) {
                Logger.getLogger(ImmunizationsFilter.class.getName()).log(Level.SEVERE, null, ex);
            }
            String key = (f instanceof Date) ? dateFormat.format((Date)f) : "DATE"+Integer.toString(++cnt);
            if(mapByDate.containsKey(key)) {
                listForDate = mapByDate.get(key);
            } else {
                listForDate = new ArrayList<>();
                mapByDate.put(key, listForDate);
            }
            listForDate.add(shm);
        }
        return mapByDate;
    }
    
    private List<SmartHashMap> filterDatesWithPreferredFac(List<SmartHashMap> list) {
        List<SmartHashMap> filteredList = new ArrayList<>();
        HashMap<String, List<SmartHashMap>> mapByDates = createMapByDate(list);
        List<SmartHashMap> filteredItems = new ArrayList<>();
        for(List<SmartHashMap> listByDate: mapByDates.values()) {
            filteredItems.clear();
            if(listByDate.size() > 1) {
                for(SmartHashMap shm : listByDate) {
                    Object f = null;
                    try {
                        f = PropertyUtils.getProperty(shm, facilityField);
                    } catch (Exception ex) {
                        Logger.getLogger(ImmunizationsFilter.class.getName()).log(Level.SEVERE, null, ex);
                    }
                    String key = (f instanceof String) ? (String) f : f == null ? "FAC"+Integer.toString(++cnt) : f.toString();
                    if(key.equals(preferredFacility)){
                        filteredItems.add(shm);
                        break;
                    }
                }
            }
            if(filteredItems.size() > 0) {
                sortByDate(filteredItems, dateTimeField);
                filteredList.add(filteredItems.get(0));
            } else {
                sortByDate(listByDate, dateTimeField);
                filteredList.add(listByDate.get(0));
            }
        }
        
        return filteredList;
    }
    
    private void sortByDate(List<SmartHashMap> list, String field) {
        //Sort List
        Collections.sort( list, new Comparator<SmartHashMap>() {
            @Override
            public int compare( SmartHashMap o1, SmartHashMap o2 ) {
                LocalDateTime d1 = getDate( o1, field );
                LocalDateTime d2 = getDate( o2, field );
                return d2.compareTo( d1 );
            }
        });
    }
    
    private HashMap<String, List<SmartHashMap>> createCVXGroupMap(List<SmartHashMap> list) {
        HashMap<String, List<SmartHashMap>> mapByCVXGroup = new HashMap<>();
        List<SmartHashMap> listForCVX;
        for(SmartHashMap shm: list) {
            Object f = null;
            try {
                f = PropertyUtils.getProperty(shm, cvxGroupField);
            } catch (Exception ex) {
                Logger.getLogger(ImmunizationsFilter.class.getName()).log(Level.SEVERE, null, ex);
            }
            String cvxGroupString = (f instanceof String && NullChecker.isNotNullOrEmpty((String) f)) ? (String) f : "CVXGA"+Integer.toString(++cnt)+",CVXGB"+Integer.toString(++cnt);
            String[] cvxGroups;
            cvxGroups = (cvxGroupString).split(",");
            for(int i = 1; i < cvxGroups.length; i++) {
                if(mapByCVXGroup.containsKey(cvxGroups[i])) {
                    listForCVX = mapByCVXGroup.get(cvxGroups[i]);
                } else {
                    listForCVX = new ArrayList<>();
                    mapByCVXGroup.put(cvxGroups[i], listForCVX);
                }
                listForCVX.add(shm);
            }
        }
        return mapByCVXGroup;
    }
    
    private List<SmartHashMap> filterCVXGroup(HashMap<String, List<SmartHashMap>> mapByGroup) {
        HashSet<SmartHashMap> filteredSet = new HashSet<>();
        
        for(List<SmartHashMap> list : mapByGroup.values()) {
            HashMap<String, List<SmartHashMap>> mapByDate = createMapByDate(list);
            for(List<SmartHashMap> listByDate: mapByDate.values()) {
                List<SmartHashMap> filteredListByDate = new ArrayList<>();
                if(listByDate.size() > 1) {
                    for(SmartHashMap shm: listByDate) {
                        Object f = null;
                        try {
                            f = PropertyUtils.getProperty(shm, cvxShortDesc);
                        } catch (Exception ex) {
                            Logger.getLogger(ImmunizationsFilter.class.getName()).log(Level.SEVERE, null, ex);
                        }
                        if(f instanceof String && !((String)f).toUpperCase().contains(filteredDesc.toUpperCase())) {
                            filteredListByDate.add(shm);
                        }
                    }
                    filteredSet.addAll(filteredListByDate);
                } else {
                    filteredSet.add(listByDate.get(0));
                }
            }
        }
        
        List<SmartHashMap> filteredList = new ArrayList<>();
        Iterator<SmartHashMap> iterator = filteredSet.iterator();
        while(iterator.hasNext()) {
            filteredList.add(iterator.next());
        }
        
        return filteredList;
    }
    
    @Override
    public List<SmartHashMap> translate( Object input, Object result,
                    Reference translation, DataQuery dataQuery ) throws IOException {

            List<SmartHashMap> translatedList = new ArrayList<>();

            setupConfigs(translation, dataQuery);
            
            if ( input != null && input instanceof List ) {

                List<SmartHashMap> list = (List<SmartHashMap>) input;
                //Create map with CVX code as key and list of immunizations as values
                HashMap<String, List<SmartHashMap>> mapByCVX = createCVXMap(list);
                //For each cvx code, Filter out most recent from preferred facility (or call function to do that)
                for(List<SmartHashMap> listByCVX: mapByCVX.values())
                {
                    sortByDate(listByCVX, dateField);
                    if(listByCVX.get(0).get(cvxField) != null) {
                        translatedList.addAll(filterDatesWithPreferredFac(listByCVX));
                    } else {
                        translatedList.addAll(listByCVX);
                    }
                }
                
                //Create map with CVX group as key and list of imm as values
                HashMap<String, List<SmartHashMap>> cvxGroupMap = createCVXGroupMap(translatedList);
            
                //For each cvx group, if more than one item in list, filter out 'unspecified formulation'
                translatedList = filterCVXGroup(cvxGroupMap);
                
            }

            return translatedList;
    }

    private void setupConfigs(Reference translation, DataQuery dataQuery) {
        if(NullChecker.isNotNullOrEmpty(translation.getProperty("cvxField"))) { this.cvxField =  translation.getProperty("cvxField"); }
        if(NullChecker.isNotNullOrEmpty(translation.getProperty("cvxGroupField"))) { this.cvxGroupField =  translation.getProperty("cvxGroupField"); }
        if(NullChecker.isNotNullOrEmpty(translation.getProperty("cvxShortDesc"))) { this.cvxShortDesc =  translation.getProperty("cvxShortDesc"); }
        if(NullChecker.isNotNullOrEmpty(translation.getProperty("filteredDesc"))) { this.filteredDesc =  translation.getProperty("filteredDesc"); }
        if(NullChecker.isNotNullOrEmpty(translation.getProperty("dateField"))) { this.dateField =  translation.getProperty("dateField"); }
        if(NullChecker.isNotNullOrEmpty(translation.getProperty("dateTimeField"))) { this.dateTimeField =  translation.getProperty("dateTimeField"); }
        if(NullChecker.isNotNullOrEmpty(translation.getProperty("facilityField"))) { this.facilityField =  translation.getProperty("facilityField"); }
        if(NullChecker.isNotNullOrEmpty(translation.getProperty("preferredFacilityField"))) { this.preferredFacilityField =  translation.getProperty("preferredFacilityField"); }
        setPreferredFacility(dataQuery);
    }

    private void setPreferredFacility(DataQuery dataQuery) {
        String field = preferredFacilityField;
        preferredFacility = "";
        String subField = "";
        if(field.contains(".")) {
            String[] s = field.split("\\.", 2);
            field = s[0];
            subField = s[1];
        }
        Object obj = dataQuery.getParameter(field);
        List l;
        if(obj instanceof List) {
            l = (List) obj;
        } else {
            l = Arrays.asList(new Object[] { obj });
        }
        for(Object i : l) {
            if(subField.length() > 0) {
                try {
                    Object me = PropertyUtils.getProperty(i, subField);
                    if(me instanceof String) {
                        preferredFacility = (String) me;
                    }
                } catch (Exception ex) {
                    Logger.getLogger(ImmunizationsFilter.class.getName()).log(Level.SEVERE, null, ex);
                }
            } else {
                if(i instanceof String) {
                    preferredFacility = (String) i;
                }
            }
        }
    }
}
