package gov.va.med.nhin.adapter.datamanager.adapters;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.naming.Context;
import javax.naming.InitialContext;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.exception.FoundationsException;
import gov.va.med.nhin.adapter.datamanager.DataAdapter;
import gov.va.med.nhin.adapter.datamanager.DataManagerException;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.utils.NullChecker;
import gov.va.med.nhin.adapter.utils.PropertiesCollectionFactory;
import gov.va.med.vistalink.adapter.cci.VistaLinkAppProxyConnectionSpec;
import gov.va.med.vistalink.adapter.cci.VistaLinkConnection;
import gov.va.med.vistalink.adapter.cci.VistaLinkConnectionFactory;
import gov.va.med.vistalink.adapter.cci.VistaLinkConnectionSpec;
import gov.va.med.vistalink.adapter.record.VistaLinkFaultException;
import gov.va.med.vistalink.rpc.RpcRequest;
import gov.va.med.vistalink.rpc.RpcRequestFactory;
import gov.va.med.vistalink.rpc.RpcResponse;

/**
 *
 * @author David Vazquez
 */
public class VistADataAdapter implements DataAdapter<String>
{
    public static final Logger logger = LoggerFactory.getLogger(VistADataAdapter.class.getName());

    @Override
    public List<String> getData(DataQuery dataQuery)
    {
        List<String> ret = null;
        VistaLinkConnection connection = null;

        // CCR 177986
        logger.info("getData() invoked");

        try {
            connection = getConnection(dataQuery);
            setConnectionAttributes(connection);
            RpcRequest rpcRequest = buildRpcRequest(dataQuery);
            RpcResponse rpcResponse = executeRpcRequest(connection, rpcRequest);
            ret = processRpcResponse(rpcResponse);
        } catch (Exception e) {
            throw new DataAdapterException("An error occurred during data retrieval.", e);
        } finally {
            if (connection != null) {
                try {
                    closeConnection(dataQuery, connection);
                } catch (Throwable t) {
                }
            }

            // CCR 177986
            logger.debug("getData() exited");
        }

        return ret;
    }

    private VistaLinkConnection getConnection(DataQuery dataQuery) throws Exception
    {
        VistaLinkConnection ret;

        // CCR 177986
        logger.info("getConnection() invoked");

        String connectionFilename = dataQuery.getProperty("connectionFilename");
        logger.debug("connection file {} ", connectionFilename);

        String connection = dataQuery.getProperty("connection");
        logger.debug("connection info {} ", connection);

        Properties propertiesCollection = PropertiesCollectionFactory.getPropertiesCollection(connectionFilename);
        Properties connectionProperties = (Properties) propertiesCollection.get(connection);
        int connectionPropertiesSize = connectionProperties.size();

        try {
            if (!NullChecker.isNullOrEmpty(connectionProperties.getProperty("JNDIName"))) {
                ret = getJNDIConnection(connectionProperties);
                if (connectionProperties.size() != connectionPropertiesSize) {
                    PropertiesCollectionFactory.storeProperties(connectionFilename, propertiesCollection);
                }
            } else {
                throw new DataManagerException("JNDIName property is not set.");
            }
        } // CCR 179231-Exceptions
        catch (Exception e) {
            throw new DataAdapterException("An error occurred while trying to obtain VistaLink connection.", e);
        }

        return ret;
    }

    private VistaLinkConnection getJNDIConnection(Properties connectionProperties) throws Exception
    {
        VistaLinkConnectionSpec connectionSpec;

        if (NullChecker.isNullOrEmpty(connectionProperties.getProperty("username"))) {
            throw new DataManagerException("username property is not set.");
        } else {
            connectionSpec = getAppProxyConnectionSpec(connectionProperties);
        }

        Context ctx = new InitialContext();
        VistaLinkConnectionFactory connectionFactory = (VistaLinkConnectionFactory) ctx.lookup(connectionProperties.getProperty("JNDIName"));
        return (VistaLinkConnection) connectionFactory.getConnection(connectionSpec);
    }

    private VistaLinkConnectionSpec getAppProxyConnectionSpec(Properties connectionProperties) throws Exception
    {
        return new VistaLinkAppProxyConnectionSpec(connectionProperties.getProperty("division"), connectionProperties.getProperty("username"));
    }

    static private final Map<String, List<VistaLinkConnection>> connectionCache = new HashMap<String, List<VistaLinkConnection>>();

    private void setConnectionAttributes(VistaLinkConnection connection)
    {
        connection.setTimeOut(60 * 10000);
    }

    private RpcRequest buildRpcRequest(DataQuery dataQuery) throws FoundationsException
    {
        String rpcContext = dataQuery.getProperty("rpcContext");
        String rpcName = dataQuery.getProperty("rpcName");
        String rpcParams = dataQuery.getProperty("rpcParams");
        
        RpcRequest rpcRequest = null;
        try { // CCR 1779231- added the try catch block
            rpcRequest = RpcRequestFactory.getRpcRequest(rpcContext, rpcName);
            ArrayList params = new ArrayList();
            
            // check rpcParams property for list of query params that hold the
            // RPC arguements.  Default is to use all query paramters.
            String[] paramNames;
            if (NullChecker.isNotNullOrEmpty(rpcParams)) {
                paramNames = rpcParams.split(",");
            }
            else {
                paramNames = dataQuery.getParameterNames();
            }
            
            for (String paramName : paramNames) {
                Object parameter = dataQuery.getParameter(paramName);
                params.add(parameter != null ? parameter : "");
            }

            rpcRequest.setParams(params);
        } catch (Exception e) {
            throw new DataAdapterException("Error setting up RPC request.", e);
        }

        return rpcRequest;
    }

    private RpcResponse executeRpcRequest(VistaLinkConnection connection, RpcRequest rpcRequest) throws FoundationsException, VistaLinkFaultException
    {
        RpcResponse rpcResponse;

        // CCR 177986
        logger.debug(getClass().getName() + "{}", "executeRpcRequest");

        try {
            rpcResponse = connection.executeRPC(rpcRequest);

            // {} if enabled CCR 177986
            logger.debug("rpc response {} ", rpcResponse.toString());
            logger.debug("Request returned: {}", rpcResponse.getRawResponse());
        } catch (Exception e) {
            logger.debug("Error getting raw response {} ", rpcRequest.toString());
            throw new DataAdapterException("Error getting raw response from request", e);
        }

        return rpcResponse;
    }

    private List<String> processRpcResponse(RpcResponse rpcResponse) throws DataManagerException
    {
        List<String> ret = new ArrayList<>();
        ret.add(rpcResponse.getResults());
        return ret;
    }

    private void closeConnection(DataQuery dataQuery, VistaLinkConnection vistaLinkConnection) throws Exception
    {
        String connectionFilename = dataQuery.getProperty("connectionFilename");
        String connection = dataQuery.getProperty("connection");

        // CCR177986
        logger.debug("connection file name {} ", connectionFilename);
        logger.debug("connection {} ", connection);

        Properties propertiesCollection = PropertiesCollectionFactory.getPropertiesCollection(connectionFilename);
        Properties connectionProperties = (Properties) propertiesCollection.get(connection);

        if (!NullChecker.isNullOrEmpty(connectionProperties.getProperty("JNDIName"))) {
            logger.debug("connection props {} ", connectionProperties.getProperty("JNDIName"));

            vistaLinkConnection.close();
        }
    }
}
