package gov.va.med.nhin.adapter.datamanager.adapters;

import gov.va.med.nhin.adapter.utils.soap.handler.SOAPHandlerResolver;
import gov.hhs.fha.nhinc.common.nhinccommon.AssertionType;

import javax.xml.namespace.QName;

import gov.va.med.nhin.adapter.datamanager.DataAdapter;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.utils.NullChecker;
import gov.va.med.nhin.adapter.utils.PropertiesCollectionFactory;
import gov.va.oit.oed.vaww.VAIdM;
import gov.va.oit.oed.vaww.VAIdMPort;
import static java.lang.Integer.parseInt;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Properties;
import java.util.TimeZone;

import javax.xml.bind.JAXBElement;
import org.hl7.v3.COCTMT090100UV01AssignedPerson;
import org.hl7.v3.COCTMT090100UV01Person;
import org.hl7.v3.CommunicationFunctionType;
import org.hl7.v3.ENExplicit;
import org.hl7.v3.EnExplicitFamily;
import org.hl7.v3.EnExplicitGiven;
import org.hl7.v3.EntityClassDevice;
import org.hl7.v3.II;
import org.hl7.v3.MCCIMT000100UV01Agent;
import org.hl7.v3.MCCIMT000100UV01Device;
import org.hl7.v3.MCCIMT000100UV01Organization;
import org.hl7.v3.MCCIMT000100UV01Receiver;
import org.hl7.v3.MCCIMT000100UV01Sender;
import org.hl7.v3.MFMIMT700701UV01DataEnterer;
import org.hl7.v3.ObjectFactory;
import org.hl7.v3.QUQIMT021001UV01DataEnterer;
import org.hl7.v3.TSExplicit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.SecureRandom;

/**
 *
 * @author PII
 * @param <T>
 */
public abstract class MVIDataAdapter<T> implements DataAdapter<T>
{
    private static final Logger logger = LoggerFactory.getLogger(MVIDataAdapter.class);

    private Properties connectionProperties;
    protected String wsdlURL;
    protected String processingCodeStr;
    protected String mviSiteKey1301;
    protected String mviSiteKey1302;
    protected String mviSiteKey1305;
    protected String mviSiteKey1309;

    // MVI timeout properties
    protected int mviRetryLimit = 5;
    protected int mviConnectTimeout = 10000;
    protected int mviRequestTimeout = 30000;

    public static final String VA_OID = "2.16.840.1.113883.4.349";

    private static SecureRandom random = null;

    static {
        random = new SecureRandom();
    }

    private VAIdM mviService = null;

    protected void initializeProperties(DataQuery dataQuery)
    {
        try {
            String connectionFilename = dataQuery.getProperty("connectionFilename");
            String connection = dataQuery.getProperty("connection");
            Properties propertiesCollection = PropertiesCollectionFactory.getPropertiesCollection(connectionFilename);
            connectionProperties = (Properties) propertiesCollection.get(connection);

            logger.debug("conn props {} :", connectionProperties);

            wsdlURL = connectionProperties.getProperty("wsdlURL");

            logger.debug("wsdl {} :", wsdlURL);

            processingCodeStr = connectionProperties.getProperty("processingCode");
            mviSiteKey1301 = connectionProperties.getProperty("mviSiteKey1301");
            mviSiteKey1302 = connectionProperties.getProperty("mviSiteKey1302");
            mviSiteKey1305 = connectionProperties.getProperty("mviSiteKey1305");
            mviSiteKey1309 = connectionProperties.getProperty("mviSiteKey1309");
            try {
                mviRetryLimit = parseInt(connectionProperties.getProperty("mviRetryLimit"));
                mviConnectTimeout = parseInt(connectionProperties.getProperty("mviConnectTimeout"));
                mviRequestTimeout = parseInt(connectionProperties.getProperty("mviRequestTimeout"));
            }
            catch (NumberFormatException nfe) {
                logger.error("Could not get MVI timeout properties, proceeding with defaults.");
            }

            logger.debug("Site keys for mviSiteKey1301 = {}, mviSiteKey1305 = {} and  mviSiteKey1309 = {} ", new Object[]{mviSiteKey1301, mviSiteKey1305, mviSiteKey1309});
        }
        catch (Exception ex) {
            throw new DataAdapterException(ex);
        }
    }

    protected synchronized VAIdMPort getVAIdMPort(String mviServiceWSDL)
    {
        if (mviService == null) {
            try {
                mviService = new VAIdM(new URL(mviServiceWSDL), new QName("http://URL", "VAIdM"));
                mviService.setHandlerResolver(new SOAPHandlerResolver(true));
            }
            catch (Exception e) {
                throw new DataAdapterException("Error getting MVI Service - VAIdMPort.", e);
            }
        }
        return mviService.getVAIdMPort();
    }

    protected II createMessageId(II id)
    {
        if (id == null) {
            id = new II();
        }
        id.setRoot("2.16.840.1.113883.3.933"); // VA OID

        // if partner passed in a message id then just use it, otherwise
        // generate one.
        if (id.getExtension() == null || id.getExtension().trim().isEmpty()) {
            id.setExtension("MCID-" + createCreationTimeString() + "-" + random.nextLong());
        }

        logger.debug("message id {} :", id);

        return id;
    }

    private String createCreationTimeString()
    {
        // Create timestamp
        String timestamp = "";
        try {
            Date now = new Date();
            SimpleDateFormat formatter = new SimpleDateFormat("yyyyMMddHHmmss");
            formatter.setTimeZone(TimeZone.getTimeZone("UTC"));
            timestamp = formatter.format(now);

            logger.debug("timestamp {} :", timestamp);
        }
        catch (Exception e) {
            logger.error("error message ", e);
            throw new DataAdapterException("Exception when creating XMLGregorian Date", e);
        }

        return timestamp;
    }

    protected TSExplicit createCreationTime()
    {
        // Set up the creation time string
        TSExplicit creationTime = new TSExplicit();
        creationTime.setValue(createCreationTimeString());

        return creationTime;
    }

    protected MCCIMT000100UV01Receiver createReceiver()
    {
        // Set the receiver
        MCCIMT000100UV01Receiver receiver = new MCCIMT000100UV01Receiver();
        receiver.setTypeCode(CommunicationFunctionType.RCV);

        MCCIMT000100UV01Device receiverDevice = new MCCIMT000100UV01Device();
        receiverDevice.setDeterminerCode("INSTANCE"); // HL7Constants.RECEIVER_DETERMINER_CODE
        receiverDevice.setClassCode(EntityClassDevice.DEV);

        II receiverId = new II();
        receiverId.setRoot("2.16.840.1.113883.3.364"); // VA OID
        receiverDevice.getId().add(receiverId);
        receiver.setDevice(receiverDevice);

        return receiver;
    }

    protected MCCIMT000100UV01Sender createSender(String mviSiteKey, String homeCommunityOID, String facilityNumber)
    {
        ObjectFactory objFactory = new ObjectFactory();

        // added trim since had issue with extraneous character in Adapter DB
        // facility table.
        facilityNumber = facilityNumber.trim();
        // perhaps the VA's record in Adapter DB facility table should use
        // "200NH"..the NHIE/Exchange's site key...but for now just map here.
        facilityNumber = (facilityNumber.equals("VA")) ? mviSiteKey : facilityNumber;

        // Set the sender
        MCCIMT000100UV01Sender sender = new MCCIMT000100UV01Sender();
        sender.setTypeCode(CommunicationFunctionType.SND);

        MCCIMT000100UV01Device senderDevice = new MCCIMT000100UV01Device();
        senderDevice.setDeterminerCode("INSTANCE"); // HL7Constants.SENDER_DETERMINER_CODE
        senderDevice.setClassCode(EntityClassDevice.DEV);

        II senderDeviceId = new II();
        senderDeviceId.setRoot(VA_OID);
        // senderDeviceId.setExtension(mviSiteKey); // SITEKEY_OBTAINED_FROM_MVI
        senderDeviceId.setExtension(facilityNumber); // Per feedback from
        // MVI/Danny Reed, we
        // should send the
        // partner's
        // station#/facility
        // number...rather than
        // the Exchange's site
        // key.
        senderDevice.getId().add(senderDeviceId);

        // Setup Sender -> Device -> Agent
        MCCIMT000100UV01Agent senderDeviceAgent = new MCCIMT000100UV01Agent();
        senderDeviceAgent.getClassCode().add("AGNT");
        MCCIMT000100UV01Organization representedOrg = new MCCIMT000100UV01Organization();
        representedOrg.setClassCode("ORG");
        representedOrg.setDeterminerCode("INSTANCE");

        II representedOrgId = new II();
        representedOrgId.setRoot(homeCommunityOID);
        representedOrgId.setExtension(facilityNumber);
        representedOrg.getId().add(representedOrgId);

        JAXBElement jaxbElementAgent = objFactory.createMCCIMT000100UV01DeviceAsAgent(senderDeviceAgent);
        JAXBElement jaxbElementRepresentedOrg = objFactory.createMCCIMT000100UV01AgentRepresentedOrganization(representedOrg);

        senderDeviceAgent.setRepresentedOrganization(jaxbElementRepresentedOrg);
        senderDevice.setAsAgent(jaxbElementAgent);
        sender.setDevice(senderDevice);

        return sender;
    }

    protected MFMIMT700701UV01DataEnterer createDataEnterer1(AssertionType assertion)
    {
        MFMIMT700701UV01DataEnterer dataEnterer = new MFMIMT700701UV01DataEnterer();
        this.createDataEnterer(assertion, dataEnterer);
        return dataEnterer;
    }

    protected QUQIMT021001UV01DataEnterer createDataEnterer2(AssertionType assertion)
    {
        QUQIMT021001UV01DataEnterer dataEnterer = new QUQIMT021001UV01DataEnterer();
        this.createDataEnterer(assertion, dataEnterer);
        return dataEnterer;
    }

    private void createDataEnterer(AssertionType assertion, Object dataEntererObj)
    {
        ObjectFactory objFactory = new ObjectFactory();

        COCTMT090100UV01AssignedPerson assignedPerson = new COCTMT090100UV01AssignedPerson();
        assignedPerson.setClassCode("ASSIGNED");
        II assignedPersonId = new II();
        assignedPersonId.setRoot("2.16.840.1.113883.3.2017");
        assignedPersonId.setExtension("122");
        assignedPerson.getId().add(assignedPersonId);

        COCTMT090100UV01Person person = new COCTMT090100UV01Person();
        person.setDeterminerCode("INSTANCE");
        person.getClassCode().add("PSN");
        ENExplicit name = new ENExplicit();

        String familyNameStr = null;
        if (assertion != null && assertion.getUserInfo() != null && assertion.getUserInfo().getPersonName() != null && NullChecker.isNotNullOrEmpty(assertion.getUserInfo().getPersonName().getFamilyName())) {
            familyNameStr = assertion.getUserInfo().getPersonName().getFamilyName();
        }
        else {
            familyNameStr = "None Provided";
        }
        EnExplicitFamily familyName = new EnExplicitFamily();
        familyName.setContent(familyNameStr);
        JAXBElement jaxbFamilyName = objFactory.createENExplicitFamily(familyName);

        String givenNameStr = null;
        if (assertion != null && assertion.getUserInfo() != null && assertion.getUserInfo().getPersonName() != null && NullChecker.isNotNullOrEmpty(assertion.getUserInfo().getPersonName().getGivenName())) {
            givenNameStr = assertion.getUserInfo().getPersonName().getGivenName();
        }
        else {
            givenNameStr = "None Provided";
        }

        EnExplicitGiven givenName = new EnExplicitGiven();
        givenName.setContent(givenNameStr);
        JAXBElement jaxbGivenName = objFactory.createENExplicitGiven(givenName);

        name.getContent().add(jaxbGivenName);
        name.getContent().add(jaxbFamilyName);
        person.getName().add(name);
        JAXBElement jaxbElementPerson = objFactory.createCOCTMT090100UV01AssignedPersonAssignedPerson(person);

        assignedPerson.setAssignedPerson(jaxbElementPerson);

        if (dataEntererObj instanceof MFMIMT700701UV01DataEnterer) {
            ((MFMIMT700701UV01DataEnterer) dataEntererObj).setContextControlCode("AP");
            ((MFMIMT700701UV01DataEnterer) dataEntererObj).getTypeCode().add("ENT");
            ((MFMIMT700701UV01DataEnterer) dataEntererObj).setAssignedPerson(assignedPerson);
        }
        else if (dataEntererObj instanceof QUQIMT021001UV01DataEnterer) {
            ((QUQIMT021001UV01DataEnterer) dataEntererObj).setContextControlCode("AP");
            ((QUQIMT021001UV01DataEnterer) dataEntererObj).getTypeCode().add("ENT");
            ((QUQIMT021001UV01DataEnterer) dataEntererObj).setAssignedPerson(assignedPerson);
        }

    }
}