package gov.va.med.nhin.adapter.policyengine;

import gov.hhs.fha.nhinc.common.nhinccommon.AssertionType;
import gov.hhs.fha.nhinc.common.nhinccommon.PersonNameType;
import gov.hhs.fha.nhinc.common.nhinccommonadapter.CheckPolicyRequestType;
import gov.hhs.fha.nhinc.common.nhinccommonadapter.CheckPolicyResponseType;
import gov.va.med.nhin.adapter.audit.Audit;
import gov.va.med.nhin.adapter.audit.AuditManager;
import gov.va.med.nhin.adapter.audit.AuditManagerLocal;
import gov.va.med.nhin.adapter.datamanager.DataManager;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.datamanager.ejb.DataManagerLocal;
import gov.va.med.nhin.adapter.documentrepository.Document;
import gov.va.med.nhin.adapter.documentrepository.DocumentRepository;
import gov.va.med.nhin.adapter.documentrepository.DocumentRepositoryLocal;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManager;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManagerLocal;
import gov.va.med.nhin.adapter.policyengine.pdpproxy.PDPProxy;
import gov.va.med.nhin.adapter.policyengine.pdpproxy.PDPProxyLocal;
import gov.va.med.nhin.adapter.propertylookup.PropertyLookup;
import gov.va.med.nhin.adapter.propertylookup.PropertyLookupLocal;
import gov.va.med.nhin.adapter.utils.AuditUtil;
import gov.va.med.nhin.adapter.utils.NullChecker;

import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.ejb.EJB;
import javax.ejb.*;
import javax.jws.WebService;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import oasis.names.tc.xacml._2_0.context.schema.os.ActionType;
import oasis.names.tc.xacml._2_0.context.schema.os.AttributeType;
import oasis.names.tc.xacml._2_0.context.schema.os.AttributeValueType;
import oasis.names.tc.xacml._2_0.context.schema.os.RequestType;
import oasis.names.tc.xacml._2_0.context.schema.os.ResourceType;
import oasis.names.tc.xacml._2_0.context.schema.os.ResponseType;
import oasis.names.tc.xacml._2_0.context.schema.os.SubjectType;

/**
 *
 * @author David Vazquez
 *
 * inputs checkPolicyRequest
 *
 * returns boolean true authorized => DecisionType.PERMIT false not authorized
 * => DecisionType.DENY
 *
 * if the passed in ICN from the request is found in the OPTEDIN patient list
 * then they are authorized
 *
 *
 */
// No need for this to be a web service right now.
//@WebService(serviceName = "AdapterPolicyEngine", portName = "AdapterPolicyEnginePortSoap", endpointInterface = "gov.hhs.fha.nhinc.adapterpolicyengine.AdapterPolicyEnginePortType", targetNamespace = "urn:gov:hhs:fha:nhinc:adapterpolicyengine", wsdlLocation = "META-INF/wsdl/AdapterPolicyEngine.wsdl")
@TransactionAttribute(value = TransactionAttributeType.SUPPORTS)
@Stateless(name = "AdapterPolicyEngine")
public class AdapterPolicyEngine implements AdapterPolicyEnginePortTypeLocal
{
    private static Logger logger = LoggerFactory.getLogger(AdapterPolicyEngine.class.getName());

    static private final String ACTION_ATTRIBUTE_ID = "urn:oasis:names:tc:xacml:1.0:action:action-id";
    static private final String RESOURCE_ATTRIBUTE_ID = "urn:oasis:names:tc:xacml:1.0:resource:resource-id";
    static private final String RESOURCE_DOCUMENT_ATTRIBUTE_ID = "urn:gov:hhs:fha:nhinc:document-id";
    static private final String RESOURCE_HOME_COMMUNITY_ATTRIBUTE_ID = "urn:gov:hhs:fha:nhinc:home-community-id";

    private PropertyLookup propertyLookup;
    private DocumentRepository documentRepository;
    private AuditManager auditManager;
    private FacilityManager facilityManager;
    private PDPProxy pdpProxy;
    private DataManager dataManager;

    @EJB(beanInterface = DocumentRepositoryLocal.class, beanName = "DocumentRepository")
    public void setDocumentRepository(DocumentRepository documentRepository)
    {
        this.documentRepository = documentRepository;
    }

    @EJB(beanInterface = PropertyLookupLocal.class, beanName = "PropertyFileLookup")
    public void setPropertyLookup(PropertyLookup propertyLookup)
    {
        this.propertyLookup = propertyLookup;
    }

    @EJB(beanInterface = FacilityManagerLocal.class, beanName = "FacilityManager")
    public void setFacilityManager(FacilityManager facilityManager)
    {
        this.facilityManager = facilityManager;
    }

    @EJB(beanInterface = AuditManagerLocal.class, beanName = "AuditManager")
    public void setAuditManager(AuditManager auditManager)
    {
        this.auditManager = auditManager;
    }

    @EJB(beanInterface = PDPProxyLocal.class, beanName = "PDPProxyVAP")
    public void setPdpProxy(PDPProxy pdpProxy)
    {
        this.pdpProxy = pdpProxy;
    }

    @EJB(beanInterface = DataManagerLocal.class, beanName = "DataManager")
    public void setDataManager(DataManager dataManager)
    {
        this.dataManager = dataManager;
    }

    @Override
    public CheckPolicyResponseType checkPolicy(CheckPolicyRequestType checkPolicyRequest) {
        return checkPolicy(checkPolicyRequest, null);
    }
    @Override
    public CheckPolicyResponseType checkPolicy(CheckPolicyRequestType checkPolicyRequest, DataQuery parentQuery)
    {
        logger.debug("checkPolicy() request {}", checkPolicyRequest);

        RequestType requestType = checkPolicyRequest.getRequest();
        String action = extractValueFromAction(requestType.getAction(), ACTION_ATTRIBUTE_ID);

        if (action.equalsIgnoreCase("DocumentRetrieveIn")) {
            fixResources(requestType);
        }

        // FIXME - use next 4 lines in DEV until connection issue is
        // resolved...and need to comment out the pdpProxy.checkPolicy line when
        // deploying to DEV.
        // Error snippet: "...but could not connect over HTTP to server:
        // 'URL', port: 'PORT'"
        // ResponseType response = new ResponseType();
        // ResultType policyResult = new ResultType();
        // policyResult.setDecision(DecisionType.PERMIT);
        // response.getResult().add(policyResult);
        ResponseType response = pdpProxy.checkPolicy(requestType);

        gov.hhs.fha.nhinc.common.nhinccommonadapter.ObjectFactory objFactory = new gov.hhs.fha.nhinc.common.nhinccommonadapter.ObjectFactory();

        CheckPolicyResponseType checkPolicyResponse = objFactory.createCheckPolicyResponseType();
        //checkPolicyResponse.setResponse(response);
        if (checkPolicyResponse != null){ checkPolicyResponse.setResponse(response);}

        // PD Audit Report - Begin
        if (action.equalsIgnoreCase("PatientDiscoveryOut") || action.equalsIgnoreCase("PatientDiscoveryIn")) {
            Audit audit = new Audit();
            audit.setAction("CheckPolicy");
            audit.setAuditTime(new Date());
            audit.setPatientId(extractValueFromResource(requestType.getResource().get(0), RESOURCE_ATTRIBUTE_ID).substring(0, 17));

            DataQuery query = dataManager.getQuery("Composite.findDemographics2", parentQuery);
            String icn = extractValueFromResource(requestType.getResource().get(0), RESOURCE_ATTRIBUTE_ID);
            query.setParameter("icn", icn);
            List<Map> results = query.getResults();
            if (!NullChecker.isNullOrEmpty(results)) {
                Map result = results.get(0);
                Map demographics = (Map)result.get("demographics");
                audit.setPatientLastName((String)demographics.get("nameFamily"));
                audit.setPatientGivenName((String)demographics.get("nameGiven"));
                audit.setPatientSSN((String)demographics.get("SSN"));
                audit.setPatientFacilityNumber((String)result.get("patientPreferredFacilityNumber"));
                audit.setPatientFacilityName((String)result.get("patientPreferredFacilityName"));
            }

            String organization = null;
            String demographicDetails = "icn=" + extractValueFromResource(requestType.getResource().get(0), RESOURCE_ATTRIBUTE_ID) + ", ";

            AssertionType assertion = checkPolicyRequest.getAssertion();
            if (assertion != null) {
                if (assertion.getHomeCommunity() != null && !NullChecker.isNullOrEmpty(assertion.getHomeCommunity().getHomeCommunityId())) {
                    organization = assertion.getHomeCommunity().getHomeCommunityId();
                }

                /*
				 * Commented out for now due to lack of data if
				 * (!NullChecker.isNullOrEmpty(assertion.getSSN())) {
				 * demographicDetails += "ssn=" + assertion.getSSN();
				 * demographicDetails += ", "; } else { demographicDetails +=
				 * "ssn=, "; }
				 * 
				 * if (assertion.getPersonName() != null &&
				 * !NullChecker.isNullOrEmpty(assertion.getPersonName().
				 * getFamilyName())) { demographicDetails += "lastName=" +
				 * assertion.getPersonName().getFamilyName(); demographicDetails
				 * += ", "; } else { demographicDetails += "lastName=, "; } if
				 * (assertion.getPersonName() != null &&
				 * !NullChecker.isNullOrEmpty(assertion.getPersonName().
				 * getGivenName())) { demographicDetails += "firstName=" +
				 * assertion.getPersonName().getGivenName(); demographicDetails
				 * += ", "; } else { demographicDetails += "firstName=, "; } if
				 * (!NullChecker.isNullOrEmpty(assertion.getDateOfBirth())) {
				 * demographicDetails += "dob=" + assertion.getDateOfBirth();
				 * demographicDetails += ", "; } else { demographicDetails +=
				 * "dob=, "; }
                 */
                audit.setUserId(assertion.getUserInfo().getUserName());

                audit.setSystemId(AuditUtil.checkSystemId(assertion));

                audit.setUserRole(assertion.getUserInfo().getRoleCoded().getCode());
                audit.setPurposeForUse(assertion.getPurposeOfDisclosureCoded().getCode());
                audit.setUserFacilityNumber(assertion.getUserInfo().getOrg().getHomeCommunityId());
                audit.setUserFacilityName(assertion.getUserInfo().getOrg().getName());
                if (assertion.getUserInfo().getPersonName() != null) {
                    PersonNameType personName = assertion.getUserInfo().getPersonName();
                    if (!NullChecker.isNullOrEmpty(personName.getFullName())) {
                        audit.setUserName(personName.getFullName());
                    }
                    else {
                        StringBuilder userName = new StringBuilder();

                        if (NullChecker.isNotNullOrEmpty(personName.getGivenName())) {
                            userName.append(personName.getGivenName());
                        }

                        if (NullChecker.isNotNullOrEmpty(personName.getFamilyName())) {
                            if (userName.length() > 0) {
                                userName.append(' ');
                            }

                            userName.append(personName.getFamilyName());
                        }

                        audit.setUserName(userName.toString());
                    }
                }
            }

            // If Assertion Organization is empty, assume VA
            if (NullChecker.isNullOrEmpty(organization)) {
                organization = facilityManager.getFacilityByFacilityNumber("VA").getFullHomeCommunityId();
            }
            audit.setOrganizationId(organization);

            // PD Out External Org
            String externalOrganization = null;
            if (action.equalsIgnoreCase("PatientDiscoveryOut")) {
                externalOrganization = "urn:oid:" + extractValueFromResource(requestType.getResource().get(0), RESOURCE_HOME_COMMUNITY_ATTRIBUTE_ID);
            }
            else {
                externalOrganization = facilityManager.getFacilityByFacilityNumber("VA").getFullHomeCommunityId();
            }
            audit.setRemoteOrganizationId(externalOrganization);

            // Get CheckPolicy Decision
            String responseValue = null;
            if (checkPolicyResponse != null && checkPolicyResponse.getResponse() != null && checkPolicyResponse.getResponse().getResult() != null && checkPolicyResponse.getResponse().getResult().get(0) != null && checkPolicyResponse.getResponse().getResult().get(0).getDecision() != null && !NullChecker.isNullOrEmpty(checkPolicyResponse.getResponse().getResult().get(0).getDecision().value())) {
                responseValue = checkPolicyResponse.getResponse().getResult().get(0).getDecision().value();
            }

            // Build Details Field
            String details = "";
            if (!NullChecker.isNullOrEmpty(responseValue)) {
                details += responseValue + " ";
            }

            if (!NullChecker.isNullOrEmpty(action)) {
                details += action + " ";
            }

            if (!NullChecker.isNullOrEmpty(externalOrganization)) {
                details += "Remote Facility=" + externalOrganization + " ";
            }

            if (!NullChecker.isNullOrEmpty(demographicDetails)) {
                details += demographicDetails + " ";
            }

            audit.setDetails(details);

            auditManager.storeAudit(audit);
        }

        // PD Audit Report - End
        return checkPolicyResponse;
    }

    private void fixResources(RequestType requestType)
    {
        for (ResourceType resourceType : requestType.getResource()) {
            String patientId = extractValueFromResource(resourceType, RESOURCE_ATTRIBUTE_ID);

            if (NullChecker.isNullOrEmpty(patientId)) {
                String documentId = extractValueFromResource(resourceType, RESOURCE_DOCUMENT_ATTRIBUTE_ID);
                if (!NullChecker.isNullOrEmpty(documentId)) {
                    Document document = documentRepository.getDocumentByDocumentUniqueId(documentId);
                    if (document != null) {
                        AttributeType attribute = new AttributeType();
                        attribute.setAttributeId(RESOURCE_ATTRIBUTE_ID);

                        AttributeValueType attributeValue = new AttributeValueType();
                        attributeValue.getContent().add(qualifyPatientId(document.getPatientId()));
                        attribute.getAttributeValue().add(attributeValue);

                        resourceType.getAttribute().add(attribute);
                    }
                }
            }
        }
    }

    private String extractValueFromAction(ActionType actionType, String key)
    {
        return extractValueFromAttributes(actionType.getAttribute(), key);
    }
    /*
    private String extractValueFromSubject(SubjectType subjectType, String key)
    {
        return extractValueFromAttributes(subjectType.getAttribute(), key);
    }
    */
    private String extractValueFromResource(ResourceType resourceType, String key)
    {
        return extractValueFromAttributes(resourceType.getAttribute(), key);
    }

    private String extractValueFromAttributes(List<AttributeType> attributes, String key)
    {
        String ret = null;

        for (AttributeType attribute : attributes) {
            if (attribute != null && attribute.getAttributeId() != null && attribute.getAttributeId().equalsIgnoreCase(key) && !NullChecker.isNullOrEmpty(attribute.getAttributeValue()) && !NullChecker.isNullOrEmpty(attribute.getAttributeValue().get(0)) && !NullChecker.isNullOrEmpty(attribute.getAttributeValue().get(0).getContent())) {
                ret = attribute.getAttributeValue().get(0).getContent().get(0).toString();
                break;
            }
        }

        return ret;
    }

    private String qualifyPatientId(String icn)
    {
        return "'" + icn + "^^^&" + propertyLookup.getProperty("HomeCommunityId") + "&ISO'";
    }
}