package gov.va.med.nhin.adapter.mvi;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ejb.EJB;
import javax.ejb.*;

import org.hl7.v3.MCCIMT000300UV01Acknowledgement;
import org.hl7.v3.MFMIMT700711UV01QueryAck;
import org.hl7.v3.PRPAIN201305UV02;
import org.hl7.v3.PRPAIN201306UV02;
import org.hl7.v3.RespondingGatewayPRPAIN201305UV02RequestType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.hhs.fha.nhinc.common.nhinccommon.AssertionType;
import gov.hhs.fha.nhinc.common.nhinccommon.PersonNameType;
import gov.hhs.fha.nhinc.patientdb.model.Address;
import gov.hhs.fha.nhinc.patientdb.model.Identifier;
import gov.hhs.fha.nhinc.patientdb.model.Patient;
import gov.hhs.fha.nhinc.patientdb.model.Personname;
import gov.hhs.fha.nhinc.patientdb.model.Phonenumber;
import gov.va.med.nhin.adapter.audit.Audit;
import gov.va.med.nhin.adapter.audit.AuditManager;
import gov.va.med.nhin.adapter.audit.AuditManagerLocal;
import gov.va.med.nhin.adapter.datamanager.DataManager;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.datamanager.ejb.DataManagerLocal;
import gov.va.med.nhin.adapter.facilitymanager.Facility;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManager;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManagerLocal;
import gov.va.med.nhin.adapter.logging.AdapterPDError;
import gov.va.med.nhin.adapter.logging.ErrorMessage;
import gov.va.med.nhin.adapter.logging.EventAuditingFactory;
import gov.va.med.nhin.adapter.logging.EventAuditingFactoryImpl;
import gov.va.med.nhin.adapter.logging.MessagingHelper;
import gov.va.med.nhin.adapter.logging.MessagingHelper.QueryType;
import gov.va.med.nhin.adapter.mvi.hl7parsers.HL7Parser201305;
import gov.va.med.nhin.adapter.mvi.hl7parsers.HL7Parser201306;
import gov.va.med.nhin.adapter.propertylookup.PropertyLookup;
import gov.va.med.nhin.adapter.propertylookup.PropertyLookupLocal;
import gov.va.med.nhin.adapter.utils.AuditUtil;
import gov.va.med.nhin.adapter.utils.NullChecker;
import org.hl7.fhir.dstu3.model.AuditEvent;

/**
 *
 * @author David Vazquez
 */
// No need to be a web service right now.
//@WebService(serviceName = "AdapterMviService", portName = "AdapterMviPort", endpointInterface = "gov.va.med.nhin.adapter.adaptermvi.AdapterMviPortType", targetNamespace = "urn:gov:va:med:nhin:adapter:adaptermvi", wsdlLocation = "META-INF/wsdl/AdapterMvi.wsdl")
@TransactionAttribute(value = TransactionAttributeType.SUPPORTS)
@Stateless(name = "AdapterMVI")
public class AdapterMVI implements AdapterMviPortTypeLocal
{
    private static final Logger logger = LoggerFactory.getLogger(AdapterMVI.class.getName());

    private PropertyLookup propertyLookup;
    private FacilityManager facilityManager;
    private AuditManager auditManager;
    private DataManager dataManager;

    @EJB(beanInterface = PropertyLookupLocal.class, beanName = "PropertyFileLookup")
    public void setPropertyLookup(PropertyLookup propertyLookup)
    {
        this.propertyLookup = propertyLookup;
    }

    @EJB(beanInterface = AuditManagerLocal.class, beanName = "AuditManager")
    public void setAuditManager(AuditManager auditManager)
    {
        this.auditManager = auditManager;
    }

    @EJB(beanInterface = FacilityManagerLocal.class, beanName = "FacilityManager")
    public void setFacilityManager(FacilityManager facilityManager)
    {
        this.facilityManager = facilityManager;
    }

    @EJB(beanInterface = DataManagerLocal.class, beanName = "DataManager")
    public void setDataManager(DataManager dataManager)
    {
        this.dataManager = dataManager;
    }

    @Override
    public PRPAIN201306UV02 findCandidates(RespondingGatewayPRPAIN201305UV02RequestType findCandidatesRequest)
    {
        PRPAIN201306UV02 ret = null;

        Map map = findCandidatesMap(findCandidatesRequest, null, null);
        if (map != null) {
            ret = (PRPAIN201306UV02)map.get("PRPAIN201306UV02");
        }
        return ret;
    }

    @Override
    public Map findCandidatesMap(RespondingGatewayPRPAIN201305UV02RequestType findCandidatesRequest,
                                 AuditEvent ae, DataQuery parentQuery)
    {
        // CCR 177986 - parameterization
        logger.info("findCandidates () invoked");

        HashMap resultsMap = null;
        String details = null;
        String auditIcn = null;

        PRPAIN201305UV02 query = findCandidatesRequest.getPRPAIN201305UV02();

        // CCR 177986
        logger.debug("Query details from 1305 {}", query);

        AssertionType assertion = findCandidatesRequest.getAssertion();

        // CCR 177986
        logger.debug("assertion {}", assertion);

        String sendingFacilityOID = getSenderOID(query);
        String sendingFacilityNumber = getSendingFacilityNumber(sendingFacilityOID);

        Patient sourcePatient = HL7Parser201305.extractMpiPatientFromMessage(query);

        // CCR 177986
        logger.debug("source patient {}", sourcePatient);

        Patient searchResultPatient = null;
        PRPAIN201306UV02 result = null;

        try {
            result = mviPatientSearch(query, assertion, sendingFacilityOID, sendingFacilityNumber, ae, parentQuery);

            // CCR 177986
            logger.debug("1306 patient search result {}", result);

            MCCIMT000300UV01Acknowledgement ack = result.getAcknowledgement().get(0);

            // CCR 177986
            logger.debug("ack code {}", ack);

            MFMIMT700711UV01QueryAck queryAck = result.getControlActProcess().getQueryAck();

            // CCR 177986
            logger.debug("ack.getTypeCode().getCode(): {}", ack.getTypeCode().getCode());
            logger.debug("queryAck.getQueryResponseCode().getCode(): {}", queryAck.getQueryResponseCode().getCode());

            if (ack.getTypeCode().getCode().equalsIgnoreCase("AE") || queryAck.getQueryResponseCode().getCode().equalsIgnoreCase("AE")) {
                // MVI returned with error code.
                String mviAckText = (String)ack.getAcknowledgementDetail().get(0).getText().getContent().get(0);
                String errorMsg = "MVI returned with error code of 'AE'.  Details: '" + mviAckText + "'";
                details = "QUERY FAILED Remote Facility=" + sendingFacilityOID + ".  " + errorMsg;
                AdapterPDError.queryError(findCandidatesRequest, ErrorMessage.IN_PD_UNKNOWN, details);

                // CCR 179231- used domain specific exception
                // TODO: Should we really be throwing an exception here?
                throw new RuntimeException(errorMsg);
            }
            else if (queryAck.getQueryResponseCode().getCode().equalsIgnoreCase("NF")) {
                details = "MATCH FAILED Remote Facility=" + sendingFacilityOID + " ";
                AdapterPDError.queryError(findCandidatesRequest, ErrorMessage.IN_PD_UNKNOWN, details);
            }
            else if (queryAck.getQueryResponseCode().getCode().equalsIgnoreCase("QE") || queryAck.getResultCurrentQuantity().getValue().intValue() > 1) {
                details = "AMBIGUOUS FAILED Remote Facility=" + sendingFacilityOID + " ";
                AdapterPDError.queryError(findCandidatesRequest, ErrorMessage.IN_PD_UNKNOWN, details);
            }
            else if (queryAck.getQueryResponseCode().getCode().equalsIgnoreCase("OK")) {
                resultsMap = new HashMap();
                resultsMap.put("PRPAIN201306UV02", result);
                searchResultPatient = getPatient(result, parentQuery);
                resultsMap.put("SEARCH_RESULT_PATIENT", searchResultPatient);
                auditIcn = getICNFromPatient(searchResultPatient);
                details = "MATCH FOUND ICN= " + auditIcn + " Remote Facility=" + sendingFacilityOID + " ";

                // CCR 177986- parameterization
                logger.debug(details + "{}");
            }
            else {
                details = "MVI response code not recognized.";
                AdapterPDError.queryError(findCandidatesRequest, ErrorMessage.IN_PD_UNKNOWN, details);
                // TODO: Should we really be throwing an exception here?
                throw new RuntimeException("MVI response code not recognized.");
            }
        }
        catch (Exception exception) {
            details = exception.getMessage();
        }
        finally {
            processAuditLogging(sourcePatient, searchResultPatient, assertion, auditIcn, sendingFacilityOID, details);
        }

        return resultsMap;
    }

    private void processAuditLogging(Patient sourcePatient, Patient searchResultPatient, AssertionType assertion, String auditIcn, String sendingFacilityOID, String details)
    {
        // PD Audit Report - Begin
        Audit audit = new Audit();
        audit.setAction("MPI findMatch");

        // Set Audit data from MVI Search Results
        if (searchResultPatient != null) {
            audit.setPatientId(auditIcn);
            audit.setPatientSSN(searchResultPatient.getSsn());

            // set Patient's name
            for (Personname name : searchResultPatient.getPersonnames()) {
                if (name.getPersonnameId().equals(HL7Parser201305.PATIENT_NAME_ID)) {
                    audit.setPatientLastName(name.getLastName());
                    audit.setPatientGivenName(name.getFirstName());
                    audit.setPatientMiddleName(name.getMiddleName());
                    continue;
                }
                if (name.getPersonnameId().equals(HL7Parser201305.MOTHERS_MAIDEN_NAME_ID)) {
                    audit.setMothersMaidenName(name.getLastName());
                    continue;
                }
            }

            // Place of Birth (City & State)
            for (Address address : searchResultPatient.getAddresses()) {
                if (address.getAddressId().equals(HL7Parser201305.POB_ADDRESS_ID)) {
                    audit.setPobCity(address.getCity());
                    audit.setPobState(address.getState());
                    break;
                }
            }

            Identifier patientFacilityId = getAndRemovePatientFacilityIdFromPatient(searchResultPatient);
            if (patientFacilityId != null) {
                String[] parts = patientFacilityId.getId().split("~", 2);
                audit.setPatientFacilityNumber(parts[0]);
                audit.setPatientFacilityName(parts[1]);
            }
        }

        // Set Audit data from Assertion (passed in via PD request)
        String organization = null;
        if (assertion != null) {
            if (assertion.getHomeCommunity() != null && NullChecker.isNotNullOrEmpty(assertion.getHomeCommunity().getHomeCommunityId())) {
                organization = assertion.getHomeCommunity().getHomeCommunityId();
            }
            audit.setUserId(assertion.getUserInfo().getUserName());

            audit.setSystemId(AuditUtil.checkSystemId(assertion));

            audit.setUserRole(assertion.getUserInfo().getRoleCoded().getCode());
            audit.setUserFacilityNumber(assertion.getUserInfo().getOrg().getHomeCommunityId());
            audit.setUserFacilityName(assertion.getUserInfo().getOrg().getName());
            audit.setPurposeForUse(assertion.getPurposeOfDisclosureCoded().getCode());
            if (assertion.getUserInfo().getPersonName() != null) {
                PersonNameType personName = assertion.getUserInfo().getPersonName();
                if (NullChecker.isNotNullOrEmpty(personName.getFullName())) {
                    audit.setUserName(personName.getFullName());
                }
                else {
                    StringBuilder userName = new StringBuilder();

                    if (NullChecker.isNotNullOrEmpty(personName.getGivenName())) {
                        userName.append(personName.getGivenName());
                    }

                    if (NullChecker.isNotNullOrEmpty(personName.getFamilyName())) {
                        if (userName.length() > 0) {
                            userName.append(' ');
                        }

                        userName.append(personName.getFamilyName());
                    }

                    audit.setUserName(userName.toString());
                }
            }
        }

        // Set Audit "Details" column data from PD Request (sourcePatient)
        // SSN
        if (NullChecker.isNotNullOrEmpty(sourcePatient.getSsn())) {
            details += "ssn=" + sourcePatient.getSsn();
            details += ", ";
        }
        else {
            details += "ssn=, ";
        }

        // Patient Name
        if (NullChecker.isNotNullOrEmpty(sourcePatient.getPersonnames())) {
            for (Personname personname : sourcePatient.getPersonnames()) {

                if (personname != null && personname.getPersonnameId().equals(HL7Parser201305.PATIENT_NAME_ID)) {
                    if (NullChecker.isNotNullOrEmpty(personname.getLastName())) {
                        details += "lastName=" + personname.getLastName();
                        details += ", ";
                    }
                    else {
                        details += "lastName=, ";
                    }
                    if (NullChecker.isNotNullOrEmpty(personname.getFirstName())) {
                        details += "firstName=" + personname.getFirstName();
                        details += ", ";
                    }
                    else {
                        details += "firstName=, ";
                    }
                    if (NullChecker.isNotNullOrEmpty(personname.getMiddleName())) {
                        details += "middleName=" + personname.getMiddleName() + ", ";
                    }
                    else {
                        details += "middleName=, ";
                    }
                    break;
                }
            }
        }
        else {
            details += "lastName=, firstName=, middleName=, ";
        }

        if (NullChecker.isNotNullOrEmpty(sourcePatient.getGender())) {
            details += "gender=" + sourcePatient.getGender();
            details += ", ";
        }
        else {
            details += "gender=, ";
        }

        if (NullChecker.isNotNullOrEmpty(sourcePatient.getDateOfBirth())) {
            details += "dob=" + sourcePatient.getDateOfBirth();
            details += ", ";
        }
        else {
            details += "dob=, ";
        }

        // Mothers Maiden Name
        String mmn = "";
        if (NullChecker.isNotNullOrEmpty(sourcePatient.getPersonnames())) {
            for (Personname personname : sourcePatient.getPersonnames()) {
                if (personname != null && personname.getPersonnameId().equals(HL7Parser201305.MOTHERS_MAIDEN_NAME_ID)) {
                    if (NullChecker.isNotNullOrEmpty(personname.getLastName())) {
                        mmn = personname.getLastName();
                    }
                    break;
                }
            }
        }
        if (mmn.isEmpty()) {
            // Audit MMN is already populated from MVI result patient
            mmn = audit.getMothersMaidenName();
        }
        details += "MMN=" + mmn + ", ";

        // Home Address
        if (NullChecker.isNotNullOrEmpty(sourcePatient.getAddresses())) {
            for (Address address : sourcePatient.getAddresses()) {

                if (address != null && address.getAddressId().equals(HL7Parser201305.HOME_ADDRESS_ID)) {

                    if (NullChecker.isNotNullOrEmpty(address.getStreet1())) {
                        details += "street1=" + address.getStreet1() + ", ";
                        if (NullChecker.isNotNullOrEmpty(address.getStreet2())) {
                            details += "street2=" + address.getStreet2() + ", ";
                        }
                    }
                    else {
                        details += "street=, ";
                    }
                    if (NullChecker.isNotNullOrEmpty(address.getCity())) {
                        details += "city=" + address.getCity();
                        details += ", ";
                    }
                    else {
                        details += "city=, ";
                    }
                    if (NullChecker.isNotNullOrEmpty(address.getState())) {
                        details += "state=" + address.getState();
                        details += ", ";
                    }
                    else {
                        details += "state=, ";
                    }
                    if (NullChecker.isNotNullOrEmpty(address.getPostal())) {
                        details += "zip=" + address.getPostal();
                        details += ", ";
                    }
                    else {
                        details += "zip=, ";
                    }
                    break;
                }
            }
        }
        else {
            details += "street=, city=, state=, zip=, ";
        }

        if (NullChecker.isNotNullOrEmpty(sourcePatient.getPhonenumbers())) {
            Phonenumber phoneNumber = sourcePatient.getPhonenumbers().get(0);
            details += "phoneNumber=" + phoneNumber.getValue();
            details += ", ";
        }
        else {
            details += "phoneNumber=, ";
        }

        // Place of Birth (City and State)
        String pobCity = "";
        String pobState = "";
        if (NullChecker.isNotNullOrEmpty(sourcePatient.getAddresses())) {
            for (Address pob : sourcePatient.getAddresses()) {
                if (pob != null && pob.getAddressId().equals(HL7Parser201305.POB_ADDRESS_ID)) {
                    if (NullChecker.isNotNullOrEmpty(pob.getCity())) {
                        pobCity = pob.getCity();
                    }
                    if (NullChecker.isNotNullOrEmpty(pob.getState())) {
                        pobState = pob.getState();
                    }
                    break;
                }
            }
        }
        if (pobCity.isEmpty()) {
            // Audit POB city is already populated from MVI result patient
            pobCity = audit.getPobCity();
        }
        if (pobState.isEmpty()) {
            // Audit POB state is already populated from MVI result patient
            pobState = audit.getPobState();
        }
        details += "pobCity=" + pobCity + ", " + "pobState=" + pobState + ", ";

        Facility homeFacility = facilityManager.getFacilityByFacilityNumber("VA");
        if (organization == null || organization.isEmpty() || organization.equals(homeFacility.getFullHomeCommunityId())) {
            organization = homeFacility.getFullHomeCommunityId();
            audit.setRemoteOrganizationId("urn:oid:" + sendingFacilityOID);
        }
        else {
            audit.setRemoteOrganizationId(facilityManager.getFacilityByFacilityNumber("VA").getFullHomeCommunityId());
        }
        audit.setOrganizationId(organization);

        if (NullChecker.isNotNullOrEmpty(details)) {
            audit.setDetails(details);
        }
        else {
            audit.setDetails("No details available");
        }

        auditManager.storeAudit(audit);
        // PD Audit Report - End
    }

    private String getSendingFacilityNumber(String senderDeviceId)
    {
        String ret;
        Facility facility = facilityManager.getFacilityByHomeCommunityId(senderDeviceId);

        if (facility != null) {
            ret = facility.getFacilityNumber();
        }
        else {
            ret = "200NHI";
        }

        logger.debug("Facility Number {}", ret);

        return ret;
    }

    private PRPAIN201306UV02 mviPatientSearch(PRPAIN201305UV02 query, AssertionType assertion,
                                              String sendingFacilityOID, String sendingFacilityNumber, AuditEvent loggingEvent, DataQuery parentQuery)
    {
        PRPAIN201306UV02 result1306 = null;
        logger.debug("Getting MVI query...");
        DataQuery dq = dataManager.getQuery("MVI.findMatch", parentQuery);

        dq.setParameter("1305Request", query);
        dq.setParameter("assertion", assertion);
        dq.setParameter("sendingFacilityOID", sendingFacilityOID);
        dq.setParameter("sendingFacilityNumber", sendingFacilityNumber);

        EventAuditingFactory<AuditEvent> afac
                                         = EventAuditingFactoryImpl.getFactory(AuditEvent.class);
        MessagingHelper<AuditEvent> msg = afac.messaging();
        if (null != loggingEvent) {
            msg.addQuery(loggingEvent, query, PRPAIN201305UV02.class,
                         QueryType.REQUEST, "mviquery");
            msg.addDetail(loggingEvent, "mviquery", "sendingFacilityOID", sendingFacilityOID);
            msg.addDetail(loggingEvent, "mviquery", "sendingFacilityNumber", sendingFacilityNumber);
        }

        logger.debug("Attempting to get MVI response...");
        List<PRPAIN201306UV02> dqResults = dq.getResults();
        if (dqResults != null) {
            if (null != loggingEvent) {
                for (PRPAIN201306UV02 o : dqResults) {
                    if (null != o) {
                        msg.addQuery(loggingEvent, o, PRPAIN201306UV02.class,
                                     QueryType.RESPONSE, null);
                    }
                }
            }

            if (dqResults.size() == 1 && dqResults.get(0) != null) {
                logger.debug("MVI response returned");
                result1306 = dqResults.get(0);
                logger.debug("Data Query Results {}", dqResults);
            }
        }

        return result1306;
    }

    private Patient getPatient(PRPAIN201306UV02 result1306, DataQuery parentQuery)
    {
        Patient dpPatient = new Patient();

        // set ICN
        String icn = HL7Parser201306.extractICN(result1306);
        List<Identifier> ids = new ArrayList<>();
        dpPatient.setIdentifiers(ids);

        Identifier id = new Identifier();
        id.setId(icn);
        id.setOrganizationId(propertyLookup.getProperty("AssigningAuthority"));
        ids.add(id);

        // set SSN
        dpPatient.setSsn(HL7Parser201306.extractSSN(result1306));

        // set patient's name
        Map<String, String> nameMap = HL7Parser201306.extractPatientPersonNames(result1306);
        Personname pName = new Personname();
        pName.setPersonnameId(HL7Parser201305.PATIENT_NAME_ID);
        pName.setPrefix(nameMap.get("namePrefix"));
        pName.setFirstName(nameMap.get("nameGiven"));
        pName.setMiddleName(nameMap.get("nameMiddle"));
        pName.setLastName(nameMap.get("nameFamily"));
        pName.setSuffix(nameMap.get("nameSuffix"));
        List<Personname> personNames = new ArrayList<>();
        personNames.add(pName);

        // set Mother's Maiden Name
        String mmnStr = HL7Parser201306.extractMothersMaidenName(result1306);
        if (NullChecker.isNotNullOrEmpty(mmnStr)) {
            Personname mmn = new Personname();
            mmn.setPersonnameId(HL7Parser201305.MOTHERS_MAIDEN_NAME_ID);
            mmn.setLastName(mmnStr);
            personNames.add(mmn);
        }

        dpPatient.setPersonnames(personNames);

        Map<String, String> pobCityStateCountryMap = HL7Parser201306.extractPlaceOfBirth(result1306);
        if (pobCityStateCountryMap != null && pobCityStateCountryMap.size() > 0) {
            Address address = new Address();
            address.setAddressId(HL7Parser201305.POB_ADDRESS_ID);
            if (NullChecker.isNotNullOrEmpty(pobCityStateCountryMap.get("pobCity"))) {
                address.setCity(pobCityStateCountryMap.get("pobCity"));
            }
            if (NullChecker.isNotNullOrEmpty(pobCityStateCountryMap.get("pobState"))) {
                address.setState(pobCityStateCountryMap.get("pobState"));
            }
            // if
            // (NullChecker.isNotNullOrEmpty(pobCityStateCountryMap.get("pobCountry")))
            // {
            // address.set(pobCityStateCountryMap.get("pobCountry"));
            // }
            dpPatient.getAddresses().add(address);
        }

        DataQuery dqDemographics = dataManager.getQuery("Composite.findESRDemographics", parentQuery);
        dqDemographics.setParameter("icn", icn);
        List<Map> dqDemographicsResults = dqDemographics.getResults();
        if (NullChecker.isNotNullOrEmpty(dqDemographicsResults)) {
            Identifier patientFacilityId = new Identifier();
            patientFacilityId.setId((String)dqDemographicsResults.get(0).get("patientPreferredFacilityNumber") + "~" + (String)dqDemographicsResults.get(0).get("patientPreferredFacilityName"));
            patientFacilityId.setOrganizationId("PATIENT_FACILITY");
            ids.add(patientFacilityId);
        }

        // CCR 177986
        logger.debug("patient details from dataquery {}", dpPatient);

        return dpPatient;
    }

    private String getSenderOID(PRPAIN201305UV02 request)
    {
        String ret = null;

        if (request.getSender() != null && request.getSender().getDevice() != null && request.getSender().getDevice().getAsAgent() != null && request.getSender().getDevice().getAsAgent().getValue() != null && request.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization() != null && request.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue() != null && request.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId() != null && NullChecker.isNotNullOrEmpty(request.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId())) {
            ret = request.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId().get(0).getRoot();

            // CCR 177986
            logger.debug("Sender OID {}", ret);
        }

        return ret;
    }

    private String getICNFromPatient(Patient patient)
    {
        String ret = null;
        String assigningAuthority = propertyLookup.getProperty("AssigningAuthority");

        for (Identifier id : patient.getIdentifiers()) {
            if (id.getOrganizationId().equals(assigningAuthority)) {
                ret = id.getId();

                // CCR 177986
                logger.debug("ICN {}", ret);

                break;
            }
        }

        return ret;
    }

    private Identifier getAndRemovePatientFacilityIdFromPatient(Patient patient)
    {
        Identifier ret = null;
        String assigningAuthority = "PATIENT_FACILITY";
        int index = 0;

        for (Identifier id : patient.getIdentifiers()) {
            if (id.getOrganizationId().equals(assigningAuthority)) {
                ret = id;
                break;
            }
            ++index;
        }

        if (ret != null) {
            patient.getIdentifiers().remove(index);
        }

        return ret;
    }
}
