package gov.va.med.nhin.adapter.docretrieve;

import gov.hhs.fha.nhinc.common.nhinccommon.AssertionType;
import gov.hhs.fha.nhinc.common.nhinccommon.PersonNameType;
import gov.hhs.fha.nhinc.common.nhinccommonadapter.RespondingGatewayCrossGatewayRetrieveRequestType;
import gov.va.med.nhin.adapter.audit.Audit;
import gov.va.med.nhin.adapter.audit.AuditManager;
import gov.va.med.nhin.adapter.audit.AuditManagerLocal;
import gov.va.med.nhin.adapter.datamanager.DataManager;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.datamanager.ejb.DataManagerLocal;
import gov.va.med.nhin.adapter.documentrepository.Document;
import gov.va.med.nhin.adapter.documentrepository.DocumentRepository;
import gov.va.med.nhin.adapter.documentrepository.DocumentRepositoryLocal;
import gov.va.med.nhin.adapter.facilitymanager.Facility;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManager;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManagerLocal;
import gov.va.med.nhin.adapter.logging.AdHocRetrieveError;
import gov.va.med.nhin.adapter.logging.ErrorMessage;
import gov.va.med.nhin.adapter.logging.EventAuditingFactory;
import gov.va.med.nhin.adapter.logging.EventAuditingFactoryImpl;
import gov.va.med.nhin.adapter.logging.LogConstants.AuditingEvent;
import gov.va.med.nhin.adapter.stylesheet.ByteDS;
import gov.va.med.nhin.adapter.utils.AuditUtil;
import gov.va.med.nhin.adapter.utils.NullChecker;
import ihe.iti.xds_b._2007.RetrieveDocumentSetRequestType;
import ihe.iti.xds_b._2007.RetrieveDocumentSetRequestType.DocumentRequest;
import ihe.iti.xds_b._2007.RetrieveDocumentSetResponseType;
import ihe.iti.xds_b._2007.RetrieveDocumentSetResponseType.DocumentResponse;

import java.util.Date;
import java.util.List;

import javax.activation.DataHandler;
import javax.ejb.EJB;
import javax.ejb.*;
import javax.jws.HandlerChain;
import javax.jws.WebService;
import javax.xml.ws.soap.MTOM;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import oasis.names.tc.ebxml_regrep.xsd.rs._3.RegistryError;
import oasis.names.tc.ebxml_regrep.xsd.rs._3.RegistryErrorList;
import oasis.names.tc.ebxml_regrep.xsd.rs._3.RegistryResponseType;
import org.hl7.fhir.dstu3.model.AuditEvent;

/**
 *
 * @author David Vazquez
 */
@WebService(serviceName = "AdapterDocRetrieve",
            portName = "AdapterDocRetrievePortSoap11",
            endpointInterface = "gov.hhs.fha.nhinc.adapterdocretrieve.AdapterDocRetrievePortType",
            targetNamespace = "urn:gov:hhs:fha:nhinc:adapterdocretrieve"/*,
            wsdlLocation = "META-INF/wsdl/AdapterDocRetrieve.wsdl"*/)
@TransactionAttribute(value = TransactionAttributeType.SUPPORTS)
@Stateless(name = "DocRetrieve")
@MTOM(enabled = true)
@HandlerChain(file = "SOAPHandlerChain.xml")
public class DocRetrieve implements AdapterDocRetrievePortTypeLocal
{
    public static final Logger logger = LoggerFactory.getLogger(DocRetrieve.class.getName());

    static public final String XDS_RETRIEVE_RESPONSE_STATUS_FAILURE = "urn:oasis:names:tc:ebxml-regrep:ResponseStatusType:Failure";
    static public final String XDS_RETRIEVE_RESPONSE_STATUS_SUCCESS = "urn:oasis:names:tc:ebxml-regrep:ResponseStatusType:Success";
    static private final String REPOSITORY_UNIQUE_ID = "1";

    static private final String ERROR_XDS_REPOSITORY_ERROR = "XDSRepositoryError";
    static private final String ERROR_DESC_XDS_REPOSITORY_ERROR = "Internal Registry/Repository Error";
    static private final String ERROR_XDS_DOCUMENT_UNIQUE_ID_ERROR = "XDSDocumentUniqueIdError";
    static private final String ERROR_DESC_XDS_DOCUMENT_UNIQUE_ID_ERROR = "Document is not Available";
    static private final String ERROR_XDS_MISSING_HOME_COMMUNITY_ID = "XDSMissingHomeCommunityId";
    static private final String ERROR_DESC_XDS_MISSING_HOME_COMMUNITY_ID = "A value for the homeCommunityId is required and has not been specified";
    static private final String ERROR_XDS_UNKNOWN_COMMUNITY = "XDSUnknownCommunity";
    static private final String ERROR_DESC_XDS_UNKNOWN_COMMUNITY = "A value for the homeCommunityId is not recognized";
    static private final String ERROR_XDS_UNKNOWN_REPOSITORY_ID = "XDSUnknownRepositoryId";
    static private final String ERROR_DESC_XDS_UNKNOWN_REPOSITORY_ID = "A value for repositoryUniqueId is not recognized";

    private AuditManager auditManager;
    private FacilityManager facilityManager;
    private DocumentRepository documentRepository;
    private DataManager dataManager;

    @EJB(beanInterface = AuditManagerLocal.class, beanName = "AuditManager")
    public void setAuditManager(AuditManager auditManager)
    {
        this.auditManager = auditManager;
    }

    @EJB(beanInterface = FacilityManagerLocal.class, beanName = "FacilityManager")
    public void setFacilityManager(FacilityManager facilityManager)
    {
        this.facilityManager = facilityManager;
    }

    @EJB(beanInterface = DocumentRepositoryLocal.class, beanName = "DocumentRepository")
    public void setDocumentRepository(DocumentRepository documentRepository)
    {
        this.documentRepository = documentRepository;
    }

    @EJB(beanInterface = DataManagerLocal.class, beanName = "DataManager")
    public void setDataManager(DataManager dataManager)
    {
        this.dataManager = dataManager;
    }

    @Override
    public RetrieveDocumentSetResponseType respondingGatewayCrossGatewayRetrieve(RespondingGatewayCrossGatewayRetrieveRequestType respondingGatewayCrossGatewayRetrieveRequest) {
        DataQuery parentQuery = dataManager.getQuery("JustCache");
        parentQuery.getResults();
        return respondingGatewayCrossGatewayRetrieve(respondingGatewayCrossGatewayRetrieveRequest, parentQuery);
    }

    @Override
    public RetrieveDocumentSetResponseType respondingGatewayCrossGatewayRetrieve(RespondingGatewayCrossGatewayRetrieveRequestType respondingGatewayCrossGatewayRetrieveRequest, DataQuery parentQuery)
    {
        EventAuditingFactory<AuditEvent> afac
                                         = EventAuditingFactoryImpl.getFactory(AuditEvent.class);
        logger.debug("respondingGatewayCrossGatewayRetrieve() invoked");

        RetrieveDocumentSetRequestType body = respondingGatewayCrossGatewayRetrieveRequest.getRetrieveDocumentSetRequest();
        RetrieveDocumentSetResponseType ret = createRetrieveDocumentSetResponse();

        AuditEvent eventGatherData = afac.newEvent(AuditingEvent.RDIN_GATHER, getClass());

        try {
            Facility homeFacility = facilityManager.getFacilityByFacilityNumber("VA");

            for (DocumentRequest dr : body.getDocumentRequest()) {
                Document doc = null;

                String error = null;
                String errdsc = null;
                ErrorMessage errmsg = null;

                if (NullChecker.isNullOrEmpty(dr.getHomeCommunityId())) {
                    errmsg = ErrorMessage.IN_DR_NO_HCID;
                    error = ERROR_XDS_MISSING_HOME_COMMUNITY_ID;
                    errdsc = ERROR_DESC_XDS_MISSING_HOME_COMMUNITY_ID;
                }
                else if (!dr.getHomeCommunityId().equals(homeFacility.getFullHomeCommunityId())) {
                    errmsg = ErrorMessage.IN_DR_UNKNOWN_HCID;
                    error = ERROR_XDS_UNKNOWN_COMMUNITY;
                    errdsc = ERROR_DESC_XDS_UNKNOWN_COMMUNITY;
                }
                else if (NullChecker.isNullOrEmpty(dr.getRepositoryUniqueId()) || !(dr.getRepositoryUniqueId().equals(REPOSITORY_UNIQUE_ID) || dr.getRepositoryUniqueId().equals(homeFacility.getHomeCommunityId() + "." + REPOSITORY_UNIQUE_ID))) {
                    errmsg = ErrorMessage.IN_DR_UNKNOWN_REPO_ID;
                    error = ERROR_XDS_UNKNOWN_REPOSITORY_ID;
                    errdsc = ERROR_DESC_XDS_UNKNOWN_REPOSITORY_ID;
                }
                else if (NullChecker.isNullOrEmpty(dr.getDocumentUniqueId()) || (doc = documentRepository.getDocumentByDocumentUniqueId(dr.getDocumentUniqueId())) == null) {
                    errmsg = ErrorMessage.IN_DR_DOC_ID;
                    error = ERROR_XDS_DOCUMENT_UNIQUE_ID_ERROR;
                    errdsc = ERROR_DESC_XDS_DOCUMENT_UNIQUE_ID_ERROR;
                }

                if (null == error) {
                    //if (doc.getRawData() == null) {
                    if (doc != null && doc.getRawData() == null) {
                        generateDocument(doc, parentQuery);

                        eventGatherData.setOutcome(AuditEvent.AuditEventOutcome._0);
                        afac.info(eventGatherData);
                    }

                    doc.setLastAccessedTime(new Date());
                    documentRepository.storeDocument(doc);

                    addDocumentToRetrieveDocumentSetResponse(ret, doc, dr);
                    auditRetrieve(doc, respondingGatewayCrossGatewayRetrieveRequest.getAssertion());
                }
                else {
                    eventGatherData.setOutcome(AuditEvent.AuditEventOutcome._4);
                    eventGatherData.setOutcomeDesc(errmsg.getMessage());
                    afac.debug(eventGatherData);

                    addErrorToRetrieveDocumentSetResponse(ret, error, errdsc);
                    AdHocRetrieveError.retrieveError(respondingGatewayCrossGatewayRetrieveRequest, dr, errmsg);
                }
            }

            setRegistryResponseStatus(ret);
        }
        // CCR 179231- no need of using Throwable to catch errors other than
        // application errors
        catch (Exception e) {
            String hcid;
            String hcidName;
            if (respondingGatewayCrossGatewayRetrieveRequest.getAssertion() != null && respondingGatewayCrossGatewayRetrieveRequest.getAssertion().getHomeCommunity() != null) {
                hcid = respondingGatewayCrossGatewayRetrieveRequest.getAssertion().getHomeCommunity().getHomeCommunityId();
                hcidName = respondingGatewayCrossGatewayRetrieveRequest.getAssertion().getHomeCommunity().getName();
            }
            else {
                hcid = "N/A";
                hcidName = "Not Available";
            }
            eventGatherData.setOutcome(AuditEvent.AuditEventOutcome._8);
            eventGatherData.setOutcomeDesc(ErrorMessage.IN_DR_UNKNOWN.getMessage());
            afac.info(eventGatherData);

            // CCR 177986 -logger with slf4j/log4j2
            logger.error("Error processing DocRetrieve from {} {} - {}", hcid, hcidName, e.getMessage());
            logger.error("Stack trace for DocRetrieve {}", e);

            addErrorToRetrieveDocumentSetResponse(ret, ERROR_XDS_REPOSITORY_ERROR, ERROR_DESC_XDS_REPOSITORY_ERROR);
            AdHocRetrieveError.retrieveError(respondingGatewayCrossGatewayRetrieveRequest, ErrorMessage.IN_DR_UNKNOWN, e.getMessage());
        }

        return ret;
    }

    private void generateDocument(Document document, DataQuery parentQuery) throws Exception
    {
        DataQuery dataQuery = dataManager.getQuery(document.getDocGenQueryName(), parentQuery);

        for (String param : document.getDocGenQueryParams().split(",")) {
            String[] p = param.split("=");
            dataQuery.setParameter(p[0], (p.length < 2 ? "" : p[1]));
        }

        List<Document> results = dataQuery.getResults(Document.class);

        if (!NullChecker.isNullOrEmpty(results)) {
            Document resultDoc = results.get(0);
            document.setRawData(resultDoc.getRawData());
            document.setHash(resultDoc.getHash());
            document.setSize(resultDoc.getSize());
            document.setAvailabilityStatus(resultDoc.getAvailabilityStatus());
        }
        else {
            throw new Exception("There was an error generating the document.");
        }
    }

    private RetrieveDocumentSetResponseType createRetrieveDocumentSetResponse()
    {
        ihe.iti.xds_b._2007.ObjectFactory retObjFactory = new ihe.iti.xds_b._2007.ObjectFactory();
        RetrieveDocumentSetResponseType ret = retObjFactory.createRetrieveDocumentSetResponseType();

        ret.setRegistryResponse(createRegistryResponse());

        return ret;
    }

    private RegistryResponseType createRegistryResponse()
    {
        oasis.names.tc.ebxml_regrep.xsd.rs._3.ObjectFactory retObjFactory = new oasis.names.tc.ebxml_regrep.xsd.rs._3.ObjectFactory();
        RegistryResponseType ret = retObjFactory.createRegistryResponseType();
        return ret;
    }

    private void addDocumentToRetrieveDocumentSetResponse(RetrieveDocumentSetResponseType ret, Document doc, DocumentRequest dr)
    {
        DocumentResponse documentResponse = new DocumentResponse();

        documentResponse.setHomeCommunityId(dr.getHomeCommunityId());
        documentResponse.setRepositoryUniqueId(dr.getRepositoryUniqueId());
        documentResponse.setDocumentUniqueId(doc.getDocumentUniqueId());
        documentResponse.setMimeType(doc.getMimeType());
        ByteDS ds = new ByteDS(doc.getRawData());
        documentResponse.setDocument(new DataHandler(ds));

        ret.getDocumentResponse().add(documentResponse);
    }

    // CCR 177986 -logger with slf4j/log4j2
    private void addErrorToRetrieveDocumentSetResponse(RetrieveDocumentSetResponseType ret, String errorCode, String codeContext)
    {
        RegistryError registryError = createRegistryError(errorCode, codeContext);

        // CCR 177986 -logger with slf4j/log4j2
        if (logger.isDebugEnabled() || logger.isErrorEnabled()) {
            if (ret.getRegistryResponse().getRegistryErrorList() == null) {
                ret.getRegistryResponse().setRegistryErrorList(createRegistryErrorList());
            }

            ret.getRegistryResponse().getRegistryErrorList().getRegistryError().add(registryError);
        }
    }

    private RegistryErrorList createRegistryErrorList()
    {
        oasis.names.tc.ebxml_regrep.xsd.rs._3.ObjectFactory retObjFactory = new oasis.names.tc.ebxml_regrep.xsd.rs._3.ObjectFactory();
        RegistryErrorList ret = retObjFactory.createRegistryErrorList();
        ret.setHighestSeverity("urn:oasis:names:tc:ebxml-regrep:ErrorSeverityType:Error");
        return ret;
    }

    private RegistryError createRegistryError(String errorCode, String codeContext)
    {
        oasis.names.tc.ebxml_regrep.xsd.rs._3.ObjectFactory retObjFactory = new oasis.names.tc.ebxml_regrep.xsd.rs._3.ObjectFactory();
        RegistryError ret = retObjFactory.createRegistryError();
        ret.setSeverity("urn:oasis:names:tc:ebxml-regrep:ErrorSeverityType:Error");
        ret.setErrorCode(errorCode);
        ret.setCodeContext(codeContext);
        ret.setLocation(getHomeFacility().getFullHomeCommunityId());
        return ret;
    }

    private void auditRetrieve(Document doc, AssertionType assertion)
    {
        Audit audit = new Audit();
        audit.setAction("Retrieve Document");
        audit.setAuditTime(new Date());
        audit.setDocumentId(doc.getDocumentUniqueId());
        audit.setPatientId(doc.getPatientId());
        audit.setPatientSSN(doc.getPatientSSN());
        audit.setPatientLastName(doc.getPatientLastName());
        audit.setPatientGivenName(doc.getPatientGivenName());
        audit.setPatientFacilityNumber(doc.getPatientPreferredFacilityNumber());
        audit.setPatientFacilityName(doc.getPatientPreferredFacilityName());
        audit.setOrganizationId(assertion.getHomeCommunity().getHomeCommunityId());
        if (assertion.getPurposeOfDisclosureCoded() != null && assertion.getPurposeOfDisclosureCoded().getCode() != null) {
            audit.setPurposeForUse(assertion.getPurposeOfDisclosureCoded().getCode());
        }
        audit.setUserId(assertion.getUserInfo().getUserName());

        audit.setSystemId(AuditUtil.checkSystemId(assertion));

        audit.setUserRole(assertion.getUserInfo().getRoleCoded().getCode());
        audit.setUserFacilityNumber(assertion.getUserInfo().getOrg().getHomeCommunityId());
        audit.setUserFacilityName(assertion.getUserInfo().getOrg().getName());
        if (assertion.getUserInfo().getPersonName() != null) {
            PersonNameType personName = assertion.getUserInfo().getPersonName();
            if (!NullChecker.isNullOrEmpty(personName.getFullName())) {
                audit.setUserName(personName.getFullName());
            }
            else {
                audit.setUserName(personName.getGivenName() + " " + personName.getFamilyName());
            }
        }

        audit.setDocSpecType(doc.getFormatCodeDisplayName());

        auditManager.storeAudit(audit);
    }

    void setRegistryResponseStatus(RetrieveDocumentSetResponseType ret)
    {
        if (ret.getRegistryResponse().getRegistryErrorList() != null) {
            ret.getRegistryResponse().setStatus(XDS_RETRIEVE_RESPONSE_STATUS_FAILURE);
        }
        else {
            ret.getRegistryResponse().setStatus(XDS_RETRIEVE_RESPONSE_STATUS_SUCCESS);
        }
    }

    private Facility getHomeFacility()
    {
        return facilityManager.getFacilityByFacilityNumber("VA");
    }
}
