package gov.va.med.nhin.adapter.adaptergateway.patientdiscovery;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;

import javax.ejb.EJB;
import javax.ejb.*;

import org.apache.commons.collections.CollectionUtils;
import org.hl7.v3.AddPatientCorrelationRequestType;
import org.hl7.v3.CommunityPRPAIN201306UV02ResponseType;
import org.hl7.v3.II;
import org.hl7.v3.MCCIMT000100UV01Receiver;
import org.hl7.v3.PRPAIN201301UV02;
import org.hl7.v3.PRPAIN201305UV02;
import org.hl7.v3.PRPAIN201306UV02;
import org.hl7.v3.PRPAIN201306UV02MFMIMT700711UV01Subject1;
import org.hl7.v3.PRPAMT201301UV02Patient;
import org.hl7.v3.PRPAMT201310UV02Patient;
import org.hl7.v3.RespondingGatewayPRPAIN201305UV02RequestType;
import org.hl7.v3.RespondingGatewayPRPAIN201306UV02ResponseType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.hhs.fha.nhinc.common.nhinccommon.AssertionType;
import gov.hhs.fha.nhinc.common.nhinccommon.NhinTargetCommunityType;
import gov.hhs.fha.nhinc.common.nhinccommon.PersonNameType;
import gov.hhs.fha.nhinc.patientdb.model.Patient;
import gov.hhs.fha.nhinc.patientdb.model.Personname;
import gov.hhs.fha.nhinc.transform.subdisc.HL7PRPA201301Transforms;
import gov.hhs.fha.nhinc.transform.subdisc.HL7PRPA201305Transforms;
import gov.hhs.fha.nhinc.transform.subdisc.HL7PatientTransforms;
import gov.hhs.fha.nhinc.transform.subdisc.HL7ReceiverTransforms;
import gov.va.med.nhin.adapter.adaptergateway.workmanager.WorkManagerExecutorServiceLocal;
import gov.va.med.nhin.adapter.audit.Audit;
import gov.va.med.nhin.adapter.audit.AuditManager;
import gov.va.med.nhin.adapter.audit.AuditManagerLocal;
import gov.va.med.nhin.adapter.audit.requests.RequestActions;
import gov.va.med.nhin.adapter.audit.requests.RequestAudit;
import gov.va.med.nhin.adapter.audit.requests.RequestAuditEntity;
import gov.va.med.nhin.adapter.datamanager.DataManager;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.datamanager.ejb.DataManagerLocal;
import gov.va.med.nhin.adapter.facilitymanager.Facility;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManager;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManagerLocal;
import gov.va.med.nhin.adapter.facilitymanager.OperationOnOff;
import gov.va.med.nhin.adapter.logging.AdapterPDError;
import gov.va.med.nhin.adapter.logging.ErrorMessage;
import gov.va.med.nhin.adapter.logging.EventAuditingFactory;
import gov.va.med.nhin.adapter.logging.EventAuditingFactoryImpl;
import gov.va.med.nhin.adapter.logging.EventAuditingThreadData;
import gov.va.med.nhin.adapter.logging.LogConstants;
import gov.va.med.nhin.adapter.logging.LogConstants.AuditingEvent;
import gov.va.med.nhin.adapter.logging.MessagingHelper;
import gov.va.med.nhin.adapter.mpi.hl7parsers.HL7DbParser201306;
import gov.va.med.nhin.adapter.mvi.AdapterMviPortTypeLocal;
import gov.va.med.nhin.adapter.mvi.hl7parsers.HL7Parser201305;
import gov.va.med.nhin.adapter.mvi.hl7parsers.HL7Parser201306;
import gov.va.med.nhin.adapter.patientcorrelation.PatientCorrelationPortTypeLocal;
import gov.va.med.nhin.adapter.patientcorrelation.model.Correlation;
import gov.va.med.nhin.adapter.patientcorrelation.model.Correlations;
import gov.va.med.nhin.adapter.utils.AuditUtil;
import gov.va.med.nhin.adapter.utils.NullChecker;
import java.util.ListIterator;
import org.hl7.fhir.dstu3.model.AuditEvent;
import org.slf4j.MDC;

/**
 *
 * @author David Vazquez
 */
// No need for this to be a web service right now.
//@WebService(serviceName = "EntityPatientDiscovery", portName = "EntityPatientDiscoveryPortSoap", endpointInterface = "gov.hhs.fha.nhinc.entitypatientdiscovery.EntityPatientDiscoveryPortType", targetNamespace = "urn:gov:hhs:fha:nhinc:entitypatientdiscovery", wsdlLocation = "META-INF/wsdl/EntityPatientDiscovery.wsdl")
//@BindingType(value = javax.xml.ws.soap.SOAPBinding.SOAP12HTTP_BINDING)
@TransactionAttribute(value = TransactionAttributeType.SUPPORTS)
@Stateless(name = "AdapterGatewayPatientDiscovery")
public class AdapterGatewayPatientDiscovery implements EntityPatientDiscoveryPortTypeLocal
{
    private static final Logger logger = LoggerFactory.getLogger(AdapterGatewayPatientDiscovery.class.getName());
    
    public class SendRequestResponse
    {
        public PRPAIN201306UV02 prpain201306UV02;
        public String remoteHomeCommunityId;
        public SendRequestException sre = null;

        public boolean hasError()
        {
            return (sre != null);
        }
    }

    public class SendRequestException extends Exception
    {

        private static final long serialVersionUID = 1L;

        public PRPAIN201305UV02 prpain201305UV02;
        public String remoteHomeCommunityId;
    }

    private class SendRequestCallable implements Callable<SendRequestResponse>
    {
        private final PRPAIN201305UV02 prpain201305UV02;
        private final AssertionType assertion;
        private final String remoteHomeCommunityId;
        private final EventAuditingFactory<AuditEvent> eventauditor;
        private final Map<String, String> mdcmap = MDC.getCopyOfContextMap();
        private String patientDemographics = "";

        public SendRequestCallable(PRPAIN201305UV02 prpain201305UV02, AssertionType assertion, String remoteHomeCommunityId,
                                   EventAuditingFactory<AuditEvent> afac, String patientDemographics)
        {
            this.prpain201305UV02 = prpain201305UV02;
            this.assertion = assertion;
            this.remoteHomeCommunityId = remoteHomeCommunityId;
            this.eventauditor = afac;
            this.patientDemographics = patientDemographics;
        }

        @Override
        public SendRequestResponse call() throws Exception
        {
            MDC.setContextMap(mdcmap);
            MessagingHelper<AuditEvent> msg = eventauditor.messaging();

            SendRequestResponse ret = new SendRequestResponse();
            RequestAuditEntity auditEntity = new RequestAuditEntity();
            auditEntity.setStartTime(new Date());
            if (prpain201305UV02 != null && prpain201305UV02.getControlActProcess() != null && prpain201305UV02.getControlActProcess().getQueryByParameter() != null && prpain201305UV02.getControlActProcess().getQueryByParameter().getValue() != null) {
                if (prpain201305UV02.getControlActProcess().getQueryByParameter().getValue().getQueryId() != null) {
                    auditEntity.setUuid(prpain201305UV02.getControlActProcess().getQueryByParameter().getValue().getQueryId().getExtension());
                }
                else {
                    auditEntity.setUuid("Error getting UUID");
                }
            }
            auditEntity.setHcid(remoteHomeCommunityId);
            auditEntity.setAction(RequestActions.PATIENT_DISCOVERY.getValue());

            AuditEvent ev = msg.reqres(AuditingEvent.PDOUT_PD, getClass(), remoteHomeCommunityId);

            try {
                ret.prpain201306UV02 = AdapterGatewayPatientDiscovery.this.nhinPatientDiscoverySender.sendRequest(prpain201305UV02, assertion, remoteHomeCommunityId, ev);
                ret.remoteHomeCommunityId = remoteHomeCommunityId;
                auditEntity.setStopTime(new Date());
                if (ret.prpain201306UV02.getControlActProcess().getQueryByParameter().getValue().getParameterList().getId() != null) {
                    auditEntity.setPid(ret.prpain201306UV02.getControlActProcess().getQueryByParameter().getValue().getParameterList().getId().getExtension());
                    msg.addPatientId(ev, auditEntity.getPid());
                }
                else {
                    auditEntity.setPid("Error getting pid");
                }

                String queryResponseCode = ret.prpain201306UV02.getControlActProcess().
                        getQueryAck().getQueryResponseCode().getCode();
                final boolean responseOk = "ok".equalsIgnoreCase(queryResponseCode);
                auditEntity.setStatus(queryResponseCode);
                // check if successful PD is false and query response code is 'ok'
                // from the partner
                if ((!AdapterGatewayPatientDiscovery.this.successfulPD) && responseOk) {
                    AdapterGatewayPatientDiscovery.this.successfulPD = true;
                }
                requestAudit.storeAudit(auditEntity);

                // Add Announce success audit if Query Response Code is 'ok'
                if (queryResponseCode != null && responseOk) {
                    ev.setOutcome(AuditEvent.AuditEventOutcome._0);
                    eventauditor.info(ev);

                    auditStatus(prpain201305UV02, assertion, getLocalPatientId(prpain201305UV02),
                                remoteHomeCommunityId, patientDemographics);
                }
                else {
                    ev.setOutcome(AuditEvent.AuditEventOutcome._4);
                    ev.setOutcomeDesc("no patient information provided");
                    eventauditor.info(ev);

                    auditStatus(prpain201305UV02, assertion, getLocalPatientId(prpain201305UV02),
                                remoteHomeCommunityId, "Failure - Partner did not respond with patient "
                                                       + "information " + patientDemographics);
                }
            }
            catch (Throwable e) {
                // Handle all errors on a per thread basis independently above.
                SendRequestException sre = new SendRequestException();
                sre.prpain201305UV02 = prpain201305UV02;
                sre.remoteHomeCommunityId = remoteHomeCommunityId;
                sre.initCause(e);
                ret.sre = sre;
                auditEntity.setStatus("Failed: " + e.getMessage());
                requestAudit.storeAudit(auditEntity);

                ev.setOutcome(AuditEvent.AuditEventOutcome._8);
                ev.setOutcomeDesc(e.getMessage());
                eventauditor.error(ev);
            }
            return ret;
        }
    }

    private PatientCorrelationPortTypeLocal adapterPatientCorrelation;
    private FacilityManager facilityManager;
    private AdapterMviPortTypeLocal adapterMvi;
    private NHINPatientDiscoverySender nhinPatientDiscoverySender;
    private ExecutorService executorService;
    private RequestAudit requestAudit;
    private DataManager dataManager;
    private boolean successfulPD = false;

    private AuditManager auditManager;

    @EJB(beanInterface = AuditManagerLocal.class, beanName = "AuditManager")
    public void setAuditManager(AuditManager auditManager)
    {
        this.auditManager = auditManager;
    }

    @EJB(beanInterface = FacilityManagerLocal.class, beanName = "FacilityManager")
    public void setFacilityManager(FacilityManager facilityManager)
    {
        this.facilityManager = facilityManager;
    }

    @EJB(beanInterface = AdapterMviPortTypeLocal.class, beanName = "AdapterMVI")
    public void setAdapterMvi(AdapterMviPortTypeLocal adapterMvi)
    {
        this.adapterMvi = adapterMvi;
    }

    @EJB(beanInterface = PatientCorrelationPortTypeLocal.class, beanName = "AdapterPatientCorrelation")
    public void setAdapterPatientCorrelation(PatientCorrelationPortTypeLocal adapterPatientCorrelation)
    {
        this.adapterPatientCorrelation = adapterPatientCorrelation;
    }

    @EJB(beanInterface = NHINPatientDiscoverySender.class, beanName = "NhincProxyNHINPatientDiscoverySender")
    public void setNhinPatientDiscoverySender(NHINPatientDiscoverySender nhinPatientDiscoverySender)
    {
        this.nhinPatientDiscoverySender = nhinPatientDiscoverySender;
    }

    @EJB(beanInterface = DataManagerLocal.class, beanName = "DataManager")
    public void setDataManager(DataManager dataManager)
    {
        this.dataManager = dataManager;
    }

    //@EJB(beanInterface = WorkManagerExecutorServiceLocal.class, beanName = "WorkManagerExecutorService") //changed 1/24/2017
    @EJB(beanInterface = WorkManagerExecutorServiceLocal.class, beanName = "OutboundPDWorkManagerExecutorService")
    public void setExecutorService(ExecutorService executorService)
    {
        this.executorService = executorService;
    }

    @EJB(beanInterface = RequestAudit.class, beanName = "RequestAudit")
    public void setRequestAudit(RequestAudit requestAudit)
    {
        this.requestAudit = requestAudit;
    }
    
    @Override
    public RespondingGatewayPRPAIN201306UV02ResponseType respondingGatewayPRPAIN201305UV02(RespondingGatewayPRPAIN201305UV02RequestType respondingGatewayPRPAIN201305UV02Request) {
        DataQuery parentQuery = dataManager.getQuery("JustCache");
        parentQuery.getResults();
        return respondingGatewayPRPAIN201305UV02(respondingGatewayPRPAIN201305UV02Request, null, parentQuery);
    }
    
    @Override
    public RespondingGatewayPRPAIN201306UV02ResponseType respondingGatewayPRPAIN201305UV02(RespondingGatewayPRPAIN201305UV02RequestType respondingGatewayPRPAIN201305UV02Request, String patientDemographics, DataQuery parentQuery)
    {
        EventAuditingFactory<AuditEvent> afac
                                         = EventAuditingFactoryImpl.getFactory(AuditEvent.class);
        afac.info(afac.newEvent(respondingGatewayPRPAIN201305UV02Request.getAssertion(),
                                LogConstants.AuditingEvent.PDOUT_BEGIN, getClass()));

        this.successfulPD = false;
        String partnerCommunityId = respondingGatewayPRPAIN201305UV02Request.getAssertion().getHomeCommunity().getHomeCommunityId();

        logger.debug("EnteringAdapterGatewayPatientDiscovery Call  and Partner homecommunity ID is {} ", partnerCommunityId);

        RespondingGatewayPRPAIN201306UV02ResponseType ret = new RespondingGatewayPRPAIN201306UV02ResponseType();
        ExecutorCompletionService<SendRequestResponse> completionService = new ExecutorCompletionService<>(executorService);

        int numSubmitted = 0;
        II localPatientId = getLocalPatientId(respondingGatewayPRPAIN201305UV02Request.getPRPAIN201305UV02());
        AssertionType assertion = respondingGatewayPRPAIN201305UV02Request.getAssertion();

        AuditEvent partnerevent = afac.newEvent(AuditingEvent.PDOUT_GETPARTNERS, getClass());
        String psource = "request";
        List<Facility> facilities = getFacilitiesFromRequest(respondingGatewayPRPAIN201305UV02Request);
        if (CollectionUtils.isEmpty(facilities)) {
            facilities = facilityManager.getAllFacilities("VA"); //remove VA facility to avoid self-querying
        }

        partnerevent.setOutcomeDesc(facilities.size() + " Facilities from " + psource);
        partnerevent.setOutcome(AuditEvent.AuditEventOutcome._0);
        afac.info(partnerevent);

        for (Facility facility : facilities) {
            String remoteHomeCommunityId = facility.getHomeCommunityId();

            AuditEvent event = afac.messaging().partnerauth(AuditingEvent.PDOUT_PARTNERAUTH,
                                                            getClass(), remoteHomeCommunityId);

            if (facilityManager.isPartnerAllowed(remoteHomeCommunityId, OperationOnOff.ONBOARD)) {
                if (facilityManager.isPartnerAllowed(remoteHomeCommunityId, OperationOnOff.OUT_PD)) {
                    afac.info(event.setOutcome(AuditEvent.AuditEventOutcome._0));

                    PRPAIN201305UV02 prpain201305UV02 = createPRPAIN201305UV02(respondingGatewayPRPAIN201305UV02Request.getPRPAIN201305UV02(), remoteHomeCommunityId);
                    logger.info("Sending PRPAIN201305UV02 request for localPatientId: '" + localPatientId.getExtension() + "' to Remote Community ID'");
                    SendRequestCallable callable = new SendRequestCallable(prpain201305UV02,
                                                                           respondingGatewayPRPAIN201305UV02Request.getAssertion(), remoteHomeCommunityId, afac, patientDemographics);

                    completionService.submit(callable);
                    ++numSubmitted;
                }
                else {
                    addResultToResponse(ret, createErrorResponse(createPRPAIN201305UV02(respondingGatewayPRPAIN201305UV02Request.getPRPAIN201305UV02(), remoteHomeCommunityId), remoteHomeCommunityId, remoteHomeCommunityId));
                    AdapterPDError.queryError(null, ErrorMessage.OUT_PD_DISABLED, remoteHomeCommunityId);

                    event.setOutcome(AuditEvent.AuditEventOutcome._4);
                    event.setOutcomeDesc(ErrorMessage.OUT_PD_DISABLED.getMessage());
                    afac.error(event);
                }
            }
            else {
                event.setOutcome(AuditEvent.AuditEventOutcome._4);
                event.setOutcomeDesc(ErrorMessage.OUT_PD_NOT_A_PARTNER.getMessage());
                afac.error(event);

                AdapterPDError.queryError(null, ErrorMessage.OUT_PD_NOT_A_PARTNER, remoteHomeCommunityId);
            }
        }

        for (int i = 0; i < numSubmitted; ++i) {
            try {
                SendRequestResponse response = completionService.take().get();
                if (!response.hasError()) {
                    processResult(response.prpain201306UV02, localPatientId, response.remoteHomeCommunityId, assertion, parentQuery);
                    addResultToResponse(ret, response.prpain201306UV02);
                }
                else {
                    SendRequestException sre = response.sre;
                    addResultToResponse(ret, createErrorResponse(sre.prpain201305UV02, sre.remoteHomeCommunityId, sre.remoteHomeCommunityId));

                    // CCR 177986
                    logger.error("Error sending PatientDiscovery to {} - {}", new Object[]{sre.remoteHomeCommunityId, sre});
                    // Log if the PD call generated an error to the partner
                    auditStatus(respondingGatewayPRPAIN201305UV02Request.getPRPAIN201305UV02(), assertion, localPatientId, sre.remoteHomeCommunityId, "Failure - Partner did not respond with patient information " + patientDemographics);
                }
            }
            catch (InterruptedException ie) {
                logger.error("processing an outbound patient discovery request was interrupted {}.", ie);
                afac.error(afac.newEvent(AuditingEvent.PDOUT_PARTNERAUTH, getClass())
                        .setOutcome(AuditEvent.AuditEventOutcome._8)
                        .setOutcomeDesc(ie.getLocalizedMessage()));
            }
            catch (ExecutionException e) {
                logger.error("An exception occured processing an outbound patient discovery request.", e);
                afac.error(afac.newEvent(AuditingEvent.PDOUT_PARTNERAUTH, getClass())
                        .setOutcome(AuditEvent.AuditEventOutcome._8)
                        .setOutcomeDesc(e.getLocalizedMessage()));
            }
        }

        // Audit Entire PD outbound process status
        // If At least one partner responds with queryResponseCode of 'OK', its
        // a success.
        try {
            auditPDStatus(respondingGatewayPRPAIN201305UV02Request.getAssertion(), localPatientId);
        }
        catch (Throwable ex) {
            logger.error("Exception while auditing PD Outbound process status to database. Please correct this.", ex);
        }

        afac.debug(afac.newEvent(respondingGatewayPRPAIN201305UV02Request.getAssertion(),
                                 LogConstants.AuditingEvent.PDOUT_END, getClass()));
        return ret;
    }

    private void auditPDStatus(AssertionType assertion, II localPatientId)
    {
        Audit audit = new Audit();
        audit.setAction("PatientDiscoveryOut");
        if (this.successfulPD) {
            audit.setDetails("Success");
        }
        else {
            audit.setDetails("Failure - None of the partners reponded with patient information");
        }
        audit.setAuditTime(new Date());
        audit.setUserId(assertion.getUserInfo().getUserName());
        audit.setSystemId(AuditUtil.checkSystemId(assertion));
        audit.setUserRole(assertion.getUserInfo().getRoleCoded().getCode());
        audit.setUserFacilityNumber(assertion.getUserInfo().getOrg().getHomeCommunityId());
        audit.setUserFacilityName(assertion.getUserInfo().getOrg().getName());
        if (assertion.getUserInfo().getPersonName() != null) {
            PersonNameType personName = assertion.getUserInfo().getPersonName();
            if (!NullChecker.isNullOrEmpty(personName.getFullName())) {
                audit.setUserName(personName.getFullName());
            }
            else {
                audit.setUserName(personName.getGivenName() + " " + personName.getFamilyName());
            }
        }
        audit.setPurposeForUse(assertion.getPurposeOfDisclosureCoded().getCode());
        audit.setOrganizationId(assertion.getHomeCommunity().getHomeCommunityId());
        audit.setPatientId(localPatientId.getExtension());
        // audit.setPatientSSN(localPatientId.getPatientSSN());
        // audit.setPatientGivenName(document.getPatientGivenName());
        // audit.setPatientLastName(document.getPatientLastName());
        // audit.setPatientFacilityNumber(document.getPatientPreferredFacilityNumber());
        // audit.setPatientFacilityName(document.getPatientPreferredFacilityName());

        // audit.setDocSpecType(document.getFormatCodeDisplayName());
        auditManager.storeAudit(audit);
    }

    // Announcement to the partner generated an error and log it
    private void auditStatus(PRPAIN201305UV02 prpaiN201305UV02, AssertionType assertion, II localPatientId, String remoteHomeCommunityId, String details)
    {
        Audit audit = new Audit();
        audit.setAction("Announce");
        audit.setDetails(details);
        audit.setAuditTime(new Date());
        audit.setUserId(assertion.getUserInfo().getUserName());

        audit.setSystemId(AuditUtil.checkSystemId(assertion));

        audit.setUserRole(assertion.getUserInfo().getRoleCoded().getCode());
        audit.setUserFacilityNumber(assertion.getUserInfo().getOrg().getHomeCommunityId());
        audit.setUserFacilityName(assertion.getUserInfo().getOrg().getName());
        if (assertion.getUserInfo().getPersonName() != null) {
            PersonNameType personName = assertion.getUserInfo().getPersonName();
            if (!NullChecker.isNullOrEmpty(personName.getFullName())) {
                audit.setUserName(personName.getFullName());
            }
            else {
                audit.setUserName(personName.getGivenName() + " " + personName.getFamilyName());
            }
        }
        audit.setPurposeForUse(assertion.getPurposeOfDisclosureCoded().getCode());
        audit.setOrganizationId(assertion.getHomeCommunity().getHomeCommunityId());
        audit.setPatientId(localPatientId.getExtension());

        Facility remoteFacility = facilityManager.getFacilityByHomeCommunityId(remoteHomeCommunityId);
        audit.setRemoteOrganization(remoteFacility);
        audit.setRemoteOrganizationId(remoteFacility.getFullHomeCommunityId());

        getPatientName(prpaiN201305UV02, audit);
        auditManager.storeAudit(audit);
    }

    private void getPatientName(PRPAIN201305UV02 request, Audit audit)
    {
        List<Personname> patientNameList = HL7Parser201305.extractPersonnames(request.getControlActProcess().getQueryByParameter().getValue().getParameterList());
        for (Personname person : patientNameList) {
            audit.setPatientGivenName(person.getFirstName());
            audit.setPatientLastName(person.getLastName());
        }

        String ssn = HL7Parser201305.extractSSNIdentifier(request.getControlActProcess().getQueryByParameter().getValue().getParameterList());
        audit.setPatientSSN(ssn);
    }

    private List<Facility> getFacilitiesFromRequest(RespondingGatewayPRPAIN201305UV02RequestType request)
    {
        List<Facility> facilities = new ArrayList<>();

        if (request.getNhinTargetCommunities() != null
            && NullChecker.isNotNullOrEmpty(request.getNhinTargetCommunities().getNhinTargetCommunity())) {
            List<NhinTargetCommunityType> nhinTargetCommunityList
                                          = request.getNhinTargetCommunities().getNhinTargetCommunity();
            for (NhinTargetCommunityType nhinTargetCommunity : nhinTargetCommunityList) {
                if (nhinTargetCommunity.getHomeCommunity() != null) {
                    if (nhinTargetCommunity.getHomeCommunity().getHomeCommunityId() != null) {
                        String remoteHomeCommunityId = nhinTargetCommunity.getHomeCommunity().getHomeCommunityId();
                        Facility facility = facilityManager.getFacilityByHomeCommunityId(remoteHomeCommunityId);
                        facilities.add(facility);
                    }
                }
            }
        }

        return facilities;
    }

    private II getLocalPatientId(PRPAIN201305UV02 request)
    {
        II ret = null;
        List<II> patientIds = request.getControlActProcess().getQueryByParameter().
                getValue().getParameterList().getLivingSubjectId().get(0).getValue();
        String aa = getAAOID(request);
        for (II patientId : patientIds) {
            if (aa.equals(patientId.getRoot())) {
                ret = patientId;
                break;
            }
        }
        return ret;
    }

    private String getAAOID(PRPAIN201305UV02 request)
    {
        // RPB: replaced old implementation with this one to make the desired value
        // a little easier to read/understand. The old implementation checked an
        // increasingly complicated series of objects for null or empty, and then
        // returned what we have here.
        //
        // old implementation:
        //
        // 		String oid = null;
        //		if ( request != null
        //				&& request.getControlActProcess() != null
        //				&& !NullChecker.isNullOrEmpty( request.getControlActProcess().getAuthorOrPerformer() )
        //				&& request.getControlActProcess().getAuthorOrPerformer().get( 0 ) != null
        //				&& request.getControlActProcess().getAuthorOrPerformer().get( 0 ).getAssignedDevice() != null
        //				&& request.getControlActProcess().getAuthorOrPerformer().get( 0 ).getAssignedDevice().getValue() != null
        //				&& request.getControlActProcess().getAuthorOrPerformer().get( 0 ).getAssignedDevice().getValue().getId().get( 0 ) != null
        //				&& !NullChecker.isNullOrEmpty( request.getControlActProcess().getAuthorOrPerformer().get( 0 ).getAssignedDevice().getValue().getId() )
        //				&& !NullChecker.isNullOrEmpty( request.getControlActProcess().getAuthorOrPerformer().get( 0 ).getAssignedDevice().getValue().getId().get( 0 ).getRoot() ) ) {
        //			oid = request.getControlActProcess().getAuthorOrPerformer().get( 0 ).
        //					getAssignedDevice().getValue().getId().get( 0 ).getRoot();
        //		}
        //		return oid;

        try {
            String val = request.getControlActProcess().
                    getAuthorOrPerformer().get(0).getAssignedDevice().getValue().
                    getId().get(0).getRoot();
            return (NullChecker.isNotNullOrEmpty(val) ? val : null);
        }
        catch (IndexOutOfBoundsException | NullPointerException e) {
            return null;
        }
    }
/*
    private void auditError(final PRPAIN201305UV02 request, final String remoteHCID, final String errorMsg)
    {
        RequestAuditEntity auditEntity = new RequestAuditEntity();
        auditEntity.setStartTime(new Date());
        II ii = getQueryId(request);
        if (ii == null) {
            auditEntity.setUuid("Error getting UUID");
        }
        else {
            auditEntity.setUuid(ii.getExtension());
        }

        ii = getPID(request);

        if (ii == null) {
            auditEntity.setPid("Error getting PID");
        }
        else {
            auditEntity.setPid(ii.getExtension());
        }

        auditEntity.setHcid(remoteHCID);
        auditEntity.setAction(RequestActions.PATIENT_DISCOVERY.getValue());

        auditEntity.setStopTime(new Date());
        auditEntity.setStatus(errorMsg);
        requestAudit.storeAudit(auditEntity);
    }
*/ /*
    private II getQueryId(final PRPAIN201305UV02 prpain201305UV02)
    {
        if (prpain201305UV02 != null && prpain201305UV02.getControlActProcess() != null && prpain201305UV02.getControlActProcess().getQueryByParameter() != null && prpain201305UV02.getControlActProcess().getQueryByParameter().getValue() != null && prpain201305UV02.getControlActProcess().getQueryByParameter().getValue().getQueryId() != null) {
            return prpain201305UV02.getControlActProcess().getQueryByParameter().getValue().getQueryId();
        }
        return null;
    }
*/ /*
    private II getPID(final PRPAIN201305UV02 prpain201305UV02)
    {
        if (prpain201305UV02 != null && prpain201305UV02.getControlActProcess() != null && prpain201305UV02.getControlActProcess().getQueryByParameter() != null && prpain201305UV02.getControlActProcess().getQueryByParameter().getValue() != null && prpain201305UV02.getControlActProcess().getQueryByParameter().getValue().getParameterList() != null && prpain201305UV02.getControlActProcess().getQueryByParameter().getValue().getParameterList().getId() != null) {
            return prpain201305UV02.getControlActProcess().getQueryByParameter().getValue().getParameterList().getId();
        }
        return null;
    }
*/
    private PRPAIN201305UV02 createPRPAIN201305UV02(PRPAIN201305UV02 fromRequest, String homeCommunityId)
    {
        PRPAIN201305UV02 ret = new PRPAIN201305UV02();
        ret.setAcceptAckCode(fromRequest.getAcceptAckCode());
        ret.setControlActProcess(fromRequest.getControlActProcess());
        ret.setCreationTime(fromRequest.getCreationTime());
        ret.setITSVersion(fromRequest.getITSVersion());
        ret.setId(fromRequest.getId());
        ret.setInteractionId(fromRequest.getInteractionId());
        ret.setProcessingCode(fromRequest.getProcessingCode());
        ret.setProcessingModeCode(fromRequest.getProcessingModeCode());
        ret.setSecurityText(fromRequest.getSecurityText());
        ret.setSender(fromRequest.getSender());
        ret.setSequenceNumber(fromRequest.getSequenceNumber());
        ret.setTypeId(fromRequest.getTypeId());
        ret.setVersionCode(fromRequest.getVersionCode());

        if (NullChecker.isNotNullOrEmpty(fromRequest.getAttachmentText())) {
            ret.getAttachmentText().addAll(fromRequest.getAttachmentText());
        }

        if (NullChecker.isNotNullOrEmpty(fromRequest.getAttentionLine())) {
            ret.getAttentionLine().addAll(fromRequest.getAttentionLine());
        }

        if (NullChecker.isNotNullOrEmpty(fromRequest.getNullFlavor())) {
            ret.getNullFlavor().addAll(fromRequest.getNullFlavor());
        }

        if (NullChecker.isNotNullOrEmpty(fromRequest.getProfileId())) {
            ret.getProfileId().addAll(fromRequest.getProfileId());
        }

        if (NullChecker.isNotNullOrEmpty(fromRequest.getRealmCode())) {
            ret.getRealmCode().addAll(fromRequest.getRealmCode());
        }

        ret.getReceiver().add(createNewReceiver(homeCommunityId));

        if (NullChecker.isNotNullOrEmpty(fromRequest.getRespondTo())) {
            ret.getRespondTo().addAll(fromRequest.getRespondTo());
        }

        if (NullChecker.isNotNullOrEmpty(fromRequest.getTemplateId())) {
            ret.getTemplateId().addAll(fromRequest.getTemplateId());
        }

        return ret;
    }

    private MCCIMT000100UV01Receiver createNewReceiver(String homeCommunityId)
    {
        return HL7ReceiverTransforms.createMCCIMT000100UV01Receiver(homeCommunityId);
    }

    private void processResult(PRPAIN201306UV02 result, II localPatientId, String homeCommunityId, AssertionType assertion, DataQuery parentQuery)
    {
        if (result.getControlActProcess() != null
            && NullChecker.isNotNullOrEmpty(result.getControlActProcess().getSubject())) {
            ListIterator<PRPAIN201306UV02MFMIMT700711UV01Subject1> it
                                                                   = result.getControlActProcess().getSubject().listIterator();
            while (it.hasNext()) {
                PRPAIN201306UV02MFMIMT700711UV01Subject1 subject = it.next();
                if (!(subject.getRegistrationEvent() != null
                      && subject.getRegistrationEvent().getSubject1() != null
                      && subject.getRegistrationEvent().getSubject1().getPatient() != null
                      && processPatient(subject.getRegistrationEvent().getSubject1().getPatient(),
                                        localPatientId, homeCommunityId, assertion, parentQuery))) {
                    it.remove();
                    logger.warn("Could NOT processResult.  Failed while looping through the PRPAIN201306UV02 result's subject.");
                }
            }
        }
        else {
            logger.warn("Could NOT processResult due to null check failures on PRPAIN201306UV02 result.");
        }
    }

    private boolean processPatient(PRPAMT201310UV02Patient prpamt201310UV02Patient, II localPatientId, String remoteHomeCommunityId, AssertionType assertion, DataQuery parentQuery)
    {
        boolean ret = false;
        Map searchResultsMap = findMatch(prpamt201310UV02Patient, remoteHomeCommunityId, assertion, parentQuery);
        PRPAIN201306UV02 mviResults = null;

        EventAuditingThreadData.setEvent(AuditingEvent.PDOUT_VALIDMATCH);
        EventAuditingFactory<AuditEvent> afac
                                         = EventAuditingFactoryImpl.getFactory(AuditEvent.class);
        AuditEvent event = afac.newEvent(AuditingEvent.PDOUT_VALIDMATCH, getClass());
        afac.messaging().addPatientId(event, localPatientId.getExtension());

        if ((searchResultsMap != null && (mviResults = (PRPAIN201306UV02)searchResultsMap.get("PRPAIN201306UV02")) != null) && doIDsMatch(localPatientId, HL7Parser201306.extractPatientIds(mviResults))) {
            ret = true;
            logger.info("Found a match for localPatientId: '" + localPatientId.getExtension() + "'");

            Patient searchResultPatient = (Patient)searchResultsMap.get("SEARCH_RESULT_PATIENT");
            
            Facility facility = facilityManager.getFacilityByHomeCommunityId(remoteHomeCommunityId);

            Correlation candidate = new Correlation(prpamt201310UV02Patient.getId().get(0).getExtension(), "NI", facility.getFacilityNumber(), remoteHomeCommunityId, "A");
            Correlations correlations = Correlations.GetFrom1306(mviResults, false);
 
            afac.messaging().addLocalRemotePatientIds(event, localPatientId.getExtension(),
                                                      ""/*Long.toString(searchResultPatient.getPatientId())*/);
            event.setOutcome(AuditEvent.AuditEventOutcome._0);
            afac.info(event);

            AuditEvent ev = afac.newEvent(correlations.toLink(candidate).size() < 1 ? AuditingEvent.PDOUT_CORRELATENOTSTORED : AuditingEvent.PDOUT_CORRELATESTORED, getClass());
            storeCorrelation(prpamt201310UV02Patient, localPatientId, remoteHomeCommunityId,
                             assertion, searchResultPatient, ev, correlations, mviResults, parentQuery);

            afac.info(ev.setOutcome(AuditEvent.AuditEventOutcome._0));
        }
        else {
            event.setOutcome(AuditEvent.AuditEventOutcome._8);
            event.setOutcomeDesc("local patient NOT matched");
            afac.info(event);
            ret = false;

            // CCR 177986
            logger.warn("Did NOT find a match for localPatientId {}: ", localPatientId.getExtension());
        }

        return ret;
    }

    @SuppressWarnings("rawtypes")
    private Map findMatch(PRPAMT201310UV02Patient prpamt201310UV02Patient, String remoteHomeCommunityId, AssertionType assertion, DataQuery parentQuery)
    {
        PRPAMT201301UV02Patient prpamt201301UV02Patient = HL7PatientTransforms.createPRPAMT201301UVPatient(prpamt201310UV02Patient);

        String localHomeCommunityId = getHomeCommunityId();
        PRPAIN201305UV02 prpain201305UV02 = HL7PRPA201305Transforms.createPRPA201305(prpamt201301UV02Patient, remoteHomeCommunityId, localHomeCommunityId, localHomeCommunityId);
        RespondingGatewayPRPAIN201305UV02RequestType request = new RespondingGatewayPRPAIN201305UV02RequestType();
        request.setPRPAIN201305UV02(prpain201305UV02);
        request.setAssertion(assertion);
        return adapterMvi.findCandidatesMap(request, null, parentQuery);
    }

    private String getHomeCommunityId()
    {
        String ret = null;
        Facility facility = facilityManager.getFacilityByFacilityNumber("VA");
        if (facility != null) {
            ret = facility.getHomeCommunityId();
        }
        return ret;
    }

    private boolean doIDsMatch(II localPatientId, List<II> ids)
    {
        boolean ret = false;
        logger.debug("doIDsMatch - localPatientId.getRoot(): " + localPatientId.getRoot());
        logger.debug("doIDsMatch - localPatientId.getExtension(): " + localPatientId.getExtension());

        for (II id : ids) {
            String icnValue = HL7Parser201306.extractICNValue(id.getExtension());
            logger.debug("id root: ''{}''; id extension: ''{}'';  id/icn value: ''{}''", id.getRoot(), id.getExtension(), icnValue); // CCR
            // 177986
            // removed id.root(i.e. the OID) equals comparison since for ICNs
            // it's always the VA's OID. And this was an issue since in each of
            // the Exchange environment
            // the VA OID's have a ".1" or ".2" etc. added to distinguish
            // environments for the connect gateway purposes.
            if (// localPatientId.getRoot().equals(id.getRoot())&&
                    localPatientId.getExtension().equals(icnValue)) {
                // CCR 177986
                logger.info("Found ID Match for ICN {}: ", icnValue);

                ret = true;
                break;
            }
        }

        if (!ret) {
            // CCR 177986
            logger.warn("Did not find ID Match for ICN {}: ", localPatientId.getExtension());
        }

        return ret;
    }

    private void storeCorrelation(PRPAMT201310UV02Patient prpamt201310UV02Patient, II localPatientId, String remoteHomeCommunityId, AssertionType assertion, Patient searchResultPatient, AuditEvent event, Correlations correlations, PRPAIN201306UV02 mviResults, DataQuery parentQuery)
    {
        PRPAMT201301UV02Patient prpamt201301UV02Patient = HL7PatientTransforms.createPRPAMT201301UVPatient(prpamt201310UV02Patient);
        String localHomeCommunityId = getHomeCommunityId();
        PRPAIN201301UV02 prpain201301UV02 = HL7PRPA201301Transforms.createPRPA201301(prpamt201301UV02Patient, localHomeCommunityId, remoteHomeCommunityId, localHomeCommunityId);
        prpain201301UV02.getControlActProcess().getSubject().get(0).getRegistrationEvent().getSubject1().getPatient().getId().add(localPatientId);
        AddPatientCorrelationRequestType request = new AddPatientCorrelationRequestType();
        request.setPRPAIN201301UV02(prpain201301UV02);
        request.setAssertion(assertion);
        // TODO the boolean s/b a flag to indicate whether or not the
        // correlation already exists so that we can avoid an
        // uncecessary 1301 call to MVI. See AdapterPatientDiscovery for example
        // of its use.
        adapterPatientCorrelation.addPatientCorrelation(request, searchResultPatient, correlations, event, mviResults, parentQuery);
    }

    private void addResultToResponse(RespondingGatewayPRPAIN201306UV02ResponseType response, PRPAIN201306UV02 result)
    {
        CommunityPRPAIN201306UV02ResponseType communityResponse = new CommunityPRPAIN201306UV02ResponseType();
        communityResponse.setPRPAIN201306UV02(result);
        response.getCommunityResponse().add(communityResponse);
    }

    private PRPAIN201306UV02 createErrorResponse(PRPAIN201305UV02 prpain201305UV02, String homeCommunityId, String assigningAuthorityId)
    {
        PRPAIN201306UV02 ret = HL7DbParser201306.BuildMessageFromMpiPatients(null, prpain201305UV02, homeCommunityId, assigningAuthorityId);
        ret.getControlActProcess().getQueryAck().getQueryResponseCode().setCode("QE");
        return ret;
    }
}
