package gov.va.med.nhin.adapter.adaptergateway.docquery;

import gov.va.med.nhin.adapter.adaptergateway.workmanager.WorkManagerExecutorServiceLocal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;

import javax.ejb.EJB;
import javax.ejb.*;

import org.apache.commons.lang3.StringUtils;
import org.hl7.v3.II;
import org.hl7.v3.PRPAIN201310UV02;
import org.hl7.v3.PRPAMT201304UV02Patient;
import org.hl7.v3.RetrievePatientCorrelationsRequestType;
import org.hl7.v3.RetrievePatientCorrelationsResponseType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.hhs.fha.nhinc.common.eventcommon.AdhocQueryRequestEventType;
import gov.hhs.fha.nhinc.common.eventcommon.AdhocQueryRequestMessageType;
import gov.hhs.fha.nhinc.common.nhinccommon.AssertionType;
import gov.hhs.fha.nhinc.common.nhinccommon.HomeCommunityType;
import gov.hhs.fha.nhinc.common.nhinccommon.NhinTargetCommunitiesType;
import gov.hhs.fha.nhinc.common.nhinccommon.NhinTargetCommunityType;
import gov.hhs.fha.nhinc.common.nhinccommonadapter.CheckPolicyRequestType;
import gov.hhs.fha.nhinc.common.nhinccommonadapter.CheckPolicyResponseType;
import gov.hhs.fha.nhinc.nhinccomponentpatientcorrelation.PatientCorrelationPortType;
import gov.hhs.fha.nhinc.nhinclib.NhincConstants;
import gov.hhs.fha.nhinc.transform.policy.PolicyEngineTransformer;
import gov.hhs.fha.nhinc.transform.subdisc.HL7PRPA201309Transforms;
import gov.va.med.nhin.adapter.audit.requests.RequestActions;
import gov.va.med.nhin.adapter.audit.requests.RequestAudit;
import gov.va.med.nhin.adapter.audit.requests.RequestAuditEntity;
import gov.va.med.nhin.adapter.datamanager.DataManager;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.datamanager.ejb.DataManagerLocal;
import gov.va.med.nhin.adapter.facilitymanager.Facility;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManager;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManagerLocal;
import gov.va.med.nhin.adapter.facilitymanager.OperationOnOff;
import gov.va.med.nhin.adapter.logging.CheckPolicy;
import gov.va.med.nhin.adapter.logging.ErrorMessage;
import gov.va.med.nhin.adapter.logging.EventAuditingFactory;
import gov.va.med.nhin.adapter.logging.EventAuditingFactoryImpl;
import gov.va.med.nhin.adapter.logging.LogConstants.AuditingEvent;
import gov.va.med.nhin.adapter.logging.MaintLog;
import gov.va.med.nhin.adapter.logging.MessagingHelper;
import gov.va.med.nhin.adapter.logging.MessagingHelper.QueryType;
import gov.va.med.nhin.adapter.patientcorrelation.PatientCorrelationPortTypeLocal;
import gov.va.med.nhin.adapter.policyengine.AdapterPolicyEnginePortTypeLocal;
import gov.va.med.nhin.adapter.utils.NullChecker;
import oasis.names.tc.ebxml_regrep.xsd.query._3.AdhocQueryRequest;
import oasis.names.tc.ebxml_regrep.xsd.query._3.AdhocQueryResponse;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.AdhocQueryType;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.RegistryObjectListType;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.SlotType1;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.ValueListType;
import oasis.names.tc.ebxml_regrep.xsd.rs._3.RegistryError;
import oasis.names.tc.ebxml_regrep.xsd.rs._3.RegistryErrorList;
import org.hl7.fhir.dstu3.model.AuditEvent;
import org.slf4j.MDC;

/**
 *
 * @author David Vazquez
 */
@TransactionAttribute(value = TransactionAttributeType.SUPPORTS)
@Stateless(name = "EntityDocQueryOrch")
public class EntityDocQueryOrchBean implements EntityDocQueryPortTypeLocal
{
    // CCR 177986
    private static final Logger logger = LoggerFactory.getLogger(EntityDocQueryOrchBean.class.getName());
    private static final String XDS_DOCQUERY_REQUEST_DOCENTRY_STATUS = "$XDSDocumentEntryStatus";
    private static final String XDS_DOCQUERY_REQUEST_DOCENTRY_TYPE = "$XDSDocumentEntryType";
    private static final String RESPONSE_STATUS_SUCCESS = "urn:oasis:names:tc:ebxml-regrep:ResponseStatusType:Success";
    private static final String XDS_DOCUMENT_ENTRY_TYPE_STABLE_URN = "urn:uuid:7edca82f-054d-47f2-a032-9b2a5b5186c1";
    private static final String XDS_DOCUMENT_ENTRY_TYPE_ON_DEMAND_URN = "urn:uuid:34268e47-fdf5-41a6-ba33-82133c465248";
    private static final String EHEX_2011_DOCUMENT_ENTRY_TYPE_QUERY_VALUES = "('" + XDS_DOCUMENT_ENTRY_TYPE_STABLE_URN + "','" + XDS_DOCUMENT_ENTRY_TYPE_ON_DEMAND_URN + "')";
    private static final String EBXML_DOCENTRY_PATIENT_ID = "$XDSDocumentEntryPatientId";
    private static final String EBXML_DOCENTRY_CLASS_CODE = "$XDSDocumentEntryClassCode";
    private static final String LOINC_CLASSIFICATION_SCHEME = "2.16.840.1.113883.6.1";

    public class SendRequestResponse
    {
        public AdhocQueryResponse adhocQueryResponse;
        public String remoteHomeCommunityId;
        public SendRequestException sre = null;

        public boolean hasError()
        {
            return sre != null;
        }
    }

    public class SendRequestException extends Exception
    {

        private static final long serialVersionUID = 1L;

        public String remoteHomeCommunityId;
    }

    public class SendRequestCallable implements Callable<SendRequestResponse>
    {

        private final AdhocQueryRequest adhocQueryRequest;
        private final AssertionType assertion;
        private final II remotePatientId;
        private final String remoteHomeCommunityId;
        private final String useSpecVersion;
        private final String auditUUID;
        private final EventAuditingFactory<AuditEvent> eventfac;
        private final Map<String, String> mdcmap = MDC.getCopyOfContextMap();

        public SendRequestCallable(AdhocQueryRequest adhocQueryRequest,
                                   AssertionType assertion, II remotePatientId, String remoteHomeCommunityId,
                                   final String useSpecVersion, final String auditUUID,
                                   EventAuditingFactory<AuditEvent> afac)
        {
            this.adhocQueryRequest = adhocQueryRequest;
            this.assertion = assertion;
            this.remotePatientId = remotePatientId;
            this.remoteHomeCommunityId = remoteHomeCommunityId;
            this.auditUUID = auditUUID;
            this.useSpecVersion = useSpecVersion;
            this.eventfac = afac;

            logger.info("SendRequestCallable Made");
        }

        @Override
        public SendRequestResponse call() throws Exception
        {
            MDC.setContextMap(mdcmap);
            logger.info("SendRequestCallable Sending Individual Request...");
            SendRequestResponse ret = new SendRequestResponse();
            RequestAuditEntity auditEntity = new RequestAuditEntity();
            auditEntity.setStartTime(new Date());
            auditEntity.setUuid(auditUUID);
            auditEntity.setHcid(remoteHomeCommunityId);
            auditEntity.setPid(remotePatientId.getExtension());
            auditEntity.setAction(RequestActions.QUERY_REQUEST.getValue());

            AuditEvent event
                       = eventfac.newEvent(this.assertion, AuditingEvent.QDOUT_QD, getClass());
            MessagingHelper<AuditEvent> msg = eventfac.messaging();
            msg.addQuery(event, adhocQueryRequest, AdhocQueryRequest.class,
                         QueryType.REQUEST, null);
            try {
                ret.adhocQueryResponse = nhinDocQueryRequestSender.sendRequest(adhocQueryRequest,
                                                                               assertion, remoteHomeCommunityId, useSpecVersion);
                msg.addQuery(event, ret.adhocQueryResponse, AdhocQueryResponse.class,
                             QueryType.RESPONSE, null);
                event.setOutcome(AuditEvent.AuditEventOutcome._0);
                eventfac.info(event);

                auditEntity.setStopTime(new Date());
                auditEntity.setStatus(ret.adhocQueryResponse.getStatus());
                requestAudit.storeAudit(auditEntity);
                ret.remoteHomeCommunityId = remoteHomeCommunityId;
                logger.debug("DQ response recieved from SSA");
            }
            catch (Throwable e) {
                event.setOutcome(AuditEvent.AuditEventOutcome._8);
                event.setOutcomeDesc(e.getLocalizedMessage());
                eventfac.error(event);

                logger.debug("Exception Occurred: {}", e.getMessage());

                SendRequestException sre = new SendRequestException();
                sre.remoteHomeCommunityId = remoteHomeCommunityId;
                sre.initCause(e);
                auditEntity.setStopTime(new Date());
                auditEntity.setStatus("Failed:" + e.getMessage());
                requestAudit.storeAudit(auditEntity);
                ret.sre = sre;
            }
            return ret;
        }
    }

    private FacilityManager facilityManager;
    private PatientCorrelationPortTypeLocal adapterPatientCorrelation;
    private AdapterPolicyEnginePortTypeLocal adapterPolicyEngine;
    private NHINDocQueryRequestSender nhinDocQueryRequestSender;
    private ExecutorService executorService;
    private RequestAudit requestAudit;
    private DataManager dataManager;

    @EJB(beanInterface = FacilityManagerLocal.class, beanName = "FacilityManager")
    public void setFacilityManager(FacilityManager facilityManager)
    {
        this.facilityManager = facilityManager;
    }

    @EJB(beanInterface = AdapterPolicyEnginePortTypeLocal.class, beanName = "AdapterPolicyEngine")
    public void setAdapterPolicyEngine(AdapterPolicyEnginePortTypeLocal adapterPolicyEngine)
    {
        this.adapterPolicyEngine = adapterPolicyEngine;
    }

    @EJB(beanInterface = PatientCorrelationPortTypeLocal.class, beanName = "AdapterPatientCorrelation")
    public void setAdapterPatientCorrelation(PatientCorrelationPortTypeLocal adapterPatientCorrelation)
    {
        this.adapterPatientCorrelation = adapterPatientCorrelation;
    }

    @EJB(beanInterface = NHINDocQueryRequestSender.class, beanName = "NhincProxyNHINDocQueryRequestSender")
    public void setNhinDocQueryRequestSender(NHINDocQueryRequestSender nhinDocQueryRequestSender)
    {
        this.nhinDocQueryRequestSender = nhinDocQueryRequestSender;
    }

    @EJB(beanInterface = DataManagerLocal.class, beanName = "DataManager")
    public void setDataManager(DataManager dataManager)
    {
        this.dataManager = dataManager;
    }

    //changed on 1/6/2017
    //@EJB(beanInterface = WorkManagerExecutorServiceLocal.class, beanName = "WorkManagerExecutorService")
    @EJB(beanInterface = WorkManagerExecutorServiceLocal.class, beanName = "OutboundQDWorkManagerExecutorService")
    public void setExecutorService(ExecutorService executorService)
    {
        this.executorService = executorService;
    }

    @EJB(beanInterface = RequestAudit.class, beanName = "RequestAudit")
    public void setRequestAudit(RequestAudit requestAudit)
    {
        this.requestAudit = requestAudit;
    }

    @Override
    public AdhocQueryResponse respondingGatewayCrossGatewayQuery(gov.hhs.fha.nhinc.common.nhinccommonentity.RespondingGatewayCrossGatewayQueryRequestType respondingGatewayCrossGatewayQueryRequest)
    {
        logger.debug("respondingGatewayCrossGatewayQuery() invoked");
        EventAuditingFactory<AuditEvent> afac
                                         = EventAuditingFactoryImpl.getFactory(AuditEvent.class);
        AdhocQueryResponse ret = createAdhocQueryResponse(respondingGatewayCrossGatewayQueryRequest.getAdhocQueryRequest());

        ExecutorCompletionService<SendRequestResponse> completionService
                                                       = new ExecutorCompletionService<>(executorService);

        II patientId = getPatientId(respondingGatewayCrossGatewayQueryRequest.getAdhocQueryRequest());
        if (patientId != null) {

            // Setup incoming request audit values.
            RequestAuditEntity requestAuditEntity = new RequestAuditEntity();
            requestAuditEntity.setPid(patientId.getExtension());
            requestAuditEntity.generateUUID();
            requestAuditEntity.setStartTime(new Date());
            requestAuditEntity.setAction(RequestActions.PARENT_REQUEST.getValue());
            logger.info("RequestAuditEntity created.");

            if (respondingGatewayCrossGatewayQueryRequest.getAssertion() != null
                && respondingGatewayCrossGatewayQueryRequest.getAssertion().getHomeCommunity() != null
                && NullChecker.isNotNullOrEmpty(respondingGatewayCrossGatewayQueryRequest.getAssertion().getHomeCommunity().getHomeCommunityId())) {
                requestAuditEntity.setHcid(respondingGatewayCrossGatewayQueryRequest.getAssertion().getHomeCommunity().getHomeCommunityId());
            }
            else {
                requestAuditEntity.setHcid("UNKNOWN");
            }

            boolean isSsa = false;
            String firstTargetCommunity = "";
            Facility f200nss = facilityManager.getFacilityByFacilityNumber("200NSS");
            if (null == f200nss) {
                logger.info("FacilityManager does not contain SSA");
            }
            else {
                try {
                    firstTargetCommunity = respondingGatewayCrossGatewayQueryRequest.getNhinTargetCommunities().getNhinTargetCommunity().get(0).getHomeCommunity().getHomeCommunityId();
                    logger.debug("Database SSA Home Community ID: {}", facilityManager.getFacilityByFacilityNumber("200NSS").getHomeCommunityId());
                    logger.debug("First target community ID: {}", firstTargetCommunity);
                    isSsa = firstTargetCommunity.contains(facilityManager.getFacilityByFacilityNumber("200NSS").getHomeCommunityId());
                }
                catch (NullPointerException e) {
                    logger.debug("No target communities found: {}", e.getMessage());
                }
            }

            logger.debug("isSsa = {}", isSsa);
            
            DataQuery parentQuery = dataManager.getQuery("JustCache");
            parentQuery.getResults();

            // Skip MVI Correlation for SSA
            RetrievePatientCorrelationsResponseType patientCorrelations = new RetrievePatientCorrelationsResponseType();
            if (!isSsa) {
                AuditEvent corrs = afac.newEvent(respondingGatewayCrossGatewayQueryRequest.getAssertion(),
                                                 AuditingEvent.QDOUT_GETCORRELATIONS, getClass());
                patientCorrelations = getPatientCorrelations(patientId, respondingGatewayCrossGatewayQueryRequest.getAssertion(), corrs, parentQuery);
                afac.info(corrs);
            }

            if (isSsa || (patientCorrelations != null && patientCorrelations.getPRPAIN201310UV02() != null)) {
                int numSubmitted = 0;

                PRPAMT201304UV02Patient patient;
                if (isSsa) {
                    patient = new PRPAMT201304UV02Patient();
                    patient.getId().add(patientId);
                }
                else {
                    patient = extractPatient(patientCorrelations.getPRPAIN201310UV02());
                }

                for (II correlatedPatientId : patient.getId()) {
                    // Set to SAML header patient ID
                    List<String> remoteHomeCommunityIds = getRemoteHomeCommunityIds(correlatedPatientId, respondingGatewayCrossGatewayQueryRequest.getNhinTargetCommunities());
                    for (String remoteHomeCommunityId : remoteHomeCommunityIds) {
                        AuditEvent policy = afac.newEvent(respondingGatewayCrossGatewayQueryRequest.getAssertion(),
                                                          AuditingEvent.QDOUT_PARTNERAUTH, getClass());

                        if (facilityManager.isPartnerAllowed(remoteHomeCommunityId, OperationOnOff.ONBOARD)) {
                            if (facilityManager.isPartnerAllowed(remoteHomeCommunityId, OperationOnOff.OUT_DQ)) {
                                logger.debug("HCID: {} onboarded for out_DQ", remoteHomeCommunityId);
                                AdhocQueryRequest adhocQueryRequest = createAdhocQueryRequest(respondingGatewayCrossGatewayQueryRequest.getAdhocQueryRequest(), remoteHomeCommunityId, correlatedPatientId);
                                logger.debug("AdhocQueryRequest created");

                                if (isSsa || checkPolicy(adhocQueryRequest, respondingGatewayCrossGatewayQueryRequest.getAssertion(), remoteHomeCommunityId, policy, parentQuery)) {
                                    afac.info(policy);

                                    logger.debug("Determining spec version...");
                                    String useSpecVersion = "2.0";
                                    if (communitySupports2011Specs(remoteHomeCommunityId)) {
                                        modifyRequestFor2011Support(adhocQueryRequest);
                                        useSpecVersion = "3.0";
                                    }

                                    logger.debug("Submitting Callables...");

                                    SendRequestCallable callable
                                                        = new SendRequestCallable(adhocQueryRequest,
                                                                                  respondingGatewayCrossGatewayQueryRequest.getAssertion(),
                                                                                  correlatedPatientId, remoteHomeCommunityId, useSpecVersion,
                                                                                  requestAuditEntity.getUuid(), afac);
                                    completionService.submit(callable);
                                    ++numSubmitted;
                                    logger.debug("numSubmitted = {}", numSubmitted);
                                }
                            }
                            else {
                                logger.debug("HCID: {} out_DQ not permitted", remoteHomeCommunityId);
                                // TODO: Determine Proper query error.
                                MaintLog.queryError(null, ErrorMessage.OUT_DQ_DISABLED_NEW + " " + remoteHomeCommunityId, logger);
                                addErrorToResponse(ret, ErrorMessage.OUT_DQ_DISABLED_NEW.getMessage(), remoteHomeCommunityId, true);
                                auditError(requestAuditEntity.getUuid(), remoteHomeCommunityId, requestAuditEntity.getPid(), "Failed: " + ErrorMessage.OUT_DQ_DISABLED_NEW.getMessage());

                                policy.setOutcome(AuditEvent.AuditEventOutcome._4);
                                afac.error(policy.setOutcomeDesc(ErrorMessage.OUT_DQ_DISABLED_NEW.getMessage()));
                            }
                        }
                        else {
                            logger.debug("HCID: {} not onboarded", remoteHomeCommunityId);
                            // TODO: Determine Proper query error.
                            MaintLog.queryError(null, ErrorMessage.OUT_DQ_NOT_A_PARTNER_NEW + " " + remoteHomeCommunityId, logger);
                            addErrorToResponse(ret, ErrorMessage.OUT_DQ_NOT_A_PARTNER_NEW.getMessage(), remoteHomeCommunityId, true);
                            auditError(requestAuditEntity.getUuid(), remoteHomeCommunityId, requestAuditEntity.getPid(), "Failed: " + ErrorMessage.OUT_DQ_NOT_A_PARTNER_NEW.getMessage());

                            policy.setOutcome(AuditEvent.AuditEventOutcome._4);
                            afac.error(policy.setOutcomeDesc(ErrorMessage.OUT_DQ_NOT_A_PARTNER_NEW.getMessage()));
                        }
                    }
                }

                if (patient.getId().size() < 1) {
                    MaintLog.queryError(respondingGatewayCrossGatewayQueryRequest, ErrorMessage.OUT_DQ_NO_CORRELATIONS, logger);

                    afac.error(afac.newEvent(respondingGatewayCrossGatewayQueryRequest.getAssertion(),
                                             AuditingEvent.QDOUT_GETCORRELATIONS, getClass())
                            .setOutcome(AuditEvent.AuditEventOutcome._4)
                            .setOutcomeDesc(ErrorMessage.OUT_DQ_NO_CORRELATIONS.getMessage()));
                }
                for (int i = 0; i < numSubmitted; ++i) {
                    try {
                        SendRequestResponse response = completionService.take().get();
                        if (!response.hasError()) {
                            logger.info("Adding result to response...");
                            addResultToResponse(ret, response.adhocQueryResponse, response.remoteHomeCommunityId);
                        }
                        else {
                            SendRequestException sre = response.sre;
                            String errorStr = "Error sending DocQuery to";
                            addErrorToResponse(ret, errorStr, sre.remoteHomeCommunityId, false);
                            logger.warn("Error sending DocQuery to {} - {}", sre.remoteHomeCommunityId, sre.getCause().getMessage());
                            logger.warn("Stack trace", sre.getCause());
                        }
                    }
                    catch (ExecutionException ee) {
                        logger.error("An Unknown error occured processing an outbound doc query request.", ee);
                        MaintLog.queryError(respondingGatewayCrossGatewayQueryRequest, ErrorMessage.OUT_DQ_UNKNOWN, "An Unknown error occured processing an outbound doc query request.", logger);
                        requestAuditEntity.setStatus("Failed: " + ErrorMessage.OUT_DQ_UNKNOWN.getMessage() + ee.getMessage());
                    }
                    catch (InterruptedException ie) {
                        logger.error("Outbound Doc Query Threads interrupted.  Return will not be complete.", ie);
                        requestAuditEntity.setStatus("Failed: " + ErrorMessage.OUT_DQ_UNKNOWN.getMessage() + ie.getMessage());
                    }
                }
            }
            else {
                MaintLog.queryError(respondingGatewayCrossGatewayQueryRequest, ErrorMessage.OUT_DQ_NO_CORRELATIONS, logger);
                requestAuditEntity.setStatus("Failed: " + ErrorMessage.OUT_DQ_NO_CORRELATIONS.getMessage());

                afac.error(afac.newEvent(respondingGatewayCrossGatewayQueryRequest.getAssertion(),
                                         AuditingEvent.QDOUT_GETCORRELATIONS, getClass())
                        .setOutcome(AuditEvent.AuditEventOutcome._4)
                        .setOutcomeDesc(ErrorMessage.OUT_DQ_NO_CORRELATIONS.getMessage()));
            }

            logger.info("Storing Audit...");
            requestAuditEntity.setStopTime(new Date());
            requestAudit.storeAudit(requestAuditEntity);
            logger.info("Audit stored.");
        }
        else {
            logger.info("No patientId.");
            MaintLog.queryError(respondingGatewayCrossGatewayQueryRequest, ErrorMessage.OUT_DQ_NO_PATIENT_ID, logger);

            afac.error(afac.newEvent(respondingGatewayCrossGatewayQueryRequest.getAssertion(),
                                     AuditingEvent.QDOUT_GETCORRELATIONS, getClass())
                    .setOutcome(AuditEvent.AuditEventOutcome._4)
                    .setOutcomeDesc(ErrorMessage.OUT_DQ_NO_PATIENT_ID.getMessage()));
        }

        logger.debug("Returning DQ response...");
        return ret;
    }

    private void auditError(final String auditUUID, final String remoteHomeCommunityId, final String pid, final String errorMsg)
    {
        RequestAuditEntity auditEntity = new RequestAuditEntity();
        auditEntity.setStartTime(new Date());
        auditEntity.setUuid(auditUUID);
        auditEntity.setHcid(remoteHomeCommunityId);
        auditEntity.setPid(pid);
        auditEntity.setAction(RequestActions.QUERY_REQUEST.getValue());

        auditEntity.setStopTime(new Date());
        auditEntity.setStatus(errorMsg);
        requestAudit.storeAudit(auditEntity);
    }

    private boolean communitySupports2011Specs(String hieCommunityOid)
    {
        logger.info("In communitySupports2011Specs");
        // fetch the facility information
        Facility facility = facilityManager.getFacilityByHomeCommunityId(hieCommunityOid);
        return Facility.SPEC_VERSION_2011.equals(facility.getUseSpecVersion());
    }

    private void modifyRequestFor2011Support(AdhocQueryRequest adhocQueryRequest)
    {
        logger.info("In modifyRequestFor2011Support");

        List<SlotType1> querySlots = adhocQueryRequest.getAdhocQuery().getSlot();
        for (SlotType1 slot : querySlots) {
            if (XDS_DOCQUERY_REQUEST_DOCENTRY_STATUS.equals(slot.getName())) {
                // replace the request status values
                slot.setValueList(createValueList("('urn:oasis:names:tc:ebxml-regrep:StatusType:Approved')"));
                break;
            }
        }

        // add the document entry slot
        SlotType1 xdsDocEntryType = new SlotType1();
        xdsDocEntryType.setName(XDS_DOCQUERY_REQUEST_DOCENTRY_TYPE);
        xdsDocEntryType.setValueList(createValueList(EHEX_2011_DOCUMENT_ENTRY_TYPE_QUERY_VALUES));
        querySlots.add(xdsDocEntryType);
    }

    private AdhocQueryResponse createAdhocQueryResponse(AdhocQueryRequest adhocQueryRequest)
    {
        logger.debug("In createAdhocQueryResponse.");
        AdhocQueryResponse ret = new AdhocQueryResponse();
        ret.setRequestId(adhocQueryRequest.getId());
        logger.debug("Response requestId set as {}", ret.getRequestId());
        ret.setStartIndex(adhocQueryRequest.getStartIndex());
        logger.debug("Response startIndex set as {}", ret.getStartIndex().toString());
        ret.setStatus("urn:oasis:names:tc:ebxml-regrep:ResponseStatusType:Success");
        logger.debug("Response set to success");
        return ret;
    }

    private II getPatientId(AdhocQueryRequest adhocQueryRequest)
    {
        II ret = new II();
        Map<String, List<String>> slotMap = getMapFromSlots(adhocQueryRequest.getAdhocQuery().getSlot());
        String patientId = slotMap.get(EBXML_DOCENTRY_PATIENT_ID).get(0);
        if (!patientId.isEmpty()) {
            ret = new II();
            String[] idSplit = patientId.split("\\^");
            ret.setExtension(idSplit[0]);
            ret.setRoot(idSplit[3].split("&")[1]);
        }
        logger.debug("ID root: {}\nID extension: {}", ret.getRoot(), ret.getExtension());
        return ret;
    }

    private Map<String, List<String>> getMapFromSlots(List<SlotType1> slots)
    {
        HashMap<String, List<String>> ret = new HashMap<>();

        for (SlotType1 slot : slots) {
            if (!NullChecker.isNullOrEmpty(slot.getName()) && !NullChecker.isNullOrEmpty(slot.getValueList()) && !NullChecker.isNullOrEmpty(slot.getValueList().getValue())) {
                List<String> values = ret.get(slot.getName());
                if (values == null) {
                    values = new ArrayList<>();
                    ret.put(slot.getName(), values);
                }
                values.addAll(parseParamFormattedStrings(slot.getValueList().getValue()));
            }
        }

        return ret;
    }

    public List<String> parseParamFormattedStrings(List<String> paramFormattedStrings)
    {
        List<String> ret = new ArrayList<>();

        for (String paramFormattedString : paramFormattedStrings) {
            parseParamFormattedString(paramFormattedString, ret);
        }

        return ret;
    }

    public void parseParamFormattedString(String paramFormattedString, List<String> resultCollection)
    {
        if ((paramFormattedString != null) && (resultCollection != null)) {
            String working = paramFormattedString;
            if (paramFormattedString.startsWith("(")) {
                working = paramFormattedString.substring(1);
                int endIndex = working.indexOf(")");
                if (endIndex != -1) {
                    working = working.substring(0, endIndex);
                }
            }
            String[] multiValueString = working.split(",");
            if (multiValueString != null) {
                for (String singleValue : multiValueString) {
                    if (singleValue != null) {
                        singleValue = singleValue.trim();
                    }
                    if (singleValue != null && singleValue.startsWith("'")) {
                        singleValue = singleValue.substring(1);
                        int endTickIndex = singleValue.indexOf("'");
                        if (endTickIndex != -1) {
                            singleValue = singleValue.substring(0, endTickIndex);
                        }
                    }
                    resultCollection.add(singleValue);
                }
            }
        }
    }

    private RetrievePatientCorrelationsResponseType getPatientCorrelations(II patientId, AssertionType assertion, AuditEvent event, DataQuery parentQuery)
    {
        RetrievePatientCorrelationsRequestType request = createRetrievePatientCorrelationsRequest(patientId);
        request.setAssertion(assertion);

        MessagingHelper<AuditEvent> msg = EventAuditingFactoryImpl.getFactory(AuditEvent.class).messaging();
        msg.addQuery(event, request, RetrievePatientCorrelationsRequestType.class,
                     QueryType.REQUEST, null);
        RetrievePatientCorrelationsResponseType resp = adapterPatientCorrelation.retrievePatientCorrelations(request, parentQuery);
        msg.addQuery(event, resp, RetrievePatientCorrelationsResponseType.class,
                     QueryType.RESPONSE, null);

        return resp;
    }

    private RetrievePatientCorrelationsRequestType createRetrievePatientCorrelationsRequest(II patientId)
    {
        RetrievePatientCorrelationsRequestType ret = new RetrievePatientCorrelationsRequestType();
        ret.setPRPAIN201309UV02(HL7PRPA201309Transforms.createPRPA201309(patientId.getRoot(), patientId.getExtension()));
        return ret;
    }

    private String getHomeCommunityId()
    {
        String ret = null;
        Facility facility = facilityManager.getFacilityByFacilityNumber("VA");
        if (facility != null) {
            ret = facility.getHomeCommunityId();
        }
        return ret;
    }

    private PRPAMT201304UV02Patient extractPatient(PRPAIN201310UV02 prpain201310UV02)
    {
        PRPAMT201304UV02Patient ret = null;

        if (prpain201310UV02.getControlActProcess() != null && !NullChecker.isNullOrEmpty(prpain201310UV02.getControlActProcess().getSubject()) && prpain201310UV02.getControlActProcess().getSubject().get(0).getRegistrationEvent() != null && prpain201310UV02.getControlActProcess().getSubject().get(0).getRegistrationEvent().getSubject1() != null && prpain201310UV02.getControlActProcess().getSubject().get(0).getRegistrationEvent().getSubject1().getPatient() != null) {
            ret = prpain201310UV02.getControlActProcess().getSubject().get(0).getRegistrationEvent().getSubject1().getPatient();
        }

        return ret;
    }

    private List<String> getRemoteHomeCommunityIds(II remotePatientId, NhinTargetCommunitiesType nhinTargetCommunities)
    {
        logger.info("In getRemoteHomeCommunityIds()");
        List<String> ret = new ArrayList<>();
        List<Facility> remoteFacilities = facilityManager.getFacilitiesContainingAssigningAuthority(remotePatientId.getRoot());
        logger.info("Got remote facilities list for remotePatient ID");

        if (NullChecker.isNotNullOrEmpty(remoteFacilities)) {
            for (Facility remoteFacility : remoteFacilities) {
                if (facilityInTargets(remoteFacility, nhinTargetCommunities)) {
                    ret.add(remoteFacility.getHomeCommunityId());
                    logger.debug("Remote Facility HCID {} added to response list", remoteFacility.getHomeCommunityId());
                }
            }
        }
        else {
            logger.info("Remote facilities list is empty");
        }

        return ret;
    }

    private boolean facilityInTargets(Facility facility, NhinTargetCommunitiesType nhinTargetCommunities)
    {
        boolean ret = false;
        
        if (nhinTargetCommunities != null && NullChecker.isNotNullOrEmpty(nhinTargetCommunities.getNhinTargetCommunity())) {
            for (NhinTargetCommunityType nhinTargetCommunity : nhinTargetCommunities.getNhinTargetCommunity()) {
                if (nhinTargetCommunity.getHomeCommunity().getHomeCommunityId().endsWith(facility.getHomeCommunityId())) {
                    ret = true;
                    break;
                }
            }
        }
        else {
            ret = true;
        }
        
        return ret;
    }
    
    private boolean checkPolicy(AdhocQueryRequest adhocQueryRequest, AssertionType assertion,
                                String homeCommunityId, AuditEvent ae, DataQuery parentQuery)
    {

        AdhocQueryRequestMessageType adhocQueryRequestMessage = new AdhocQueryRequestMessageType();
        adhocQueryRequestMessage.setAdhocQueryRequest(adhocQueryRequest);
        adhocQueryRequestMessage.setAssertion(assertion);

        HomeCommunityType receivingHomeCommunity = new HomeCommunityType();
        receivingHomeCommunity.setHomeCommunityId(homeCommunityId);

        HomeCommunityType sendingHomeCommunity = new HomeCommunityType();
        sendingHomeCommunity.setHomeCommunityId(getHomeCommunityId());

        AdhocQueryRequestEventType adhocQueryRequestEvent = new AdhocQueryRequestEventType();
        adhocQueryRequestEvent.setMessage(adhocQueryRequestMessage);
        adhocQueryRequestEvent.setDirection(NhincConstants.POLICYENGINE_OUTBOUND_DIRECTION);
        adhocQueryRequestEvent.setReceivingHomeCommunity(receivingHomeCommunity);
        adhocQueryRequestEvent.setSendingHomeCommunity(sendingHomeCommunity);
        adhocQueryRequestEvent.setInterface(NhincConstants.AUDIT_LOG_ENTITY_INTERFACE);

        PolicyEngineTransformer policyEngineTransformer = new PolicyEngineTransformer();
        CheckPolicyRequestType checkPolicyRequest = policyEngineTransformer.transformAdhocQueryToCheckPolicy(adhocQueryRequestEvent);

        MessagingHelper<AuditEvent> msg = EventAuditingFactoryImpl.getFactory(AuditEvent.class).messaging();
        msg.addQuery(ae, checkPolicyRequest, CheckPolicyRequestType.class,
                     QueryType.REQUEST, null);

        CheckPolicyResponseType checkPolicyResponse = adapterPolicyEngine.checkPolicy(checkPolicyRequest, parentQuery);
        msg.addQuery(ae, checkPolicyResponse, CheckPolicyResponseType.class,
                     QueryType.RESPONSE, null);

        return CheckPolicy.checkPolicy(checkPolicyRequest, checkPolicyResponse);
    }

    private AdhocQueryRequest createAdhocQueryRequest(AdhocQueryRequest fromAdhocQueryRequest, String homeCommunityId, II remotePatientId)
    {
        logger.debug("In createAdhocQueryRequest");
        AdhocQueryRequest ret = new AdhocQueryRequest();
        ret.setFederation(fromAdhocQueryRequest.getFederation());
        logger.debug("Federation: {}", fromAdhocQueryRequest.getFederation());
        ret.setMaxResults(fromAdhocQueryRequest.getMaxResults());
        logger.debug("MaxResults: {}", fromAdhocQueryRequest.getMaxResults());
        ret.setStartIndex(fromAdhocQueryRequest.getStartIndex());
        logger.debug("StartIndex: {}", fromAdhocQueryRequest.getStartIndex());
        ret.setId(fromAdhocQueryRequest.getId());
        logger.debug("Id: {}", fromAdhocQueryRequest.getId());
        ret.setComment(fromAdhocQueryRequest.getComment());
        logger.debug("Comment: {}", fromAdhocQueryRequest.getComment());
        ret.setRequestSlotList(fromAdhocQueryRequest.getRequestSlotList());
        logger.debug("RequestSlotList: {}", fromAdhocQueryRequest.getRequestSlotList());
        ret.setAdhocQuery(createAdhocQuery(fromAdhocQueryRequest.getAdhocQuery(), homeCommunityId, remotePatientId));
        ret.setResponseOption(fromAdhocQueryRequest.getResponseOption());
        logger.debug("ResponseOption: {}", fromAdhocQueryRequest.getResponseOption());
        return ret;
    }

    private AdhocQueryType createAdhocQuery(AdhocQueryType fromAdhocQuery, String homeCommunityId, II remotePatientId)
    {
        AdhocQueryType ret = new AdhocQueryType();
        ret.setDescription(fromAdhocQuery.getDescription());
        ret.setHome("urn:oid:" + homeCommunityId);
        ret.setId(fromAdhocQuery.getId());
        ret.setLid(fromAdhocQuery.getLid());
        ret.setName(fromAdhocQuery.getName());
        ret.setObjectType(fromAdhocQuery.getObjectType());
        ret.setQueryExpression(fromAdhocQuery.getQueryExpression());
        ret.setStatus(fromAdhocQuery.getStatus());
        ret.setVersionInfo(fromAdhocQuery.getVersionInfo());
        ret.getExternalIdentifier().addAll(fromAdhocQuery.getExternalIdentifier());

        // Additional patient ID processing is required for SSA
        if (facilityManager.getFacilityByFacilityNumber("200NSS") != null && homeCommunityId.contains(facilityManager.getFacilityByFacilityNumber("200NSS").getHomeCommunityId())) {
            ret.getSlot().addAll(createSlotsNonPreprocess(fromAdhocQuery.getSlot(), remotePatientId));
        }
        else {
            ret.getSlot().addAll(createSlots(fromAdhocQuery.getSlot(), remotePatientId));
        }

        ret.getClassification().addAll(fromAdhocQuery.getClassification());
        return ret;
    }

    private List<SlotType1> createSlots(List<SlotType1> slots, II remotePatientId)
    {
        ArrayList<SlotType1> ret = new ArrayList<>();

        for (SlotType1 slot : slots) {
            SlotType1 newSlot;
            if (slot.getName().equals(EBXML_DOCENTRY_PATIENT_ID)) {
                newSlot = new SlotType1();
                newSlot.setName(EBXML_DOCENTRY_PATIENT_ID);
                newSlot.setValueList(createValueList("'" + remotePatientId.getExtension() + "^^^&" + remotePatientId.getRoot() + "&ISO'"));
            }
            else {
                newSlot = slot;
            }
            ret.add(newSlot);
        }

        return ret;
    }

    private List<SlotType1> createSlotsNonPreprocess(List<SlotType1> slots, II remotePatientId)
    {
        ArrayList<SlotType1> ret = new ArrayList<>();

        for (SlotType1 slot : slots) {
            SlotType1 newSlot;
            switch (slot.getName()) {
                case EBXML_DOCENTRY_PATIENT_ID:
                    newSlot = new SlotType1();
                    newSlot.setName(EBXML_DOCENTRY_PATIENT_ID);
                    newSlot.setValueList(createValueList("'" + remotePatientId.getExtension() + "^^^&" + remotePatientId.getRoot() + "&ISO'"));
                    break;
                case EBXML_DOCENTRY_CLASS_CODE:
                    newSlot = new SlotType1();
                    newSlot.setName(EBXML_DOCENTRY_CLASS_CODE);
                    newSlot.setValueList(createValueList(slot.getValueList().getValue().get(0) + "^^" + LOINC_CLASSIFICATION_SCHEME));
                    break;
                default:
                    newSlot = slot;
                    break;
            }
            ret.add(newSlot);
        }

        return ret;
    }

    private ValueListType createValueList(String value)
    {
        ValueListType ret = new ValueListType();
        ret.getValue().add(value);
        return ret;
    }

    private void addResultToResponse(AdhocQueryResponse adhocQueryResponse, AdhocQueryResponse result, String remoteHomeCommunityId)
    {
        logger.debug("In addResultToResponse");

        Facility facility = this.facilityManager.getFacilityByHomeCommunityId(remoteHomeCommunityId);
        logger.debug("Facility: {}", facility.getFacilityName());

        // if Success from partner
        if (result.getStatus() != null && result.getStatus().equalsIgnoreCase(RESPONSE_STATUS_SUCCESS)) {
            logger.debug("Result Status: {}", result.getStatus());
            // if success but no documents.
            if (result.getRegistryObjectList() == null || result.getRegistryObjectList().getIdentifiable() == null || result.getRegistryObjectList().getIdentifiable().isEmpty()) {
                // if success but no documents and no errors from partner.
                if (result.getRegistryErrorList() == null || result.getRegistryErrorList().getRegistryError() == null || result.getRegistryErrorList().getRegistryError().isEmpty()) {
                    RegistryError regError = new RegistryError();
                    regError.setErrorCode("XDSRegistryError");
                    regError.setLocation("Entity Document Query");
                    regError.setValue(facility.getFacilityName() + " responded to the query but did not provide a document for the patient.");
                    if (adhocQueryResponse.getRegistryObjectList() == null) {
                        adhocQueryResponse.setRegistryObjectList(new RegistryObjectListType());
                    }
                    if (adhocQueryResponse.getRegistryErrorList() == null) {
                        adhocQueryResponse.setRegistryErrorList(new RegistryErrorList());
                    }
                    adhocQueryResponse.getRegistryErrorList().getRegistryError().add(regError);
                }
                else {
                    // if success but no documents and errors from partner.
                    List<RegistryError> regErrorList = result.getRegistryErrorList().getRegistryError();
                    for (RegistryError regError : regErrorList) {
                        String value = regError.getValue();
                        if (StringUtils.isEmpty(value)) {
                            value = regError.getCodeContext();
                        }
                        // regError.setCodeContext(facility.getFacilityName() +
                        // " responded to the query with an error. ( " +
                        // regError.getErrorCode() + ": " +
                        // regError.getCodeContext());
                        regError.setValue(facility.getFacilityName() + " responded to the query with an error. ( " + regError.getErrorCode() + ": " + value + " )");
                        if (adhocQueryResponse.getRegistryErrorList() == null) {
                            adhocQueryResponse.setRegistryErrorList(new RegistryErrorList());
                        }
                        adhocQueryResponse.getRegistryErrorList().getRegistryError().add(regError);
                    }
                }

            }
        }
        else {
            logger.info("Partner returned failure");
            // Partner returned failure
            if (result.getRegistryErrorList() != null && result.getRegistryErrorList().getRegistryError() != null && !result.getRegistryErrorList().getRegistryError().isEmpty()) {
                List<RegistryError> regErrorList = result.getRegistryErrorList().getRegistryError();
                for (RegistryError regError : regErrorList) {
                    String value = regError.getValue();
                    if (StringUtils.isEmpty(value)) {
                        value = regError.getCodeContext();
                    }

                    regError.setValue(facility.getFacilityName() + " returned an error. ( " + regError.getErrorCode() + ": " + value + " )");
                    if (adhocQueryResponse.getRegistryErrorList() == null) {
                        adhocQueryResponse.setRegistryErrorList(new RegistryErrorList());
                    }
                    adhocQueryResponse.getRegistryErrorList().getRegistryError().add(regError);
                }

            }
        }
        logger.info("add all the document details returned by the partner");
        // add all the document details returned by the partner
        if (result.getRegistryObjectList() != null && result.getRegistryObjectList().getIdentifiable() != null && !result.getRegistryObjectList().getIdentifiable().isEmpty()) {
            if (adhocQueryResponse.getRegistryObjectList() == null) {
                adhocQueryResponse.setRegistryObjectList(new RegistryObjectListType());
            }
            adhocQueryResponse.getRegistryObjectList().getIdentifiable().addAll(result.getRegistryObjectList().getIdentifiable());

            if (adhocQueryResponse.getTotalResultCount() == null) {
                adhocQueryResponse.setTotalResultCount(BigInteger.valueOf(result.getRegistryObjectList().getIdentifiable().size()));
            }
            else {
                adhocQueryResponse.setTotalResultCount(adhocQueryResponse.getTotalResultCount().add(BigInteger.valueOf(result.getRegistryObjectList().getIdentifiable().size())));
            }

        }

    }

    private void addErrorToResponse(final AdhocQueryResponse adhocQueryResponse, String errorMessage, final String remoteHomeCommunityId, boolean prependPartnerName)
    {
        if (adhocQueryResponse.getRegistryErrorList() == null) {
            adhocQueryResponse.setRegistryErrorList(new RegistryErrorList());
        }

        Facility facility = facilityManager.getFacilityByHomeCommunityId(remoteHomeCommunityId);

        if (prependPartnerName) {
            errorMessage = facility.getFacilityName() + " " + errorMessage;
        }
        else {
            errorMessage = errorMessage + " " + facility.getFacilityName();
        }

        adhocQueryResponse.getRegistryErrorList().setHighestSeverity("urn:oasis:names:tc:ebxml-regrep:ErrorSeverityType:Error");

        RegistryError registryError = new RegistryError();
        registryError.setErrorCode("XDSRegistryError");
        registryError.setCodeContext(errorMessage);
        registryError.setValue(errorMessage);

        if (StringUtils.isNotEmpty(remoteHomeCommunityId)) {
            registryError.setLocation(remoteHomeCommunityId);
        }

        registryError.setSeverity("urn:oasis:names:tc:ebxml-regrep:ErrorSeverityType:Error");
        adhocQueryResponse.getRegistryErrorList().getRegistryError().add(registryError);
    }
}
