/**
 * RESTful web-service which returns data from the ADAPTER.DQ_TRANSMISSION_LOG database 
 * table's DAO class: DqTransmissionLogManager.
 * 
 * Uses HTTP URL context:
 * <base>/dataquality/DqTransmissionLogService?<optionalQueryParameters>
 * 
 */
package gov.va.med.nhin.adapter.dataquality;


import java.util.Date;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.List;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.QueryParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Response;
import javax.inject.Inject;
import javax.enterprise.context.RequestScoped;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.core.JsonProcessingException;
import gov.va.med.nhin.adapter.dataquality.common.CrossSiteScriptingXSSUtil;

import gov.va.med.nhin.adapter.dataquality.common.NullChecker;
import gov.va.med.nhin.adapter.dataquality.transmissionlog.DqTransmissionLog;
import gov.va.med.nhin.adapter.dataquality.transmissionlog.DqTransmissionLogManagerBean;
import gov.va.med.nhin.adapter.dataquality.transmissionlog.DqTransmissionLogManager;
import java.util.ArrayList;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


@Path("/DqTransmissionLogService")
@RequestScoped
public class DQTransmissionLogService {
    
    private static final Logger LOG = LoggerFactory.getLogger(DqTransmissionLogManagerBean.class.getName());
        
    @Inject DqTransmissionLogManager dqtlm;
    
    @GET
    @Produces("application/json") 
    public Response getDqTransmissionLogData(@QueryParam("runBeginStartDate") String runBeginStartDateStr, 
            @QueryParam("runFinishStartDate") String runFinishStartDateStr, @QueryParam("filename") String filename, 
            @QueryParam("eventType") String eventType, @QueryParam("status") String status) {
        LOG.debug("Received GET request...");   
        
        // verify and re-format Date input parameters          
        Date runBeginStartDate = null;      
        if (!NullChecker.isNullOrEmpty(runBeginStartDateStr)) {
            runBeginStartDate = this.getDateFromStr(runBeginStartDateStr);
            if(NullChecker.isNullOrEmpty(runBeginStartDate)) {       
                LOG.error("Unknown runBeginStartDate query parameter date-time format received - returning 400/Bad Request Error response");
                // return an error response instead
                String responseBodyStr = "{\"error\": \"HTTP 1.1 400/Bad Request Error\"}";
                return Response.status(400).entity(responseBodyStr).build();
            }
        }
        Date runFinishStartDate = null;
        if (!NullChecker.isNullOrEmpty(runFinishStartDateStr)) {
            runFinishStartDate =  this.getDateFromStr(runFinishStartDateStr);
            if(NullChecker.isNullOrEmpty(runFinishStartDate)) {
                LOG.error("Unknown runFinishStartDate query parameter date-time format received - returning 400/Bad Request Error response");
                // return an error response instead
                String responseBodyStr = "{\"error\": \"HTTP 1.1 400/Bad Request Error\"}";
                return Response.status(400).entity(responseBodyStr).build();
            }          
        } 
        // attempt to get the response body with database data as a JSON string
        String responseBodyStr = getLogResults(runBeginStartDate, runFinishStartDate, filename, eventType, status);  
        
        if(NullChecker.isNullOrEmpty(responseBodyStr)) {
            // return an error response instead
            responseBodyStr = "{\"error\": \"HTTP 1.1 500/Internal Server Error\"}";
            LOG.warn("From error: returning 500/Internal Server Error response");
            return Response.status(500).entity(responseBodyStr).build();
        } else { 
            LOG.debug("Success: returning 200/OK response");
            return Response.status(200).entity(CrossSiteScriptingXSSUtil.stripXSSAttack(responseBodyStr)).build();
        }
    }
    
    private String getLogResults(final Date runBeginStartDate, final Date runFinishStartDate, final String filename, final String eventType, final String status) { 
  
        // get query response with table date
        List<DqTransmissionLog> dqtlList = dqtlm.findByVarious(eventType, runBeginStartDate,runFinishStartDate, null, null, null, 
                null, null, filename, null, status, null);
        
        // count of results being returned
        int resultCount = dqtlList.size();
        
        DqTransmissionLogDataResponse responseObj = new DqTransmissionLogDataResponse();
        responseObj.setTableName("DqTransmissionLog");
        responseObj.setResultCount(resultCount);
        responseObj.setData(dqtlList);

        // transform whole of response metadata and data list of DqTransmissionLog results 
        // into an individual JSON-formatted String object
        String responseJsonStr = null;
        try {        
            ObjectMapper objectMapper = new ObjectMapper();        
            responseJsonStr = objectMapper.writeValueAsString(responseObj);
        } catch (JsonProcessingException e) {
            LOG.error("Error transforming DqTransmissionLog query response object into a JSON string!: ",e);
            // abort processing if error
            responseJsonStr = null;
        }        
        return responseJsonStr;
    }
     
    /**
     * RESTful DqTransmissionLog table data response to be transformed into a 
     * JSON String for returning DqTransmissionLog table query results.
     * 
     */
    public class DqTransmissionLogDataResponse {
        
        private String tableName;
        
        private int resultCount;
        
        private List<DqTransmissionLog> data;   

        public String getTableName() {
            return tableName;
        }

        public void setTableName(String tableName) {
            this.tableName = tableName;
        }

        public int getResultCount() {
            return resultCount;
        }

        public void setResultCount(int resultCount) {
            this.resultCount = resultCount;
        }

        public List<DqTransmissionLog> getData() {
            return data;
        }

        public void setData(List<DqTransmissionLog> data) {
            this.data = data;
        }
        
    }
    
    private Date getDateFromStr(String dateStr) {        
        if(NullChecker.isNotNullOrEmpty(dateStr)) {
            List<SimpleDateFormat> knownPatterns = new ArrayList<>();
            knownPatterns.add(new SimpleDateFormat("yyyy-MM-dd.HH:mm:ss"));
            knownPatterns.add(new SimpleDateFormat("yyyy-MM-dd"));
            knownPatterns.add(new SimpleDateFormat("yyyy-MM-dd.HH:mm"));
            knownPatterns.add(new SimpleDateFormat("yyyy-MM-dd.HH"));        

            for (SimpleDateFormat pattern : knownPatterns) {
                try {
                    Date responseDate = pattern.parse(dateStr);
                    return responseDate;
                } catch (ParseException e) {
                    LOG.debug("dateStr input failed to be parsed with current format: "+pattern.toPattern(), e);
                }
            }
        }
        return null;
    }    
}

