package gov.va.med.nhin.adapter.dataquality.util;

import gov.va.med.nhin.adapter.dataquality.common.DatabaseUtils;

import gov.va.med.nhin.adapter.dataquality.common.Utils;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.Properties;
import org.apache.commons.vfs2.FileObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.apache.commons.vfs2.provider.sftp.SftpFileSystemConfigBuilder;
import org.apache.commons.vfs2.FileSystemOptions;
import org.apache.commons.vfs2.FileSystemException;
import org.apache.commons.vfs2.Selectors;
import org.apache.commons.vfs2.impl.StandardFileSystemManager;

/**
 * Utility function to transfer the zip file from local directory to SFTP Shared Location for DQ Team
 * @author Sricharan Reddy
 * 
 */
public class SftpUtil
{
    public static final String DQ_EXPORT_PROPS = "dqExport.properties";
    public static final String SFTP_URL_CONSTANT= "sftp://";
    
    private static final Logger LOGGER = LoggerFactory.getLogger(SftpUtil.class);
    
    /**
     * Utility function to generate the sftp connection string from parameters
     * 
     * @param hostName -- The hostname of the DQ FTP Share server
     * @param username -- The username for the SFTP server
     * @param password -- The password for the SFTP server
     * @param remoteFilePath -- The directory on the remote server to push the local file 
     * @return the connection string representing the SFTP location
     */
    private static String createConnectionString(String hostName,String username, String password, String remoteFilePath) 
    {
    
    return SFTP_URL_CONSTANT + username + ":" + password + "@" + hostName + "/" + remoteFilePath;
    
    }
    
    /**
     * Function to return the default options on an SFTP connection
     * @return options object containing all the SFTP options
     * @throws FileSystemException 
     */
    private static FileSystemOptions createDefaultOptions()
        throws FileSystemException
    {
    // Initialize SFTP options object opts
    FileSystemOptions opts = new FileSystemOptions();
 
       // set SSH strict Key checking to No
     SftpFileSystemConfigBuilder.getInstance().setStrictHostKeyChecking(opts, "no");
 
    // Root directory set to user home
    SftpFileSystemConfigBuilder.getInstance().setUserDirIsRoot(opts, true);
 
    // Timeout is count by Milliseconds
    SftpFileSystemConfigBuilder.getInstance().setTimeout(opts, 50000);


    return opts;
   }


    /**
     * function to upload the file
     * 
     * @param filename of the local zip file to be uploaded
     */ 
    public static void uploadFile(String filename)
    {
        InputStream propInputStream = null;
           
        try
        {
            propInputStream = Utils.getResourceAsStream(DQ_EXPORT_PROPS);
            Properties props = new Properties();
            props.load(propInputStream);
            
            String hostname   = props.getProperty("ftpHostname");
            String username   = props.getProperty("ftpuser");
            String password   = props.getProperty("ftpPassword");
            String remotePath = props.getProperty("remoteDirectory");
            
            
           
            
              File f = new File(filename);
              if (!f.exists())
              {
                 throw new FileNotFoundException("Error. Local file not found");
              }
              
              StandardFileSystemManager manager = new StandardFileSystemManager();
              manager.init();
              
              FileObject localFile = manager.resolveFile(f.getAbsolutePath());
              
              FileObject remoteFile = manager.resolveFile(createConnectionString(hostname,username,password,remotePath), createDefaultOptions());
              
              remoteFile.copyFrom(localFile, Selectors.SELECT_SELF);

        } catch (Exception ex) {            
            LOGGER.error("Create DQ Export timer failed.", ex);                    
        } finally {
            DatabaseUtils.closeIO(propInputStream, LOGGER);
        }
        
        
    }
    
}
