package gov.va.med.nhin.adapter.dataquality.export;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Blob;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.sql.DataSource;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.nhin.adapter.dataquality.common.DatabaseUtils;


public class BulkExportLib  
{
   public static final String YEAR_MONTH_DAY = "yyyyMMdd";
   public static final String DISCLOSURE_ACTION = "Retrieve Document";
   public static final String RETRIEVE_FROM_PARTNER_ACTION = "RetrieveDocumentOut";

   public static final String ORG_URI_SCHEME = "urn:oid:";
   // this is the real org ID, but in SQA environments, .1 or .2 can be
   // appended to make a unique community ID for testing
   public static final String VA_ORG_NUMBER_PREFIX = "2.16.840.1.113883.4.349";
   public static final String VA_LONG_NAME = "DEPARTMENT OF VETERANS AFFAIRS";

   public static final String PROP_START_AUDIT_DATETIME = "start.audit.datetime";
   public static final String PROP_END_AUDIT_DATETIME = "end.audit.datetime";
   public static final String PROP_EXPORT_ROOT_DIR = "export.root.dir";
   public static final String PROP_CREATOR_ORG_ID = "creator.org.id";

   /**
    * The main query.
    * <table border="1" cellpadding="3px" cellspacing="0">
    *  <tr style="font-weight: bold">
    *   <td>Action Type</td>
    *   <td>Logical Action</td>
    *   <td>ORGANIZATION_ID</td>
    *   <td>REMOTE_ORGANIZATION_ID</td>
    *   <td>Doc Creator</td>
    *  </tr>
    * <tr>
    *   <td>RetrieveDocumentOut</td>
    *   <td>VA Downloads *from* Partner (Receipt)</td>
    *   <td>Dept. of Vet Affairs</td>
    *   <td>Partner</td>
    *   <td>REMOTE_ORGANIZATION_ID</td>
    *  </tr>
    *  <tr>
    *   <td>Retrieve Document</td>
    *   <td>Partner Downloads *from* VA (Disclosure)</td>
    *   <td>Partner (Maybe Internal VA)</td>
    *   <td>NULL</td>
    *   <td>Dept. of Vet Affairs</td>
    *  </tr>
    * </table>
    */
   private static final String EXPORT_SQL =
      "SELECT ACTION_NAME, ORGANIZATION_ID /* all same for RetrieveDocumentOut (same initiator, DoVA) */, " +
      "REMOTE_ORGANIZATION_ID /* use for RetrieveDocumentOut, null for Retrieve Document */, " +
      "REMOTE_ORG_FAC.FACILITY_NAME AS REMOTE_ORG_FACILITY_NAME, " +
      "AUDIT_TIME, DOCUMENTS.DOCUMENT_ID INTERNAL_DOC_ID, DOCUMENTS.DOCUMENT_UNIQUE_ID LOCAL_DOC_ID, CLASS_CODE, FORMAT_CODE, RAW_DATA, RAW_DATA_SIZE, AUDIT_ID " +   //removed RAW_DATA_SIZE, DOCUMENT_UNIQUE_ID - JMC 7/20/17
      "FROM AUDITS JOIN DOCUMENTS " +
      "ON DOCUMENTS.DOCUMENT_UNIQUE_ID = AUDITS.DOCUMENT_ID " +
      "LEFT JOIN FACILITIES REMOTE_ORG_FAC " +
      "ON AUDITS.REMOTE_ORGANIZATION_ID = REMOTE_ORG_FAC.FULL_HOME_COMMUNITY_ID " +      
      "WHERE AUDIT_TIME >= ? AND AUDIT_TIME < ? " +
      "AND ACTION_NAME IN ('RetrieveDocumentOut', 'Retrieve Document') ";
	
    private static final String AND_CREATOR_SQL =
        "AND (" +
        "   (ACTION_NAME = 'RetrieveDocumentOut' AND REMOTE_ORGANIZATION_ID = ?) OR " +
        "   (ACTION_NAME = 'Retrieve Document' AND 1 = ?)" + // will be 1 iff they want documents created by VA
             ") ";

    private static final String AND_PATIENT_ID_SQL =
    "AND AUDITS.PATIENT_ID = ? ";

    private static final String AND_PATIENT_SSN_SQL =
     "AND AUDITS.PATIENT_SSN = ? ";

    private static final String AND_PATIENT_LAST_NAME_SQL =
    "AND UPPER(AUDITS.PATIENT_LAST_NAME) = UPPER(?) ";

    private static final String AND_PATIENT_GIVEN_NAME_SQL =
     "AND UPPER(AUDITS.PATIENT_GIVEN_NAME) = UPPER(?) ";

    private static final String ORDER_BY_SQL =
    "ORDER BY AUDIT_TIME";

   private static final Logger LOGGER = LoggerFactory.getLogger(BulkExportLib.class);

   private final Date exportDate = new Date();
   private final DataSource dataSource;
   private final boolean doZip;

   public BulkExportLib(DataSource dataSource)
   {
      this.dataSource = dataSource;
      doZip = true;
   }

   public BulkExportLib(DataSource dataSource, boolean doZip)
   {
      this.dataSource = dataSource;
      this.doZip = doZip;
   }

   public void exportReceivedDocs(BulkExportParams params, BulkFileProcessor processor)
      throws IOException, SQLException 
   {
       processor.start();
       
       Connection conn = null;
       PreparedStatement stmt = null;
       //Blob rawData = null;
       
       try
       {
      	conn = dataSource.getConnection();

        StringBuilder sqlBldr = new StringBuilder(EXPORT_SQL);
        if(!StringUtils.isBlank(params.creatorOrgId)) {
                sqlBldr.append(AND_CREATOR_SQL);
        }
        if(!StringUtils.isBlank(params.patientIdentifier)) {
                sqlBldr.append(AND_PATIENT_ID_SQL);
        }
        if(!StringUtils.isBlank(params.ssn)) {
                sqlBldr.append(AND_PATIENT_SSN_SQL);
        }
        if(!StringUtils.isBlank(params.lastName)) {
                sqlBldr.append(AND_PATIENT_LAST_NAME_SQL);
        }
        if(!StringUtils.isBlank(params.firstName)) {
                sqlBldr.append(AND_PATIENT_GIVEN_NAME_SQL);
        }
        sqlBldr.append(ORDER_BY_SQL);

			// minimize locking via forward-only + read only ResultSet
         stmt = conn.prepareStatement(sqlBldr.toString(), ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
         int nextParamIdx = 1;
	 stmt.setTimestamp(nextParamIdx++, new Timestamp(params.startAuditedTime.getTime()));
         stmt.setTimestamp(nextParamIdx++, new Timestamp(params.endAuditedTime.getTime()));
         
        //***
        // The ordering here of conditionally setting parameters based on filter
        // criteria existing or not must follow the ordering above on conditionally
        // appending AND clauses to the SQL. The whole SQL has to be built before
        // we can prepare a statement object to set parameters on, so these same
        // conditional checks have to be repeated.
        //***
        if(!StringUtils.isBlank(params.creatorOrgId))
        {
            stmt.setString(nextParamIdx++, ORG_URI_SCHEME + params.creatorOrgId);
            boolean isFilterForVACreator = params.creatorOrgId != null &&
                        params.creatorOrgId.startsWith(VA_ORG_NUMBER_PREFIX);
            stmt.setInt(nextParamIdx++, isFilterForVACreator ? 1 : 0);
        }
        if(!StringUtils.isBlank(params.patientIdentifier)) {
            stmt.setString(nextParamIdx++, params.patientIdentifier);             
        }
        if(!StringUtils.isBlank(params.ssn)) {
            stmt.setString(nextParamIdx++, params.ssn);             
        }
        if(!StringUtils.isBlank(params.lastName)) {
            stmt.setString(nextParamIdx++, params.lastName);             
        }
        if(!StringUtils.isBlank(params.firstName)) {
            stmt.setString(nextParamIdx++, params.firstName);             
        }

         ResultSet rst = stmt.executeQuery();

         while(rst.next())
         {
            String actionName = rst.getString("ACTION_NAME");
            Timestamp auditDateTime = rst.getTimestamp("AUDIT_TIME");
            long internalDocId = rst.getLong("INTERNAL_DOC_ID");
            String docClassCode = rst.getString("CLASS_CODE");
            String formatCode = rst.getString("FORMAT_CODE"); 
            //rawData = rst.getBlob("RAW_DATA");
            //long blobLength = rawData.length();
            //LOGGER.debug("current blobLength: "+blobLength);
            byte[] rawData = rst.getBytes("RAW_DATA");
            long rawDataSize = rst.getLong("RAW_DATA_SIZE");
//            long rawDataSize = 8675309; //FIX THIS BEFORE PUSHING --> testing SQL issues...             
            String uniqueDocID = rst.getString("LOCAL_DOC_ID");
//            String uniqueDocID = "Test_DOCUMENT_ID"; //FIX THIS BEFORE PUSHING ---> testing SQL issues...
            long auditId = rst.getLong("AUDIT_ID");
            
            String docCreatorOrgId;
            String docCreatorName;
            if(DISCLOSURE_ACTION.equals(actionName))
            {
                // Doesn't really matter as of now, because docCreatorName
                // is not null. But just being defensively consistent.
            	docCreatorOrgId = ORG_URI_SCHEME + VA_ORG_NUMBER_PREFIX;
            	docCreatorName = VA_LONG_NAME;
            }
            else
            {
            	docCreatorOrgId = rst.getString("REMOTE_ORGANIZATION_ID");
            	docCreatorName = rst.getString("REMOTE_ORG_FACILITY_NAME");
            }
            
            // make sure docCreatorName is populated
            if(docCreatorName == null) {
            	docCreatorName = docCreatorOrgId == null ? "UNKNOWN ORG" : docCreatorOrgId;
            }
            // TODO: fix file name to use the DH convention
            String documentType = mapToDocumentTypeName(docClassCode, formatCode);
            String fileSubPath = buildFileSubPath(auditDateTime, docCreatorName, auditId, "" + internalDocId, documentType);
            //if (rawData == null) {
            //    LOGGER.warn("BLOB for RAW_DATA was null - skipping processing of internal doc ID " + internalDocId);
            //} else {
                //processor.processFile(uniqueDocID, rawDataSize, fileSubPath, rawData.getBinaryStream()); 
                processor.processFile(uniqueDocID, rawDataSize, fileSubPath, rawData, formatCode, docClassCode);
                //DatabaseUtils.closeBlob(rawData, LOGGER);
            //}
         }
      }
      finally
      {
        //DatabaseUtils.closeBlob(rawData, LOGGER);
        DatabaseUtils.closeStatement(stmt, LOGGER);
        DatabaseUtils.closeConnection(conn, LOGGER);
        processor.finish();
      }
       
   }

   public static Date parseRangeDateStr(String rangeDateStr) throws ParseException
   {
   	SimpleDateFormat dateOnly = new SimpleDateFormat("yyyy-MM-dd");
	   SimpleDateFormat dateTime = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm");

	   Date rangeDate;
      try
      {
         if(rangeDateStr.contains("T"))
         {
            rangeDate = dateTime.parse(rangeDateStr);
         }
         else
         {
            rangeDate = dateOnly.parse(rangeDateStr);
         }
      }
      catch(ParseException ex)
      {
         throw new ParseException("Start date and end date must be of the " +
           "format 'yyyy-MM-dd' or 'yyyy-MM-dd'T'HH:mm'.", ex.getErrorOffset());
      }
      return rangeDate;
   }
   
   /**
    * Maps to friendly name that business likes.
    * @param docClassCode LOINC
    * @param formatCode from FORMAT_CODE in Documents table
    * @return
    */
   static String mapToDocumentTypeName(String docClassCode, String formatCode)
   {
   	if(StringUtils.isBlank(docClassCode))
   	{
   		return "UNK-DOC-TYPE";
   	}
   	
   	// obviate null checks below
   	if(formatCode == null) {
   		formatCode = "";
   	}
   	
   	if(docClassCode.equals("34133-9"))
   	{
   		if(formatCode.equals("urn:ihe:pcc:xphr:2007"))
   		{
   			return "C32";
   		}
   		
   		if(
   			formatCode.equals("urn:hl7-org:sdwg:ccda-structuredBody:1.1") ||
   			formatCode.equals("urn:hl7-org:sdwg:CCDA:1.1") ||
   			formatCode.equals("urn:hl7-org:sdwg:ccda-structuredBody:2.1"))
   		{
   			return "CCDACCD";
   		}
   	}
   	
   	if(docClassCode.equals("11506-3"))
   	{
   		if(
   			formatCode.equals("urn:hl7-org:sdwg:ccda-structuredBody:1.1") ||
   			formatCode.equals("urn:hl7-org:sdwg:ccda-structuredBody:2.1"))
   		{
   			return "CCDAPN";
   		}
   	}
   	
		if(formatCode.equals("urn:ihe:iti:xds-sd:pdf:2008"))
		{
			return "C62PDF";
		}

		if(formatCode.equals("urn:ihe:iti:xds-sd:text:2008"))
		{
			return "C62TXT";
		}
		
		// no mapping...default to LOINC
		return docClassCode;
   }

   static void exportFile(File destinationRoot, String fileSubPath, InputStream input) throws IOException, SQLException
   {
   	File targetFile = new File(destinationRoot, FilenameUtils.normalize(fileSubPath));
   	if(targetFile.exists())
      {
         return;
      }
      
      FileUtils.forceMkdir(targetFile.getParentFile());
      FileOutputStream fileOut = null;
      
      try
      {
         fileOut = new FileOutputStream(targetFile);
         IOUtils.copy(input, fileOut);
         fileOut.flush();
      }
      finally
      {
         DatabaseUtils.closeIO(fileOut, LOGGER);
      }
   }

   static void exportFile(ZipOutputStream zipOut, Set<String> pathsWritten, String fileSubPath, InputStream input)
   		throws IOException, SQLException
   {
   	if(pathsWritten.contains(fileSubPath))
		{
			return;
		}
   	pathsWritten.add(fileSubPath);
   	
   	ZipEntry entry = new ZipEntry(fileSubPath);
   	zipOut.putNextEntry(entry);
      IOUtils.copy(input, zipOut);
   }

   /**
    * Naming Convention for Downloaded Patient Files
    * The parent folder will the date the file was generated.
    * The sub parent folders will be separated by document creator.
    * Naming convention for specific files: <index><type><dateStored>.xml
    * E.g. 0001_C32_20161201.xml
	 * Date = date generated or received
    */
   String buildFileSubPath(Date auditDateTime, String docCreatorName, long auditId, String index, String docType)
   {
      SimpleDateFormat sdf = new SimpleDateFormat(YEAR_MONTH_DAY);
      //StringBuilder path = new StringBuilder(sdf.format(exportDate));
      StringBuilder path = new StringBuilder(StringUtils.EMPTY);
   	
      docCreatorName = docCreatorName.replace("urn:oid:", "oid_");
      docCreatorName = cleanFileName(docCreatorName);
      //path.append("/").append(docCreatorName);
      
      docType = cleanFileName(docType);
      String auditDateStr = sdf.format(auditDateTime);
      //String fileName = index + "_" + docType + "_" + auditDateStr + ".xml"; -- Original code 
      String fileName = docCreatorName + "_" + docType + "_" + auditDateStr + "_" + auditId + "_" + index + ".xml";
       if (fileName.length()>255)
       {
          int diff = 255 -(fileName.length()); // This is the excess length to be trimmed off from the Partner name
          String truncatedFileName =  docCreatorName.substring(0, docCreatorName.length()-diff-1) + "_" + docType + "_" + auditDateStr + "_" + index + ".xml";
          path.append(truncatedFileName);
       }
       else
       {
          path.append(fileName);
       }
      return path.toString();
   }
   
   static final String cleanFileName(String fileName)
   {
   	char[] cleaned = null;
   	int fileLen = fileName.length();
   	for(int i = 0; i < fileLen; i++)
   	{
   		char charAt = fileName.charAt(i);
   		boolean ok =
   				('A' <= charAt && charAt <= 'Z') ||
   				('a' <= charAt && charAt <= 'z') ||
   				('0' <= charAt && charAt <= '9') ||
   				charAt == '-' ||
   				charAt == '_' ||
   				charAt == '.' ||
   				charAt == ' ';
   		
   		if(ok) {
   			continue;
   		}
   		
   		// lazy create copy...most names will be fine in reality
   		if(cleaned == null) {
   			cleaned = fileName.toCharArray();
   		}
   		cleaned[i] = '-';
   	}
   	
   	if(cleaned == null) {
   		return fileName;
   	} else {
   		return new String(cleaned);
   	}
   }

	public static class BulkExportParams
	{
		public Date startAuditedTime;
		public Date endAuditedTime;
		public String creatorOrgId;
		public String patientIdentifier;
		public String ssn;
		public String lastName;
		public String firstName;
                public String zipFilename;
	}
}
