package gov.va.med.nhin.adapter.dataquality.export;


import java.io.InputStream;
import java.util.Properties;
import javax.annotation.Resource;
import javax.ejb.*;
import javax.ejb.Timeout;
import javax.ejb.Timer;
import javax.ejb.TimerService;
import javax.annotation.PostConstruct;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.nhin.adapter.dataquality.common.DatabaseUtils;
import gov.va.med.nhin.adapter.dataquality.common.LogUtil;
import gov.va.med.nhin.adapter.dataquality.common.NullChecker;
import gov.va.med.nhin.adapter.dataquality.common.Utils;
import gov.va.med.nhin.adapter.dataquality.reporting.TransmissionReport;
import gov.va.med.nhin.adapter.dataquality.reporting.TransmissionReportLocal;
import gov.va.med.nhin.adapter.dataquality.transmissionlog.DqTransmissionLogManager;
import gov.va.med.nhin.adapter.dataquality.transmissionlog.DqTransmissionLogManagerLocal;
import javax.sql.DataSource;
import org.apache.commons.validator.routines.RegexValidator;


@Singleton
@Startup
@TransactionAttribute(value = TransactionAttributeType.NOT_SUPPORTED)
public class BulkExportBean implements BulkExport
{
   
   @Resource
   TimerService timerService;
   
   @EJB(beanInterface = DqTransmissionLogManagerLocal.class, beanName = "DqTransmissionLogManager")
   private DqTransmissionLogManager dqlm;
    
   @EJB(beanInterface = TransmissionReportLocal.class, beanName = "TransmissionReport")
   private TransmissionReport trep;
   
   @Resource(name = "jdbc/adapter")
   private DataSource dataSource;
   
   private static final Logger LOGGER = LoggerFactory.getLogger(BulkExportBean.class);
   private static final String DQ_EXPORT_RUN_PROCESS = "dqExport.runTimer";

   /**
    * Creates a repeating timer to export files to DH and DQ server
    */
   @PostConstruct
   public void createTimer() {
       
   	// remove any old timers first
   	for(Object timerObj: timerService.getTimers())
   	{
            Timer timer = (Timer)timerObj;
            timer.cancel();
   	}
        
   	InputStream propInputStream = null;
        try
        {
            boolean setupTimer = true;
            
            // get properties file 
            propInputStream = Utils.getResourceAsStream(DQExportProperties.DQ_EXPORT_PROPS);
            Properties props = new Properties();
            props.load(propInputStream);
            ScheduleExpression schedule = new ScheduleExpression();

            String dqExportRunIfPropertyType = props.getProperty(DQExportProperties.PROP_DQEXPORT_RUNIF_PROPERTY_TYPE, "SYS");
            String dqExportRunIfPropertyName = props.getProperty(DQExportProperties.PROP_DQEXPORT_RUNIF_PROPERTY_NAME);
            String dqExportRunIfPropertyValue = props.getProperty(DQExportProperties.PROP_DQEXPORT_RUNIF_PROPERTY_VALUE, "");
            String value;
            if (NullChecker.isNotNullOrEmpty(dqExportRunIfPropertyName)
                && NullChecker.isNotNullOrEmpty(dqExportRunIfPropertyValue)) {
                if (dqExportRunIfPropertyType.equalsIgnoreCase("ENV")) {
                    value = System.getenv(dqExportRunIfPropertyName);
                }
                else {
                    value = System.getProperty(dqExportRunIfPropertyName);
                }
                //setupTimer = value.matches(dqExportRunIfPropertyValue);
                //Apache-commons validator f/w is implemented to remidiate the Fortify issue- Denial ofService - regexp. 
                //RegEx is case sensitive as in the below instance
                RegexValidator regexValidator = new RegexValidator(dqExportRunIfPropertyValue);
                setupTimer = regexValidator.isValid(value);
            }
            
            if (true == setupTimer) {
                String schedulerDayofWeek = props.getProperty(DQExportProperties.PROP_SCHED_DAY_OF_WEEK);
                if(schedulerDayofWeek != null && schedulerDayofWeek.length() > 0) {
                      schedule.dayOfWeek(schedulerDayofWeek);
                }

                String schedulerHour = props.getProperty(DQExportProperties.PROP_SCHED_HOUR);
                if(schedulerHour != null && schedulerHour.length() > 0) {
                      schedule.hour(schedulerHour);
                }

                String schedulerMinute = props.getProperty(DQExportProperties.PROP_SCHED_MINUTE);
                if(schedulerMinute != null && schedulerMinute.length() > 0) {
                      schedule.minute(schedulerMinute);
                }

                String schedulerSecond = props.getProperty(DQExportProperties.PROP_SCHED_SECOND);
                if(schedulerSecond != null && schedulerSecond.length() > 0) {
                      schedule.second(schedulerSecond);
                }

                // create the actual repeating Timer with schedule settings
                timerService.createCalendarTimer(schedule);

                LOGGER.info(LogUtil.cleanLogMessage("Data Quality BulkExportBean will process files using TimerService schedule: " + schedule));
            }
            else {
                LOGGER.info(LogUtil.cleanLogMessage("Data Quality BulkExportBean WILL NOT process files."));
            }
        }
        catch (Exception ex)
        {
            LOGGER.error("Data Quality BulkExportBean createTimer failed: ", ex);
        }
        finally
        {
            DatabaseUtils.closeIO(propInputStream, LOGGER);
        }
   } // end createTimer()
   
   @Timeout
   public void timeout(Timer timer) {
       Orchestrator orchestrator = new Orchestrator();
       orchestrator.setDqTransmissionLogManager(dqlm);
       orchestrator.setTransmissionReport(trep);
       orchestrator.setDataSource(dataSource);
       orchestrator.orchestrate();
   }
   
}
