/**
 * SMPT email client to create and send email messages 
 * using the email.properties values loaded in EmailBean. 
 * Copied from VAP and modified. 
 * 
 */
package gov.va.med.nhin.adapter.dataquality.common;


import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import javax.mail.BodyPart;

import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.PasswordAuthentication;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.InternetHeaders;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMultipart;
import javax.mail.internet.MimeMessage;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class Emailer {
    
    private static final Logger LOG = LoggerFactory.getLogger(Emailer.class.getName());
     
    
    private final static String SMTP_AUTH = "mail.smtp.auth";
    private final static String SMTP_STARTTLS = "mail.smtp.starttls.enable";
    private final static String SMTP_HOST = "mail.smtp.host";
    private final static String SMTP_PORT = "mail.smtp.port";

    private final static String SUBJECT = "subject";
    private final static String BODY = "body";
    private final static String REPORT_PATTERN = "%report%";
    

    /**
     * @param distributionList List of email addresses and/or contact groups separated by semicolon
     * @param messageAttributes Map of message attributes, keyed by attribute name. Currently supported keys are:
     *                          "subject"
     *                          "body" 
     * @param report List of items in the report. If "body" attribute contains "%report%" stub, generated tabular representation
     *                of the report will substitute the stub; otherwise, it will be appended at the end of the "body" attribute.
     * @return true, if email messages are sent correctly;
     *          false, if at least one send operation fails
     */
    final public boolean emailReport(String distributionList, Map<String, String> messageAttributes,
    		List<Map<String, Object>> report) {
        boolean retValue = true;
        EmailBean emailBean = EmailBean.getInstance();
        final String username = emailBean.getFrom();
        final String password = emailBean.getPassword();
        final String reportLink = emailBean.getLink();

        Properties props = new Properties();
        props.put(SMTP_AUTH, "true");
        props.put(SMTP_STARTTLS, "true");
        props.put(SMTP_HOST, emailBean.getHost());
        props.put(SMTP_PORT, emailBean.getPort());

        Session session = Session.getInstance(props,
            new javax.mail.Authenticator() {
                @Override
                protected PasswordAuthentication getPasswordAuthentication() {
                       return new PasswordAuthentication(username, password);
                }
        });
 
        try {
            MimeMessage message = new MimeMessage(session);
            message.setFrom(new InternetAddress(emailBean.getFrom()));                               
            addRecipients(message, distributionList);   	                        
            message.setSubject(messageAttributes.get(SUBJECT));                
            MimeBodyPart bodyText = new MimeBodyPart();             
            InternetHeaders headers = new InternetHeaders();
            headers.addHeader("Content-type", "text/html; charset=UTF-8");
            bodyText.setText(messageAttributes.get(BODY) + reportLink);
            MimeMultipart body = new MimeMultipart();
            body.addBodyPart(bodyText);                                    	      
            message.setContent(body);
            Transport.send(message);
        } catch (MessagingException e) {
            LOG.error("emailReport MessagingException: ",e);
            //e.printStackTrace(System.out);
            retValue = false;
        }
        return retValue;
    }
    
    /**
     * Sends an notification email to target email addresses separated by semicolon in distributionList String, 
     * where messageAttributes Map contains "subject" and "body" keys. Uses information from email.properties 
     * along with the above input parameters.
     * 
     * Example:
     *       String bodyString = "VAP Bulk Data Upload Failure!" 
     *       Map<String, String> messageAttributes = new HashMap<String, String>();
     *       messageAttributes.put("subject", "VAP Data Bulk Upload Failure Notice");
     *       messageAttributes.put("body", bodyString);
     *
     *       Emailer emailer = new Emailer();
     *       emailer.emailNotification("PII                ; PII                ",
     *       messageAttributes);
     *
     * 
     * @param distributionList List of email addresses and/or contact groups separated by semicolon
     * @param messageAttributes Map of message attributes, keyed by attribute name. Currently supported keys are:
     *                          "subject"
     *                          "body"
     * @return true, if email messages are sent correctly;
     *          false, if at least one send operation fails
     */
    final public boolean emailNotification(String distributionList, Map<String, String> messageAttributes) {
        boolean retValue = true;
        //System.out.println("in emailNotification() ...");
        EmailBean emailBean = EmailBean.getInstance();
        final String username = emailBean.getUsername();
        final String password = emailBean.getPassword();
        
        Session session = null;
                
        //try {
        // attempt to retrieve session from JNDI:
        //InitialContext ic = new InitialContext();
        //session = (Session) ic.lookup("myMailSession");        
        //} catch(NamingException e) {
            // TODO: Log exception as WARN?            
        //}
        
//        if(null == session) {
            // create a session from the EmailBean/email.properties file
            Properties props = new Properties();
            props.put(SMTP_AUTH, "true");
            props.put(SMTP_STARTTLS, "true");
            //props.put("mail.transport.protocol", "smtp");
            props.put(SMTP_HOST, emailBean.getHost());
            props.put(SMTP_PORT, emailBean.getPort());
            //props.put("mail.from", emailBean.getFrom());
            //TODO: should this also be an empty check?
            if(null != username && null != password) {
                session = Session.getInstance(props,
                new javax.mail.Authenticator() {
                      @Override
                      protected PasswordAuthentication getPasswordAuthentication() {
                             return new PasswordAuthentication(username, password);
                      }
                });
            } else {            
                session = Session.getInstance(props);            
            }
//        }        
      
        try {            
            // create message
            Message message = new MimeMessage(session);
            // set message properties
            message.setFrom(new InternetAddress(emailBean.getFrom()));                          
            addRecipients(message, distributionList); 
            message.setSentDate(new Date()); 
            message.setSubject(messageAttributes.get(SUBJECT));                                
            // set the body
            BodyPart bodyText = new MimeBodyPart(); 
            bodyText.setText(messageAttributes.get(BODY));
            bodyText.addHeader("Content-type", "text/plain; charset=UTF-8");
            //bodyText.addHeader("Content-type", "text/html; charset=UTF-8");                        
            Multipart body = new MimeMultipart();
            body.addBodyPart(bodyText);                       
            message.setContent(body);            
            //System.out.println("about to send the message...");
            LOG.debug(" about to send the message...");
            // send the message            
            Transport.send(message); 
            //System.out.println("... message sent.");
            LOG.debug("... message sent.");
        } catch (MessagingException e) {            
            LOG.error("emailNotification MessagingException: ",e);
            //e.printStackTrace(System.out);
            retValue = false;
        }        
        return retValue;
    }
    
    /**
     * Sends an email containing the Data Quality health document daily upload summary information 
     * to the email.toList distribution list in email.properties. The only messageAttributes map 
     * value which needs to be set programmatically is the "body" key/value item, 
     * which contains the contents of the email. The subject from the properties 
     * file is used instead.
     * 
     * Example:
     *       String bodyString = "Diameter Health bulk data upload daily summary contents here"; 
     *       Map<String, String> messageAttributes = new HashMap<>();       
     *       messageAttributes.put("body", bodyString);
     *
     *       Emailer emailer = new Emailer();
     *       boolean success = emailer.emailNotificationDQUpload(messageAttributes);
     * 
     * @param messageAttributes Map of message attributes, keyed by attribute name. Currently supported keys are:
     *                          "body"
     * @return true, if email messages are sent correctly;
     *          false, if at least one send operation fails
     */
    final public boolean emailNotificationDQUpload(Map<String, String> messageAttributes) {                
        EmailBean emailBean = EmailBean.getInstance();
        String distributionList = emailBean.getToList();
        String subject = emailBean.getSubject();
        messageAttributes.put("subject", subject);        
        // call the wrapped method 
        boolean retValue = this.emailNotification(distributionList, messageAttributes);        
        return retValue;        
    }

    /**
     * @param distributionList List of email addresses and/or contact groups separated by semicolon
     * @param messageAttributes Map of message attributes, keyed by attribute name. Currently supported keys are:
     *                          "subject"
     *                          "body" 
     */
    private void addRecipients(Message message, String distributionList)
    		throws MessagingException{    	
    	String[] list = distributionList.split(";");
    	for (int i=0; i<list.length; i++){
    		 message.addRecipient(Message.RecipientType.TO,
                     new InternetAddress(list[i]));
    	}
    }

    /**
     * @param reportList List of items in the report. If "body" attribute contains "%report%" stub, generated tabular representation
     *                of the report will substitute the stub; otherwise, it will be appended at the end of the "body" attribute.
     * @return The report;
     */
    /*
    private String getTabReport(List<Map<String, Object>> reportList){
    	Iterator<Map<String, Object>> iter = reportList.iterator();
		StringBuilder report = new StringBuilder();
    	while(iter.hasNext()){
			Map<String, Object> map = iter.next();
			for(Map.Entry<String, Object> entry : map.entrySet()){
				report.append(entry.getValue().toString()).append("\n");
			}
		}
    	return report.toString();
    }
    */
}
