package gov.va.med.nhin.adapter.facilitymanager;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ejb.*;
import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.nhin.adapter.utils.NullChecker;

/**
 * DAO for the FACILITIES table.
 * 
 * @author David Vazquez
 */
@TransactionAttribute(value = TransactionAttributeType.SUPPORTS)
@Stateless(name = "FacilityManager", mappedName = "FacilityManager")
public class FacilityManagerBean implements FacilityManagerRemote, FacilityManagerLocal
{
	private static final Logger logger = LoggerFactory.getLogger(FacilityManagerBean.class.getName());

	private EntityManager entityManager;

	@PersistenceContext
	public void setEntityManager(EntityManager entityManager)
	{
		this.entityManager = entityManager;
	}

	@Override
	public List<Facility> getAllFacilities()
	{
		Query query = entityManager.createNamedQuery("Facility.findAll");
		
		return query.getResultList();
	}

	public Facility getMyFacility()
	{
		Query query = entityManager.createNamedQuery("Facility.findByFacilityNumber");
		query.setParameter("facilityNumber", "VA");
		return getSingleFacilityResult(query);
	}

	@Override
	public List<Facility> getAllFacilities(String... excludeFacilityNumbers)
	{
		StringBuilder whereClause = new StringBuilder();
		Map<String, Object> setParams = new HashMap<>();

		if(!NullChecker.isNullOrEmpty(excludeFacilityNumbers))
		{
			whereClause.append(" WHERE f.facilityNumber NOT IN (:facility0");
			setParams.put("facility0", excludeFacilityNumbers[0]);

			for(int i = 1; i < excludeFacilityNumbers.length; ++i)
			{
				whereClause.append(",:facility").append(i);
				setParams.put("facility" + i, excludeFacilityNumbers[i]);
			}

			whereClause.append(")");
		}

		Query query = entityManager.createQuery("SELECT f from Facility f" + whereClause.toString());

		for(Map.Entry<String, Object> e : setParams.entrySet())
		{
			query.setParameter(e.getKey(), e.getValue());
		}

		return query.getResultList();
	}

	@Override
	public Facility getFacilityByFacilityNumber(String facilityNumber)
	{
		Query query = entityManager.createNamedQuery("Facility.findByFacilityNumber");
		query.setParameter("facilityNumber", facilityNumber);
		return getSingleFacilityResult(query);
	}

	public Facility getFacilityByHomeCommunityId(String oid)
	{
		Query query = entityManager.createNamedQuery("Facility.findByHomeCommunityId");
		query.setParameter("homeCommunityId", oid);
		return getSingleFacilityResult(query);
	}

	@Override
	public Facility getFacilityByFullHomeCommunityId(String fullHomeCommunityId)
	{
		Query query = entityManager.createNamedQuery("Facility.findByFullHomeCommunityId");
		query.setParameter("fullHomeCommunityId", fullHomeCommunityId);
		return getSingleFacilityResult(query);
	}

	@Override
	public List<Facility> getFacilitiesContainingAssigningAuthority(String assigningAuthorityOID)
	{
		logger.info("In FacilityManagerBean::getFacilitiesContainingAssigningAuthority");
		List<Facility> ret = new ArrayList<>();
		Query query = entityManager.createNamedQuery("AssigningAuthority.findByAssigningAuthorityOid");
		query.setParameter("assigningAuthorityOid", assigningAuthorityOID);
		AssigningAuthority result = getSingleAAResult(query);
		if(result != null)
		{
			for(FacilityToAA f2aa : result.getFacilitiesToAAs())
			{
				ret.add(f2aa.getFacility());
				// logger.info("Facility added to return"); //CCR 177986
			}
		}
		else
		{
			logger.info("AssiningAuthority query result is null");
		}
		return ret;
	}

	@SuppressWarnings("unchecked")
	public String getDocumentTypeToBeGenerated(String loincode, String partnerHomeCommunityId)
	{
		Facility myFacility;
		List<FacSupDoctypes> partnerFacSupDoctypesList;

		Query query = entityManager.createQuery("SELECT fsd FROM FacSupDoctypes fsd WHERE fsd.homeCommunityId = :hci AND fsd.loincCode = :code AND fsd.isPrefType = :pref");
		query.setParameter("hci", partnerHomeCommunityId);
		query.setParameter("code", loincode);
        query.setParameter("pref", "Y");
		partnerFacSupDoctypesList = query.getResultList();

		return partnerFacSupDoctypesList.get(0).getQueryName();
	}

	/**
	 * Get all possible queryNames for a particular Home Community.
	 * 
	 * @param partnerHomeCommunityId
	 * @return
	 */
	@SuppressWarnings("unchecked")
    @Override
	public List<String> getQueryNamesToGenerateDocuments(final String partnerHomeCommunityId)
	{
		List<FacSupDoctypes> partnerFacSupDoctypesList;

		List<String> commonQueryNames = new ArrayList<>();

		Query query = entityManager.createQuery("SELECT fsd FROM FacSupDoctypes fsd WHERE fsd.homeCommunityId = :hci AND fsd.isPrefType = :pref");
		query.setParameter("hci", normalizeHCID(partnerHomeCommunityId));
        query.setParameter("pref", "Y");
		partnerFacSupDoctypesList = query.getResultList();

		for(FacSupDoctypes partnerFSD : partnerFacSupDoctypesList)
		{
			String queryName = partnerFSD.getQueryName();
			if(StringUtils.isNotBlank(queryName))
			{
				commonQueryNames.add(queryName);
			}
		}

		return commonQueryNames;
	}

	@SuppressWarnings("unchecked")
    @Override
	public List<String> getQueryNamesToGenerateDocuments(String partnerHomeCommunityId, List<String> loinCodeListIn)
	{
		List<String> validatedLoinCdList = new ArrayList<>();

		List<String> parsedLoincs = listFromQueryParam(loinCodeListIn);

		Query query = entityManager.createQuery("SELECT fsd FROM FacSupDoctypes fsd WHERE fsd.homeCommunityId = :hci AND fsd.isPrefType = :pref");
		query.setParameter("hci", normalizeHCID(partnerHomeCommunityId));
        query.setParameter("pref", "Y");
		List<FacSupDoctypes> partnerFacSupDoctypesList = query.getResultList();

        for(FacSupDoctypes partnerFSD : partnerFacSupDoctypesList)
		{
            if(parsedLoincs.contains(partnerFSD.getLoincCode())) {
                validatedLoinCdList.add(partnerFSD.getQueryName());
            }
        }

		return validatedLoinCdList;
	}

        @Override
        @TransactionAttribute(value = TransactionAttributeType.REQUIRED)
	public void addAssigningAuthorityToFacility(String homeCommunityId, String assigningAuthorityOid, String assigningAuthorityName)
	{
		logger.debug("addAssigningAuthorityToFacility() invoked");

		Query query = entityManager.createNamedQuery("Facility.findByHomeCommunityId");
		query.setParameter("homeCommunityId", homeCommunityId);
		Facility facility = getSingleFacilityResult(query);

		if(facility != null)
		{
			FacilityToAA facilityToAA = null;
			if(!NullChecker.isNullOrEmpty(facility.getFacilitiesToAAs()))
			{
				for(FacilityToAA f2aa : facility.getFacilitiesToAAs())
				{
					if(f2aa.getAssigningAuthority().getAssigningAuthorityOid().equals(assigningAuthorityOid))
					{
						facilityToAA = f2aa;
						break;
					}
				}
			}

			if(facilityToAA == null)
			{
				Query aaQuery = entityManager.createNamedQuery("AssigningAuthority.findByAssigningAuthorityOid");
				aaQuery.setParameter("assigningAuthorityOid", assigningAuthorityOid);
				AssigningAuthority assigningAuthority = getSingleAAResult(aaQuery);

				if(assigningAuthority == null)
				{
					assigningAuthority = new AssigningAuthority();
					assigningAuthority.setAssigningAuthorityOid(assigningAuthorityOid);
					assigningAuthority.setAssigningAuthorityName(assigningAuthorityName);
					assigningAuthority.setFacilitiesToAAs(new ArrayList<FacilityToAA>());
					entityManager.persist(assigningAuthority);
				}

				facilityToAA = new FacilityToAA();
				facilityToAA.setAssigningAuthority(assigningAuthority);
				facilityToAA.setFacility(facility);
				facility.getFacilitiesToAAs().add(facilityToAA);
				assigningAuthority.getFacilitiesToAAs().add(facilityToAA);
				entityManager.persist(facilityToAA);
			}
		}

		logger.debug("addAssigningAuthorityToFacility() exited");
	}

	private Facility getSingleFacilityResult(Query query)
	{
		Facility ret;

		try
		{
			ret = (Facility) query.getSingleResult();
		}
		catch(NoResultException nre)
		{
			ret = null;
		}

		return ret;
	}

	private AssigningAuthority getSingleAAResult(Query query)
	{
		AssigningAuthority ret;

		try
		{
			ret = (AssigningAuthority) query.getSingleResult();
		}
		catch(NoResultException nre)
		{
			ret = null;
		}

		return ret;
	}

	/**
	 * Incoming queryParameters can be of type list with a single element. Or
	 * can be a list of values depending on spec version. Need to normalize the
	 * data into a list of LOINCS.
	 * 
	 * @param incoming
	 * @return
	 */
	private List<String> listFromQueryParam(final List<String> incoming)
	{
		List<String> loincs = new ArrayList<>();

		if(CollectionUtils.isEmpty(incoming))
		{
			return Collections.emptyList();
		}

		for(String value : incoming)
		{
			String[] values = parseOutAllQueryParam(value);

			for(String single : values)
			{
				loincs.add(extractLoinc(single));
			}
		}

		return loincs;
	}

	private String[] parseOutAllQueryParam(final String value)
	{
		String val = parseSingleQueryParam(value);
		String[] parsed = StringUtils.split(val, ",");
		return parsed;
	}

	private String parseSingleQueryParam(final String value)
	{
		String parsed = value;
		// Remove leading paren.
		parsed = StringUtils.remove(parsed, "(");
		// Remove trailing paren.
		parsed = StringUtils.remove(parsed, ")");
		// Remove single qoutes.
		parsed = StringUtils.remove(parsed, "'");
		return parsed;
	}

	/**
	 * Will return just the LOINC from an inbound request. Some values come in
	 * as just LOINC some come in as LOINC^^TEMPLATID.
	 * 
	 * @param value
	 * @return
	 */
	private String extractLoinc(final String value)
	{
		if(!StringUtils.contains(value, "^^"))
		{
			// Did not find a delimiter return value.
			return value;
		}

		// There is a delimiter need to parse out the LOINC.
		String[] split = value.split("\\^\\^");
		return split[0];

	}

	private String normalizeHCID(final String hcid)
	{
		return StringUtils.remove(hcid, "urn:oid:");
	}

	/**
	 * TODO
	 * @param hcid
	 * @param operation
	 * @return true if
	 */
	@Override
	public boolean isPartnerAllowed(final String hcid, final OperationOnOff operation)
	{
		if(StringUtils.isBlank(hcid) || operation == null)
		{
			return false;
		}
		
		String fullHCID = hcid;
		
		if(!fullHCID.startsWith("urn:oid:"))
		{
			fullHCID = "urn:oid:" + fullHCID;
		}
		
		StringBuilder querySb = new StringBuilder();
		querySb.append("SELECT f, fo from Facility as f, Operations as fo ");
		querySb.append("WHERE f.fullHomeCommunityId = :fullHomeCommunityId AND f.facilityId = fo.id ");
		querySb.append("AND fo.");
		querySb.append(operation.getParam());
		querySb.append(" = :opValue");
		
		Query query = entityManager.createQuery(querySb.toString());
		query.setParameter("fullHomeCommunityId", fullHCID);
		query.setParameter("opValue", "Y");
		
		return NullChecker.isNotNullOrEmpty( query.getResultList() );
	}
}
