package gov.va.nvap.web.util.xls;

import gov.va.nvap.common.validation.Assert;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.svc.consentmgmt.stub.dao.DelayReasonDAO;
import gov.va.nvap.web.dao.FacilityDAO;
import gov.va.nvap.web.facility.Facility;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.util.Constants;
import java.io.ByteArrayOutputStream;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.lang.StringUtils;

import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.CreationHelper;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddress;
import org.springframework.beans.factory.annotation.Required;

/**
 * Helper class to export a set of data rows to excel. It encapsulates the
 * apache POI excel workbook modules. It serves as a helper class to create VAP
 * specific excel reports which can be reused for all reports.
 *
 * The class can be refactored to follow a more Object Oriented structure when
 * the reports are also migrated to follow a an object structure with Title,
 * Headers and Data Rows.
 *
 * @author Asha Amritraj
 *
 */
public class ExcelExporter {

    // Static fields

	private static final String EXCEL_CACHE_CONTROL = "Cache-Control";
	private static final String EXCEL_CONTENT_DISPOSITION = "Content-Disposition";

    // Private constants

    private final int MAX_ROWS = 65536;

    // Public static methods

    public static Object getFilterValue(Object filterValue) {
        return NullChecker.isNullOrEmpty(filterValue) ? "ALL" : filterValue;
    }

    public static String getConsentTypeFilter(String consentType) {
        if (NullChecker.isNullOrEmpty(consentType)) {
            return "ALL";
        } else{
            if (consentType.contains("NwHIN")) {
                consentType = consentType.replace("NwHIN", Constants.getOrganizationName());
            }
            return consentType;
        }
    }

    public static String getFacilitiesFilter(FacilityDAO dao, String stationNumbers) {
        if (NullChecker.isNullOrEmpty(stationNumbers)) {
            return "";
        } else if ("ALL".equals(stationNumbers)) {
            return stationNumbers;
        } else {
            String text = "";
            List<Facility> facilities = dao.findByStationNumbers(stationNumbers);
            if (facilities.size() < 12) {
                for (int i = 0; i < facilities.size(); i++) {
                    text += i == 0 ? "" : "\n";
                    text += facilities.get(i).getFacilityName();
                }
            } else {
                for (int i = 0; i < 10; i++) {
                    text += facilities.get(i).getFacilityName() + "\n";
                }
                text += "and " + (facilities.size() - 10) + " other facilities";
            }
            return text;
        }
    }

    public static String getPatientPreferredFacilityFilter(FacilityDAO dao, String facilityName) {
        Facility facility = dao.findByStationNumber(facilityName);
        return facility == null ? "ALL" : facility.getFacilityName();
    }

    public static String getReasonsForDelayFilter(DelayReasonDAO dao, String delayReasonIds) {
        if (NullChecker.isEmpty(delayReasonIds)) {
            return "";
        } else if ("ALL".equals(delayReasonIds)) {
            return delayReasonIds;
        } else {
            String text = "";
            List<String> delayReasonNamesList = new ArrayList<String>();
            for (final String delayReasonId : delayReasonIds.split(",")) {
                delayReasonNamesList.add(dao.findByDelayReasonId(Long.parseLong(delayReasonId)).getName());
            }
            text = delayReasonNamesList.toString().replaceAll("\\[|\\]", "");
            return text;
        }
    }

    public static void populateFilterMapForExport(HttpServletRequest request, final LinkedHashMap<String, Object> filters,
        final Map<String, List<Object>> filterMap, String patientTypes) {

        if ("1".equals(patientTypes)) {
            // Read patients only
            filters.put("Patient Types", "Real Patients");
        } else if ("2".equals(patientTypes)) {
            // Test patients only
            filters.put("Patient Types", "Test Patients");
        } else if ("3".equals(patientTypes)) {
            // Both real and test patients
            filters.put("Patient Types", "ALL");
        } else {
            // Don't output this filter
        }

        for (Map.Entry<String, Object> filter : filters.entrySet()) {
            final String filterName = filter.getKey();
            final Object value = filter.getValue();
            List<Object> headingValue = new ArrayList<Object>() {{ add(value); }};
            filterMap.put(filterName, headingValue);
        }
    }

    // Private fields

	/**
	 * The HTTP application type for Excel. Note: it will be different for
	 * Office 2003, 2011 etc.
	 */
	private String streamApplicationType;

	/**
	 * Specify the cache control that is needed when streaming excel export.
	 */
	private String streamCacheControl;

	/**
	 * The title font size.
	 */
	private short titleFontSize;

    Date date;
    SimpleDateFormat df = new SimpleDateFormat("yyyyMMdd_hhmmss");
    SimpleDateFormat dateCell = new SimpleDateFormat("MM/dd/yyyy");
    SimpleDateFormat timeStamp = new SimpleDateFormat("MM/dd/yyyy hh:mm a");

    String formattedDate;

    /**
     * Boolean for determining if max row count has been hit
     */
    private static Boolean maxRows = false;

    /**
	 * Auto size the sheet. We do not wants columns to be hidden in the excel
	 * sheet.
	 *
	 * @param s
	 *            the sheet
	 * @param totalColumns
	 *            the total number of columns
	 * @return the sheet after auto sizing.
	 */
	private Sheet autoSizeSheet(final Sheet s, final int totalColumns) {

		Assert.assertNotEmpty(s, "Sheet cannot be empty!");

		if (totalColumns < 0) {
			throw new RuntimeException("Total columns cannot be less than 0");
		}
		// Iterate through all columns
		for (int i = 0; i < totalColumns; i++) {
			// Auto size each column
			try {
				s.autoSizeColumn(i);
			} catch (Exception ex) {
				// TODO: Fix an error with Apache POI component with autosize
				// BAD TEMPORARY FIX!
				//Ignore Exception
				//ex.printStackTrace();
			}
		}
		// Return the sheet after auto sizing
		return s;
	}

	/**
	 * Create the Column headings in the Excel Spreadsheet starting at the
	 * specified row numbers.
	 *
	 * @param sheet
	 *            The excel document sheet
	 * @param rowNumber
	 *            The row number to create the heading
	 * @param headings
	 *            The headers like patient first name, last name etc. in a
	 *            collection of strings. The headers will be created in the same
	 *            order in the list.
	 * @return the excel sheet
	 */
	protected Sheet createColumnHeadings(final Sheet sheet,
			final int rowNumber, final Collection<String> headings) {
		Assert.assertNotEmpty(sheet, "Sheet cannot be empty!");
		if (rowNumber < 0) {
			throw new RuntimeException("Row number cannot be less than 0");
		}
		if (NullChecker.isNotEmpty(headings)) {
			// Create the heading row based on the row number.
			final Row heading = sheet.createRow(rowNumber);
			int i = 0;
			// Loop through all the headings
			for (final String headingText : headings) {
				// Create Font
				final Font boldFont = sheet.getWorkbook().createFont();
				// Make font bold
				boldFont.setBoldweight(Font.BOLDWEIGHT_BOLD);
				// Create Style for headers
				final CellStyle cs = sheet.getWorkbook().createCellStyle();
				// Set the Font to the style
				cs.setFont(boldFont);
				cs.setAlignment(CellStyle.ALIGN_CENTER);
				// Create heading cell
				final Cell cell = heading.createCell(i);
				// Set the bold font cell style to the cell
				cell.setCellStyle(cs);
				cell.setCellValue(headingText);
				i++;
			}
		}
		// Return the created sheet
		return sheet;
	}

	/**
	 * Create the data rows (the list of records). The data rows are expected in
	 * a list. The items in the List would contain a map<columnKey, value>. For
	 * example: List Item 1: patientSSN, 6660000001 List Item 2:
	 * patientFirstName: Asha
	 *
	 * @param sheet
	 *            The excel workbook sheet
	 * @param startRowNumber
	 *            the starting row number
	 * @param headings
	 *            the headings where the key for the data is pulled from
	 * @param results
	 *            the list of records
	 * @return the sheet with all the data rows created
	 */
	protected Sheet createDataRows(final Sheet sheet, int startRowNumber,
			final Map<String, String> headings,
			final List<Map<String, Object>> results) {

		Assert.assertNotEmpty(sheet, "Sheet cannot be empty!");

		if (startRowNumber < 0) {
			throw new RuntimeException("Row number cannot be less than 0");
		}

        CreationHelper createHelper = sheet.getWorkbook().getCreationHelper();

        //Any date column
        final CellStyle dateStyle = sheet.getWorkbook().createCellStyle();
        dateStyle.setDataFormat(createHelper.createDataFormat().getFormat("mm/dd/yyyy"));
        //Audit time column
        final CellStyle timeStyle = sheet.getWorkbook().createCellStyle();
        timeStyle.setDataFormat(createHelper.createDataFormat().getFormat("mm/dd/yyyy hh:mm AM/PM"));

        int emptyRowNumber = startRowNumber - 2;

		// Iterate through results and create Cell
		if (NullChecker.isNotEmpty(results)) {
			// Create Style
			final CellStyle cs = sheet.getWorkbook().createCellStyle();
			// Set the Alignment
			cs.setAlignment(CellStyle.ALIGN_LEFT);
			// Wrap the text
			//cs.setWrapText(true);
			for (int i = 0; i < results.size(); i++) {

                            //Implementing check to account for the row limitation of the exporter.
                            if (startRowNumber == MAX_ROWS) {
                                outputMaxRowNotice(sheet, emptyRowNumber);
                                return sheet;
                            }

				final Row r = sheet.createRow(startRowNumber);
				// Get the data row from the results
				final Map<String, Object> resultEntries = results.get(i);
				int j = 0;
				// Create the heading first
				for (final String headingKey : headings.keySet()) {
					String valueStr = "";
					// For every heading key, ex. patientSSN, get the value
                    Object value = null;
					if (resultEntries.containsKey(headingKey)) {
						value = resultEntries.get(headingKey);
						if (NullChecker.isNotEmpty(value)) {
							// Get the string for Excel Cell
							valueStr = value.toString();
						}
					}
					// Make a cell for each value
					final Cell c = r.createCell(j);
					// Create Style
					//final CellStyle cs = sheet.getWorkbook().createCellStyle();
					// Set the Alignment
					//cs.setAlignment(CellStyle.ALIGN_LEFT);
					// Wrap the text
					//cs.setWrapText(true);
                    if (headingKey.contains("Date")) {
                        c.setCellType(0);
                        c.setCellStyle(dateStyle);
                        try {
                            Date date = dateCell.parse((String)value);
                            c.setCellValue(date);
                        } catch (ParseException ex) {
                            c.setCellValue("");
                        }
                    }
                    else if (headingKey.contains("auditTime")) {
                        c.setCellType(0);
                        c.setCellStyle(timeStyle);
                        try {
                            Date date = timeStamp.parse((String)value);
                            c.setCellValue(date);
                        } catch (ParseException ex) {
                            c.setCellValue("");
                        }
                    }
                    else if (!headingKey.toLowerCase().contains("ssn") && valueStr.length() > 0 && StringUtils.isNumeric(valueStr)) {
                        c.setCellType(0);
                        c.setCellStyle(cs);
                        c.setCellValue(Long.parseLong(valueStr));
                    }
                    else if (headingKey.toLowerCase().contains("oid") && !NullChecker.isNullOrEmpty(valueStr)) {
                        c.setCellStyle(cs);
                        c.setCellValue(ReportHelper.trimOrgOid(valueStr));
                    }
                    else {
                        c.setCellStyle(cs);
                        c.setCellValue(valueStr);
                    }
					// Go to the next cell
					j++;
				}
				// Go to the next row
				startRowNumber++;
			}
		} else {
            // Create Style
			final CellStyle cs = sheet.getWorkbook().createCellStyle();
			// Set the Alignment
			cs.setAlignment(CellStyle.ALIGN_LEFT);
            //add a row
            final Row r = sheet.createRow(startRowNumber);
            //add the cell and the value for no records found
            final Cell c = r.createCell(0);
            c.setCellValue("No records were found");
        }
		// The sheet with the rows created
		return sheet;
	}

	/**
	 * Create optional rows if needed like selected options Ex. start date, end
	 * date, total etc.
	 *
	 * @param sheet
	 *            the workbook sheet
	 * @param startNumber
	 *            the number where to start creating the optional rows
	 * @param optionalRows
	 *            the key and the list of data columns Ex. <key Column> <value1
	 *            Column> <value2 Column>
	 * @return the workbook sheet
	 */
	protected Sheet createOptionalRows(final Sheet sheet, int startNumber,
			final Map<String, List<Object>> optionalRows) {
		Assert.assertNotEmpty(sheet, "Sheet cannot be empty!");

		if (startNumber < 0) {
			throw new RuntimeException(
					"Starting row number cannot be less than 0");
		}

        int emptyRowNumber = startNumber - 2;

		// Iterate through results and create Cell
		if (NullChecker.isNotEmpty(optionalRows)) {
			// Start with the row number that came into this method
			for (final Entry<String, List<Object>> entry : optionalRows
					.entrySet()) {

                //Implementing check to account for the row limitation of the exporter.
                if (startNumber == MAX_ROWS) {
                    outputMaxRowNotice(sheet, emptyRowNumber);
                    return sheet;
                }

				// Create the row with the starting row number
				final Row r = sheet.createRow(startNumber);
				// Main Column
				final CellStyle mainColumnStyle = sheet.getWorkbook()
						.createCellStyle();
				// Set the Alignment
				mainColumnStyle.setAlignment(CellStyle.ALIGN_LEFT);
				// Create Font
				final Font boldFont = sheet.getWorkbook().createFont();
				// Make font bold
				boldFont.setBoldweight(Font.BOLDWEIGHT_BOLD);
				mainColumnStyle.setFont(boldFont);
				//mainColumnStyle.setWrapText(true);
				// Create the key in the first column
				final Cell c = r.createCell(0);
				c.setCellValue(entry.getKey() + ":");
				c.setCellStyle(mainColumnStyle);
				// Create list of data columns with values
				int j = 1;
				for (final Object value : entry.getValue()) {
					final Cell col = r.createCell(j);
					final CellStyle cs = sheet.getWorkbook().createCellStyle();
					// Set the Alignment
					cs.setAlignment(CellStyle.ALIGN_LEFT);
					//cs.setWrapText(true);
					col.setCellStyle(cs);
					if (NullChecker.isNotEmpty(value)) {
                        if (entry.getKey().contains("Total")) {
                            col.setCellType(0);
                            col.setCellValue(Integer.parseInt(value.toString()));
                        }
                        else {
                            col.setCellValue(value.toString());
                        }
					} else {
						// Default to N/A if it's not a total row
						if (!entry.getKey().contains("Total")) {
                            col.setCellValue("n/a");
                        }
					}
					// Next data column
					j++;
				}
				// Next row
				startNumber++;
			}
		}
		// Return excel sheet
		return sheet;
	}

	/**
	 * A convenience method to create an excel sheet with a sheet name and
	 * number.
	 *
	 * @param sheetName
	 *            the sheet name
	 * @param sheetNumber
	 *            the sheet number (cannot be less than 0)
	 * @return the created sheet
	 */
	private Sheet createSheet(final String sheetName, final int sheetNumber) {

		Assert.assertNotEmpty(sheetName, "Sheet name cannot be empty!");

		if (sheetNumber < 0) {
			throw new RuntimeException("Sheet number cannot be less than 0");
		}

		// Create Workbook
		final Workbook wb = new HSSFWorkbook();
		// create a new sheet
		final Sheet s = wb.createSheet();
		// Set Sheet name
		wb.setSheetName(sheetNumber, sheetName);
		// Return created sheet
		return s;
	}

	/**
	 * Create the title of the report in the excel sheet based on the row and
	 * center is based on the number of columns.
	 *
	 * @param s
	 *            the sheet to create the title
	 * @param rowNumber
	 *            the row number to create the title
	 * @param colSize
	 *            the column sizes to center the title
	 * @param title
	 *            the title string
	 * @return the sheet with the title created
	 */
	protected Sheet createTitle(final Sheet s, final int rowNumber,
			final int colSize, final String title) {

		Assert.assertNotEmpty(s, "Sheet cannot be empty!");

		if (rowNumber < 0) {
			throw new RuntimeException("Sheet number cannot be less than 0");
		}

		final Row titleRow = s.createRow(rowNumber);
		final Font superBoldFont = s.getWorkbook().createFont();
		// Make font bold
		superBoldFont.setBoldweight(Font.BOLDWEIGHT_BOLD);
		superBoldFont.setFontHeightInPoints(this.titleFontSize);
		// Create Style
		final CellStyle cs = s.getWorkbook().createCellStyle();
		// Set the Font to the style
		cs.setFont(superBoldFont);
		cs.setAlignment(CellStyle.ALIGN_CENTER);
		//cs.setWrapText(true);
		// Create heading cell
		final Cell cell = titleRow.createCell(0);
		// Set the bold font cell style to the cell
		cell.setCellStyle(cs);
		cell.setCellValue(title);
		// Merge to the column size, so the title looks centered
		s.addMergedRegion(new CellRangeAddress(0, 0, 0, colSize));
		return s;
	}

	/**
	 * The main export to excel method. This method takes a sheet name and
	 * creates a default work book with the title, headers, data rows and
	 * optional rows.
	 *
	 * @param sheetName
	 *            the name of the sheet to create
	 * @param headingText
	 *            the heading text (title)
	 * @param reportHeadingMap
	 *            the headers for the data rows. The key element of the result
	 *            set and the column beading value.
	 * @param results
	 *            the data set result map
	 * @param headingParameters
	 *            the parameters that are below the title of the report like
	 *            start date, end date etc.
	 * @param optionalRows
	 *            Optional rows that are needed like the Grand Total, Total
	 *            count etc.
	 * @return the excel workbook object
	 */
	public Workbook exportToExcel(final String sheetName,
			final String headingText,
			final Map<String, String> reportHeadingMap,
			final List<Map<String, Object>> results,
			final Map<String, List<Object>> headingParameters,
			final Map<String, List<Object>> optionalRows) {
                int row = 1;

        Assert.assertNotEmpty(sheetName, "Sheet name cannot be empty!");

        // Create first Sheet - index 0
        final Sheet s = this.createSheet(sheetName, 0);

        // Create Title - Row 0
        Map<String, List<Object>> title = new HashMap<String, List<Object>>();
        List<Object> titleText = new ArrayList<Object>();
        titleText.add(headingText);
        title.put("Report", titleText);
        this.createOptionalRows(s, 0, title);

        // Create Date row - Row 1
        Map<String, List<Object>> generated = new HashMap<String, List<Object>>();
        List<Object> dateText = new ArrayList<Object>();
        dateText.add(new SimpleDateFormat("MM/dd/yyyy").format(new Date()));
        generated.put("Date Generated", dateText);
        this.createOptionalRows(s, 1, generated);

        // Create filters - Row 2
        this.createOptionalRows(s, 2, headingParameters);

        // Create column headings
        row = 3 + headingParameters.size();
        if(results != null && results.size() > 0){
            this.createColumnHeadings(s, row, reportHeadingMap.values());
        }
        
        // Create data rows
        this.createDataRows(s, row + 1, reportHeadingMap, results);

        // Optional Rows after the results with a few rows after if we aren't at and won't exceed the max row count.
        if(results != null){
            this.createOptionalRows(s, row + results.size() + 2, optionalRows);
        } else {
            //add 1 for No Results Found row
            this.createOptionalRows(s, row + 3, optionalRows);
        }
        
        // Do not autosize due to perfomance
        // Autosize all Columns
        //if (reportHeadingMap.values().size() > 1) {
        //    this.autoSizeSheet(s, reportHeadingMap.values().size());
        //} else {
        //    this.autoSizeSheet(s, 2);
        //}

        int numOfCols = reportHeadingMap.values().size() > 1 ? reportHeadingMap.values().size() : 2;
		for (int i = 0; i < numOfCols; i++) {
			try {
                s.setColumnWidth(i, 19 * 256);
			} catch (Exception ex) {
				// Ignore Exception
			}
		}
        
        // return the workbook
        return s.getWorkbook();
    }

	@Required
	public void setStreamApplicationType(final String streamApplicationType) {
		this.streamApplicationType = streamApplicationType;
	}

	@Required
	public void setStreamCacheControl(final String streamCacheControl) {
		this.streamCacheControl = streamCacheControl;
	}

	@Required
	public void setTitleFontSize(final short titleFontSize) {
		this.titleFontSize = titleFontSize;
	}

	/**
	 * Write the excel to a HTTP stream.
	 * @param reportType
     *          the name of the report being exported
	 * @param wb
	 *            the workbook
	 * @param response
	 *            the HTTPServletResponse object to write to its output stream
	 * @throws ServletException
	 *             if an error (IO) occurs
	 */
	public void writeExcelToStream(String reportType, final Workbook wb,
			final HttpServletResponse response) throws ServletException {

		Assert.assertNotEmpty(wb, "Workbook cannot be empty!");
		Assert.assertNotEmpty(response, "HTTP Response cannot be empty!");
        Date d = new Date();
        String formattedDate = df.format(d);
		// Output Excel to Browser
		// Get the application type from Spring - typicaly
		// application/vnd.ms-excel
		response.setContentType(this.streamApplicationType);
		// Set the content disposition filename to send to the browser
		response.setHeader(ExcelExporter.EXCEL_CONTENT_DISPOSITION,
				"attachment; filename=\"" + reportType + "_" + formattedDate + ".xls" + "\"");
		// Set the cache control in the header (typically no-cache
		response.setHeader(ExcelExporter.EXCEL_CACHE_CONTROL,
				this.streamCacheControl);
		try {
			// Get the output stream
			final ServletOutputStream os = response.getOutputStream();
			Assert.assertNotEmpty(os, "Output Stream cannot be null!");
			// Write the response
			wb.write(os);
			// Close the output stream
			try {
				// Close stream
				os.close();
			} catch (final IOException ex) {
				// Ignore.. Pass it on to the application server to handle the
				// thread
			}
		} catch (final IOException ex) {
			throw new ServletException(ex);
		}
	}

    /**
	 * Get the Excel document as a byte array.
     *
	 * @param wb
	 *            the workbook
     * @return
	 * @throws ServletException
	 *             if an error (IO) occurs
	 */
	public byte[] getExcelAsBytes(final Workbook wb) throws ServletException {

		Assert.assertNotEmpty(wb, "Workbook cannot be empty!");
        date = new Date();
        formattedDate = df.format(date);
        ByteArrayOutputStream byteStream = new ByteArrayOutputStream();
        byte[] docBytes = null;

		try {
			// Write the response
			wb.write(byteStream);
            // Get document as byte array
            docBytes = byteStream.toByteArray();
			// Close the output stream
			try {
				// Close stream
				byteStream.close();
			} catch (final IOException ex) {
				// Ignore.. Pass it on to the application server to handle the
				// thread
			}
		} catch (final IOException ex) {
			throw new ServletException(ex);
		}

        return docBytes;
	}

    private void outputMaxRowNotice(final Sheet sheet, int emptyRowNumber) {
        final Row r = sheet.createRow(emptyRowNumber);
        final Cell c = r.createCell(0);
        final CellStyle cs = sheet.getWorkbook().createCellStyle();
        final Font font = sheet.getWorkbook().createFont();

        font.setColor(IndexedColors.RED.getIndex());

        cs.setFont(font);
        //cs.setWrapText(false);

        c.setCellStyle(cs);
        c.setCellValue("Note: Report has been truncated to fit " + MAX_ROWS + " rows.");

        maxRows = true;
    }

}
