package gov.va.nvap.web.util.xls;

import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.web.helper.report.ReportHelper;
import java.io.IOException;
import java.io.OutputStream;
import java.text.SimpleDateFormat;
import java.util.LinkedHashMap;
import java.util.Date;
import java.util.List;
import java.util.Map;
import javax.servlet.http.HttpServletResponse;

/**
 *
 * @author Zachary Tubb
 */
public class CsvExporter {

    Date date;
    SimpleDateFormat df = new SimpleDateFormat("yyyyMMdd_hhmmss");
    String formattedDate;
    
    public void exportToCSV(final HttpServletResponse response, String reportType, List<Map<String, Object>> results,
        LinkedHashMap<String, String> reportMap) throws IOException {

        OutputStream outputStream = null;
        
        date = new Date();
        formattedDate = df.format(date);
        
        response.setContentType("text/csv");
        response.setHeader("Content-Disposition", "attachment; filename=\""
            + reportType + "_" + formattedDate + ".csv" + "\"");

        try {
            StringBuilder sbOutput;
            
            outputStream = response.getOutputStream();

            if (NullChecker.isNullOrEmpty(results)) {
                sbOutput = new StringBuilder();
                sbOutput.append("No records were found.");
            } else {
                SimpleDateFormat dt = new SimpleDateFormat("MM/dd/yyyy");
                // Allocate string builder based on the number of rows, but no larger than 20MB.
                if (results.size() > 100000) {
                    sbOutput =  new StringBuilder(20000000);
                } else {
                    sbOutput =  new StringBuilder(results.size() * 200);
                }

                // Append column headers
                for (String heading : reportMap.keySet()) {
                    sbOutput.append(heading);
                    sbOutput.append(",");
                }
                sbOutput.append("\n");

                // Append report data
                for (Map<String, Object> row : results) {
                    // Create the row of data
                    String separator = "";
                    for (String dataField : reportMap.values()) {
                        sbOutput.append(separator);
                        separator = ",";
                        // Include escape character.
                        sbOutput.append("\"");

                        if (NullChecker.isNullOrEmpty(row.get(dataField))) {
                            sbOutput.append("");
                        } else {
                            if (dataField.contains("Date")) {
                                // Format dates. patientDateFormatted is an exception - comes pre-formatted
                                if (dataField.equals("patientDateFormatted")) {
                                    sbOutput.append(row.get(dataField));
                                } else {
                                    Date oldDate = (Date) row.get(dataField);
                                    sbOutput.append(dt.format(oldDate));
                                }
                            } else if (dataField.toLowerCase().contains("oid") && !NullChecker.isNullOrEmpty(row.get(dataField))) {
                                sbOutput.append(ReportHelper.trimOrgOid((String)row.get(dataField)));
                            } else {
                                // Else print the data as-is.
                                sbOutput.append(row.get(dataField));
                            }
                        }
                        // Include closing escape character.
                        sbOutput.append("\"");
                    }
                    sbOutput.append("\n");
                }
            }

            outputStream.write(String.valueOf(sbOutput).getBytes());
            
            // Blank out this string builder so nothing is in memory (i.e. ssn, other PII). Even though sbOutput is scoped to this method
            // and will be destroyed and set to be garbage collected.
            sbOutput.setLength(0);
        } catch (Exception e) {
            throw new IOException("Error exporting to CSV");
        } finally {
            if (outputStream != null) {
                outputStream.flush();
                outputStream.close();                
            }
        }
    }
}
